﻿# backend/app/measures/generic_measures.py
"""
Medidas especializadas para análisis de genéricos - Estilo Power BI farmacéutico.

Medidas incluidas:
- GenericSavingsOpportunity: Ahorro potencial por cambio a genéricos
- PartnerLabSales: Ventas de laboratorios partners
- GenericVsBrandedRatio: Ratio genéricos vs marca
- HomogeneousGroupSales: Ventas por conjunto homogéneo
- TherapeuticCategorySales: Ventas por categoría terapéutica
"""

from datetime import timedelta
from typing import Any, Dict

from sqlalchemy import func

from app.external_data.nomenclator_integration import pvp_to_pvl
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from ..utils.datetime_utils import utc_now
from .base import BaseMeasure, QueryContext


class GenericSavingsOpportunity(BaseMeasure):
    """
    Cálculo de ahorro potencial por cambio de marca a genéricos.
    Medida fundamental para estudios de genéricos.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ahorro potencial por cambio de marca a genéricos"
        self.unit = "€"
        self.category = "Genéricos"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular oportunidades de ahorro usando la medida UnidadesNoPartner.

        LÓGICA SIMPLIFICADA:
        1. Usa UnidadesNoPartner para obtener unidades de oportunidad
        2. Calcula PVL base usando nomenclator (PVP mínimo -> PVL)
        3. Frontend aplica % descuento del slider sobre PVL base
        """
        from app.external_data.nomenclator_integration import pvp_to_pvl

        # 1. Obtener unidades NO-partner usando la medida especializada
        unidades_measure = UnidadesNoPartner()
        unidades_result = unidades_measure.calculate(context)

        if unidades_result.get("error") or unidades_result.get("non_partner_units", 0) == 0:
            return {
                "total_opportunities": 0,
                "total_potential_savings": 0.0,
                "opportunities_detail": [],
                "avg_savings_per_opportunity": 0.0,
                "message": unidades_result.get("error") or unidades_result.get("message", "No hay oportunidades"),
            }

        # 2. Obtener PVP representativo de conjuntos homogéneos partners
        # (usar PVP mínimo de productos en ALTA como en el servicio)
        partner_labs = unidades_result["partner_laboratories"]

        partner_homogeneous_codes = (
            context.db.query(ProductCatalog.nomen_codigo_homogeneo)
            .filter(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.nomen_laboratorio.in_(partner_labs),
                ProductCatalog.nomen_estado == "ALTA",
            )
            .distinct()
            .all()
        )

        partner_codes_list = [code[0] for code in partner_homogeneous_codes if code[0]]

        # 3. Calcular PVL promedio de estos conjuntos homogéneos
        if partner_codes_list:
            avg_pvp = (
                context.db.query(func.avg(ProductCatalog.nomen_pvp))
                .filter(
                    ProductCatalog.nomen_codigo_homogeneo.in_(partner_codes_list),
                    ProductCatalog.nomen_estado == "ALTA",
                    ProductCatalog.nomen_pvp.isnot(None),
                )
                .scalar()
                or 0
            )
        else:
            avg_pvp = 0

        # 4. Convertir a PVL
        avg_pvl = pvp_to_pvl(float(avg_pvp)) if avg_pvp > 0 else 0

        # 5. Calcular base de ahorro potencial
        non_partner_units = unidades_result["non_partner_units"]
        total_pvl_base = non_partner_units * avg_pvl

        # 6. Preparar resultado compatible con frontend
        return {
            "total_opportunities": non_partner_units,
            "total_potential_savings": round(total_pvl_base, 2),  # Base para aplicar % descuento
            "opportunities_detail": [
                {
                    "partner_groups_count": unidades_result["partner_groups_count"],
                    "partner_penetration": unidades_result["partner_penetration"],
                    "avg_pvl_per_unit": round(avg_pvl, 2),
                    "total_units_in_partner_groups": unidades_result["total_units_in_partner_groups"],
                    "partner_laboratories": partner_labs,
                }
            ],
            "avg_savings_per_opportunity": round(avg_pvl, 2),
            "calculation_method": "unidades_no_partner_measure",
            "calculation_note": "Ahorro final = total_potential_savings * (descuento_slider / 100)",
        }


class PartnerLabSales(BaseMeasure):
    """
    Ventas específicas de laboratorios partners.
    Fundamental para análisis de partners vs otros laboratorios.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ventas de laboratorios partners"
        self.unit = "€"
        self.category = "Partners"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular ventas de laboratorios partners vs otros laboratorios.
        Requiere que el contexto incluya partner_laboratories en filtros adicionales.
        """
        # Obtener lista de partners desde el contexto o farmacia
        partner_labs = getattr(context.filters, "partner_laboratories", [])

        if not partner_labs:
            # Intentar obtener desde la farmacia
            from app.models.pharmacy import Pharmacy

            pharmacy = context.db.query(Pharmacy).filter(Pharmacy.id == context.filters.pharmacy_id).first()

            if pharmacy and pharmacy.partner_laboratories:
                partner_labs = [p["laboratory_name"] for p in pharmacy.partner_laboratories]

        if not partner_labs:
            return {
                "partner_sales": 0.0,
                "other_sales": 0.0,
                "total_sales": 0.0,
                "partner_percentage": 0.0,
                "error": "No partner laboratories found",
            }

        # Calcular ventas partners usando nomenclator oficial
        from app.models.nomenclator_local import NomenclatorLocal

        # Ventas partners
        partner_sales = (
            context.base_query.join(
                NomenclatorLocal,
                SalesData.codigo_nacional == NomenclatorLocal.national_code,
            )
            .filter(NomenclatorLocal.laboratory.in_(partner_labs))
            .with_entities(func.sum(SalesData.total_amount))
            .scalar()
            or 0.0
        )

        # Ventas totales
        total_sales = context.base_query.with_entities(func.sum(SalesData.total_amount)).scalar() or 0.0

        other_sales = total_sales - partner_sales
        partner_percentage = (partner_sales / total_sales * 100) if total_sales > 0 else 0

        return {
            "partner_sales": round(float(partner_sales), 2),
            "other_sales": round(float(other_sales), 2),
            "total_sales": round(float(total_sales), 2),
            "partner_percentage": round(partner_percentage, 2),
            "partner_laboratories": partner_labs,
        }


class UnidadesNoPartner(BaseMeasure):
    """
    Unidades vendidas que NO son de laboratorios partners, pero en conjuntos homogéneos donde SÍ hay partners.
    Fundamental para calcular oportunidades de ahorro con descuentos partners.
    """

    def __init__(self):
        super().__init__()
        self.description = "Unidades NO-partner en conjuntos con partners"
        self.unit = "unidades"
        self.category = "Partners"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular unidades vendidas que no son de laboratorios partners,
        pero pertenecen a conjuntos homogéneos donde sí hay partners.

        Lógica:
        1. Obtener laboratorios partners
        2. Identificar conjuntos homogéneos que ofertan esos partners
        3. Filtrar ventas solo de esos conjuntos homogéneos
        4. Contar unidades que NO son de partners
        """
        # Obtener lista de partners desde el contexto
        partner_labs = getattr(context.filters, "partner_laboratories", [])

        if not partner_labs:
            # Intentar obtener desde la farmacia usando PharmacyPartner
            from app.models.pharmacy_partners import PharmacyPartner

            partners = (
                context.db.query(PharmacyPartner)
                .filter(
                    PharmacyPartner.pharmacy_id == context.filters.pharmacy_id,
                    PharmacyPartner.is_selected.is_(True),
                )
                .all()
            )
            partner_labs = [p.laboratory_name for p in partners]

        if not partner_labs:
            return {
                "non_partner_units": 0,
                "total_units_in_partner_groups": 0,
                "partner_groups_count": 0,
                "partner_penetration": 0.0,
                "error": "No hay laboratorios partners seleccionados",
            }

        # 1. Identificar conjuntos homogéneos que ofertan laboratorios partners
        partner_homogeneous_codes = (
            context.db.query(ProductCatalog.nomen_codigo_homogeneo)
            .filter(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.nomen_laboratorio.in_(partner_labs),
                ProductCatalog.nomen_estado == "ALTA",  # Solo productos dados de alta
            )
            .distinct()
            .all()
        )

        partner_codes_list = [code[0] for code in partner_homogeneous_codes if code[0]]

        if not partner_codes_list:
            return {
                "non_partner_units": 0,
                "total_units_in_partner_groups": 0,
                "partner_groups_count": 0,
                "partner_penetration": 0.0,
                "message": "No hay conjuntos homogéneos con laboratorios partners",
            }

        # 2. Filtrar ventas solo de conjuntos homogéneos donde hay partners
        partner_groups_query = context.enriched_query.filter(
            ProductCatalog.nomen_codigo_homogeneo.in_(partner_codes_list)
        )

        # 3. Total unidades en estos conjuntos homogéneos
        total_units_in_partner_groups = partner_groups_query.with_entities(func.sum(SalesData.quantity)).scalar() or 0

        # 4. Unidades vendidas DE laboratorios partners
        partner_units = (
            partner_groups_query.filter(ProductCatalog.nomen_laboratorio.in_(partner_labs))
            .with_entities(func.sum(SalesData.quantity))
            .scalar()
            or 0
        )

        # 5. Unidades NO-partner = Total - Partners
        non_partner_units = total_units_in_partner_groups - partner_units

        # 6. Penetración de partners
        partner_penetration = (partner_units / max(total_units_in_partner_groups, 1)) * 100

        return {
            "non_partner_units": non_partner_units,
            "partner_units": partner_units,
            "total_units_in_partner_groups": total_units_in_partner_groups,
            "partner_groups_count": len(partner_codes_list),
            "partner_penetration": round(partner_penetration, 2),
            "partner_laboratories": partner_labs,
            "calculation_note": "Solo cuenta unidades en conjuntos homogéneos donde hay laboratorios partners",
        }


class GenericVsBrandedRatio(BaseMeasure):
    """
    Ratio de ventas genéricos vs marca.
    Medida clave para análisis de comportamiento farmacéutico.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ratio de ventas genéricos vs productos de marca"
        self.unit = "%"
        self.category = "Análisis Genéricos"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular distribución genéricos vs marca"""

        # Ventas genéricos
        generic_sales = (
            context.enriched_query.filter(ProductCatalog.nomen_tipo_farmaco == "GENERICO")
            .with_entities(
                func.sum(SalesData.total_amount),
                func.sum(SalesData.quantity),
                func.count(SalesData.id),
            )
            .first()
        )

        # Ventas marca
        branded_sales = (
            context.enriched_query.filter(ProductCatalog.nomen_tipo_farmaco == "MARCA")
            .with_entities(
                func.sum(SalesData.total_amount),
                func.sum(SalesData.quantity),
                func.count(SalesData.id),
            )
            .first()
        )

        generic_amount = float(generic_sales[0] or 0)
        generic_units = int(generic_sales[1] or 0)
        generic_transactions = int(generic_sales[2] or 0)

        branded_amount = float(branded_sales[0] or 0)
        branded_units = int(branded_sales[1] or 0)
        branded_transactions = int(branded_sales[2] or 0)

        total_amount = generic_amount + branded_amount
        total_units = generic_units + branded_units
        total_transactions = generic_transactions + branded_transactions

        return {
            "generic_sales": {
                "amount": round(generic_amount, 2),
                "units": generic_units,
                "transactions": generic_transactions,
                "percentage_amount": round((generic_amount / total_amount * 100) if total_amount > 0 else 0, 2),
                "percentage_units": round((generic_units / total_units * 100) if total_units > 0 else 0, 2),
            },
            "branded_sales": {
                "amount": round(branded_amount, 2),
                "units": branded_units,
                "transactions": branded_transactions,
                "percentage_amount": round((branded_amount / total_amount * 100) if total_amount > 0 else 0, 2),
                "percentage_units": round((branded_units / total_units * 100) if total_units > 0 else 0, 2),
            },
            "totals": {
                "amount": round(total_amount, 2),
                "units": total_units,
                "transactions": total_transactions,
            },
        }


class HomogeneousGroupSales(BaseMeasure):
    """
    Ventas agrupadas por conjunto homogéneo.
    Esencial para análisis de oportunidades de descuentos.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ventas agrupadas por conjunto homogéneo"
        self.unit = "€"
        self.category = "Conjuntos Homogéneos"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular ventas por conjunto homogéneo"""

        # Obtener parámetros adicionales del contexto si están disponibles
        discount_percentage = getattr(context.filters, "discount_percentage", 25.0)
        top_groups = getattr(context.filters, "top_groups", 20)

        # Ventas por conjunto homogéneo
        group_sales = (
            context.enriched_query.filter(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.nomen_pvp.isnot(None),
            )
            .with_entities(
                ProductCatalog.nomen_codigo_homogeneo,
                func.sum(SalesData.total_amount).label("total_sales"),
                func.sum(SalesData.quantity).label("total_quantity"),
                func.count(func.distinct(ProductCatalog.nomen_laboratorio)).label("laboratories_count"),
                func.avg(ProductCatalog.nomen_pvp).label("avg_pvp"),
            )
            .group_by(ProductCatalog.nomen_codigo_homogeneo)
            .order_by(func.sum(SalesData.total_amount).desc())
            .limit(top_groups)
            .all()
        )

        groups_analysis = []
        total_sales_analyzed = 0
        total_potential_savings = 0

        for group in group_sales:
            total_sales = float(group.total_sales or 0)
            total_quantity = int(group.total_quantity or 0)
            avg_pvp = float(group.avg_pvp or 0)

            # Calcular ahorro potencial con descuento
            if avg_pvp > 0:
                common_pvl = pvp_to_pvl(avg_pvp)
                potential_savings = (common_pvl * discount_percentage / 100) * total_quantity
            else:
                potential_savings = 0

            total_sales_analyzed += total_sales
            total_potential_savings += potential_savings

            groups_analysis.append(
                {
                    "homogeneous_group": group.nomen_codigo_homogeneo,
                    "total_sales": round(total_sales, 2),
                    "total_quantity": total_quantity,
                    "laboratories_count": group.laboratories_count,
                    "avg_pvp": round(avg_pvp, 2),
                    "potential_savings": round(potential_savings, 2),
                    "savings_percentage": round(
                        ((potential_savings / total_sales * 100) if total_sales > 0 else 0),
                        2,
                    ),
                }
            )

        return {
            "groups_analysis": groups_analysis,
            "summary": {
                "total_groups": len(groups_analysis),
                "total_sales_analyzed": round(total_sales_analyzed, 2),
                "total_potential_savings": round(total_potential_savings, 2),
                "avg_savings_percentage": round(
                    ((total_potential_savings / total_sales_analyzed * 100) if total_sales_analyzed > 0 else 0),
                    2,
                ),
                "discount_applied": discount_percentage,
            },
        }


class TherapeuticCategorySales(BaseMeasure):
    """
    Ventas por categoría terapéutica ATC.
    Análisis farmacológico de la cartera de ventas.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ventas por categoría terapéutica ATC"
        self.unit = "€"
        self.category = "Análisis Terapéutico"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular distribución de ventas por categoría terapéutica"""

        # Análisis por grupo ATC (primeros 3 caracteres)
        atc_analysis = (
            context.enriched_query.filter(
                ProductCatalog.cima_atc_code.isnot(None),
                ProductCatalog.cima_atc_code != "",
            )
            .with_entities(
                func.substring(ProductCatalog.cima_atc_code, 1, 3).label("atc_group"),
                SalesEnrichment.therapeutic_category,
                func.count(SalesData.id).label("sales_count"),
                func.sum(SalesData.total_amount).label("total_amount"),
                func.sum(SalesData.quantity).label("total_quantity"),
                func.avg(SalesData.total_amount).label("avg_sale_amount"),
            )
            .group_by(
                func.substring(ProductCatalog.cima_atc_code, 1, 3),
                SalesEnrichment.therapeutic_category,
            )
            .order_by(func.sum(SalesData.total_amount).desc())
            .limit(15)
            .all()
        )

        categories_analysis = []
        total_sales = sum(float(cat.total_amount or 0) for cat in atc_analysis)

        for category in atc_analysis:
            cat_sales = float(category.total_amount or 0)

            categories_analysis.append(
                {
                    "atc_group": category.atc_group,
                    "therapeutic_category": category.therapeutic_category,
                    "sales_count": category.sales_count,
                    "total_amount": round(cat_sales, 2),
                    "total_quantity": category.total_quantity,
                    "avg_sale_amount": round(float(category.avg_sale_amount or 0), 2),
                    "percentage_of_total": round((cat_sales / total_sales * 100) if total_sales > 0 else 0, 2),
                }
            )

        return {
            "categories_analysis": categories_analysis,
            "summary": {
                "total_categories": len(categories_analysis),
                "total_sales_analyzed": round(total_sales, 2),
                "top_category": categories_analysis[0] if categories_analysis else None,
            },
        }


class GenericContextTreemap(BaseMeasure):
    """
    Medida compuesta para treemap de contexto de genéricos (Issue #472).

    Genera estructura jerárquica usando TotalVentas con diferentes filtros:
    - Total Ventas
      - Prescripción (requiere receta)
        - Sustituible (grupos homogéneos con genéricos)
          - Analizable (cubierto por partners)
            - Ya Partners (ventas actuales partners)
            - Oportunidad (ventas no-partners en grupos analizables)
          - No Analizable (sin partners)
        - No Sustituible (sin genéricos disponibles)
      - Venta Libre (OTC)

    Patrón Power BI: Una medida base (TotalVentas) + filtros composables.
    """

    def __init__(self):
        super().__init__()
        self.description = "Treemap jerárquico de contexto de genéricos"
        self.unit = "€"
        self.category = "Visualización Genéricos"
        self.dependencies = ["TotalVentas"]

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular estructura jerárquica para treemap de Plotly.

        Usa TotalVentas internamente con diferentes combinaciones de filtros
        para calcular cada segmento de la jerarquía.
        """
        from .core_measures import TotalVentas
        from .base import FilterContext, QueryContext as QC
        from app.models.pharmacy_partners import PharmacyPartner

        # Obtener partners seleccionados de la farmacia (códigos y nombres)
        partner_info = self._get_pharmacy_partner_info(context)
        partner_codes = partner_info["codes"]
        partner_names = partner_info["names"]

        # =====================================================================
        # Calcular cada segmento usando TotalVentas + filtros
        # =====================================================================

        # 1. TOTAL - Sin filtros adicionales
        total_sales = TotalVentas().calculate(context)

        # 2. PRESCRIPCIÓN - is_prescription=True
        rx_filters = self._clone_filters(context.filters, is_prescription=True)
        rx_context = QC(context.db, rx_filters)
        rx_sales = TotalVentas().calculate(rx_context)

        # 3. VENTA LIBRE (OTC) - is_prescription=False
        otc_filters = self._clone_filters(context.filters, is_prescription=False)
        otc_context = QC(context.db, otc_filters)
        otc_sales = TotalVentas().calculate(otc_context)

        # 4. SUSTITUIBLE (dentro de Rx) - is_prescription=True, is_substitutable=True
        rx_substitutable_filters = self._clone_filters(
            context.filters, is_prescription=True, is_substitutable=True
        )
        rx_substitutable_context = QC(context.db, rx_substitutable_filters)
        rx_substitutable_sales = TotalVentas().calculate(rx_substitutable_context)

        # 5. NO SUSTITUIBLE (dentro de Rx) - is_prescription=True, is_substitutable=False
        rx_non_substitutable_filters = self._clone_filters(
            context.filters, is_prescription=True, is_substitutable=False
        )
        rx_non_substitutable_context = QC(context.db, rx_non_substitutable_filters)
        rx_non_substitutable_sales = TotalVentas().calculate(rx_non_substitutable_context)

        # 6. ANALIZABLE (dentro de Sustituible) - con cobertura de partners
        rx_analyzable_sales = 0.0
        rx_partner_sales = 0.0
        rx_opportunity_sales = 0.0
        rx_non_analyzable_sales = rx_substitutable_sales

        if partner_codes:
            # Analizable = sustituible + cubierto por partners
            rx_analyzable_filters = self._clone_filters(
                context.filters,
                is_prescription=True,
                is_substitutable=True,
                partner_laboratory_codes=partner_codes,
            )
            rx_analyzable_context = QC(context.db, rx_analyzable_filters)
            rx_analyzable_sales = TotalVentas().calculate(rx_analyzable_context)

            # Ya Partners = ventas DE laboratorios partners en grupos analizables
            rx_partner_filters = self._clone_filters(
                context.filters,
                is_prescription=True,
                is_substitutable=True,
                partner_laboratory_codes=partner_codes,
                laboratories=partner_names,  # Filtrar por NOMBRES (nomen_laboratorio)
            )
            rx_partner_context = QC(context.db, rx_partner_filters)
            rx_partner_sales = TotalVentas().calculate(rx_partner_context)

            # Oportunidad = Analizable - Ya Partners
            rx_opportunity_sales = rx_analyzable_sales - rx_partner_sales

            # No Analizable = Sustituible - Analizable
            rx_non_analyzable_sales = rx_substitutable_sales - rx_analyzable_sales

        # =====================================================================
        # Construir estructura treemap para Plotly
        # =====================================================================
        treemap_data = self._build_treemap_structure(
            total_sales=total_sales,
            rx_sales=rx_sales,
            otc_sales=otc_sales,
            rx_substitutable_sales=rx_substitutable_sales,
            rx_non_substitutable_sales=rx_non_substitutable_sales,
            rx_analyzable_sales=rx_analyzable_sales,
            rx_partner_sales=rx_partner_sales,
            rx_opportunity_sales=rx_opportunity_sales,
            rx_non_analyzable_sales=rx_non_analyzable_sales,
        )

        return {
            "treemap_data": treemap_data,
            "hierarchy": {
                "label": "Total Ventas",
                "value": total_sales,
                "children": [
                    {
                        "label": "Prescripción",
                        "value": rx_sales,
                        "percentage": (rx_sales / total_sales * 100) if total_sales > 0 else 0,
                    },
                    {
                        "label": "Venta Libre",
                        "value": otc_sales,
                        "percentage": (otc_sales / total_sales * 100) if total_sales > 0 else 0,
                    },
                ],
            },
            "summary": {
                "total_sales": round(total_sales, 2),
                "prescription_sales": round(rx_sales, 2),
                "otc_sales": round(otc_sales, 2),
                "substitutable_sales": round(rx_substitutable_sales, 2),
                "analyzable_sales": round(rx_analyzable_sales, 2),
                "partner_sales": round(rx_partner_sales, 2),
                "opportunity_sales": round(rx_opportunity_sales, 2),
                "has_partners": len(partner_codes) > 0,
                "partner_count": len(partner_codes),
            },
        }

    def _get_pharmacy_partner_info(self, context: QueryContext) -> dict:
        """
        Obtener info de laboratorios partners de la farmacia.

        Retorna dict con:
        - codes: códigos de laboratorio (para filtro partner_laboratory_codes)
        - names: nombres de laboratorio (para filtro laboratories)
        """
        from sqlalchemy import text
        from app.models.pharmacy_partners import PharmacyPartner

        # 1. Obtener nombres de laboratorios partners seleccionados
        partners = (
            context.db.query(PharmacyPartner)
            .filter(
                PharmacyPartner.pharmacy_id == context.filters.pharmacy_id,
                PharmacyPartner.is_selected.is_(True),
            )
            .all()
        )

        if not partners:
            return {"codes": [], "names": []}

        lab_names = [p.laboratory_name for p in partners]

        if not lab_names:
            return {"codes": [], "names": []}

        # 2. Buscar códigos de laboratorio en product_catalog
        codes_query = text(
            """
            SELECT DISTINCT nomen_codigo_laboratorio
            FROM product_catalog
            WHERE nomen_laboratorio IN :lab_names
              AND nomen_codigo_laboratorio IS NOT NULL
              AND nomen_codigo_laboratorio != ''
            """
        )
        result = context.db.execute(codes_query, {"lab_names": tuple(lab_names)})
        codes = [row[0] for row in result.fetchall()]

        return {"codes": codes, "names": lab_names}

    def _clone_filters(self, base_filters: "FilterContext", **overrides) -> "FilterContext":
        """Clonar filtros con overrides específicos."""
        from .base import FilterContext

        filter_dict = base_filters.to_dict()

        # Convertir tuplas de vuelta a listas
        for key, value in filter_dict.items():
            if isinstance(value, tuple):
                filter_dict[key] = list(value)

        # Aplicar overrides
        filter_dict.update(overrides)

        # Crear nuevo FilterContext
        return FilterContext(
            pharmacy_id=filter_dict["pharmacy_id"],
            start_date=base_filters.start_date,
            end_date=base_filters.end_date,
            product_codes=filter_dict.get("product_codes"),
            therapeutic_categories=filter_dict.get("therapeutic_categories"),
            laboratories=filter_dict.get("laboratories"),
            requires_prescription=filter_dict.get("requires_prescription"),
            is_generic=filter_dict.get("is_generic"),
            min_amount=filter_dict.get("min_amount"),
            max_amount=filter_dict.get("max_amount"),
            partner_laboratories=filter_dict.get("partner_laboratories"),
            discount_percentage=filter_dict.get("discount_percentage"),
            top_groups=filter_dict.get("top_groups"),
            months_back=filter_dict.get("months_back"),
            homogeneous_group=filter_dict.get("homogeneous_group"),
            product_type=filter_dict.get("product_type"),
            min_sales_count=filter_dict.get("min_sales_count"),
            snapshot_date=base_filters.snapshot_date,
            min_stock=filter_dict.get("min_stock"),
            max_stock=filter_dict.get("max_stock"),
            days_without_sale=filter_dict.get("days_without_sale"),
            abc_class=filter_dict.get("abc_class"),
            employee_names=filter_dict.get("employee_names"),
            # Nuevos filtros Issue #472
            is_prescription=overrides.get("is_prescription", filter_dict.get("is_prescription")),
            is_substitutable=overrides.get("is_substitutable", filter_dict.get("is_substitutable")),
            partner_laboratory_codes=overrides.get(
                "partner_laboratory_codes", filter_dict.get("partner_laboratory_codes")
            ),
        )

    def _build_treemap_structure(
        self,
        total_sales: float,
        rx_sales: float,
        otc_sales: float,
        rx_substitutable_sales: float,
        rx_non_substitutable_sales: float,
        rx_analyzable_sales: float,
        rx_partner_sales: float,
        rx_opportunity_sales: float,
        rx_non_analyzable_sales: float,
    ) -> Dict[str, list]:
        """
        Construir estructura plana para Plotly treemap.

        Colores semánticos:
        - Verde: Oportunidad / Positivo
        - Azul: Analizable / Partners
        - Naranja: Atención requerida
        - Gris: No aplica / Base
        """
        labels = []
        parents = []
        values = []
        text = []
        colors = []

        # Colores semánticos (Issue #426)
        COLOR_TOTAL = "#6c757d"  # Gris - Base
        COLOR_RX = "#0d6efd"  # Azul - Prescripción
        COLOR_OTC = "#6f42c1"  # Púrpura - Venta Libre
        COLOR_SUBSTITUTABLE = "#198754"  # Verde - Sustituible
        COLOR_NON_SUBSTITUTABLE = "#adb5bd"  # Gris claro
        COLOR_ANALYZABLE = "#20c997"  # Verde agua - Analizable
        COLOR_PARTNER = "#0dcaf0"  # Cyan - Ya Partners
        COLOR_OPPORTUNITY = "#ffc107"  # Amarillo - Oportunidad
        COLOR_NON_ANALYZABLE = "#fd7e14"  # Naranja - No Analizable

        def pct(part, whole):
            return f"{(part / whole * 100):.1f}%" if whole > 0 else "0%"

        def fmt(value):
            if value >= 1000:
                return f"€{value/1000:.1f}k"
            return f"€{value:.0f}"

        # Nivel 0: Total (raíz invisible para treemap)
        # No se añade porque Plotly usa parents="" para nivel raíz

        # Nivel 1: Prescripción y Venta Libre
        labels.append("Prescripción")
        parents.append("")
        values.append(rx_sales)
        text.append(f"{fmt(rx_sales)} ({pct(rx_sales, total_sales)})")
        colors.append(COLOR_RX)

        labels.append("Venta Libre")
        parents.append("")
        values.append(otc_sales)
        text.append(f"{fmt(otc_sales)} ({pct(otc_sales, total_sales)})")
        colors.append(COLOR_OTC)

        # Nivel 2: Dentro de Prescripción
        if rx_substitutable_sales > 0:
            labels.append("Con Genéricos")
            parents.append("Prescripción")
            values.append(rx_substitutable_sales)
            text.append(f"{fmt(rx_substitutable_sales)} ({pct(rx_substitutable_sales, rx_sales)})")
            colors.append(COLOR_SUBSTITUTABLE)

        if rx_non_substitutable_sales > 0:
            labels.append("Sin Genéricos")
            parents.append("Prescripción")
            values.append(rx_non_substitutable_sales)
            text.append(f"{fmt(rx_non_substitutable_sales)} ({pct(rx_non_substitutable_sales, rx_sales)})")
            colors.append(COLOR_NON_SUBSTITUTABLE)

        # Nivel 3: Dentro de Con Genéricos (Sustituible)
        if rx_analyzable_sales > 0:
            labels.append("En Vademécum")
            parents.append("Con Genéricos")
            values.append(rx_analyzable_sales)
            text.append(f"{fmt(rx_analyzable_sales)} ({pct(rx_analyzable_sales, rx_substitutable_sales)})")
            colors.append(COLOR_ANALYZABLE)

        if rx_non_analyzable_sales > 0:
            labels.append("Fuera Vademécum")
            parents.append("Con Genéricos")
            values.append(rx_non_analyzable_sales)
            text.append(f"{fmt(rx_non_analyzable_sales)} ({pct(rx_non_analyzable_sales, rx_substitutable_sales)})")
            colors.append(COLOR_NON_ANALYZABLE)

        # Nivel 4: Dentro de En Vademécum (Analizable)
        if rx_partner_sales > 0:
            labels.append("Ya Partners")
            parents.append("En Vademécum")
            values.append(rx_partner_sales)
            text.append(f"{fmt(rx_partner_sales)} ({pct(rx_partner_sales, rx_analyzable_sales)})")
            colors.append(COLOR_PARTNER)

        if rx_opportunity_sales > 0:
            labels.append("Oportunidad")
            parents.append("En Vademécum")
            values.append(rx_opportunity_sales)
            text.append(f"{fmt(rx_opportunity_sales)} ({pct(rx_opportunity_sales, rx_analyzable_sales)})")
            colors.append(COLOR_OPPORTUNITY)

        return {
            "labels": labels,
            "parents": parents,
            "values": values,
            "text": text,
            "colors": colors,
        }


class SubstitutableUniverseSummary(BaseMeasure):
    """
    Resumen del universo sustituible - Métricas para Context Panel (Issue #478 Fase 2).

    Calcula:
    - total_substitutable_groups: Número de conjuntos homogéneos con al menos un genérico
    - total_substitutable_units: Unidades vendidas en esos conjuntos
    - total_substitutable_revenue: Ingresos de ventas en esos conjuntos
    - total_pharmacy_revenue: Total de ventas de la farmacia (para calcular %)

    DEFINICIÓN UNIVERSO SUSTITUIBLE:
    - Solo conjuntos homogéneos que tienen AL MENOS un laboratorio genérico disponible
    - Métricas que NO cambian con la selección de partners
    - Base para todos los análisis posteriores de oportunidades

    Equivale al CTE `generic_homogeneous` + `pharmacy_sales_in_period` del servicio
    partner_analysis_service.get_substitutable_universe_context(), pero usando
    el patrón de medidas Power BI-style.
    """

    def __init__(self):
        super().__init__()
        self.description = "Resumen del universo sustituible para panel de contexto"
        self.unit = "€"
        self.category = "Genéricos Context"
        self.dependencies = ["TotalVentas"]

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular métricas del universo sustituible.

        Uses the base_query with JOIN to product_catalog filtered by
        substitutable groups (has at least one generic product available).

        Returns structure compatible with existing context-store consumers:
        - universe_summary: Métricas agregadas
        - homogeneous_groups: Lista vacía (para backward compatibility)
        - laboratories_in_universe: Laboratorios con productos en el universo sustituible
        """
        from sqlalchemy import text, func

        # 1. TOTAL PHARMACY REVENUE - Todas las ventas de la farmacia (sin filtros de catálogo)
        from .core_measures import TotalVentas
        total_pharmacy_revenue = TotalVentas().calculate(context)

        # 2. UNIVERSO SUSTITUIBLE - Solo ventas en conjuntos homogéneos con genéricos
        # Usamos SQL raw para eficiencia (el servicio usa CTEs materializados)
        substitutable_query = text(
            """
            WITH generic_homogeneous AS MATERIALIZED (
                -- Conjuntos que tienen AL MENOS un producto genérico activo
                SELECT DISTINCT pc.nomen_codigo_homogeneo
                FROM product_catalog pc
                WHERE pc.nomen_codigo_homogeneo IS NOT NULL
                  AND pc.nomen_codigo_homogeneo != ''
                  AND pc.nomen_tipo_farmaco = 'GENERICO'
                  AND pc.nomen_estado = 'ALTA'
            ),
            pharmacy_sales_in_period AS MATERIALIZED (
                -- Ventas de la farmacia en período para conjuntos sustituibles
                SELECT
                    pc.nomen_codigo_homogeneo,
                    -- Cantidad neta considerando devoluciones (total_amount < 0)
                    SUM(CASE WHEN sd.total_amount >= 0 THEN sd.quantity ELSE -sd.quantity END) as total_units,
                    SUM(sd.total_amount) as total_revenue,
                    COUNT(DISTINCT sd.id) as total_transactions
                FROM sales_data sd
                JOIN sales_enrichment se ON sd.id = se.sales_data_id
                JOIN product_catalog pc ON se.product_catalog_id = pc.id
                JOIN generic_homogeneous gh ON pc.nomen_codigo_homogeneo = gh.nomen_codigo_homogeneo
                WHERE sd.pharmacy_id = :pharmacy_id
                  AND (:start_date IS NULL OR sd.sale_date >= :start_date)
                  AND (:end_date IS NULL OR sd.sale_date <= :end_date)
                  AND pc.xfarma_prescription_category IS NOT NULL
                GROUP BY pc.nomen_codigo_homogeneo
            )
            SELECT
                COUNT(*) as total_substitutable_groups,
                COALESCE(SUM(total_units), 0) as total_substitutable_units,
                COALESCE(SUM(total_revenue), 0) as total_substitutable_revenue,
                COALESCE(SUM(total_transactions), 0) as total_substitutable_transactions
            FROM pharmacy_sales_in_period
            """
        )

        result = context.db.execute(
            substitutable_query,
            {
                "pharmacy_id": str(context.filters.pharmacy_id),
                "start_date": context.filters.start_date,
                "end_date": context.filters.end_date,
            },
        ).fetchone()

        total_substitutable_groups = int(result.total_substitutable_groups or 0)
        total_substitutable_units = int(result.total_substitutable_units or 0)
        total_substitutable_revenue = float(result.total_substitutable_revenue or 0)
        total_substitutable_transactions = int(result.total_substitutable_transactions or 0)

        # 3. Calcular porcentajes
        substitutable_percentage = (
            (total_substitutable_revenue / total_pharmacy_revenue * 100)
            if total_pharmacy_revenue > 0
            else 0
        )

        # 4. LABORATORIOS EN UNIVERSO SUSTITUIBLE - Para dropdown partners.py
        # Lista de laboratorios que tienen productos genéricos en conjuntos homogéneos
        labs_query = text(
            """
            SELECT DISTINCT
                pc.nomen_laboratorio as laboratory,
                pc.nomen_codigo_laboratorio as laboratory_code
            FROM product_catalog pc
            WHERE pc.nomen_codigo_homogeneo IS NOT NULL
              AND pc.nomen_codigo_homogeneo != ''
              AND pc.nomen_tipo_farmaco = 'GENERICO'
              AND pc.nomen_estado = 'ALTA'
              AND pc.nomen_laboratorio IS NOT NULL
            ORDER BY pc.nomen_laboratorio
            """
        )
        labs_result = context.db.execute(labs_query).fetchall()
        laboratories_in_universe = [
            {
                "laboratory": row.laboratory,
                "laboratory_code": row.laboratory_code,
            }
            for row in labs_result
        ]

        return {
            "universe_summary": {
                "total_substitutable_groups": total_substitutable_groups,
                "total_substitutable_units": total_substitutable_units,
                "total_substitutable_revenue": round(total_substitutable_revenue, 2),
                "total_substitutable_transactions": total_substitutable_transactions,
                "total_pharmacy_revenue": round(total_pharmacy_revenue, 2),
                "substitutable_percentage": round(substitutable_percentage, 2),
            },
            # Backward compatibility: homogeneous_groups vacío (detalle en análisis dinámico)
            "homogeneous_groups": [],
            # Laboratorios para dropdown de partners.py
            "laboratories_in_universe": laboratories_in_universe,
            # Metadata
            "filters_applied": {
                "pharmacy_id": str(context.filters.pharmacy_id),
                "start_date": context.filters.start_date.isoformat() if context.filters.start_date else None,
                "end_date": context.filters.end_date.isoformat() if context.filters.end_date else None,
            },
            "measure_used": self.name,
        }


class MonthlyPartnerTrend(BaseMeasure):
    """
    Tendencia mensual de ventas de laboratorios partners.
    Para análisis temporal de evolución de partners.
    """

    def __init__(self):
        super().__init__()
        self.description = "Tendencia mensual de ventas partners vs otros"
        self.unit = "%"
        self.category = "Tendencias Partners"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular evolución mensual de % partners durante período especificado"""

        # Obtener partners
        partner_labs = getattr(context.filters, "partner_laboratories", [])
        months_back = getattr(context.filters, "months_back", 12)

        if not partner_labs:
            # Intentar obtener desde la farmacia
            from app.models.pharmacy import Pharmacy

            pharmacy = context.db.query(Pharmacy).filter(Pharmacy.id == context.filters.pharmacy_id).first()

            if pharmacy and pharmacy.partner_laboratories:
                partner_labs = [p["laboratory_name"] for p in pharmacy.partner_laboratories]

        if not partner_labs:
            return {"error": "No partner laboratories found"}

        from app.models.nomenclator_local import NomenclatorLocal

        # Análisis mensual
        end_date = context.filters.end_date or utc_now().date()
        start_date = end_date - timedelta(days=months_back * 30)

        monthly_data = []

        for i in range(months_back):
            month_start = start_date + timedelta(days=i * 30)
            month_end = month_start + timedelta(days=30)
            if month_end > end_date:
                month_end = end_date

            # Crear contexto mensual
            from .base import FilterContext, QueryContext

            monthly_filters = FilterContext(
                pharmacy_id=context.filters.pharmacy_id,
                start_date=month_start,
                end_date=month_end,
            )
            monthly_context = QueryContext(context.db, monthly_filters)

            # Ventas partners del mes
            partner_sales = (
                monthly_context.base_query.join(
                    NomenclatorLocal,
                    SalesData.codigo_nacional == NomenclatorLocal.national_code,
                )
                .filter(NomenclatorLocal.laboratory.in_(partner_labs))
                .with_entities(func.sum(SalesData.total_amount))
                .scalar()
                or 0.0
            )

            # Ventas totales del mes
            total_sales = monthly_context.base_query.with_entities(func.sum(SalesData.total_amount)).scalar() or 0.0

            partner_percentage = (partner_sales / total_sales * 100) if total_sales > 0 else 0

            monthly_data.append(
                {
                    "month": month_start.strftime("%Y-%m"),
                    "month_name": month_start.strftime("%B %Y"),
                    "partner_sales": round(float(partner_sales), 2),
                    "total_sales": round(float(total_sales), 2),
                    "partner_percentage": round(partner_percentage, 2),
                }
            )

        # Calcular tendencia
        avg_percentage = sum(m["partner_percentage"] for m in monthly_data) / len(monthly_data) if monthly_data else 0

        return {
            "monthly_trend": monthly_data,
            "summary": {
                "avg_partner_percentage": round(avg_percentage, 2),
                "months_analyzed": len(monthly_data),
                "partner_laboratories": partner_labs,
            },
        }
