# backend/app/measures/core_measures.py
"""
Medidas farmacéuticas básicas - Equivalentes a medidas core de Power BI.

Medidas incluidas:
- TotalVentas: Suma de importes de venta
- TotalUnidades: Suma de unidades vendidas
- NumTransacciones: Número de operaciones únicas
- TicketPromedio: Venta promedio por transacción
- MargenBruto: Margen de beneficio calculado
"""


from sqlalchemy import func

from app.models.sales_data import SalesData

from .base import BaseMeasure, QueryContext


class TotalVentas(BaseMeasure):
    """
    Total de ventas en importe - Medida fundamental farmacéutica.
    Equivale a SUM(total_amount) en Power BI.

    NOTA (Issue #472): Usa enriched_query automáticamente cuando hay filtros
    que requieren JOIN con ProductCatalog (is_prescription, is_substitutable,
    laboratories, partner_laboratory_codes).
    """

    def __init__(self):
        super().__init__()
        self.description = "Importe total de ventas"
        self.unit = "€"
        self.category = "Ventas"

    def _needs_enriched_query(self, context: QueryContext) -> bool:
        """Determinar si se necesita enriched_query (JOIN completo a ProductCatalog)."""
        filters = context.filters
        return any([
            filters.is_substitutable is not None,
            filters.laboratories,
            filters.partner_laboratory_codes,
            filters.therapeutic_categories,
            filters.requires_prescription is not None,
            filters.is_generic is not None,
        ])

    def _needs_sales_enrichment_query(self, context: QueryContext) -> bool:
        """
        Determinar si solo necesita sales_enrichment_query (JOIN a SalesEnrichment).

        Issue #472: Para is_prescription, usar sales_enrichment_query preserva
        registros venta_libre que no tienen product_catalog_id.
        """
        filters = context.filters
        return filters.is_prescription is not None and not self._needs_enriched_query(context)

    def calculate(self, context: QueryContext) -> float:
        """Calcular suma total de importes de venta"""
        # Issue #472: Seleccionar query apropiado según filtros
        if self._needs_enriched_query(context):
            # Requiere JOIN completo (ProductCatalog) para filtros de catálogo
            query = context.enriched_query
        elif self._needs_sales_enrichment_query(context):
            # Solo requiere JOIN a SalesEnrichment para product_type
            query = context.sales_enrichment_query
        else:
            # Sin filtros de enriquecimiento
            query = context.base_query

        result = query.with_entities(func.sum(SalesData.total_amount).label("total_ventas")).scalar()

        return float(result or 0)


class TotalUnidades(BaseMeasure):
    """
    Total de unidades vendidas - Medida de volumen.
    Equivale a SUM(quantity) en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Número total de unidades vendidas"
        self.unit = "unidades"
        self.category = "Volumen"

    def calculate(self, context: QueryContext) -> int:
        """Calcular suma total de unidades"""
        result = context.base_query.with_entities(func.sum(SalesData.quantity).label("total_unidades")).scalar()

        return int(result or 0)


class NumTransacciones(BaseMeasure):
    """
    Número de transacciones únicas - Medida de frecuencia.
    Equivale a DISTINCTCOUNT(operation_id) en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Número de transacciones/tickets únicos"
        self.unit = "transacciones"
        self.category = "Frecuencia"

    def calculate(self, context: QueryContext) -> int:
        """Calcular número de operaciones distintas"""
        # Contar operation_id únicos, o total de registros si no hay operation_id
        result = context.base_query.with_entities(
            func.count(func.distinct(SalesData.operation_id)).label("num_transacciones")
        ).scalar()

        # Fallback si operation_id es null - contar registros individuales
        if not result:
            result = context.base_query.count()

        return int(result or 0)


class TicketPromedio(BaseMeasure):
    """
    Ticket promedio por transacción - Medida calculada.
    Equivale a [TotalVentas] / [NumTransacciones] en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Importe promedio por ticket/transacción"
        self.unit = "€"
        self.category = "Promedio"
        self.dependencies = ["TotalVentas", "NumTransacciones"]

    def calculate(self, context: QueryContext) -> float:
        """Calcular ticket promedio"""
        # Usar las medidas base para mantener consistencia
        total_ventas = TotalVentas().calculate(context)
        num_transacciones = NumTransacciones().calculate(context)

        if num_transacciones == 0:
            return 0.0

        return round(total_ventas / num_transacciones, 2)


class MargenBruto(BaseMeasure):
    """
    Margen bruto calculado - Medida financiera.
    Equivale a ([TotalVentas] - [TotalCostos]) / [TotalVentas] en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Margen bruto de beneficio (%)"
        self.unit = "%"
        self.category = "Rentabilidad"

    def calculate(self, context: QueryContext) -> float:
        """
        Calcular margen bruto.

        Nota: Depende de tener datos de purchase_price.
        Si no están disponibles, retorna 0.
        """
        # Calcular ventas y costos totales
        result = context.base_query.with_entities(
            func.sum(SalesData.total_amount).label("total_ventas"),
            func.sum(SalesData.purchase_price * SalesData.quantity).label("total_costos"),
        ).first()

        if not result or not result.total_ventas or result.total_ventas == 0:
            return 0.0

        total_ventas = float(result.total_ventas or 0)
        total_costos = float(result.total_costos or 0)

        if total_costos == 0:
            # Si no hay datos de coste, no se puede calcular margen
            return 0.0

        margen = ((total_ventas - total_costos) / total_ventas) * 100
        return round(margen, 2)


class PrecioPromedio(BaseMeasure):
    """
    Precio promedio por unidad - Medida de pricing.
    Equivale a [TotalVentas] / [TotalUnidades] en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Precio promedio por unidad vendida"
        self.unit = "€/unidad"
        self.category = "Pricing"
        self.dependencies = ["TotalVentas", "TotalUnidades"]

    def calculate(self, context: QueryContext) -> float:
        """Calcular precio promedio por unidad"""
        total_ventas = TotalVentas().calculate(context)
        total_unidades = TotalUnidades().calculate(context)

        if total_unidades == 0:
            return 0.0

        return round(total_ventas / total_unidades, 2)


class UnidadesPorTransaccion(BaseMeasure):
    """
    Unidades promedio por transacción - Medida de comportamiento.
    Equivale a [TotalUnidades] / [NumTransacciones] en Power BI.
    """

    def __init__(self):
        super().__init__()
        self.description = "Unidades promedio por transacción"
        self.unit = "unidades/ticket"
        self.category = "Comportamiento"
        self.dependencies = ["TotalUnidades", "NumTransacciones"]

    def calculate(self, context: QueryContext) -> float:
        """Calcular unidades promedio por ticket"""
        total_unidades = TotalUnidades().calculate(context)
        num_transacciones = NumTransacciones().calculate(context)

        if num_transacciones == 0:
            return 0.0

        return round(total_unidades / num_transacciones, 2)
