# backend/app/measures/__init__.py
"""
Framework de Medidas para xFarma - Estilo Power BI

Sistema de medidas reutilizables para análisis farmacéutico consistente.
Replica el paradigma de Power BI: Tablas relacionadas + Medidas calculadas.
"""

from .base import BaseMeasure, MeasureRegistry, QueryContext
from .core_measures import (
    MargenBruto,
    NumTransacciones,
    PrecioPromedio,
    TicketPromedio,
    TotalUnidades,
    TotalVentas,
    UnidadesPorTransaccion,
)
from .generic_measures import (
    GenericContextTreemap,
    GenericSavingsOpportunity,
    GenericVsBrandedRatio,
    HomogeneousGroupSales,
    MonthlyPartnerTrend,
    PartnerLabSales,
    SubstitutableUniverseSummary,
    TherapeuticCategorySales,
)
from .inventory_measures import (
    DeadStockValue,
    StockAlerts,
    StockCoverageDays,
    StockValue,
)
from .profitability_measures import (
    ContributionMargin,
    GMROI,
    ROIProduct,
)
from .rotation_measures import (
    ABCClassification,
    DaysInventory,
    RotationIndex,
)
from .temporal_measures import MAT  # Moving Annual Total
from .temporal_measures import YTD  # Year to Date
from .temporal_measures import MoM  # Month over Month
from .temporal_measures import QoQ  # Quarter over Quarter
from .temporal_measures import QTD_YoY  # Quarter-to-Date Year-over-Year
from .temporal_measures import MTD_YoY  # Month-to-Date Year-over-Year
from .temporal_measures import WTD_YoY  # Week-to-Date Year-over-Year
from .prescription_measures import (  # Issue #484
    PrescriptionSales,
    PrescriptionUnits,
    PrescriptionPercentage,
    PrescriptionCategoryDistribution,
    PrescriptionATCDistribution,
    PrescriptionKPIs,
)

# Registro global de medidas disponibles
measure_registry = MeasureRegistry()

# Registrar medidas core
measure_registry.register("total_ventas", TotalVentas())
measure_registry.register("total_unidades", TotalUnidades())
measure_registry.register("num_transacciones", NumTransacciones())
measure_registry.register("ticket_promedio", TicketPromedio())
measure_registry.register("margen_bruto", MargenBruto())
measure_registry.register("precio_promedio", PrecioPromedio())
measure_registry.register("unidades_por_transaccion", UnidadesPorTransaccion())

# Registrar medidas temporales
measure_registry.register("mat", MAT())
measure_registry.register("ytd", YTD())
measure_registry.register("mom", MoM())
measure_registry.register("qoq", QoQ())

# Registrar medidas YoY (Issue #472)
measure_registry.register("qtd_yoy", QTD_YoY())
measure_registry.register("mtd_yoy", MTD_YoY())
measure_registry.register("wtd_yoy", WTD_YoY())

# Registrar medidas especializadas para genéricos
measure_registry.register("generic_savings_opportunity", GenericSavingsOpportunity())
measure_registry.register("partner_lab_sales", PartnerLabSales())
measure_registry.register("generic_vs_branded_ratio", GenericVsBrandedRatio())
measure_registry.register("homogeneous_group_sales", HomogeneousGroupSales())
measure_registry.register("therapeutic_category_sales", TherapeuticCategorySales())
measure_registry.register("monthly_partner_trend", MonthlyPartnerTrend())
measure_registry.register("generic_context_treemap", GenericContextTreemap())  # Issue #472
measure_registry.register("substitutable_universe_summary", SubstitutableUniverseSummary())  # Issue #478

# Registrar medidas de inventario (Issue #470)
measure_registry.register("stock_value", StockValue())
measure_registry.register("dead_stock_value", DeadStockValue())
measure_registry.register("stock_coverage_days", StockCoverageDays())
measure_registry.register("stock_alerts", StockAlerts())

# Registrar medidas de rotación (Issue #470)
measure_registry.register("rotation_index", RotationIndex())
measure_registry.register("days_inventory", DaysInventory())
measure_registry.register("abc_classification", ABCClassification())

# Registrar medidas de rentabilidad (Issue #470)
measure_registry.register("roi_product", ROIProduct())
measure_registry.register("gmroi", GMROI())
measure_registry.register("contribution_margin", ContributionMargin())

# Registrar medidas de prescripción (Issue #484)
measure_registry.register("prescription_sales", PrescriptionSales())
measure_registry.register("prescription_units", PrescriptionUnits())
measure_registry.register("prescription_percentage", PrescriptionPercentage())
measure_registry.register("prescription_category_distribution", PrescriptionCategoryDistribution())
measure_registry.register("prescription_atc_distribution", PrescriptionATCDistribution())
measure_registry.register("prescription_kpis", PrescriptionKPIs())

__all__ = [
    "BaseMeasure",
    "QueryContext",
    "MeasureRegistry",
    "measure_registry",
    "TotalVentas",
    "TotalUnidades",
    "NumTransacciones",
    "TicketPromedio",
    "MargenBruto",
    "PrecioPromedio",
    "UnidadesPorTransaccion",
    "MAT",
    "YTD",
    "MoM",
    "QoQ",
    # YoY measures (Issue #472)
    "QTD_YoY",
    "MTD_YoY",
    "WTD_YoY",
    "GenericContextTreemap",
    "GenericSavingsOpportunity",
    "PartnerLabSales",
    "GenericVsBrandedRatio",
    "HomogeneousGroupSales",
    "SubstitutableUniverseSummary",
    "TherapeuticCategorySales",
    "MonthlyPartnerTrend",
    # Inventory measures (Issue #470)
    "StockValue",
    "DeadStockValue",
    "StockCoverageDays",
    "StockAlerts",
    # Rotation measures (Issue #470)
    "RotationIndex",
    "DaysInventory",
    "ABCClassification",
    # Profitability measures (Issue #470)
    "ROIProduct",
    "GMROI",
    "ContributionMargin",
    # Prescription measures (Issue #484)
    "PrescriptionSales",
    "PrescriptionUnits",
    "PrescriptionPercentage",
    "PrescriptionCategoryDistribution",
    "PrescriptionATCDistribution",
    "PrescriptionKPIs",
]
