﻿# backend/app/main.py
import asyncio
import os
from contextlib import asynccontextmanager
from typing import Any, AsyncGenerator, Dict

import httpx

# Configurar structured logging
from dotenv import load_dotenv
from fastapi import FastAPI, HTTPException, Request, Response
from fastapi.middleware.cors import CORSMiddleware
from sqlalchemy import text

# Configurar logging avanzado antes de importar otros módulos
from app.core.logging_config import configure_advanced_logging

logger = configure_advanced_logging()

# Cargar variables de entorno
load_dotenv()

from app.api.auth import router as auth_router
from app.api.auth_local import router as auth_local_router  # Pivot 2026: Local PIN auth
from app.api.auth_local import license_router  # Pivot 2026: License status
from app.api.employees import router as employees_router
from app.api.sales import router as sales_router
from app.api.simple_version_test import router as version_router
from app.api.upload import router as upload_router

# Importaciones locales
from app.database import check_database_connection

# Middleware de versionado
from app.middleware import APIVersioningMiddleware

# Importación thread-safe de modelos para evitar conflictos SQLAlchemy en Render
from app.models.base import safe_model_registry
from app.utils.datetime_utils import utc_now

# Solo importar enums específicos que se necesitan globalmente


# Lifecycle manager para startup/shutdown
@asynccontextmanager
async def lifespan(app: FastAPI) -> AsyncGenerator[None, None]:
    # Startup
    logger.info(
        "startup.beginning",
        version="0.1.0",
        environment=os.getenv("ENVIRONMENT", "development"),
    )

    # PASO 0: Verificar integridad de .env ANTES de continuar
    # Implementado para Issue #[INVESTIGACIÓN]: Pérdida de datos usuario dgruiz@gmail.com
    try:
        from app.core.env_integrity import verify_env_on_startup

        logger.info("env_integrity.check.starting")
        verify_env_on_startup(strict=False)  # strict=False: warnings no bloquean startup
        logger.info("env_integrity.check.passed")

    except Exception as e:
        logger.critical(
            "env_integrity.check.CRITICAL_FAILURE",
            error=str(e),
            message="Sistema NO PUEDE ARRANCAR por configuración .env inválida",
        )
        # No continuamos el startup si .env tiene errores críticos
        raise

    # Registrar modelos de manera thread-safe
    try:
        safe_model_registry()
        logger.info("models.registry.initialized")
    except Exception as e:
        logger.error("models.registry.failed", error=str(e))

    # Inicializar Redis cache
    try:
        from app.core.cache_config import init_redis

        if init_redis():
            logger.info("redis.connection.established")
        else:
            logger.warning("redis.connection.failed - Graceful degradation enabled")
    except Exception as e:
        logger.error("redis.initialization.failed", error=str(e))

    # Verificar conexión a BD
    if check_database_connection():
        logger.info("database.connection.established")

        # ❌ DESHABILITADO (Issue: Pérdida de datos con Alembic)
        # init_db() crea tablas con Base.metadata.create_all() sin registrar en alembic_version
        # Esto causa inconsistencias cuando se generan migraciones con --autogenerate
        # SOLUCIÓN: Alembic debe ser la ÚNICA fuente de verdad para el esquema
        # if init_db():
        #     logger.info("database.initialized")

        logger.info("database.schema.managed_by_alembic - Manual migrations only")

        # PASO 1: Validar esquema de base de datos (CRÍTICO - Previene arranque con esquema inconsistente)
        try:
            from app.core.schema_validator import validate_schema_on_startup
            from app.database import SessionLocal

            db = SessionLocal()
            try:
                validation_result = validate_schema_on_startup(db, strict=True)
                logger.info(
                    "schema.validation.passed",
                    alembic_version=validation_result.get("alembic_version"),
                    tables_count=validation_result.get("tables_count"),
                )
            finally:
                db.close()

        except RuntimeError as e:
            logger.critical(
                "schema.validation.CRITICAL_FAILURE",
                error=str(e),
                message="Sistema NO PUEDE ARRANCAR - Esquema de BD inconsistente",
            )
            # NO continuamos el startup si el esquema es inválido
            raise

        # Limpiar estados zombie de sincronizaciones anteriores
        try:
            from app.database import SessionLocal
            from app.models.system_status import SystemStatus

            db = SessionLocal()
            # Resetear cualquier estado "INITIALIZING" a "ERROR" con mensaje
            zombie_states = db.query(SystemStatus).filter(SystemStatus.status == "INITIALIZING").all()

            for state in zombie_states:
                state.status = "ERROR"
                state.message = "Proceso interrumpido por reinicio del servidor"
                state.progress = 0
                logger.warning("cleanup.zombie_state", component=state.component)

            if zombie_states:
                db.commit()
                logger.info("cleanup.zombie_states.cleaned", count=len(zombie_states))

            # Pivot 2026: Reset zombie ERP sync state (if stuck in SYNCING)
            try:
                from app.models.erp_sync_state import ERPSyncState, SyncStatusEnum

                erp_state = db.query(ERPSyncState).first()
                if erp_state and erp_state.status == SyncStatusEnum.SYNCING.value:
                    erp_state.status = SyncStatusEnum.ERROR.value
                    erp_state.last_error = "Sync interrupted by server restart"
                    db.commit()
                    logger.warning(
                        "cleanup.erp_sync_zombie",
                        last_started=erp_state.last_sync_started_at,
                    )
            except Exception as erp_e:
                logger.debug("cleanup.erp_sync_check.skipped", reason=str(erp_e))

            db.close()

        except Exception as e:
            logger.error("cleanup.zombie_states.failed", error=str(e))

        # Pivot 2026: Initialize local security manager (only in KAIFARMA_LOCAL mode)
        if os.getenv("KAIFARMA_LOCAL", "").lower() == "true":
            try:
                from app.core.security_local.local_state import (
                    LicenseValidationResult,
                    security_manager,
                )
                from app.database import SessionLocal
                from app.services.license_client import LicenseClientService

                license_key = os.getenv("KAIFARMA_LICENSE_KEY", "DEMO-0000-0000")
                pharmacy_name = os.getenv("KAIFARMA_PHARMACY_NAME", "Farmacia Local")

                # Validate license against Hub (or use cache if offline)
                license_result = None
                db = None
                try:
                    db = SessionLocal()
                    license_client = LicenseClientService(db)
                    client_result = await license_client.validate_license(license_key)

                    # Convert to LicenseValidationResult for security_manager
                    license_result = LicenseValidationResult(
                        valid=client_result.is_valid,
                        tier=client_result.tier if client_result.tier in ("free", "pro", "enterprise") else "pro",
                        expires_at=client_result.expires_at,
                        in_grace_period=client_result.in_grace_period,
                        days_remaining=client_result.days_remaining,
                        kill_switch_active=client_result.kill_switch_active,
                    )

                    # Update pharmacy_name from Hub if available
                    if client_result.pharmacy_name:
                        pharmacy_name = client_result.pharmacy_name

                    # Store kill-switch status in app state for middleware
                    app.state.read_only_mode = client_result.kill_switch_active

                    if client_result.kill_switch_active:
                        logger.critical(
                            "license.kill_switch.activated",
                            message="License expired - entering read-only mode",
                            security_violation=client_result.security_violation,
                        )
                    elif client_result.in_grace_period:
                        logger.warning(
                            "license.grace_period.active",
                            days_remaining=client_result.days_remaining,
                            message="Offline mode - please connect to internet to revalidate",
                        )
                    else:
                        logger.info(
                            "license.validation.success",
                            pharmacy_name=pharmacy_name,
                            tier=client_result.tier,
                        )

                except Exception as license_error:
                    logger.error("license.validation.failed", error=str(license_error), exc_info=True)
                    # Fall back to stub validation in security_manager
                    app.state.read_only_mode = False
                finally:
                    # Ensure database session is always closed
                    if db is not None:
                        db.close()

                security_manager.initialize(
                    license_key,
                    pharmacy_name,
                    license_result=license_result,
                )
                logger.info(
                    "security.local.initialized",
                    pharmacy_name=pharmacy_name,
                    message="Terminal locked - awaiting PIN",
                )
            except Exception as e:
                logger.error("security.local.init_failed", error=str(e))
                app.state.read_only_mode = False

        # Cache warming para laboratory mappings
        try:
            from app.services.laboratory_cache_service import laboratory_cache_service

            # Ejecutar cache warming en background
            def warm_laboratory_cache():
                from app.database import SessionLocal

                try:
                    db = SessionLocal()
                    try:
                        success = laboratory_cache_service.warm_cache_on_startup(db)
                        if success:
                            logger.info("cache.warming.laboratory.completed")
                        else:
                            logger.warning("cache.warming.laboratory.failed")
                    finally:
                        db.close()
                except Exception as e:
                    logger.error("cache.warming.laboratory.error", error=str(e))

            # Ejecutar warming en thread background sin bloquear startup
            import threading

            warming_thread = threading.Thread(target=warm_laboratory_cache, daemon=True)
            warming_thread.start()
            logger.info("cache.warming.laboratory.started_background")

        except Exception as e:
            logger.error("cache.warming.laboratory.initialization_failed", error=str(e))

        # Inicializar monitor automático CIMA (Issue #109 - Servicio consolidado)
        try:
            from app.services.cima_sync_monitor import start_cima_monitoring

            # Iniciar monitoreo CIMA con el servicio moderno consolidado
            monitor_started = start_cima_monitoring()

            if monitor_started:
                logger.info(
                    "cima.monitor.started_background",
                    service="CimaSyncMonitor",
                    note="Sistema consolidado desde Issue #109",
                )
            else:
                logger.warning("cima.monitor.start_failed", reason="Redis no disponible o error de inicialización")

        except Exception as e:
            logger.error("cima.monitor.initialization_failed", error=str(e))

    else:
        logger.error("database.connection.failed")

    # DESHABILITADO - Solo sincronización manual
    # await check_and_update_catalog_on_startup()
    logger.info("catalog.startup.disabled - Manual sync only to avoid startup issues")

    # Iniciar tarea de limpieza automática de syncs
    cleanup_task = None
    try:
        cleanup_task = asyncio.create_task(periodic_sync_cleanup())
        logger.info("cleanup.periodic.started", interval_minutes=30)
    except Exception as e:
        logger.error("cleanup.periodic.failed_to_start", error=str(e))

    # Iniciar tarea de limpieza diaria de datos FREE tier (Issue #142)
    free_tier_cleanup_task = None
    try:
        free_tier_cleanup_task = asyncio.create_task(periodic_free_tier_cleanup())
        logger.warning(
            "cleanup.free_tier.started",
            schedule="daily at 10:30 UTC (11:30 Madrid)",
            message="FREE tier data cleanup task initialized"
        )
    except Exception as e:
        logger.error("cleanup.free_tier.failed_to_start", error=str(e))

    # Iniciar tarea de sincronización automática CIMA/Nomenclator cada 15 días
    catalog_sync_task = None
    try:
        catalog_sync_task = asyncio.create_task(periodic_catalog_sync())
        logger.warning(
            "catalog.sync.periodic.started",
            schedule="Every 15 days at 02:00 Madrid (days 1 and 16)",
            message="CIMA/Nomenclator automatic sync task initialized"
        )
    except Exception as e:
        logger.error("catalog.sync.periodic.failed_to_start", error=str(e))

    # Pivot 2026: Iniciar tarea de sincronización automática ERP (si configurado)
    erp_sync_task = None
    if os.getenv("ERP_TYPE"):
        try:
            erp_sync_task = asyncio.create_task(periodic_erp_sync())
            logger.warning(
                "erp_sync.periodic.task_started",
                erp_type=os.getenv("ERP_TYPE"),
                interval=os.getenv("ERP_SYNC_INTERVAL", "15"),
                message="ERP Direct Access sync task initialized"
            )
        except Exception as e:
            logger.error("erp_sync.periodic.failed_to_start", error=str(e))

    # Pivot 2026: Iniciar dropzone watcher (folder monitoring para archivos ERP)
    try:
        from app.services.dropzone_watcher import dropzone_watcher

        if dropzone_watcher.start():
            logger.warning(
                "dropzone.watcher.started",
                paths=os.getenv("DROPZONE_PATHS", ""),
                message="Dropzone file watcher initialized"
            )
        else:
            logger.info(
                "dropzone.watcher.not_started",
                reason="DROPZONE_ENABLED=false or no paths configured"
            )
    except Exception as e:
        logger.error("dropzone.watcher.failed_to_start", error=str(e))

    # Log de middlewares configurados (WARNING level para visibilidad en producción)
    # IMPORTANTE: Nivel WARNING obligatorio porque production config usa WARNING level (logging_config.py:45)
    logger.warning("middleware.rate_limit.enabled", message="Auth endpoint protection active")
    logger.warning("middleware.authentication.enabled", message="Global JWT validation active")
    logger.warning("middleware.audit.enabled", message="Critical operations logging active")
    logger.warning(
        "middleware.slowapi.enabled",
        component="SlowAPI",
        affected_endpoints=["system/sync", "admin/pharmacies/{id}", "admin/tools/vacuum"],
        message="Rate limiting configured for critical admin endpoints"
    )

    yield

    # Cancelar tarea de limpieza de syncs al hacer shutdown
    if cleanup_task:
        cleanup_task.cancel()
        try:
            await cleanup_task
        except asyncio.CancelledError:
            logger.info("cleanup.periodic.cancelled")

    # Cancelar tarea de limpieza FREE tier al hacer shutdown
    if free_tier_cleanup_task:
        free_tier_cleanup_task.cancel()
        try:
            await free_tier_cleanup_task
        except asyncio.CancelledError:
            logger.info("cleanup.free_tier.cancelled")

    # Cancelar tarea de sync automático CIMA/Nomenclator
    if catalog_sync_task:
        catalog_sync_task.cancel()
        try:
            await catalog_sync_task
        except asyncio.CancelledError:
            logger.info("catalog.sync.periodic.cancelled")

    # Cancelar tarea de sync automático ERP
    if erp_sync_task:
        erp_sync_task.cancel()
        try:
            await erp_sync_task
        except asyncio.CancelledError:
            logger.info("erp_sync.periodic.cancelled")

    # Detener dropzone watcher
    try:
        from app.services.dropzone_watcher import dropzone_watcher

        if dropzone_watcher.is_running():
            dropzone_watcher.stop()
            logger.info("dropzone.watcher.stopped")
    except Exception as e:
        logger.error("dropzone.watcher.shutdown_error", error=str(e))

    # Detener monitor automático CIMA
    try:
        from app.services.cima_sync_monitor import stop_cima_monitoring

        stop_cima_monitoring()
        logger.info("cima.monitoring.stopped_on_shutdown")
    except Exception as e:
        logger.error("cima.monitoring.shutdown_error", error=str(e)[:200])

    # Cerrar conexiones Redis
    try:
        from app.core.cache_config import close_redis

        close_redis()
        logger.info("redis.connections.closed")
    except Exception as e:
        logger.error("redis.shutdown.error", error=str(e))

    # Shutdown
    logger.info("shutdown.complete", version="0.1.0")


# Crear aplicación FastAPI
app = FastAPI(
    title="xfarma API",
    description="API para análisis de datos farmacéuticos",
    version="1.0.0",  # Actualizado a 1.0.0
    lifespan=lifespan,
)

# ============================================================================
# CRÍTICO: Inicializar SlowAPI limiter para rate limiting de admin endpoints
# ============================================================================
# Fix para Issue: Endpoints con @admin_sync_limit fallaban silenciosamente
# sin ejecutar la función (respondían 200 OK pero nunca ejecutaban el código).
#
# Root cause: SlowAPI requiere app.state.limiter inicializado para funcionar.
# Sin esto, los decoradores @limiter.limit() y @admin_sync_limit FALLAN SILENCIOSAMENTE.
#
# Endpoints afectados (antes del fix):
# - POST /api/v1/system/sync (sincronización CIMA/nomenclator) ✅ FIJO
# - DELETE /api/v1/admin/pharmacies/{id} (borrado de farmacias) ✅ FIJO
# - POST /api/v1/admin/tools/vacuum (mantenimiento DB) ✅ FIJO
#
# Documentación completa: docs/SYNC_ENDPOINT_INVESTIGATION.md
# ============================================================================
from app.core.rate_limiting import limiter

app.state.limiter = limiter
# Nota: Log de inicialización movido a línea ~387 con otros logs de middleware


# Configurar CORS para desarrollo y producción
def get_allowed_origins() -> list[str]:
    """
    Obtener orígenes permitidos según el entorno.

    Returns:
        list[str]: Lista de orígenes permitidos para CORS
    """
    environment = os.getenv("ENVIRONMENT", "development")

    if environment == "production":
        # Orígenes específicos para producción en Render (URL real)
        return [
            "https://xfarma.onrender.com",
            # Agregar dominios personalizados aquí si los tienes
        ]
    else:
        # Orígenes específicos para desarrollo local (sin wildcards)
        return [
            "http://localhost:8050",
            "http://127.0.0.1:8050",
            "http://frontend:8050",
            "http://localhost:3000",  # Por si usas otro frontend
        ]


app.add_middleware(
    CORSMiddleware,
    allow_origins=get_allowed_origins(),
    allow_credentials=True,
    allow_methods=["GET", "POST", "PUT", "DELETE"],
    allow_headers=["*"],
)

# Añadir middleware de versionado de API
app.add_middleware(APIVersioningMiddleware, default_version="v1")

# CAPA 2: Middleware UTF-8 explícito (defensa en profundidad contra corruption de encoding)
# Fuerza Content-Type: application/json; charset=utf-8 en todas las respuestas JSON
@app.middleware("http")
async def force_utf8_headers(request: Request, call_next):
    """
    Middleware que fuerza UTF-8 en respuestas JSON.

    Parte de la estrategia de 3 capas contra corruption de encoding en cloud providers:
    - Capa 1: Normalización en puntos de ingesta (nomenclator, CIMA, parsers)
    - Capa 2: Headers UTF-8 explícitos (middleware, gunicorn, nginx) ← AQUÍ
    - Capa 3: Tests de regresión de encoding

    Previene que proxies intermedios (Render, AWS ALB, Cloudflare) malinterpreten
    el encoding y corrompan caracteres españoles (Ñ, tildes).
    """
    response = await call_next(request)

    # Solo aplicar a respuestas JSON (no static files, HTML, etc.)
    content_type = response.headers.get("content-type", "")
    if "application/json" in content_type:
        # Forzar charset=utf-8 incluso si hay otro charset presente (ej: ISO-8859-1)
        # Esto reemplaza charsets incorrectos que podrían causar doble encoding
        base_type = content_type.split(';')[0].strip()
        response.headers["content-type"] = f"{base_type}; charset=utf-8"

    return response

# Añadir middleware de logging inteligente
from app.middleware.logging_middleware import create_smart_logging_middleware

smart_logging_middleware = create_smart_logging_middleware(environment=os.getenv("ENVIRONMENT", "development"))
app._smart_logging_middleware = smart_logging_middleware  # Guardar referencia para stats
app.middleware("http")(smart_logging_middleware)

# Añadir middleware de rate limiting para laboratory endpoints
from app.core.rate_limiter import rate_limit_middleware

app.middleware("http")(rate_limit_middleware)

# Añadir exception handler para slowapi (admin rate limits)
from slowapi.errors import RateLimitExceeded

from app.core.rate_limiting import create_rate_limit_error_handler

app.add_exception_handler(RateLimitExceeded, create_rate_limit_error_handler)

# Añadir middleware de autenticación y auditoría
from app.middleware.auth_middleware import AuditLoggingMiddleware, AuthenticationMiddleware, RateLimitMiddleware

# IMPORTANTE: El orden de los middlewares importa
# 1. Primero rate limiting (para prevenir ataques de fuerza bruta)
# 2. Luego audit logging (para capturar todo)
# 3. Finalmente autenticación (para validar acceso)
app.add_middleware(AuditLoggingMiddleware)
app.add_middleware(AuthenticationMiddleware)
app.add_middleware(
    RateLimitMiddleware,
    max_requests=5,  # 5 intentos
    time_window=60,  # en 60 segundos
    lockout_time=300,  # bloqueo de 5 minutos
)

# Pivot 2026: Local mode middlewares
if os.getenv("KAIFARMA_LOCAL", "").lower() == "true":
    from app.middleware.autolock_middleware import AutoLockMiddleware
    from app.middleware.license_guard import LicenseGuardMiddleware

    # LicenseGuard enforces read-only mode when kill-switch is active
    app.add_middleware(LicenseGuardMiddleware)
    # AutoLock locks terminal after inactivity
    app.add_middleware(AutoLockMiddleware)

# Nota: Logs de middleware movidos al lifespan startup (líneas ~215-224)
# para garantizar visibilidad en producción con nivel WARNING

# Endpoint de health check
# NOTA: Removido el endpoint raíz "/" para permitir que el proxy sirva el frontend


@app.get("/api/info")
async def service_info() -> Dict[str, str]:
    """Endpoint para información básica del servicio (usado por tests)"""
    return {"status": "online", "service": "xfarma-backend", "version": "1.0.0"}


@app.get("/health/simple")
async def health_check_simple() -> Dict[str, str]:
    """Endpoint ligero para verificar que el servicio está activo"""
    # Solo verificar que podemos ejecutar una query simple
    try:
        from app.database import SessionLocal

        db = SessionLocal()
        try:
            # Query mínima para verificar conexión
            db.execute(text("SELECT 1"))
            return {"status": "healthy", "service": "xfarma-backend"}
        finally:
            db.close()
    except Exception as e:
        logger.error(f"Database connection check failed: {e}")
        return {"status": "unhealthy", "service": "xfarma-backend"}


@app.get("/health")
async def health_check() -> Dict[str, Any]:
    """Endpoint para verificar el estado del servicio"""

    # Verificar BD
    db_status = "healthy" if check_database_connection() else "unhealthy"

    # Verificar Redis cache
    try:
        from app.core.cache_config import is_cache_available

        redis_status = "healthy" if is_cache_available() else "unhealthy"
    except Exception as e:
        redis_status = "error"
        logger.error(f"Error checking Redis health: {e}")

    # Verificar servicios externos de datos
    try:
        from app.database import SessionLocal
        from app.external_data.nomenclator_integration import nomenclator_integration

        db = SessionLocal()
        try:
            product_count = nomenclator_integration.count_products_in_db(db)
            is_recent = nomenclator_integration.is_database_recent(db, max_days=15)
            external_data_status = "healthy" if product_count > 0 and is_recent else "degraded"
        finally:
            db.close()

    except Exception as e:
        external_data_status = "error"
        logger.error(f"Error checking external data health: {e}")

    overall_status = "healthy"
    if db_status != "healthy" or external_data_status not in ["healthy", "degraded"]:
        overall_status = "degraded"

    # Incluir información de timezone Madrid
    try:
        from app.utils.timezone import get_madrid_timezone_info, now_madrid

        timezone_info = get_madrid_timezone_info()
        server_time_madrid = now_madrid().isoformat()
    except ImportError:
        timezone_info = {"error": "timezone utils not available"}
        server_time_madrid = "unavailable"

    return {
        "status": overall_status,
        "database": db_status,
        "redis": redis_status,
        "external_data": external_data_status,
        "timezone": timezone_info,
        "server_time_madrid": server_time_madrid,
    }


# Incluir routers - solo versión v1 para evitar duplicación
# Eliminada duplicación /api/ legacy para simplificar

# Routers principales
app.include_router(upload_router, prefix="/api/v1/upload", tags=["upload"])

app.include_router(sales_router, prefix="/api/v1", tags=["sales"])

app.include_router(auth_router, prefix="/api/v1", tags=["auth"])

# Pivot 2026: Local PIN authentication (only in local mode)
if os.getenv("KAIFARMA_LOCAL", "").lower() == "true":
    app.include_router(auth_local_router, prefix="/api/v1", tags=["auth-local"])
    app.include_router(license_router, prefix="/api/v1", tags=["license"])

# Router para employee filtering (Issue #402 - PRO feature)
app.include_router(employees_router, prefix="/api/v1", tags=["employees"])

# Sistema de invitaciones de usuarios
from app.api.invitations import router as invitations_router

app.include_router(invitations_router, prefix="/api/v1/invitations", tags=["invitations"])

# Nuevo sistema de medidas estilo Power BI
from app.api.measures import router as measures_router

app.include_router(measures_router, prefix="/api/v1", tags=["measures"])

from app.api.admin import router as admin_router
from app.api.admin_catalog import router as admin_catalog_router
from app.api.admin_prescription import router as admin_prescription_router
from app.api.prescription_analytics import router as prescription_analytics_router
from app.api.prescription_seasonality import router as prescription_seasonality_router

# Importar y registrar routers de análisis de genéricos
from app.api.generic_analysis import router as generic_analysis_router
from app.api.pharmacy import router as pharmacy_router
from app.api.pharmacy_partners import router as pharmacy_partners_router
from app.api.pharmacy_targets import router as pharmacy_targets_router  # Issue #510

app.include_router(generic_analysis_router, prefix="/api/v1", tags=["generic-analysis"])

app.include_router(pharmacy_partners_router, prefix="/api/v1", tags=["pharmacy-partners"])
app.include_router(pharmacy_targets_router, prefix="/api/v1", tags=["pharmacy-targets"])  # Issue #510

app.include_router(admin_router, prefix="/api/v1", tags=["admin"])
app.include_router(admin_catalog_router, prefix="/api/v1", tags=["admin-catalog"])
app.include_router(admin_prescription_router, prefix="/api/v1", tags=["admin-prescription"])
app.include_router(prescription_analytics_router, prefix="/api/v1", tags=["prescription-analytics"])
app.include_router(prescription_seasonality_router, prefix="/api/v1", tags=["prescription-seasonality"])

# Normalizar pharmacy -> pharmacies para consistencia
app.include_router(pharmacy_router, prefix="/api/v1", tags=["pharmacy"])

# Nuevo router para oportunidades de genéricos por farmacia
from app.api.generic_opportunities import router as generic_opportunities_router

app.include_router(generic_opportunities_router, prefix="/api/v1", tags=["generic-opportunities"])

# Router para estado del sistema
from app.api.system import router as system_router

app.include_router(system_router, prefix="/api/v1", tags=["system"])

# Router de ejemplo para versionado
app.include_router(version_router, prefix="/api/v1", tags=["versioning"])

# Router para re-enriquecimiento
from app.api.reenrichment import router as reenrichment_router

app.include_router(reenrichment_router, prefix="/api/v1", tags=["reenrichment"])

from app.api.admin_tools import router as admin_tools_router

# NUEVOS ROUTERS - Reorganización del panel de estado
from app.api.system_unified import router as system_unified_router

app.include_router(system_unified_router, tags=["system-unified"])
app.include_router(admin_tools_router, tags=["admin-tools"])

# Router para Partner Analysis (Panel Partners Redesign)
from app.api.partner_analysis import router as partner_analysis_router
from app.api.optimal_partners import router as optimal_partners_router

app.include_router(partner_analysis_router, prefix="/api/v1", tags=["partner-analysis"])
app.include_router(optimal_partners_router, prefix="/api/v1/optimal-partners", tags=["optimal-partners"])

# Router para Laboratory Mapping (Códigos ↔ Nombres)
from app.api.laboratory_mapping import router as laboratory_mapping_router

app.include_router(laboratory_mapping_router, prefix="/api/v1", tags=["laboratory-mapping"])

# Router para métricas Prometheus (Issue #114 - Fase 3)
from app.api.metrics import router as metrics_router

app.include_router(metrics_router, tags=["metrics"])

# Issue #348 FASE 2: Admin routers para RBAC
from app.api.admin_users import router as admin_users_router
from app.api.admin_database import router as admin_database_router

app.include_router(admin_users_router, prefix="/api/v1/admin", tags=["admin-users"])
app.include_router(admin_database_router, prefix="/api/v1/admin", tags=["admin-database"])

# Issue #444: Subscription expiration management
from app.api.subscriptions import router as subscriptions_router

app.include_router(subscriptions_router, prefix="/api/v1", tags=["subscriptions"])

# Issue #446: Grupos intercambiables de venta libre
from app.api.intercambiable_groups import router as intercambiable_groups_router

app.include_router(intercambiable_groups_router, prefix="/api/v1", tags=["intercambiable-groups"])

# Issue #457: M4 Feedback Loop - Human-in-the-Loop
from app.api.feedback import router as feedback_router

app.include_router(feedback_router, prefix="/api/v1/feedback", tags=["feedback"])

# Issue #458: M5 Clustering Visualization
from app.api.clustering import router as clustering_router

app.include_router(clustering_router, prefix="/api/v1", tags=["clustering"])

# Issue #461: Dashboard Venta Libre - Análisis por NECESIDAD
from app.api.ventalibre import router as ventalibre_router

app.include_router(ventalibre_router, prefix="/api/v1", tags=["ventalibre"])

# Issue #477: Admin UI para duplicados VentaLibre
from app.api.admin_duplicates import router as admin_duplicates_router

app.include_router(admin_duplicates_router, prefix="/api/v1", tags=["admin-duplicates"])

# Issue #449: Admin UI para keywords NECESIDAD
from app.api.admin_keywords import router as admin_keywords_router

app.include_router(admin_keywords_router, prefix="/api/v1", tags=["admin-keywords"])

# Issue #459: Admin UI para category aliases (DB-backed)
from app.api.admin_category_aliases import router as admin_category_aliases_router

app.include_router(admin_category_aliases_router, prefix="/api/v1", tags=["admin-category-aliases"])

# Issue #XXX: Admin UI para brand aliases (corrección de marcas)
from app.api.admin_brand_aliases import router as admin_brand_aliases_router

app.include_router(admin_brand_aliases_router, prefix="/api/v1", tags=["admin-brand-aliases"])

# Issue #506: Insight Engine v2.0 - Motor de detección automática de anomalías
from app.api.insights import router as insights_router

app.include_router(insights_router, prefix="/api/v1", tags=["insights"])

# Issue #509: Executive Summary con NLG (NarrativeService)
from app.api.narrative import router as narrative_router

app.include_router(narrative_router, prefix="/api/v1", tags=["narrative"])

# Issue #521: Admin UI para grupos intercambiables curados
from app.api.admin_curated_groups import router as admin_curated_groups_router

app.include_router(admin_curated_groups_router, prefix="/api/v1", tags=["admin-curated-groups"])

# Issue #514: ROI Tracker - Feedback Loop de Acciones
from app.api.roi_tracker import router as roi_tracker_router

app.include_router(roi_tracker_router, prefix="/api/v1", tags=["roi-tracker"])

# Issue #511: Sistema de Doble Reporte (Dirección + Suelo)
from app.api.reports import router as reports_router

app.include_router(reports_router, prefix="/api/v1", tags=["reports"])

# Pivot 2026: ERP Direct Access sync management
from app.api.erp_sync import router as erp_sync_router

app.include_router(erp_sync_router, prefix="/api/v1", tags=["erp-sync"])

# Pivot 2026: Dropzone file watcher (folder monitoring)
from app.api.dropzone import router as dropzone_router

app.include_router(dropzone_router, prefix="/api/v1", tags=["dropzone"])

# Pivot 2026: Customer Order Tracking (Modulo 1: Cazador de Encargos)
from app.api.customer_orders import router as customer_orders_router

app.include_router(customer_orders_router, prefix="/api/v1", tags=["customer-orders"])

# Router para APIs del dashboard


# Funciones auxiliares para manejo del nomenclátor
async def check_and_update_catalog_on_startup() -> None:
    """
    Verifica y actualiza el catálogo completo durante el startup de la aplicación
    Incluye nomenclator, CIMA y grupos homogéneos
    Implementación mejorada para Issues #12 y #14 - Inicialización transparente
    """
    try:
        from app.database import SessionLocal
        from app.models.product_catalog import ProductCatalog
        from app.models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
        from app.services.catalog_startup import check_and_update_catalog_if_needed

        logger.info("catalog.check.startup")

        # Verificar si el catálogo está vacío
        db = SessionLocal()
        try:
            catalog_count = db.query(ProductCatalog).count()

            if catalog_count == 0:
                logger.warning(
                    "catalog.empty.detected",
                    message="Catálogo vacío detectado, marcando para inicialización",
                )

                # Actualizar estado del sistema para indicar que necesita inicialización
                for component in [
                    SystemComponent.CATALOG,
                    SystemComponent.NOMENCLATOR,
                    SystemComponent.CIMA,
                ]:
                    status = db.query(SystemStatus).filter_by(component=component).first()
                    if not status:
                        status = SystemStatus(
                            component=component,
                            status=SystemStatusEnum.NOT_INITIALIZED,
                            message=f"Sistema requiere inicialización de {component.value}",
                        )
                        db.add(status)
                    else:
                        status.status = SystemStatusEnum.NOT_INITIALIZED
                        status.message = f"Sistema requiere inicialización de {component.value}"
                db.commit()

                # NO inicializar automáticamente - dejar que el usuario lo haga desde la UI
                logger.info(
                    "catalog.initialization.required",
                    message="Sistema requiere inicialización manual desde la interfaz",
                )
            else:
                logger.info("catalog.check.products", count=catalog_count)
                # Verificar si necesita actualización
                await check_and_update_catalog_if_needed()

        finally:
            db.close()

    except Exception as e:
        logger.error("catalog.check.startup.error", error=str(e))


async def check_and_update_nomenclator_on_login() -> None:
    """
    Verifica y actualiza el nomenclátor cuando un usuario hace login
    """
    try:
        logger.info("nomenclator.check.login")
        await check_and_update_nomenclator_if_needed()
    except Exception as e:
        logger.error("nomenclator.check.login.error", error=str(e))


async def check_and_update_nomenclator_if_needed() -> None:
    """
    Lógica común para verificar si el nomenclátor necesita actualizarse
    Si los datos tienen más de 10 días, los actualiza automáticamente
    """
    try:
        import threading

        from app.database import SessionLocal
        from app.external_data.nomenclator_integration import nomenclator_integration

        db = SessionLocal()
        try:
            # Verificar si necesita actualización
            needs_update = not nomenclator_integration.is_database_recent(db, max_days=15)
            product_count = nomenclator_integration.count_products_in_db(db)

            if product_count == 0:
                needs_update = True
                logger.warning("[NOMENCLATOR] No hay datos del nomenclátor, iniciando descarga inicial")
            elif needs_update:
                logger.warning("[NOMENCLATOR] Datos del nomenclátor antiguos (>10 días), actualizando...")
            else:
                logger.info("[NOMENCLATOR] Datos del nomenclátor actualizados")
        finally:
            db.close()

        if needs_update:
            # Ejecutar actualización en thread separado
            def update_nomenclator():
                try:
                    logger.info("[NOMENCLATOR] Iniciando descarga y actualización automática...")
                    db = SessionLocal()
                    try:
                        result = nomenclator_integration.update_if_needed(db, force_update=True)
                        if result.get("status") == "updated":
                            logger.info("[NOMENCLATOR] Actualización completada exitosamente")
                        elif result.get("status") == "up_to_date":
                            logger.info("[NOMENCLATOR] Datos ya están actualizados")
                        else:
                            logger.error(f"[NOMENCLATOR] Error en actualización: {result.get('message')}")
                    finally:
                        db.close()
                except Exception as e:
                    logger.error(f"[NOMENCLATOR] Error en actualización automática: {e}")

            # Ejecutar en background SIN BLOQUEAR
            # Usar threading.Thread en lugar de ThreadPoolExecutor con 'with'
            # para evitar que el context manager espere a que termine
            background_thread = threading.Thread(target=update_nomenclator, daemon=True)
            background_thread.start()
            logger.info("[NOMENCLATOR] Actualización iniciada en background (thread daemon)")

    except Exception as e:
        logger.error(f"[NOMENCLATOR] Error en verificación de nomenclátor: {e}")


def classify_generic_laboratories():
    """
    Clasifica laboratorios como genéricos o no genéricos basándose en los datos del nomenclátor

    Returns:
        bool: True si la clasificación fue exitosa
    """
    try:
        from app.external_data.nomenclator_integration import NomenclatorIntegrationService

        logger.info("[CLASSIFICATION] Iniciando clasificación de laboratorios genéricos...")

        nomenclator_service = NomenclatorIntegrationService()

        # Cargar datos del nomenclátor si no están cargados
        if nomenclator_service.nomenclator_data is None:
            if not nomenclator_service.load_nomenclator_data():
                logger.error("[CLASSIFICATION] No se pudieron cargar datos del nomenclátor")
                return False

        # Lista de palabras clave que indican medicamentos genéricos
        generic_keywords = [
            "EFG",
            "GENERICO",
            "GENERICOS",
            "NORMON",
            "CINFA",
            "RATIOPHARM",
            "TEVA",
            "SANDOZ",
            "MYLAN",
            "STADA",
            "KERN PHARMA",
            "ACTAVIS",
        ]

        # Analizar datos del nomenclátor
        data = nomenclator_service.nomenclator_data
        total_products = len(data) if data is not None else 0

        generic_labs = set()
        brand_labs = set()

        if data is not None and total_products > 0:
            for _, row in data.iterrows():
                nombre = str(row.get("Nombre del producto farmacéutico", "")).upper()
                laboratorio = str(row.get("Nombre del laboratorio ofertante", "")).upper()
                tipo_farmaco = str(row.get("Tipo de fármaco", "")).upper()

                # Clasificar como genérico si contiene palabras clave
                is_generic = any(
                    keyword in nombre or keyword in laboratorio or keyword in tipo_farmaco
                    for keyword in generic_keywords
                )

                if is_generic:
                    generic_labs.add(laboratorio)
                else:
                    brand_labs.add(laboratorio)

            # Log resultados
            logger.info(f"[CLASSIFICATION] Procesados {total_products} productos del nomenclátor")
            logger.info(f"[CLASSIFICATION] Laboratorios genéricos identificados: {len(generic_labs)}")
            logger.info(f"[CLASSIFICATION] Laboratorios de marca identificados: {len(brand_labs)}")

            # Mostrar algunos ejemplos
            if generic_labs:
                logger.info(f"[CLASSIFICATION] Ejemplos laboratorios genéricos: {list(generic_labs)[:5]}")
            if brand_labs:
                logger.info(f"[CLASSIFICATION] Ejemplos laboratorios marca: {list(brand_labs)[:5]}")

            return True
        else:
            logger.warning("[CLASSIFICATION] No hay datos del nomenclátor para clasificar")
            return False

    except Exception as e:
        logger.error(f"[CLASSIFICATION] Error en clasificación de laboratorios: {e}")
        return False


async def periodic_sync_cleanup():
    """
    Tarea en background que ejecuta limpieza automática de sincronizaciones enganchadas
    cada 30 minutos.
    """
    while True:
        try:
            await asyncio.sleep(30 * 60)  # Esperar 30 minutos

            # Importar las funciones necesarias
            from app.api.system import cleanup_stale_syncs
            from app.database import SessionLocal

            # Ejecutar limpieza con timeout de 60 minutos
            db = SessionLocal()
            try:
                result = cleanup_stale_syncs(db, timeout_minutes=60)

                if result["cleaned_count"] > 0:
                    logger.info(
                        "cleanup.periodic.executed",
                        cleaned_count=result["cleaned_count"],
                        components=[c["component"] for c in result["cleaned_components"]],
                    )
                else:
                    logger.debug("cleanup.periodic.no_stale_syncs")

            finally:
                db.close()

        except asyncio.CancelledError:
            logger.info("cleanup.periodic.cancelled")
            break
        except Exception as e:
            logger.error("cleanup.periodic.error", error=str(e))
            # Continuar con el siguiente ciclo a pesar del error


async def periodic_free_tier_cleanup():
    """
    Tarea en background que ejecuta limpieza diaria de datos de usuarios FREE.

    Issue #142: Usuarios FREE no deben mantener datos de ventas en el servidor.
    Se ejecuta diariamente a las 10:30 UTC (11:30/12:30 Madrid según horario).

    La limpieza elimina:
    - SalesData (con CASCADE automático a SalesEnrichment via FK)
    - FileUpload asociados a la farmacia
    """
    from datetime import datetime, timezone

    TARGET_HOUR_UTC = 10  # 10:30 UTC = 11:30 Madrid (12:30 en verano)
    TARGET_MINUTE = 30

    while True:
        try:
            # Calcular tiempo hasta las 10:30 UTC (11:30 Madrid)
            now = datetime.now(timezone.utc)
            target_today = now.replace(hour=TARGET_HOUR_UTC, minute=TARGET_MINUTE, second=0, microsecond=0)

            if now >= target_today:
                # Ya pasó la hora objetivo hoy, programar para mañana
                from datetime import timedelta
                target = target_today + timedelta(days=1)
            else:
                target = target_today

            seconds_until_target = (target - now).total_seconds()

            logger.info(
                "cleanup.free_tier.scheduled",
                next_run_utc=target.isoformat(),
                seconds_until_run=int(seconds_until_target),
            )

            # Esperar hasta la hora programada
            await asyncio.sleep(seconds_until_target)

            # Ejecutar limpieza
            logger.info("cleanup.free_tier.starting")

            from app.database import SessionLocal
            from app.scripts.cleanup_free_tier_data import cleanup_free_tier_sales_data

            db = SessionLocal()
            try:
                # Ejecutar limpieza real (no dry_run)
                stats = cleanup_free_tier_sales_data(db, dry_run=False)

                logger.warning(
                    "cleanup.free_tier.completed",
                    users_processed=stats["users_processed"],
                    sales_deleted=stats["total_sales_deleted"],
                    uploads_deleted=stats["total_uploads_deleted"],
                    partners_deleted=stats["total_partners_deleted"],
                )

            finally:
                db.close()

        except asyncio.CancelledError:
            logger.info("cleanup.free_tier.cancelled")
            break
        except Exception as e:
            logger.error("cleanup.free_tier.error", error=str(e))
            # Esperar 1 hora antes de reintentar en caso de error
            await asyncio.sleep(3600)


async def periodic_erp_sync():
    """
    Tarea en background que sincroniza datos desde el ERP cada N minutos.

    Pivot 2026: ERP Direct Access - sync automático incremental.
    Solo se activa si ERP_TYPE está configurado en el entorno.
    """
    # Check if ERP sync is configured
    erp_type = os.getenv("ERP_TYPE")
    if not erp_type:
        logger.info("erp_sync.periodic.disabled", reason="ERP_TYPE not configured")
        return

    interval_minutes = int(os.getenv("ERP_SYNC_INTERVAL", "15"))

    logger.warning(
        "erp_sync.periodic.started",
        erp_type=erp_type,
        interval_minutes=interval_minutes,
    )

    # Initial delay to let the system stabilize
    await asyncio.sleep(60)  # Wait 1 minute after startup

    while True:
        try:
            # Run sync
            logger.info("erp_sync.periodic.running")

            from app.services.erp_sync_service import get_erp_sync_service

            service = get_erp_sync_service()
            result = service.sync()

            if result.get("skipped"):
                logger.debug(
                    "erp_sync.periodic.skipped",
                    reason=result.get("reason"),
                )
            elif result.get("success"):
                logger.info(
                    "erp_sync.periodic.completed",
                    records_synced=result.get("records_synced", 0),
                    duration=result.get("duration_seconds", 0),
                )
            else:
                logger.warning(
                    "erp_sync.periodic.failed",
                    error=result.get("error"),
                )

            # Wait for next interval
            await asyncio.sleep(interval_minutes * 60)

        except asyncio.CancelledError:
            logger.info("erp_sync.periodic.cancelled")
            break
        except Exception as e:
            logger.error("erp_sync.periodic.error", error=str(e))
            # Wait 5 minutes before retrying on error
            await asyncio.sleep(5 * 60)


async def periodic_catalog_sync():
    """
    Tarea en background que sincroniza CIMA y Nomenclator cada 15 días.

    Se ejecuta a las 2:00 hora de Madrid (1:00 UTC en invierno, 0:00 UTC en verano).
    Usa zona horaria Europe/Madrid para manejar cambios de horario automáticamente.
    """
    from datetime import datetime, timedelta
    from zoneinfo import ZoneInfo

    MADRID_TZ = ZoneInfo("Europe/Madrid")
    TARGET_HOUR = 2  # 2:00 AM Madrid
    TARGET_MINUTE = 0
    SYNC_INTERVAL_DAYS = 15

    while True:
        try:
            # Calcular próxima ejecución
            now_madrid = datetime.now(MADRID_TZ)

            # Buscar la próxima fecha de sync (cada 15 días desde el día 1 y 16 del mes)
            day_of_month = now_madrid.day
            if day_of_month < 16:
                next_sync_day = 16 if day_of_month >= 1 else 1
            else:
                # Próximo mes, día 1
                next_month = now_madrid.replace(day=1) + timedelta(days=32)
                next_sync_day = 1
                now_madrid = next_month.replace(day=1, hour=0, minute=0, second=0)

            target = now_madrid.replace(
                day=next_sync_day if day_of_month < 16 else 1,
                hour=TARGET_HOUR,
                minute=TARGET_MINUTE,
                second=0,
                microsecond=0
            )

            # Si ya pasó la hora objetivo hoy/este periodo, ir al siguiente
            if datetime.now(MADRID_TZ) >= target:
                if target.day == 1:
                    target = target.replace(day=16)
                else:
                    next_month = target.replace(day=1) + timedelta(days=32)
                    target = next_month.replace(day=1, hour=TARGET_HOUR, minute=TARGET_MINUTE, second=0)

            seconds_until_target = (target - datetime.now(MADRID_TZ)).total_seconds()

            logger.info(
                "catalog.sync.scheduled",
                next_run_madrid=target.isoformat(),
                days_until_run=int(seconds_until_target / 86400),
            )

            # Esperar hasta la hora programada
            await asyncio.sleep(max(seconds_until_target, 60))  # Mínimo 60s para evitar loops

            # Ejecutar sincronización
            logger.warning("catalog.sync.periodic.starting", schedule="Every 15 days at 02:00 Madrid")

            from app.database import SessionLocal
            from app.services.catalog_maintenance_service import CatalogMaintenanceService

            db = SessionLocal()
            try:
                service = CatalogMaintenanceService(db)

                # 1. Sync CIMA
                logger.info("catalog.sync.periodic.cima.starting")
                cima_result = await service.sync_cima_chunked(db, chunk_size=300)
                logger.warning(
                    "catalog.sync.periodic.cima.completed",
                    status=cima_result.get("status"),
                    products=cima_result.get("total_processed", 0),
                )

                # 2. Sync Nomenclator (incluye grupos homogéneos)
                logger.info("catalog.sync.periodic.nomenclator.starting")
                nomenclator_result = service.sync_nomenclator_only(db, force_update=False)
                logger.warning(
                    "catalog.sync.periodic.nomenclator.completed",
                    status=nomenclator_result.get("status"),
                )

                # 3. Auto-clasificar productos de prescripción
                logger.info("catalog.sync.periodic.classification.starting")
                try:
                    from app.models.product_catalog import ProductCatalog
                    from app.services.prescription_classification_service import PrescriptionClassificationService

                    # Query productos de prescripción sin clasificar
                    productos_sin_clasificar = db.query(ProductCatalog).filter(
                        ProductCatalog.xfarma_prescription_category.is_(None),
                        (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
                        (ProductCatalog.cima_requiere_receta == True)
                    ).all()

                    if productos_sin_clasificar:
                        classification_service = PrescriptionClassificationService(db=db)
                        classification_result = classification_service.bulk_classify(
                            productos_sin_clasificar, dry_run=False
                        )
                        logger.warning(
                            "catalog.sync.periodic.classification.completed",
                            classified=classification_result.get("classified_count", 0),
                            total=classification_result.get("total_products", 0),
                        )
                    else:
                        logger.info("catalog.sync.periodic.classification.skipped", reason="No unclassified products")

                except Exception as class_error:
                    logger.error("catalog.sync.periodic.classification.error", error=str(class_error))

                # 4. Re-enriquecer ventas en manual_review que ahora tienen match en catálogo
                logger.info("catalog.sync.periodic.reenrichment.starting")
                reenrich_updates = 0
                try:
                    from app.services.reenrichment_service import ReEnrichmentService

                    reenrich_service = ReEnrichmentService(db)
                    reenrich_result = reenrich_service.schedule_re_enrichment()
                    reenrich_updates = reenrich_result.get("updates_performed", 0)

                    if reenrich_updates > 0:
                        logger.warning(
                            "catalog.sync.periodic.reenrichment.completed",
                            updates_performed=reenrich_updates,
                            pharmacies_analyzed=reenrich_result.get("pharmacies_analyzed", 0),
                        )
                    else:
                        logger.info(
                            "catalog.sync.periodic.reenrichment.skipped",
                            reason="No stale enrichments found",
                        )

                except Exception as reenrich_error:
                    logger.error("catalog.sync.periodic.reenrichment.error", error=str(reenrich_error))

                logger.warning(
                    "catalog.sync.periodic.completed",
                    cima_status=cima_result.get("status"),
                    nomenclator_status=nomenclator_result.get("status"),
                    reenrichment_updates=reenrich_updates,
                )

            finally:
                db.close()

        except asyncio.CancelledError:
            logger.info("catalog.sync.periodic.cancelled")
            break
        except Exception as e:
            logger.error("catalog.sync.periodic.error", error=str(e))
            # Esperar 6 horas antes de reintentar en caso de error
            await asyncio.sleep(6 * 3600)


# Endpoint de prueba para desarrollo
@app.get("/api/test")
async def test_endpoint():
    return {
        "message": "API funcionando correctamente",
        "database_connected": check_database_connection(),
    }


# Endpoint para verificar optimización de logging
@app.get("/api/logging/stats")
async def logging_optimization_stats():
    """Endpoint para verificar estado de optimización de logging"""
    try:
        from app.core.logging_config import get_logging_stats

        logging_config = get_logging_stats()

        # Estadísticas del middleware si está disponible
        middleware_stats = {}
        if hasattr(app, "_smart_logging_middleware") and hasattr(app._smart_logging_middleware, "stats"):
            middleware_stats = app._smart_logging_middleware.stats.copy()

        return {
            "status": "active",
            "logging_config": logging_config,
            "middleware_stats": middleware_stats,
            "optimizations": {
                "health_checks_filtered": logging_config.get("health_check_filtered", False),
                "dash_internals_filtered": logging_config.get("dash_internals_filtered", False),
                "json_output": logging_config.get("json_output", False),
                "environment_optimized": logging_config.get("environment") == "production",
            },
            "recommendations": get_optimization_recommendations(),
        }
    except Exception as e:
        return {"status": "error", "error": str(e), "message": "Error retrieving logging optimization stats"}


def get_optimization_recommendations() -> Dict[str, str]:
    """Obtener recomendaciones de optimización basadas en el entorno actual"""
    environment = os.getenv("ENVIRONMENT", "development")
    is_render = os.getenv("RENDER") == "true"

    recommendations = {}

    if environment == "production":
        recommendations["logging"] = "✅ Optimizado para producción - filtros activos"
        if is_render:
            recommendations["render"] = "✅ Configuración específica Render aplicada"
        else:
            recommendations["render"] = "ℹ️ Para Render, considerar configurar health checks con intervalo 30s+"
    else:
        recommendations["logging"] = "ℹ️ Modo desarrollo - todos los logs visibles para debugging"
        recommendations["render"] = "ℹ️ En producción, filtros se activarán automáticamente"

    return recommendations


# Endpoint para testing de timezone Madrid
@app.get("/api/timezone")
async def timezone_test():
    """Endpoint para verificar configuración de timezone Madrid"""
    try:

        from app.utils.timezone import (
            format_madrid_datetime,
            get_madrid_timezone_info,
            madrid_to_utc,
            now_madrid,
            utc_to_madrid,
        )

        # Obtener tiempo actual en diferentes formatos
        madrid_now = now_madrid()
        utc_now_dt = utc_now()  # _dt suffix to avoid shadowing imported function

        return {
            "status": "success",
            "timezone_info": get_madrid_timezone_info(),
            "current_times": {
                "madrid_iso": madrid_now.isoformat(),
                "madrid_formatted": format_madrid_datetime(madrid_now),
                "madrid_spanish": format_madrid_datetime(madrid_now, "%d/%m/%Y %H:%M:%S"),
                "utc_iso": utc_now_dt.isoformat(),
                "utc_to_madrid": utc_to_madrid(utc_now_dt).isoformat(),
                "madrid_to_utc": madrid_to_utc(madrid_now).isoformat(),
            },
            "conversions_test": "OK - All timezone conversions working",
        }
    except Exception as e:
        return {
            "status": "error",
            "error": str(e),
            "message": "Timezone utilities not available or error in configuration",
        }


# Endpoint para trigger manual de verificación de nomenclátor (útil para testing)
@app.post("/api/admin/check-nomenclator")
async def trigger_nomenclator_check():
    """
    Endpoint para trigger manual de verificación del nomenclátor
    Útil para testing y administración
    """
    try:
        await check_and_update_nomenclator_if_needed()
        return {
            "status": "success",
            "message": "Verificación del nomenclátor iniciada",
            "timestamp": utc_now(),
        }
    except Exception as e:
        logger.error(f"Error triggering nomenclator check: {e}")
        raise HTTPException(status_code=500, detail=str(e))


# ========================================================================
# PROXY PARA SERVIR DASH FRONTEND
# ========================================================================
# En producción (Render), FastAPI actúa como servidor principal en puerto 8000
# y hace proxy al frontend Dash que corre en puerto interno 8050

# Cliente HTTP para proxy
frontend_client = None


def get_frontend_client():
    """Obtiene o crea el cliente HTTP para el frontend"""
    global frontend_client
    if frontend_client is None:
        frontend_port = os.getenv("FRONTEND_PORT", "8050")
        frontend_url = f"http://localhost:{frontend_port}"
        frontend_client = httpx.AsyncClient(base_url=frontend_url, timeout=30.0, follow_redirects=True)
    return frontend_client


# Proxy catch-all para el frontend (debe ir al final)
@app.api_route("/{path:path}", methods=["GET", "POST", "PUT", "DELETE", "OPTIONS", "HEAD", "PATCH"])
async def proxy_to_frontend(request: Request, path: str):
    """
    Proxy para todas las rutas no manejadas por FastAPI hacia el frontend Dash.
    Esto permite que FastAPI sirva como servidor principal.
    """

    # Si es una ruta de API, devolver 404 (no debería llegar aquí)
    if path.startswith("api/") or path == "health" or path == "docs" or path == "openapi.json":
        raise HTTPException(status_code=404, detail="API endpoint not found")

    try:
        client = get_frontend_client()

        # Construir URL completa para el frontend
        url = f"/{path}"
        if request.url.query:
            url = f"{url}?{request.url.query}"

        # Preparar headers (filtrar algunos que no deben ser reenviados)
        headers = dict(request.headers)
        headers.pop("host", None)
        headers.pop("content-length", None)

        # Leer body si existe
        body = await request.body() if request.method in ["POST", "PUT", "PATCH"] else None

        # Hacer la petición al frontend
        response = await client.request(
            method=request.method,
            url=url,
            headers=headers,
            content=body,
            follow_redirects=True,
        )

        # Devolver la respuesta del frontend
        return Response(
            content=response.content,
            status_code=response.status_code,
            headers=dict(response.headers),
        )

    except httpx.ConnectError:
        # El frontend no está disponible
        logger.warning(f"Frontend not available at port {os.getenv('FRONTEND_PORT', '8050')}")
        return Response(
            content="<h1>Frontend temporarily unavailable</h1><p>The dashboard is starting up. Please refresh in a few seconds.</p>",
            status_code=503,
            media_type="text/html",
        )
    except Exception as e:
        logger.error(f"Proxy error: {e}")
        raise HTTPException(status_code=500, detail="Internal proxy error")
