"""
Filtros especializados para análisis de partners farmacéuticos.
Sistema de filtrado para identificar universo sustituible y análisis de laboratorios.
"""

from datetime import date
from typing import Dict, List, Optional, Set

from sqlalchemy import and_, case, distinct, func, or_
from sqlalchemy.orm import Query, Session

from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment


class SubstitutableUniverseFilter:
    """
    Filtro para identificar el universo de productos sustituibles.
    Se considera sustituible cuando existe al menos un genérico en el conjunto homogéneo.
    """

    def __init__(self, db: Session):
        self.db = db

    def get_substitutable_groups(self) -> Set[str]:
        """
        Obtiene los códigos de grupos homogéneos que tienen al menos un genérico.

        Returns:
            Set de códigos homogéneos sustituibles
        """
        # Buscar grupos homogéneos con al menos un producto donde tipo_farmaco indique genérico
        # En nomenclator, tipo_farmaco = 'EFG' indica genérico
        result = (
            self.db.query(ProductCatalog.nomen_codigo_homogeneo)
            .filter(
                and_(
                    ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                    ProductCatalog.nomen_tipo_farmaco == "EFG",
                )
            )
            .distinct()
            .all()
        )

        return {r[0] for r in result if r[0]}

    def get_substitutable_products(self) -> Query:
        """
        Obtiene query de productos que pertenecen a grupos sustituibles.

        Returns:
            Query de ProductCatalog filtrado por productos sustituibles
        """
        substitutable_groups = self.get_substitutable_groups()

        if not substitutable_groups:
            # Si no hay grupos sustituibles, devolver query vacío
            return self.db.query(ProductCatalog).filter(False)

        return self.db.query(ProductCatalog).filter(ProductCatalog.nomen_codigo_homogeneo.in_(substitutable_groups))

    def calculate_substitutable_universe(
        self,
        pharmacy_id: str,
        start_date: Optional[date] = None,
        end_date: Optional[date] = None,
    ) -> Dict:
        """
        Calcula el universo sustituible para una farmacia específica.

        Args:
            pharmacy_id: ID de la farmacia
            start_date: Fecha inicio del período (opcional)
            end_date: Fecha fin del período (opcional)

        Returns:
            Diccionario con métricas del universo sustituible
        """
        # Obtener grupos sustituibles
        substitutable_groups = self.get_substitutable_groups()

        # Query base de ventas enriquecidas
        base_query = (
            self.db.query(SalesData, SalesEnrichment, ProductCatalog)
            .join(SalesEnrichment, SalesEnrichment.sales_data_id == SalesData.id)
            .join(ProductCatalog, ProductCatalog.id == SalesEnrichment.product_catalog_id)
            .filter(SalesData.pharmacy_id == pharmacy_id)
        )

        # Aplicar filtros de fecha si se proporcionan
        if start_date:
            base_query = base_query.filter(SalesData.sale_date >= start_date)
        if end_date:
            base_query = base_query.filter(SalesData.sale_date <= end_date)

        # Calcular total de ventas
        total_sales = base_query.with_entities(
            func.sum(SalesData.total_amount).label("total"),
            func.count(distinct(SalesData.id)).label("count"),
        ).first()

        # Calcular ventas sustituibles (productos en grupos homogéneos con genéricos)
        substitutable_sales = (
            base_query.filter(ProductCatalog.nomen_codigo_homogeneo.in_(substitutable_groups))
            .with_entities(
                func.sum(SalesData.total_amount).label("total"),
                func.count(distinct(SalesData.id)).label("count"),
            )
            .first()
        )

        # Calcular métricas por laboratorio
        lab_metrics = (
            base_query.filter(ProductCatalog.nomen_codigo_homogeneo.in_(substitutable_groups))
            .with_entities(
                ProductCatalog.nomen_laboratorio,
                func.sum(SalesData.total_amount).label("sales_total"),
                func.count(distinct(SalesData.id)).label("sales_count"),
                func.count(distinct(ProductCatalog.nomen_codigo_homogeneo)).label("unique_groups"),
            )
            .group_by(ProductCatalog.nomen_laboratorio)
            .having(ProductCatalog.nomen_laboratorio.isnot(None))
            .all()
        )

        return {
            "total_sales": {
                "amount": float(total_sales.total or 0) if total_sales else 0,
                "count": int(total_sales.count or 0) if total_sales else 0,
            },
            "substitutable_sales": {
                "amount": (float(substitutable_sales.total or 0) if substitutable_sales else 0),
                "count": (int(substitutable_sales.count or 0) if substitutable_sales else 0),
            },
            "substitutable_percentage": (
                (float(substitutable_sales.total or 0) / float(total_sales.total) * 100)
                if total_sales and total_sales.total and total_sales.total > 0
                else 0
            ),
            "laboratory_breakdown": (
                [
                    {
                        "laboratory": lab[0],
                        "sales_amount": float(lab[1] or 0),
                        "sales_count": int(lab[2] or 0),
                        "unique_homogeneous_groups": int(lab[3] or 0),
                    }
                    for lab in lab_metrics
                ]
                if lab_metrics
                else []
            ),
            "total_substitutable_groups": len(substitutable_groups),
            "analysis_period": {
                "start_date": start_date.isoformat() if start_date else None,
                "end_date": end_date.isoformat() if end_date else None,
            },
        }


class PartnerDynamicFilter:
    """
    Filtro dinámico para análisis de partners basado en criterios configurables.
    """

    def __init__(self, db: Session):
        self.db = db
        self.base_filter = SubstitutableUniverseFilter(db)

    def filter_by_generic_percentage(self, min_percentage: float = 50.0) -> List[Dict]:
        """
        Filtra laboratorios por porcentaje mínimo de genéricos.

        Args:
            min_percentage: Porcentaje mínimo de productos genéricos

        Returns:
            Lista de laboratorios que cumplen el criterio
        """
        # Calcular estadísticas por laboratorio
        lab_stats = (
            self.db.query(
                ProductCatalog.nomen_laboratorio,
                func.count(ProductCatalog.id).label("total_products"),
                func.sum(case((ProductCatalog.nomen_tipo_farmaco == "EFG", 1), else_=0)).label("generic_products"),
            )
            .filter(ProductCatalog.nomen_laboratorio.isnot(None))
            .group_by(ProductCatalog.nomen_laboratorio)
            .all()
        )

        result = []
        for lab in lab_stats:
            if lab.total_products > 0:
                generic_percentage = (lab.generic_products or 0) / lab.total_products * 100
                if generic_percentage >= min_percentage:
                    result.append(
                        {
                            "laboratory": lab.nomen_laboratorio,
                            "total_products": int(lab.total_products),
                            "generic_products": int(lab.generic_products or 0),
                            "generic_percentage": round(generic_percentage, 2),
                            "is_generic_specialist": generic_percentage >= 80,  # >80% se considera especialista
                        }
                    )

        # Ordenar por porcentaje de genéricos descendente
        result.sort(key=lambda x: x["generic_percentage"], reverse=True)

        return result

    def filter_by_therapeutic_groups(self, atc_codes: Optional[List[str]] = None) -> List[Dict]:
        """
        Filtra laboratorios por grupos terapéuticos específicos.

        Args:
            atc_codes: Lista de códigos ATC a filtrar (primeros caracteres)

        Returns:
            Lista de laboratorios con presencia en esos grupos
        """
        query = self.db.query(
            ProductCatalog.nomen_laboratorio,
            ProductCatalog.cima_atc_code,
            func.count(ProductCatalog.id).label("product_count"),
        ).filter(ProductCatalog.nomen_laboratorio.isnot(None))

        if atc_codes:
            # Filtrar por códigos ATC
            atc_conditions = []
            for code in atc_codes:
                atc_conditions.append(ProductCatalog.cima_atc_code.like(f"{code}%"))
            query = query.filter(or_(*atc_conditions))

        lab_groups = query.group_by(ProductCatalog.nomen_laboratorio, ProductCatalog.cima_atc_code).all()

        # Agrupar por laboratorio
        lab_summary = {}
        for lab in lab_groups:
            if lab.nomen_laboratorio not in lab_summary:
                lab_summary[lab.nomen_laboratorio] = {
                    "laboratory": lab.nomen_laboratorio,
                    "therapeutic_groups": [],
                    "total_products": 0,
                }

            if lab.cima_atc_code:
                # Obtener grupo terapéutico principal (primeros 3 caracteres del ATC)
                group = lab.cima_atc_code[:3] if len(lab.cima_atc_code) >= 3 else lab.cima_atc_code
                if group not in lab_summary[lab.nomen_laboratorio]["therapeutic_groups"]:
                    lab_summary[lab.nomen_laboratorio]["therapeutic_groups"].append(group)

            lab_summary[lab.nomen_laboratorio]["total_products"] += lab.product_count

        result = list(lab_summary.values())
        # Ordenar por número de productos descendente
        result.sort(key=lambda x: x["total_products"], reverse=True)

        return result

    def get_strategic_partners(self, pharmacy_id: str, criteria: Optional[Dict] = None) -> List[Dict]:
        """
        Identifica partners estratégicos basado en múltiples criterios.

        Args:
            pharmacy_id: ID de la farmacia
            criteria: Criterios de filtrado (min_generic_percentage, atc_codes, etc.)

        Returns:
            Lista de partners estratégicos recomendados
        """
        criteria = criteria or {}
        min_generic_percentage = criteria.get("min_generic_percentage", 50.0)
        atc_codes = criteria.get("atc_codes", None)

        # Obtener laboratorios genéricos
        generic_labs = self.filter_by_generic_percentage(min_generic_percentage)
        generic_lab_names = {lab["laboratory"] for lab in generic_labs}

        # Si hay filtros ATC, aplicarlos
        if atc_codes:
            therapeutic_labs = self.filter_by_therapeutic_groups(atc_codes)
            # Intersección de laboratorios que cumplen ambos criterios
            strategic_labs = [
                lab for lab in generic_labs if lab["laboratory"] in {t["laboratory"] for t in therapeutic_labs}
            ]
        else:
            strategic_labs = generic_labs

        # Enriquecer con datos del universo sustituible
        universe_data = self.base_filter.calculate_substitutable_universe(pharmacy_id)
        lab_sales_map = {lab["laboratory"]: lab for lab in universe_data["laboratory_breakdown"]}

        # Combinar información
        for lab in strategic_labs:
            if lab["laboratory"] in lab_sales_map:
                lab["current_sales"] = lab_sales_map[lab["laboratory"]]["sales_amount"]
                lab["current_transactions"] = lab_sales_map[lab["laboratory"]]["sales_count"]
            else:
                lab["current_sales"] = 0
                lab["current_transactions"] = 0

        # Clasificar partners
        for lab in strategic_labs:
            # Clasificación basada en porcentaje genérico y ventas actuales
            if lab["generic_percentage"] >= 80:
                if lab["current_sales"] > 0:
                    lab["partner_type"] = "CORE_PARTNER"  # Ya vendemos sus productos
                else:
                    lab["partner_type"] = "HIGH_POTENTIAL"  # Alto potencial, no vendemos aún
            elif lab["generic_percentage"] >= 50:
                if lab["current_sales"] > 0:
                    lab["partner_type"] = "ACTIVE_PARTNER"
                else:
                    lab["partner_type"] = "OPPORTUNITY"
            else:
                lab["partner_type"] = "COMPLEMENTARY"

        return strategic_labs
