﻿# backend/app/external_data/nomenclator_integration.py
"""
Sistema de integración con datos oficiales del nomenclátor del Ministerio de Sanidad
Automatiza la descarga y procesamiento del CSV oficial cada 10 días
"""

import logging
import os
import tempfile
import zipfile
from datetime import timezone
from typing import Any, Dict, List, Optional

import pandas as pd
import requests
from sqlalchemy import func
from sqlalchemy.orm import Session

from ..utils.datetime_utils import utc_now

# Configuración del logger
logger = logging.getLogger(__name__)


class NomenclatorIntegration:
    """
    Servicio para integración con el nomenclátor oficial del Ministerio de Sanidad

    Automatiza:
    - Descarga del CSV oficial cada 10 días
    - Procesamiento de datos oficiales (laboratorios, precios, genéricos)
    - Actualización de la base de datos nomenclator_local
    - Consultas rápidas de información oficial
    """

    def __init__(self):
        # URL oficial del nomenclátor (Ministerio de Sanidad)
        self.base_url = "https://www.sanidad.gob.es"
        self.nomenclator_url = f"{self.base_url}/profesionales/nomenclator.do"
        self.temp_dir = tempfile.gettempdir()

    def _update_system_status(self, db: Session, component: str, status: str, progress: int, message: str) -> None:
        """
        Actualiza el estado del sistema para un componente específico

        Args:
            db: Sesión de base de datos
            component: Nombre del componente (nomenclator, cima, catalog)
            status: Estado del componente (NOT_INITIALIZED, INITIALIZING, READY, ERROR)
            progress: Progreso de 0 a 100
            message: Mensaje descriptivo del estado actual
        """
        try:
            from app.models.system_status import SystemComponent, SystemStatus

            # Mapear string a enum
            component_enum = SystemComponent[component.upper()]

            # Buscar o crear registro de estado
            system_status = db.query(SystemStatus).filter(SystemStatus.component == component_enum).first()

            if not system_status:
                system_status = SystemStatus(
                    component=component_enum,
                    status=status,
                    progress=progress,
                    message=message,
                )
                db.add(system_status)
            else:
                system_status.status = status
                system_status.progress = progress
                system_status.message = message
                system_status.last_check = utc_now()

                if status == "ERROR":
                    system_status.error_count = (system_status.error_count or 0) + 1
                elif status == "READY":
                    system_status.last_success = utc_now()
                    system_status.error_count = 0

            db.commit()
            logger.info(f"[SYSTEM_STATUS] {component}: {status} ({progress}%) - {message}")

        except Exception as e:
            logger.error(f"Error updating system status: {str(e)}")
            db.rollback()

    def download_nomenclator_csv(self) -> Optional[str]:
        """
        Descarga el CSV oficial del nomenclátor desde el Ministerio de Sanidad
        Maneja sesión web requerida para acceder al enlace "formato CSV"

        Returns:
            Ruta al archivo CSV descargado o None si falla
        """

        logger.info("[NOMENCLATOR] Iniciando descarga desde Ministerio de Sanidad con sesión web")

        try:
            # Crear sesión web persistente con headers realistas
            session = requests.Session()
            session.headers.update(
                {
                    "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36",
                    "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8",
                    "Accept-Language": "es-ES,es;q=0.8,en-US;q=0.5,en;q=0.3",
                    "Accept-Encoding": "gzip, deflate, br",
                    "DNT": "1",
                    "Connection": "keep-alive",
                    "Upgrade-Insecure-Requests": "1",
                }
            )

            # 1. Establecer sesión accediendo primero a la página principal
            logger.info(f"[NOMENCLATOR] Estableciendo sesión en: {self.nomenclator_url}")
            main_response = session.get(self.nomenclator_url, timeout=30)
            main_response.raise_for_status()
            logger.info(f"[NOMENCLATOR] Sesión establecida (status: {main_response.status_code})")

            # 2. URL para descargar el CSV completo del nomenclátor
            csv_download_url = f"{self.base_url}/profesionales/nomenclator.do?metodo=buscarProductos&especialidad=%25%25%25&d-4015021-e=1&6578706f7274=1%20%C2%A0"
            logger.info(f"[NOMENCLATOR] Descargando CSV completo desde: {csv_download_url}")

            # Headers adicionales para la descarga
            session.headers.update({"Referer": self.nomenclator_url})

            file_response = session.get(csv_download_url, timeout=300)  # 5 minutos timeout
            file_response.raise_for_status()

            logger.info(
                f"[NOMENCLATOR] Descarga exitosa (status: {file_response.status_code}, size: {len(file_response.content)} bytes)"
            )

            # 4. Determinar tipo de archivo por content-type y extensión
            content_type = file_response.headers.get("content-type", "").lower()
            is_zip = (
                "zip" in content_type or csv_download_url.lower().endswith(".zip") or "application/zip" in content_type
            )

            logger.info(
                f"[NOMENCLATOR] Tipo de archivo detectado: {'ZIP' if is_zip else 'CSV'} (content-type: {content_type})"
            )

            if is_zip:
                # Manejar archivo ZIP
                zip_path = os.path.join(
                    self.temp_dir,
                    f"nomenclator_{utc_now().strftime('%Y%m%d_%H%M%S')}.zip",
                )

                with open(zip_path, "wb") as f:
                    f.write(file_response.content)

                logger.info(f"[NOMENCLATOR] ZIP guardado temporalmente: {zip_path}")

                # Extraer CSV del ZIP
                with zipfile.ZipFile(zip_path, "r") as zip_ref:
                    csv_files = [f for f in zip_ref.namelist() if f.lower().endswith(".csv")]

                    if not csv_files:
                        logger.error(
                            f"[NOMENCLATOR] No se encontró CSV dentro del ZIP. Archivos disponibles: {zip_ref.namelist()}"
                        )
                        os.remove(zip_path)
                        return None

                    # Usar el primer CSV encontrado
                    csv_filename = csv_files[0]
                    logger.info(f"[NOMENCLATOR] Extrayendo CSV del ZIP: {csv_filename}")

                    csv_path = os.path.join(
                        self.temp_dir,
                        f"nomenclator_{utc_now().strftime('%Y%m%d_%H%M%S')}.csv",
                    )

                    with zip_ref.open(csv_filename) as csv_file:
                        with open(csv_path, "wb") as output_file:
                            output_file.write(csv_file.read())

                # Limpiar ZIP temporal
                os.remove(zip_path)
                logger.info("[NOMENCLATOR] ZIP temporal eliminado")

            else:
                # Archivo CSV directo
                csv_path = os.path.join(
                    self.temp_dir,
                    f"nomenclator_{utc_now().strftime('%Y%m%d_%H%M%S')}.csv",
                )

                with open(csv_path, "wb") as f:
                    f.write(file_response.content)

            # Verificar que el archivo se creó correctamente
            if os.path.exists(csv_path) and os.path.getsize(csv_path) > 0:
                logger.info(
                    f"[NOMENCLATOR] Descarga completada exitosamente: {csv_path} ({os.path.getsize(csv_path)} bytes)"
                )
                return csv_path
            else:
                logger.error(f"[NOMENCLATOR] Error: archivo CSV vacío o no creado: {csv_path}")
                return None

        except requests.exceptions.RequestException as e:
            logger.error(f"[NOMENCLATOR] Error de conexión durante descarga: {str(e)}")
            return None
        except zipfile.BadZipFile as e:
            logger.error(f"[NOMENCLATOR] Error: archivo ZIP corrupto: {str(e)}")
            return None
        except Exception as e:
            logger.error(f"[NOMENCLATOR] Error inesperado durante descarga: {str(e)}")
            return None

    def parse_nomenclator_csv(self, csv_path: str) -> List[Dict[str, Any]]:
        """
        Parsea el CSV del nomenclátor oficial

        Args:
            csv_path: Ruta al archivo CSV

        Returns:
            Lista de productos procesados
        """

        logger.info(f"[NOMENCLATOR] Parseando CSV: {csv_path}")

        # Estadísticas de procesamiento para transparencia total (Issue #235)
        stats = {
            "total_rows": 0,
            "valid_products": 0,
            "discarded": {"invalid_code": 0, "missing_code": 0, "parsing_errors": 0, "skipped_bad_lines": 0},
        }

        try:
            # Intentar diferentes encodings comunes
            # CRÍTICO: UTF-8 primero para preservar caracteres españoles (Ñ, tildes)
            # latin1 NUNCA falla (lee cualquier byte), pero corrompe UTF-8 multibyte
            # Ver INFORME_ENCODING_NOMENCLATOR.md para detalles del root cause
            encodings = ["utf-8", "cp1252", "latin1", "iso-8859-1"]
            df = None

            # Cambio 2 (Issue #235): Callback personalizado para tracking de líneas descartadas
            def bad_lines_callback(bad_line):
                """Callback para contar líneas descartadas por pandas"""
                stats["discarded"]["skipped_bad_lines"] += 1
                # Log solo las primeras 3 líneas malas para no saturar logs
                if stats["discarded"]["skipped_bad_lines"] <= 3:
                    logger.warning(
                        "[NOMENCLATOR-DISCARD] Tipo: bad_line",
                        extra={
                            "discard_type": "bad_line",
                            "reason": "Línea mal formateada detectada por pandas",
                            "example_value": str(bad_line)[:200] if bad_line else None,
                            "issue": "#235",
                        },
                    )
                return None  # Skip the bad line

            for encoding in encodings:
                try:
                    df = pd.read_csv(
                        csv_path,
                        sep=",",
                        encoding=encoding,
                        engine="python",  # Issue #235: Required for on_bad_lines callback
                        on_bad_lines=bad_lines_callback,  # Issue #235: Usar callback para tracking
                        quoting=1,  # QUOTE_ALL para manejar comillas correctamente
                    )
                    logger.info(f"[NOMENCLATOR] CSV leído exitosamente con encoding {encoding}")
                    break
                except (UnicodeDecodeError, pd.errors.ParserError) as e:
                    logger.warning(f"[NOMENCLATOR] Error con encoding {encoding}: {e}")
                    continue

            if df is None:
                logger.error("[NOMENCLATOR] No se pudo leer el CSV con ningún encoding")
                return {"products": [], "stats": stats}  # Issue #235: Retornar con stats

            logger.info(f"[NOMENCLATOR] CSV parseado: {len(df)} filas encontradas")

            # Cambio 3 (Issue #235): Contar total_rows del DataFrame
            stats["total_rows"] = len(df)

            # Mapear columnas EXACTAS del nomenclátor oficial (Ministerio de Sanidad)
            # Estructura validada: 20 columnas oficiales del CSV del nomenclátor
            # FIXED: UTF-8 correcto para match con CSV del Ministerio
            official_columns = {
                # Identificación básica
                "codigo_nacional": "Código Nacional",
                "nombre_producto": "Nombre del producto farmacéutico",
                "tipo_farmaco": "Tipo de fármaco",
                "nombre_generico": "Nombre genérico efecto y accesorio",
                # Laboratorio
                "codigo_laboratorio": "Código del laboratorio ofertante",
                "nombre_laboratorio": "Nombre del laboratorio ofertante",
                # Estado y fechas
                "estado": "Estado",
                "fecha_alta": "Fecha de alta en el nomenclátor",
                "fecha_baja": "Fecha de baja en el nomenclátor",
                # Precios y aportación
                "aportacion_usuario": "Aportación del beneficiario",
                "principio_activo": "Principio activo o asociación de principios activos",
                "pvp": "Precio venta al público con IVA",
                "precio_referencia": "Precio de referencia",
                "menor_precio_homogeneo": "Menor precio de la agrupación homogénea del producto sanitario",
                # Agrupación homogénea (genéricos)
                "codigo_homogeneo": "Código de la agrupación homogénea del producto sanitario",
                "nombre_homogeneo": "Nombre de la agrupación homogénea del producto sanitario",
                # Indicadores especiales
                "diagnostico_hospitalario": "Diagnóstico hospitalario",
                "tratamiento_larga_duracion": "Tratamiento de larga duración",
                "especial_control_medico": "Especial control médico",
                "medicamento_huerfano": "Medicamento huérfano",
            }

            # Verificar que tenemos las columnas oficiales del nomenclátor
            available_columns = df.columns.tolist()
            logger.info(f"[NOMENCLATOR] Columnas encontradas: {len(available_columns)}")

            # Mapear las columnas que están disponibles
            column_mapping = {}
            for field, official_name in official_columns.items():
                if official_name in available_columns:
                    column_mapping[field] = official_name

            logger.info(f"[NOMENCLATOR] Columnas mapeadas exitosamente: {len(column_mapping)}/20")
            logger.info(f"[NOMENCLATOR] Campos disponibles: {list(column_mapping.keys())}")

            products = []
            processed = 0

            for index, row in df.iterrows():
                try:
                    # Código nacional es obligatorio
                    codigo_nacional = None
                    if "codigo_nacional" in column_mapping:
                        codigo_nacional = str(row[column_mapping["codigo_nacional"]]).strip()

                    if not codigo_nacional or codigo_nacional == "nan":
                        # Cambio 4 (Issue #235): Structured logging para invalid_code
                        if not codigo_nacional:
                            stats["discarded"]["missing_code"] += 1
                        else:
                            stats["discarded"]["invalid_code"] += 1

                        # Log solo los primeros ejemplos para no saturar logs
                        if stats["discarded"]["invalid_code"] + stats["discarded"]["missing_code"] <= 5:
                            logger.warning(
                                "[NOMENCLATOR-DISCARD] Tipo: invalid_code",
                                extra={
                                    "discard_type": "invalid_code" if codigo_nacional == "nan" else "missing_code",
                                    "row_index": index,
                                    "reason": "Código nacional faltante o inválido",
                                    "example_value": codigo_nacional,
                                    "issue": "#235",
                                },
                            )
                        continue

                    # CAPA 1: Normalización defensiva de encoding UTF-8
                    # Importar normalize_text para prevenir corruption en cloud providers
                    from app.utils.text_normalization import normalize_text

                    # Extraer datos disponibles del nomenclátor oficial
                    # Aplicar normalize_text() a TODOS los campos de texto
                    product_data = {
                        "national_code": codigo_nacional,
                        "product_name": normalize_text(
                            str(row.get(column_mapping.get("nombre_producto", ""), "")),
                            max_length=255,
                            context="product_name"
                        ),
                        "laboratory": normalize_text(
                            str(row.get(column_mapping.get("nombre_laboratorio", ""), "")),
                            max_length=255,
                            context="laboratory"
                        ),
                        "active_ingredient": normalize_text(
                            str(row.get(column_mapping.get("principio_activo", ""), "")),
                            max_length=255,
                            context="active_ingredient"
                        ),
                        "updated_at": utc_now(),
                    }

                    # CAMPOS HOMOGÉNEOS CRÍTICOS - NECESARIOS PARA ESTUDIO DE GENÉRICOS
                    if "codigo_homogeneo" in column_mapping:
                        value = str(row.get(column_mapping["codigo_homogeneo"], "")).strip()
                        product_data["homogeneous_code"] = value[:50] if value and value != "nan" else None

                    if "nombre_homogeneo" in column_mapping:
                        value = str(row.get(column_mapping["nombre_homogeneo"], "")).strip()
                        # Aplicar normalización también a nombre_homogeneo
                        product_data["homogeneous_name"] = normalize_text(
                            value,
                            max_length=500,
                            context="homogeneous_name"
                        ) if value and value != "nan" else None

                    if "estado" in column_mapping:
                        value = str(row.get(column_mapping["estado"], "")).strip()
                        product_data["status"] = value[:20] if value and value != "nan" else None

                    if "pvp" in column_mapping:
                        try:
                            pvp_str = str(row[column_mapping["pvp"]]).replace(",", ".")
                            if pvp_str and pvp_str != "nan":
                                pvp_value = float(pvp_str)
                                product_data["pvp"] = pvp_value
                                product_data["pvp_iva"] = pvp_value  # PVP ya incluye IVA
                        except (ValueError, TypeError):
                            pass

                    if "aportacion_usuario" in column_mapping:
                        try:
                            aport_str = str(row[column_mapping["aportacion_usuario"]]).replace(",", ".")
                            if aport_str and aport_str != "nan":
                                product_data["user_contribution"] = float(aport_str)
                        except (ValueError, TypeError):
                            pass

                    # Campos adicionales útiles
                    if "codigo_laboratorio" in column_mapping:
                        value = str(row.get(column_mapping["codigo_laboratorio"], "")).strip()
                        # Limpiar códigos decimales (ej: "1237.0" -> "1237")
                        if value and value != "nan":
                            if value.endswith(".0"):
                                value = value[:-2]
                            product_data["lab_code"] = value[:20]
                        else:
                            product_data["lab_code"] = None

                    if "tipo_farmaco" in column_mapping:
                        value = str(row.get(column_mapping["tipo_farmaco"], "")).strip()
                        product_data["drug_type"] = value[:50] if value and value != "nan" else None

                    # Procesar precio de referencia (es PVP máximo que paga SS)
                    if "precio_referencia" in column_mapping:
                        try:
                            price_str = str(row[column_mapping["precio_referencia"]]).replace(",", ".")
                            if price_str and price_str != "nan":
                                product_data["reference_price"] = float(price_str)
                            else:
                                product_data["reference_price"] = None
                        except (ValueError, TypeError):
                            product_data["reference_price"] = None
                    else:
                        product_data["reference_price"] = None

                    # Detectar si es genérico usando el campo "Tipo de fármaco"
                    is_generic = False
                    if "tipo_farmaco" in column_mapping:
                        tipo_farmaco = str(row[column_mapping["tipo_farmaco"]]).strip().lower()
                        is_generic = "generico" in tipo_farmaco or "genérico" in tipo_farmaco

                    product_data["is_generic"] = is_generic

                    # Determinar si requiere receta (mayoría de productos del nomenclátor la requieren)
                    product_data["requires_prescription"] = True

                    products.append(product_data)
                    processed += 1
                    # Cambio 6a (Issue #235): Actualizar contador valid_products
                    stats["valid_products"] += 1

                    if processed % 5000 == 0:
                        logger.info(f"[NOMENCLATOR] Procesados {processed} productos...")

                except Exception as e:
                    # Cambio 5 (Issue #235): Structured logging para parsing_errors
                    stats["discarded"]["parsing_errors"] += 1

                    # Log solo los primeros errores para no saturar logs
                    if stats["discarded"]["parsing_errors"] <= 5:
                        logger.warning(
                            f"[NOMENCLATOR-DISCARD] Tipo: parsing_error - Fila {index}: {str(e)}",
                            extra={
                                "discard_type": "parsing_error",
                                "row_index": index,
                                "reason": "Error durante procesamiento de fila",
                                "example_value": str(e),
                                "issue": "#235",
                            },
                        )
                    continue

            # Log final con estadísticas completas (Issue #235)
            logger.info(f"[NOMENCLATOR] Parsing completado: {len(products)} productos válidos")
            logger.info(
                f"[NOMENCLATOR-STATS] Issue #235 - Total filas: {stats['total_rows']}, "
                f"Válidos: {stats['valid_products']}, "
                f"Descartados: {sum(stats['discarded'].values())} "
                f"(sin código: {stats['discarded']['missing_code']}, "
                f"código inválido: {stats['discarded']['invalid_code']}, "
                f"errores parsing: {stats['discarded']['parsing_errors']}, "
                f"líneas mal formateadas: {stats['discarded']['skipped_bad_lines']})"
            )

            # Cambio 6b (Issue #235): Retornar con estadísticas
            return {"products": products, "stats": stats}

        except Exception as e:
            logger.error(f"[NOMENCLATOR] Error parseando CSV: {str(e)}")
            # Issue #235: Retornar estructura consistente con indicador de error catastrófico
            return {
                "products": [],
                "stats": {
                    "total_rows": 0,
                    "valid_products": 0,
                    "discarded": {
                        "invalid_code": 0,
                        "missing_code": 0,
                        "parsing_errors": 0,
                        "skipped_bad_lines": 0,
                        "catastrophic_failure": 1,  # Indicador de error catastrófico
                    },
                    "error": str(e),
                },
            }

    def get_product_by_code(self, db: Session, codigo_nacional: str) -> Optional[Dict[str, Any]]:
        """
        Busca producto por código nacional en la base de datos local

        Args:
            db: Sesión de base de datos
            codigo_nacional: Código nacional del medicamento

        Returns:
            Información del producto desde base de datos o None
        """
        try:
            from app.models.nomenclator_local import NomenclatorLocal

            product = (
                db.query(NomenclatorLocal)
                .filter(NomenclatorLocal.national_code == str(codigo_nacional).strip())
                .first()
            )

            if product:
                return {
                    "codigo_nacional": product.national_code,
                    "nombre_producto": product.product_name,
                    "laboratorio": product.laboratory,
                    "principio_activo": product.active_ingredient,
                    "es_generico": product.is_generic,
                    "precio_referencia": (float(product.reference_price) if product.reference_price else None),
                    "requiere_receta": product.requires_prescription,
                    "grupo_terapeutico": product.therapeutic_group,
                }

            return None

        except Exception as e:
            logger.error(f"Error searching nomenclator_local by code '{codigo_nacional}': {e}")
            return None

    def count_products_in_db(self, db: Session) -> int:
        """
        Cuenta productos en la base de datos

        Args:
            db: Sesión de base de datos

        Returns:
            Número de productos en nomenclator_local
        """
        try:
            from app.models.nomenclator_local import NomenclatorLocal

            return db.query(func.count(NomenclatorLocal.id)).scalar() or 0
        except Exception as e:
            logger.error(f"Error counting products in nomenclator_local: {e}")
            return 0

    def is_database_recent(self, db: Session, max_days: int = 10) -> bool:
        """
        Verifica si los datos en base de datos están actualizados

        Args:
            db: Sesión de base de datos
            max_days: Días máximos para considerar datos frescos (por defecto 10)

        Returns:
            True si los datos son recientes
        """
        try:
            from app.models.nomenclator_local import NomenclatorLocal

            # Obtener fecha de actualización más reciente
            latest_update = db.query(func.max(NomenclatorLocal.updated_at)).scalar()

            if not latest_update:
                return False

            # Asegurar que latest_update sea timezone-aware
            if latest_update.tzinfo is None:
                latest_update = latest_update.replace(tzinfo=timezone.utc)

            # Verificar si es reciente
            # Fix: Usar < en lugar de <= para evitar considerar "reciente" datos con exactamente max_days
            # Ejemplo: Si max_days=10, datos con 10 días (10d 16h truncado a 10d) deben considerarse obsoletos
            current_time = utc_now()
            days_old = (current_time - latest_update).days
            result = days_old < max_days  # ← FIX: Cambio de <= a <

            return result

        except Exception as e:
            logger.error(f"Error checking database freshness: {e}")
            return False

    def load_csv_to_database(self, db: Session, csv_file_path: str) -> Dict[str, Any]:
        """
        Carga datos del CSV del nomenclator a la base de datos usando BULK UPSERT

        Args:
            db: Sesión de base de datos
            csv_file_path: Ruta al archivo CSV

        Returns:
            Diccionario con estadísticas del proceso
        """
        try:
            from sqlalchemy.dialects.postgresql import insert

            from app.models.nomenclator_local import NomenclatorLocal

            logger.info(f"Loading nomenclator data from CSV to database using BULK UPSERT: {csv_file_path}")

            # Actualizar estado: parseando
            self._update_system_status(db, "nomenclator", "INITIALIZING", 60, "Analizando archivo CSV...")

            # Cambio 7a (Issue #235): Actualizar para manejar el nuevo formato de retorno
            # Parsear CSV
            parse_result = self.parse_nomenclator_csv(csv_file_path)
            products = parse_result.get("products", [])
            discard_stats = parse_result.get("stats", {})

            if not products:
                self._update_system_status(db, "nomenclator", "ERROR", 60, "No se encontraron productos en CSV")
                # Issue #235: Incluir estadísticas de descarte incluso cuando no hay productos válidos
                return {
                    "error": "No products parsed from CSV",
                    "processed": 0,
                    "errors": 0,
                    "discard_stats": discard_stats,
                }

            total_products = len(products)
            processed = 0
            errors = 0

            logger.info(f"[NOMENCLATOR] Procesando {total_products} productos con BULK UPSERT...")
            self._update_system_status(
                db,
                "nomenclator",
                "INITIALIZING",
                70,
                f"Procesando {total_products} productos...",
            )

            # Tamaño del chunk para operaciones bulk
            CHUNK_SIZE = 1000

            # Procesar en chunks para evitar usar demasiada memoria
            for i in range(0, len(products), CHUNK_SIZE):
                chunk = products[i : i + CHUNK_SIZE]
                chunk_size = len(chunk)

                try:
                    # Preparar statement UPSERT con PostgreSQL
                    stmt = insert(NomenclatorLocal).values(chunk)

                    # En caso de conflicto (código nacional ya existe), actualizar todos los campos
                    # Fix: Usar stmt.excluded[c.name] en lugar de c para obtener el valor real
                    update_dict = {c.name: stmt.excluded[c.name] for c in stmt.excluded if c.name != "id" and c.name != "created_at"}

                    # Ejecutar UPSERT masivo
                    stmt = stmt.on_conflict_do_update(index_elements=["national_code"], set_=update_dict)

                    # Ejecutar la operación bulk
                    db.execute(stmt)
                    db.commit()

                    processed += chunk_size

                    # Actualizar progreso
                    progress = 70 + int((processed / total_products) * 25)  # 70-95%
                    self._update_system_status(
                        db,
                        "nomenclator",
                        "INITIALIZING",
                        progress,
                        f"Procesados {processed}/{total_products} productos...",
                    )
                    logger.info(f"[NOMENCLATOR BULK] Procesados {processed}/{total_products} productos")

                except Exception as e:
                    errors += chunk_size
                    logger.error(f"[NOMENCLATOR BULK] Error procesando chunk {i//CHUNK_SIZE + 1}: {str(e)}")
                    db.rollback()
                    # Intentar procesar uno por uno este chunk como fallback
                    for product_data in chunk:
                        try:
                            national_code = product_data["national_code"]
                            existing = (
                                db.query(NomenclatorLocal)
                                .filter(NomenclatorLocal.national_code == national_code)
                                .first()
                            )

                            if existing:
                                for key, value in product_data.items():
                                    if key != "national_code":
                                        setattr(existing, key, value)
                            else:
                                new_product = NomenclatorLocal(**product_data)
                                db.add(new_product)

                            db.commit()
                            errors -= 1  # Reducir contador de errores si se procesa exitosamente

                        except Exception as inner_e:
                            logger.warning(
                                f"[NOMENCLATOR] Error procesando producto individual {product_data.get('national_code', 'unknown')}: {str(inner_e)}"
                            )
                            db.rollback()

            # Contar registros finales para estadísticas
            final_count = db.query(func.count(NomenclatorLocal.id)).scalar() or 0

            # Cambio 7b (Issue #235): Agregar estadísticas de descarte al resultado
            result = {
                "success": True,
                "processed": processed,
                "total_in_db": final_count,
                "errors": errors,
                "timestamp": utc_now().isoformat(),
                "method": "BULK_UPSERT",
                "discard_stats": discard_stats,  # Issue #235: Incluir estadísticas de descarte
            }

            # HOOK AUTOMÁTICO: Actualizar conjuntos homogéneos tras actualización del nomenclator
            try:
                logger.info("[NOMENCLATOR] Actualizando conjuntos homogéneos tras cambios en nomenclator...")
                self._update_homogeneous_groups_after_nomenclator(db, result)
            except Exception as e:
                logger.error(f"[NOMENCLATOR] Error actualizando conjuntos homogéneos: {str(e)}")
                # No fallar la actualización del nomenclator por esto

            logger.info(f"[NOMENCLATOR] Database update completed con BULK UPSERT: {result}")
            return result

        except Exception as e:
            logger.error(f"Error loading CSV to database: {str(e)}")
            db.rollback()
            return {"error": str(e), "processed": 0, "errors": 1}

    def update_if_needed(self, db: Session, force_update: bool = False) -> Dict[str, Any]:
        """
        Actualiza automáticamente si los datos son antiguos (>10 días)

        Args:
            db: Sesión de base de datos
            force_update: Fuerza la actualización independientemente de la antigüedad

        Returns:
            Resultado de la actualización
        """

        try:
            # Actualizar estado del sistema al inicio
            self._update_system_status(db, "nomenclator", "INITIALIZING", 0, "Verificando nomenclátor...")

            # Verificar si necesita actualización (cambio a 10 días como solicitó el usuario)
            if not force_update and self.is_database_recent(db, max_days=10):
                logger.info("Nomenclator database is up to date")
                self._update_system_status(db, "nomenclator", "READY", 100, "Nomenclátor actualizado")
                return {"status": "up_to_date", "message": "No update needed"}

            logger.info("Nomenclator database is outdated (>10 days old), updating...")

            # Actualizar estado: descargando
            self._update_system_status(
                db,
                "nomenclator",
                "INITIALIZING",
                10,
                "Descargando nomenclátor del Ministerio...",
            )

            # Descargar CSV
            csv_path = self.download_nomenclator_csv()

            if not csv_path:
                self._update_system_status(db, "nomenclator", "ERROR", 10, "Error en descarga")
                return {
                    "status": "error",
                    "message": "Failed to download nomenclator CSV",
                }

            # Actualizar estado: procesando
            self._update_system_status(db, "nomenclator", "INITIALIZING", 50, "Procesando archivo CSV...")

            # Cargar a base de datos
            result = self.load_csv_to_database(db, csv_path)

            # Limpiar archivo temporal
            if os.path.exists(csv_path):
                os.remove(csv_path)

            # Actualizar estado: completado
            self._update_system_status(db, "nomenclator", "READY", 100, "Nomenclátor actualizado exitosamente")

            return {
                "status": "updated",
                "message": "Nomenclator database updated successfully",
                "details": result,
            }

        except Exception as e:
            logger.error(f"Error in auto-update: {str(e)}")
            return {"status": "error", "message": str(e)}

    def manual_update(self, db: Session) -> Dict[str, Any]:
        """
        Fuerza una actualización manual del nomenclator

        Args:
            db: Sesión de base de datos

        Returns:
            Resultado de la actualización
        """
        logger.info("[NOMENCLATOR] Manual update requested")
        return self.update_if_needed(db, force_update=True)

    def _update_homogeneous_groups_after_nomenclator(self, db: Session, nomenclator_result: Dict) -> None:
        """
        Actualiza automáticamente conjuntos homogéneos tras actualización del nomenclator.

        Se ejecuta automáticamente cuando:
        - Se actualiza el nomenclator
        - Se crean/modifican productos en el nomenclator_local

        Args:
            db: Sesión de base de datos
            nomenclator_result: Resultado de la actualización del nomenclator
        """
        try:
            from app.models.pharmacy import Pharmacy
            from app.models.sales_data import SalesData
            from app.services.homogeneous_groups_service import HomogeneousGroupsService

            # Obtener todas las farmacias activas que pueden tener conjuntos homogéneos
            pharmacies = db.query(Pharmacy).filter(Pharmacy.is_active.is_(True)).all()

            if not pharmacies:
                logger.info("[NOMENCLATOR-HOOK] No hay farmacias activas para actualizar conjuntos homogéneos")
                return

            logger.info(f"[NOMENCLATOR-HOOK] Actualizando conjuntos homogéneos para {len(pharmacies)} farmacias")

            # Actualizar conjuntos homogéneos para cada farmacia
            homogeneous_service = HomogeneousGroupsService(db)
            updated_pharmacies = 0
            total_groups_updated = 0

            for pharmacy in pharmacies:
                try:
                    # Solo actualizar si la farmacia tiene datos de ventas
                    sales_count = (
                        db.query(func.count(SalesData.id)).filter(SalesData.pharmacy_id == pharmacy.id).scalar() or 0
                    )

                    if sales_count == 0:
                        logger.debug(f"[NOMENCLATOR-HOOK] Omitiendo farmacia {pharmacy.code} (sin datos de ventas)")
                        continue

                    logger.info(f"[NOMENCLATOR-HOOK] Actualizando conjuntos para farmacia {pharmacy.code}")

                    # Repoblar conjuntos homogéneos con datos actualizados del nomenclator
                    result = homogeneous_service.populate_homogeneous_groups_for_pharmacy(str(pharmacy.id))

                    # Actualizar cálculos PVL
                    homogeneous_service.update_pvl_base_calculations(str(pharmacy.id))

                    updated_pharmacies += 1
                    total_groups_updated += result.get("groups_processed", 0)

                    logger.info(
                        f"[NOMENCLATOR-HOOK] Farmacia {pharmacy.code}: {result.get('groups_processed', 0)} grupos actualizados"
                    )

                except Exception as e:
                    logger.error(f"[NOMENCLATOR-HOOK] Error actualizando farmacia {pharmacy.code}: {str(e)}")
                    continue

            logger.info(
                f"[NOMENCLATOR-HOOK] Actualización completada: {updated_pharmacies} farmacias, {total_groups_updated} grupos totales"
            )

            # Actualizar metadatos del nomenclator con info de los conjuntos homogéneos
            nomenclator_result["homogeneous_groups_update"] = {
                "pharmacies_updated": updated_pharmacies,
                "total_groups_updated": total_groups_updated,
                "update_timestamp": utc_now().isoformat(),
            }

        except Exception as e:
            logger.error(f"[NOMENCLATOR-HOOK] Error crítico actualizando conjuntos homogéneos: {str(e)}")
            # No relanzar la excepción para evitar fallar la actualización del nomenclator


# Instancia global del servicio
nomenclator_integration = NomenclatorIntegration()


def pvp_to_pvl(pvp_con_iva: float, con_recargo_equivalencia: bool = True) -> float:
    """
    Convierte PVP con IVA a PVL con IVA y recargo de equivalencia.
    Implementa la normativa oficial española de márgenes farmacéuticos.

    Args:
        pvp_con_iva: Precio de Venta al Público con IVA
        con_recargo_equivalencia: Si aplica recargo equivalencia (0.5%)

    Returns:
        PVL con IVA y recargo de equivalencia
    """
    if pvp_con_iva is None or pvp_con_iva <= 0:
        return 0.0

    # Constantes oficiales
    IVA = 0.04
    RECARGO = 0.005 if con_recargo_equivalencia else 0.0

    # PVP sin IVA
    pvp_sin_iva = pvp_con_iva / 1.04

    # Margen farmacia según tramos oficiales (RD 823/2008)
    if pvp_sin_iva <= 91.63:
        margen_farmacia = pvp_sin_iva * 0.279
    elif pvp_sin_iva <= 182.88:
        margen_farmacia = 25.56 + (pvp_sin_iva - 91.63) * 0.10
    else:
        margen_farmacia = 34.69 + (pvp_sin_iva - 182.88) * 0.085

    # Precio sin margen farmacia
    precio_sin_farmacia = pvp_sin_iva - margen_farmacia

    # Margen distribución oficial
    if precio_sin_farmacia <= 91.63:
        margen_distribucion = precio_sin_farmacia * 0.076
    else:
        margen_distribucion = min(precio_sin_farmacia * 0.076, 6.96)

    # PVL final con IVA y recargo
    pvl_sin_iva = precio_sin_farmacia - margen_distribucion
    pvl_con_iva = pvl_sin_iva * (1 + IVA + RECARGO)

    return round(pvl_con_iva, 2)
