﻿# backend/app/external_data/data_updater.py
"""
Servicio para actualización automática de fuentes de datos externas
Gestiona la sincronización periódica de CIMA, nomenclator y otras fuentes
"""

import logging
from typing import Any, Dict

from ..utils.datetime_utils import utc_now
from .cima_integration import CIMAIntegrationService
from .nomenclator_integration import nomenclator_integration

logger = logging.getLogger(__name__)


class ExternalDataUpdater:
    """
    Servicio para actualización automática de fuentes externas

    Funcionalidades:
    - Actualización periódica de nomenclator (semanal/mensual)
    - Verificación de estado de API CIMA
    - Logs de actualizaciones y errores
    - Health checks de todas las fuentes
    """

    def __init__(self):
        """Inicializa el actualizador de datos externos"""
        self.cima_service = None
        self.nomenclator_service = None
        self.last_update_check = None

    def initialize_services(self):
        """Inicializa los servicios externos"""
        try:
            self.cima_service = CIMAIntegrationService()
            self.nomenclator_service = nomenclator_integration
            logger.info("External data services initialized")
            return True
        except Exception as e:
            logger.error(f"Failed to initialize external data services: {e}")
            return False

    def update_all_sources(self, force_refresh: bool = False) -> Dict[str, Any]:
        """
        Actualiza todas las fuentes de datos externas

        Args:
            force_refresh: Forzar actualización aunque no sea necesaria

        Returns:
            Diccionario con resultado de las actualizaciones
        """
        results = {
            "timestamp": utc_now(),
            "nomenclator_updated": False,
            "cima_checked": False,
            "errors": [],
            "success": False,
        }

        try:
            if not self.cima_service or not self.nomenclator_service:
                if not self.initialize_services():
                    results["errors"].append("Failed to initialize services")
                    return results

            # 1. Actualizar nomenclator
            logger.info("Updating nomenclator data...")
            nomenclator_success = self.nomenclator_service.download_nomenclator(force_refresh)
            results["nomenclator_updated"] = nomenclator_success

            if not nomenclator_success:
                results["errors"].append("Failed to update nomenclator")

            # 2. Verificar CIMA API
            logger.info("Checking CIMA API status...")
            cima_health = self.cima_service.health_check()
            results["cima_checked"] = True
            results["cima_status"] = cima_health["status"]

            if cima_health["status"] != "healthy":
                results["errors"].append(f"CIMA API not healthy: {cima_health}")

            # 3. Evaluar éxito general
            results["success"] = nomenclator_success and cima_health["status"] == "healthy"

            self.last_update_check = utc_now()

            if results["success"]:
                logger.info("All external data sources updated successfully")
            else:
                logger.warning(f"Update completed with issues: {results['errors']}")

            return results

        except Exception as e:
            logger.error(f"Error updating external data sources: {e}")
            results["errors"].append(str(e))
            return results

    def health_check_all(self) -> Dict[str, Any]:
        """
        Verifica el estado de todas las fuentes de datos

        Returns:
            Diccionario con estado de salud de todas las fuentes
        """
        try:
            if not self.cima_service or not self.nomenclator_service:
                if not self.initialize_services():
                    return {
                        "overall_status": "unhealthy",
                        "error": "Failed to initialize services",
                        "timestamp": utc_now(),
                    }

            # Health check de todos los servicios
            cima_health = self.cima_service.health_check()
            nomenclator_health = self.nomenclator_service.health_check()

            # Evaluar estado general
            all_healthy = cima_health["status"] == "healthy" and nomenclator_health["status"] == "healthy"

            return {
                "overall_status": "healthy" if all_healthy else "degraded",
                "cima": cima_health,
                "nomenclator": nomenclator_health,
                "last_update_check": self.last_update_check,
                "timestamp": utc_now(),
            }

        except Exception as e:
            return {
                "overall_status": "unhealthy",
                "error": str(e),
                "timestamp": utc_now(),
            }

    def should_update_nomenclator(self, max_age_days: int = 7) -> bool:
        """
        Determina si es necesario actualizar el nomenclator

        Args:
            max_age_days: Edad máxima en días antes de requerir actualización

        Returns:
            True si necesita actualización
        """
        try:
            if not self.nomenclator_service:
                return True

            nomenclator_health = self.nomenclator_service.health_check()

            if not nomenclator_health.get("file_exists", False):
                return True

            last_updated = nomenclator_health.get("last_updated")
            if not last_updated:
                return True

            age = utc_now() - last_updated
            return age.days >= max_age_days

        except Exception as e:
            logger.error(f"Error checking nomenclator update need: {e}")
            return True

    def get_update_schedule(self) -> Dict[str, Any]:
        """
        Obtiene información sobre el cronograma de actualizaciones

        Returns:
            Información sobre próximas actualizaciones programadas
        """
        return {
            "nomenclator": {
                "frequency": "weekly",
                "next_update": "manual_trigger_required",
                "last_update": (self.nomenclator_service.last_updated if self.nomenclator_service else None),
            },
            "cima": {
                "frequency": "real_time_api",
                "status": "always_current",
                "last_check": self.last_update_check,
            },
        }
