﻿# backend/app/external_data/cima_integration.py
"""
Sistema de integración con CIMA (Centro de Información de Medicamentos - AEMPS)
Descarga batch completa de datos clínicos y regulatorios

Pivot 2026: Conditional Redis import for local mode (no Redis in local)
"""

import asyncio
import os
import uuid
from datetime import datetime
from typing import Any, AsyncIterator, Dict, List, Optional, Tuple

import httpx
import structlog
from sqlalchemy import case, func, text
from sqlalchemy.orm import Session

# Pivot 2026: Redis is optional in local mode (no caching needed)
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"
if IS_LOCAL_MODE:
    redis = None  # type: ignore
else:
    import redis

from app.exceptions import APIError, RateLimitError
from app.utils.retry_helper import exponential_backoff

from ..utils.datetime_utils import utc_now

# Initialize logger before use
logger = structlog.get_logger(__name__)

# Métricas Prometheus para monitoring (Issue #196 - Item #6)
try:
    from app.metrics.cima_metrics import (
        cima_sync_active,
        cima_sync_duration_seconds,
        cima_sync_errors_total,
        cima_sync_items_total,
        cima_sync_progress_percent,
    )

    METRICS_AVAILABLE = True
except ImportError:
    # Si no hay métricas disponibles, usar no-ops
    METRICS_AVAILABLE = False
    logger.warning("cima.metrics.unavailable", msg="Prometheus metrics not available")

# NO USAR aempsconn - Solo API REST directa
AEMPSCONN_AVAILABLE = False
logger.info("cima.integration.initialization", status="rest_api_direct", aempsconn_enabled=False)


class CIMAIntegrationService:
    """
    Servicio de integración con API de CIMA

    Implementa:
    - Descarga batch completa de medicamentos usando aempsconn
    - Cache en Redis para optimizar consultas
    - Rate limiting para respetar límites de API
    - Mapeo de campos a product_catalog
    """

    def __init__(self):
        """Inicializa conexión con CIMA API"""

        # NO USAR aempsconn - Solo API REST
        self.aemps = None
        logger.info("cima.rest_api.initialized", api_type="direct")

        # Redis para caché (si está disponible)
        # Pivot 2026: Skip Redis entirely in local mode
        if IS_LOCAL_MODE or redis is None:
            self.redis_client = None
            self.cache_enabled = False
            logger.info("cima.cache.disabled", reason="local_mode")
        else:
            try:
                redis_url = os.getenv("REDIS_URL", "redis://localhost:6379")
                self.redis_client = redis.from_url(redis_url, decode_responses=True)
                self.redis_client.ping()
                self.cache_enabled = True
                logger.info("cima.cache.enabled", cache_type="redis")
            except (redis.RedisError, redis.ConnectionError, Exception) as e:
                self.redis_client = None
                self.cache_enabled = False
                logger.warning("cima.cache.disabled", reason="redis_unavailable", error=str(e))

        # Rate limiting
        self.requests_per_second = 10
        self.last_request_time = 0

    def search_by_name(self, product_name: str, max_results: int = 100) -> List[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        logger.warning("[CIMA] search_by_name no implementado, usar fetch_all_presentations_batch")
        return []

    def get_by_registration_number(self, nregistro: str) -> Optional[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        return None

    def get_by_national_code(self, national_code: str) -> Optional[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        return None

    def search_by_active_ingredient(self, active_ingredient: str, max_results: int = 50) -> List[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        return []

    def get_generic_alternatives(self, nregistro: str) -> List[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        return []

    def get_atc_medications(self, atc_code: str, max_results: int = 100) -> List[Dict[str, Any]]:
        """
        NO IMPLEMENTADO - Solo usamos fetch_all_presentations_batch
        """
        return []

    def _extract_medication_data(self, med) -> Dict[str, Any]:
        """
        Extrae y normaliza datos de un objeto medicamento de la API

        Args:
            med: Objeto medicamento de aempsconn

        Returns:
            Diccionario con datos normalizados del medicamento
        """
        try:
            data = {
                "source": "CIMA",
                "nregistro": getattr(med, "nregistro", None),
                "nombre": getattr(med, "nombre", None),
                "laboratorio": getattr(med, "laboratorio", None),
                "principio_activo": getattr(med, "principio_activo", None),
                "atc_code": getattr(med, "atc", None),
                "forma_farmaceutica": getattr(med, "forma_farmaceutica", None),
                "dosis": getattr(med, "dosis", None),
                "estado": getattr(med, "estado", None),
                "fecha_autorizacion": getattr(med, "fecha_autorizacion", None),
                "prescripcion_necesaria": self._requires_prescription(med),
                "es_generico": self._is_generic_medication_raw(med),
                "updated_at": utc_now(),
            }

            # Limpiar campos None o vacíos
            return {k: v for k, v in data.items() if v is not None and v != ""}

        except Exception as e:
            logger.error(f"Error extracting medication data: {e}")
            return {"source": "CIMA", "error": str(e)}

    def _extract_medication_data_full(self, med, national_code: str) -> Dict[str, Any]:
        """
        Extrae datos completos de un objeto medicamento de la API usando el modelo completo

        Args:
            med: Objeto medicamento de aempsconn
            national_code: Código nacional buscado

        Returns:
            Diccionario con datos normalizados del medicamento
        """
        try:
            # Buscar la presentación que coincida con el CN
            target_presentation = None
            for pres in med.presentaciones:
                if pres.cn == national_code:
                    target_presentation = pres
                    break

            # Extraer código ATC principal
            atc_code = None
            therapeutic_group = None
            if med.atcs and len(med.atcs) > 0:
                atc_code = med.atcs[0].codigo  # Primer código ATC
                therapeutic_group = med.atcs[0].nombre

            data = {
                "source": "CIMA",
                "nregistro": str(med.nregistro),
                "nombre": med.nombre,
                "laboratorio": med.labtitular,
                "principio_activo": med.pactivos,
                "atc_code": atc_code,
                "therapeutic_group": therapeutic_group,
                "forma_farmaceutica": (med.formaFarmaceutica.nombre if med.formaFarmaceutica else None),
                "dosis": med.dosis,
                "estado": (str(med.estado.aut) if med.estado and med.estado.aut else None),
                "prescripcion_necesaria": med.receta,
                "es_generico": self._is_generic_medication_from_full_model(med),
                "comercializado": med.comerc,
                "updated_at": utc_now(),
                # Información específica de la presentación
                "codigo_nacional": national_code,
                "presentacion_nombre": (target_presentation.nombre if target_presentation else med.nombre),
                "presentacion_comercializado": (target_presentation.comerc if target_presentation else med.comerc),
            }

            # Limpiar campos None o vacíos
            return {k: v for k, v in data.items() if v is not None and v != ""}

        except Exception as e:
            logger.error(f"Error extracting full medication data: {e}")
            return {"source": "CIMA", "error": str(e)}

    def _normalize_search_term(self, term: str) -> str:
        """Normaliza términos de búsqueda"""
        if not term:
            return ""
        return term.strip().lower()

    def _requires_prescription(self, med) -> Optional[bool]:
        """Determina si el medicamento requiere prescripción"""
        try:
            # Buscar indicadores de prescripción en diferentes campos
            nombre = str(getattr(med, "nombre", "")).upper()
            estado = str(getattr(med, "estado", "")).upper()

            # Patrones que indican prescripción
            prescription_indicators = ["R1", "R2", "RECETA", "EFG"]
            for indicator in prescription_indicators:
                if indicator in nombre or indicator in estado:
                    return True

            return None  # No se puede determinar

        except Exception:
            return None

    def _is_generic_medication_raw(self, med) -> Optional[bool]:
        """Determina si es medicamento genérico desde objeto raw"""
        try:
            nombre = str(getattr(med, "nombre", "")).upper()

            # Indicadores de genérico
            generic_indicators = ["EFG", "GENERICO", "GENERIC"]
            for indicator in generic_indicators:
                if indicator in nombre:
                    return True

            return False

        except Exception:
            return None

    def _is_generic_medication(self, med_data: Dict[str, Any]) -> bool:
        """Determina si es medicamento genérico desde datos procesados"""
        nombre = med_data.get("nombre", "").upper()
        generic_indicators = ["EFG", "GENERICO", "GENERIC"]

        for indicator in generic_indicators:
            if indicator in nombre:
                return True

        return False

    def _is_generic_medication_from_full_model(self, med) -> Optional[bool]:
        """Determina si es medicamento genérico desde el modelo completo"""
        try:
            nombre = str(getattr(med, "nombre", "")).upper()

            # Indicadores de genérico
            generic_indicators = ["EFG", "GENERICO", "GENERIC"]
            for indicator in generic_indicators:
                if indicator in nombre:
                    return True

            return False

        except Exception:
            return None

    def _contains_active_ingredient(self, med_data: Dict[str, Any], active_ingredient: str) -> bool:
        """Verifica si el medicamento contiene el principio activo especificado"""
        try:
            pa = med_data.get("principio_activo", "").upper()
            nombre = med_data.get("nombre", "").upper()

            active_upper = active_ingredient.upper()

            return active_upper in pa or active_upper in nombre

        except Exception:
            return False

    def health_check(self) -> Dict[str, Any]:
        """
        Verifica el estado de la conexión con CIMA API

        Returns:
            Diccionario con estado de salud del servicio
        """
        try:
            import requests

            # Hacer una petición simple para verificar conectividad
            response = requests.get(
                "https://cima.aemps.es/cima/rest/presentaciones",
                params={"tamañoPagina": 1, "pagina": 1},
                timeout=5,
            )

            return {
                "service": "CIMA",
                "status": "healthy" if response.status_code == 200 else "degraded",
                "api_accessible": True,
                "last_check": utc_now(),
                "status_code": response.status_code,
            }

        except Exception as e:
            return {
                "service": "CIMA",
                "status": "unhealthy",
                "api_accessible": False,
                "last_check": utc_now(),
                "error": str(e),
            }

    def fetch_all_presentations_batch(self) -> List[Dict[str, Any]]:
        """
        DEPRECATED: Este método carga todo en memoria.
        Usar fetch_presentations_streaming() para mejor eficiencia.

        Mantiene compatibilidad hacia atrás recolectando todo desde el generador.
        """
        logger.warning("[CIMA] ⚠️  fetch_all_presentations_batch está deprecado. Usar fetch_presentations_streaming()")
        all_presentations = []

        # Recolectar todo desde el generador para mantener compatibilidad
        for batch in self.fetch_presentations_streaming():
            all_presentations.extend(batch)

        return all_presentations

    async def fetch_presentations_streaming(
        self, batch_size: int = 200, start_page: int = 1
    ) -> AsyncIterator[Tuple[List[Dict[str, Any]], int]]:
        """
        Generador asíncrono que descarga presentaciones de CIMA en lotes.
        Más eficiente en memoria que fetch_all_presentations_batch.

        IMPORTANTE: Usa httpx.AsyncClient para no bloquear el event loop de FastAPI.

        Args:
            batch_size: Tamaño del lote (máximo 200 por límite de API)
            start_page: Página desde donde iniciar (para reanudar después de checkpoint)

        Yields:
            Tuple[List[Dict], int]: Lote de presentaciones y número de página actual
        """
        # Marcar inicio de sync en métricas (Issue #196 - Item #6)
        operation = "streaming"
        sync_timer = None
        if METRICS_AVAILABLE:
            cima_sync_active.labels(operation=operation).set(1)
            sync_timer = cima_sync_duration_seconds.labels(operation=operation).time()
            sync_timer.__enter__()  # Inicializar el timer correctamente

        logger.info(
            f"[CIMA] Iniciando descarga streaming ASYNC de presentaciones vía API REST desde página {start_page}"
        )

        # Validar batch_size
        if batch_size > 200:
            logger.warning(f"[CIMA] batch_size {batch_size} excede límite de API (200). Ajustando a 200.")
            batch_size = 200

        page_num = start_page
        total_downloaded = 0

        # Límite configurable según entorno
        environment = os.getenv("ENVIRONMENT", "development")
        dev_limit = None

        # Límite de desarrollo eliminado - descargar todos los productos
        # Para reactivar el límite, establecer CIMA_DEV_LIMIT en el .env
        if environment == "development" and os.getenv("CIMA_DEV_LIMIT"):
            dev_limit = int(os.getenv("CIMA_DEV_LIMIT"))
            logger.info(f"[CIMA] ⚠️  Modo desarrollo: limitando a {dev_limit} productos")

        # Timeout configurable vía env var (Issue #196 - Item #5)
        timeout = float(os.getenv("CIMA_API_TIMEOUT", "30.0"))

        # Usar httpx.AsyncClient para requests async (DENTRO del generador para prevenir resource leaks)
        async with httpx.AsyncClient(timeout=timeout) as client:
            while True:
                # URL base de la API REST de CIMA para presentaciones
                url = "https://cima.aemps.es/cima/rest/presentaciones"

                # Ajustar batch_size si estamos cerca del límite de desarrollo
                current_batch_size = batch_size
                if dev_limit and total_downloaded + batch_size > dev_limit:
                    current_batch_size = min(batch_size, dev_limit - total_downloaded)
                    if current_batch_size <= 0:
                        logger.info(f"[CIMA] Límite de desarrollo alcanzado: {total_downloaded} productos")
                        break

                params = {"tamañoPagina": current_batch_size, "pagina": page_num}

                # Reintentos con exponential backoff
                data = None

                @exponential_backoff(
                    max_retries=5,
                    initial_delay=2.0,
                    max_delay=60.0,
                    exceptions=(httpx.HTTPError, APIError, RateLimitError),
                )
                async def fetch_page():
                    logger.info(f"[CIMA] Descargando página {page_num} (batch de {current_batch_size} productos)")
                    response = await client.get(url, params=params)

                    # Detectar rate limiting
                    if response.status_code == 429:
                        retry_after = response.headers.get("Retry-After", 60)
                        raise RateLimitError(endpoint=url, retry_after=int(retry_after))

                    # Detectar otros errores de API
                    if response.status_code >= 500:
                        raise APIError(endpoint=url, status_code=response.status_code, response=response.text[:500])

                    response.raise_for_status()
                    return response.json()

                try:
                    data = await fetch_page()
                except RateLimitError as e:
                    if METRICS_AVAILABLE:
                        cima_sync_errors_total.labels(operation=operation, error_type="rate_limit").inc()
                    logger.error(f"[CIMA] Rate limit en página {page_num}: {str(e)}")
                    return  # Terminar el generador
                except APIError as e:
                    if METRICS_AVAILABLE:
                        cima_sync_errors_total.labels(operation=operation, error_type="api_error").inc()
                    logger.error(f"[CIMA] API error en página {page_num}: {str(e)}")
                    return  # Terminar el generador
                except Exception as e:
                    if METRICS_AVAILABLE:
                        cima_sync_errors_total.labels(operation=operation, error_type="unknown").inc()
                    logger.error(f"[CIMA] Error tras reintentos en página {page_num}: {str(e)}")
                    return  # Terminar el generador

                # Procesar respuesta
                if not data or "resultados" not in data:
                    logger.error(
                        f"[CIMA] Formato de respuesta inesperado. Claves: {list(data.keys()) if data else 'None'}"
                    )
                    break

                presentations = data["resultados"]

                if not presentations:
                    logger.info(f"[CIMA] No hay más presentaciones en página {page_num}")
                    break

                # CAPA 1: Importar normalización para prevenir corruption de encoding
                from app.utils.text_normalization import normalize_text

                # Preparar lote para yield
                batch = []
                for i, pres in enumerate(presentations):
                    # Extraer y normalizar campos de texto (prevenir corruption UTF-8)
                    pres_data = {
                        "source": "CIMA",
                        "cn": pres.get("cn"),  # Código Nacional - campo crítico (numérico, no normalizar)
                        "nregistro": pres.get("nregistro"),  # Para obtener ATC después (numérico)
                        "nombre": normalize_text(
                            pres.get("nombre", ""),
                            max_length=500,
                            context="product_name_cima"
                        ),
                        "labtitular": normalize_text(
                            pres.get("labtitular", ""),
                            max_length=255,
                            context="laboratory_cima"
                        ),
                        "pactivos": normalize_text(
                            pres.get("pactivos", ""),
                            max_length=500,
                            context="active_ingredient_cima"
                        ) if pres.get("pactivos") else None,
                        "receta": pres.get("receta"),  # Boolean, no normalizar
                        "comerc": pres.get("comerc"),
                        "updated_at": utc_now(),
                    }

                    # Limpiar campos None o vacíos
                    pres_data = {k: v for k, v in pres_data.items() if v is not None and v != ""}
                    batch.append(pres_data)

                    # Yield control cada 50 items para batches grandes (Issue #196 - Item #8)
                    if i > 0 and i % 50 == 0:
                        await asyncio.sleep(0)  # Yield al event loop sin delay

                total_downloaded += len(batch)

                # Actualizar métricas de items procesados (Issue #196 - Item #6)
                if METRICS_AVAILABLE:
                    cima_sync_items_total.labels(operation=operation, status="success").inc(len(batch))
                    # Actualizar progreso (estimado 67k total)
                    estimated_total = 67000
                    progress = min(100, (total_downloaded / estimated_total) * 100)
                    cima_sync_progress_percent.labels(operation=operation).set(progress)

                # Yield el lote para procesamiento inmediato con número de página para checkpoint
                logger.info(f"[CIMA] Página {page_num}: {len(batch)} productos. Total acumulado: {total_downloaded}")
                yield batch, page_num

                # Si recibimos menos del tamaño solicitado, hemos llegado al final
                if len(presentations) < current_batch_size:
                    logger.info(f"[CIMA] Última página alcanzada (página {page_num})")
                    break

                # Verificar límite de desarrollo (solo si está explícitamente configurado)
                if dev_limit and total_downloaded >= dev_limit:
                    logger.info(
                        f"[CIMA] Límite de desarrollo alcanzado ({total_downloaded}/{dev_limit} presentaciones)"
                    )
                    break

                page_num += 1

                # CRÍTICO: Pausa async para no saturar la API y liberar el event loop
                # Esto permite que FastAPI procese requests del admin panel
                await asyncio.sleep(0.1)  # 100ms - yield control al event loop

        # Log final y métricas
        logger.info(f"[CIMA] ✅ Descarga streaming ASYNC completa: {total_downloaded} presentaciones procesadas")

        # Marcar fin de sync en métricas (Issue #196 - Item #6)
        if METRICS_AVAILABLE:
            if sync_timer:
                sync_timer.__exit__(None, None, None)
            cima_sync_active.labels(operation=operation).set(0)
            cima_sync_progress_percent.labels(operation=operation).set(100)

    def _fetch_all_presentations_batch_legacy(self) -> List[Dict[str, Any]]:
        """
        Versión legacy del método - mantener por si algo lo necesita.
        """
        logger.info("[CIMA] Iniciando descarga batch completa de presentaciones vía API REST")
        all_presentations = []

        try:
            import requests

            # Usar la API REST directa de CIMA para presentaciones
            page_size = 200  # Máximo permitido por la API
            page_num = 1
            total_downloaded = 0
            max_retries = 3

            while True:
                # URL base de la API REST de CIMA para presentaciones
                url = "https://cima.aemps.es/cima/rest/presentaciones"
                params = {"tamañoPagina": page_size, "pagina": page_num}

                # Reintentos para robustez
                for attempt in range(max_retries):
                    try:
                        logger.info(
                            f"[CIMA] Descargando página {page_num} (registros {(page_num-1)*page_size + 1} a {page_num*page_size})"
                        )

                        response = requests.get(url, params=params, timeout=30)
                        response.raise_for_status()

                        data = response.json()
                        break
                    except requests.exceptions.RequestException as e:
                        if attempt < max_retries - 1:
                            logger.warning(f"[CIMA] Error en intento {attempt + 1}, reintentando...")
                            continue
                        else:
                            logger.error(f"[CIMA] Error tras {max_retries} intentos: {str(e)}")
                            return all_presentations

                # Procesar respuesta
                if "resultados" in data:
                    presentations = data["resultados"]

                    if not presentations:
                        logger.info(f"[CIMA] No hay más presentaciones en página {page_num}")
                        break

                    for pres in presentations:
                        # Extraer solo los campos específicos solicitados
                        pres_data = {
                            "source": "CIMA",
                            "cn": pres.get("cn"),  # Código Nacional - campo crítico
                            "nregistro": pres.get("nregistro"),  # Para obtener ATC después
                            "nombre": pres.get("nombre"),
                            "labtitular": pres.get("labtitular"),
                            "pactivos": pres.get("pactivos"),
                            "receta": pres.get("receta"),
                            "comerc": pres.get("comerc"),
                            "updated_at": utc_now(),
                        }

                        # Limpiar campos None o vacíos
                        pres_data = {k: v for k, v in pres_data.items() if v is not None and v != ""}
                        all_presentations.append(pres_data)
                        total_downloaded += 1

                    if total_downloaded % 1000 == 0:
                        logger.info(f"[CIMA] Descargadas {total_downloaded} presentaciones...")

                    # Si no hay más páginas o hemos alcanzado el límite
                    if len(presentations) < page_size:
                        logger.info(f"[CIMA] Última página alcanzada (página {page_num})")
                        break

                    # Límite configurable según entorno
                    import os

                    environment = os.getenv("ENVIRONMENT", "development")

                    # Límite de desarrollo eliminado - descargar todos los productos
                    # Para reactivar el límite, establecer CIMA_DEV_LIMIT en el .env
                    if environment == "development" and os.getenv("CIMA_DEV_LIMIT"):
                        dev_limit = int(os.getenv("CIMA_DEV_LIMIT"))
                        if total_downloaded >= dev_limit:
                            logger.info(
                                f"[CIMA] Límite de desarrollo alcanzado ({total_downloaded}/{dev_limit} presentaciones)"
                            )
                            logger.info(
                                "[CIMA] Para descargar todos los ~67,283 productos, usar ENVIRONMENT=production"
                            )
                            break
                    # En producción, descargar todos los datos sin límite

                    page_num += 1

                else:
                    logger.error(f"[CIMA] Formato de respuesta inesperado. Claves disponibles: {list(data.keys())}")
                    logger.error(f"[CIMA] Respuesta completa: {str(data)[:500]}")
                    break

            # Validar descarga antes de retornar
            if len(all_presentations) == 0:
                logger.error("[CIMA] ¡DESCARGA FALLÓ! No se descargó ninguna presentación")
                return []
            elif len(all_presentations) < 1000:
                logger.warning(f"[CIMA] Descarga sospechosamente pequeña: {len(all_presentations)} presentaciones")

            logger.info(f"[CIMA] Descarga completada exitosamente: {len(all_presentations)} presentaciones")
            return all_presentations

        except Exception as e:
            logger.error(f"[CIMA] Error crítico en descarga batch: {str(e)}")
            import traceback

            logger.error(f"[CIMA] Traceback completo: {traceback.format_exc()}")

            # Si hay presentaciones parciales, las devolvemos con advertencia
            if all_presentations:
                logger.warning(f"[CIMA] Retornando descarga parcial: {len(all_presentations)} presentaciones")
            return all_presentations

    def map_to_product_catalog(self, pres_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Mapea datos de presentación CIMA a campos de product_catalog

        Args:
            pres_data: Datos de la presentación desde CIMA

        Returns:
            Diccionario con campos mapeados para product_catalog
        """
        mapped = {}

        try:
            # === CAMPOS ESPECÍFICOS SOLICITADOS ===

            # Código nacional (identificador principal) - CRÍTICO
            mapped["national_code"] = str(pres_data.get("cn", "")).strip()
            if not mapped["national_code"]:
                # Si no hay CN, no podemos hacer matching
                return {}

            # Nombre completo con presentación
            mapped["cima_nombre_comercial"] = pres_data.get("nombre", "")[:500]

            # Laboratorio titular
            mapped["cima_laboratorio_titular"] = pres_data.get("labtitular", "")[:200]

            # Principios activos (string)
            pactivos = pres_data.get("pactivos", "")
            if pactivos:
                mapped["cima_principios_activos"] = [pactivos]  # Convertir a array para compatibilidad
            else:
                mapped["cima_principios_activos"] = []

            # Requiere prescripción médica
            mapped["cima_requiere_receta"] = bool(pres_data.get("receta", False))

            # Estado de comercialización - se mapea más abajo a cima_estado_registro

            # Número de registro del medicamento (necesario para backfill de ATCs)
            mapped["cima_nregistro"] = pres_data.get("nregistro", None)

            # === CAMPOS OPCIONALES DE CIMA ===
            # Inicializar todos los campos opcionales para evitar errores SQL
            mapped["cima_vmp"] = None
            mapped["cima_vmpp"] = None
            mapped["cima_forma_farmaceutica"] = None
            mapped["cima_via_administracion"] = None
            mapped["cima_dosis"] = None
            mapped["cima_condiciones_prescripcion"] = None
            mapped["cima_grupo_terapeutico"] = None
            mapped["cima_es_generico"] = False  # Default False para booleanos
            mapped["cima_uso_veterinario"] = False

            # Estado de registro basado en comercialización
            comerc = pres_data.get("comerc", False)
            mapped["cima_estado_registro"] = "COMERCIALIZADO" if comerc else "NO_COMERCIALIZADO"

            # Obtener códigos ATC del detalle del medicamento (si disponible)
            # IMPORTANTE: Siempre inicializar los campos para evitar errores de SQL
            mapped["cima_atc_codes"] = []
            mapped["cima_atc_code"] = None  # Usar None en lugar de '' para campos nullable

            # Por ahora, deshabilitamos la búsqueda de ATC para evitar timeouts y errores
            # que están causando problemas en la sincronización masiva
            # TODO: Implementar búsqueda ATC asíncrona o en batch separado
            """
            nregistro = pres_data.get('nregistro')
            if nregistro:
                atc_codes = self._get_atc_codes_from_medication(nregistro)
                if atc_codes:  # Solo actualizar si obtuvimos datos válidos
                    mapped['cima_atc_codes'] = atc_codes
                    # ATC principal (el más específico, nivel 5)
                    for atc in atc_codes:
                        if atc.get('nivel') == 5:
                            mapped['cima_atc_code'] = atc.get('codigo', None)
                            break
                    if not mapped.get('cima_atc_code') and atc_codes:
                        # Si no hay nivel 5, usar el último
                        mapped['cima_atc_code'] = atc_codes[-1].get('codigo', None)
            """

            # Timestamp de actualización
            mapped["updated_at"] = utc_now()

            return mapped

        except Exception as e:
            logger.error(f"[CIMA] Error mapeando datos presentación: {str(e)}")
            return {}

    def _get_atc_codes_from_medication(self, nregistro: str) -> List[Dict[str, Any]]:
        """
        Obtiene códigos ATC del detalle del medicamento

        Args:
            nregistro: Número de registro del medicamento

        Returns:
            Lista de códigos ATC con estructura: [{"codigo": "B01AC06", "nombre": "...", "nivel": 5}]
        """
        if not nregistro:
            return []

        try:
            import requests

            url = "https://cima.aemps.es/cima/rest/medicamento"
            response = requests.get(url, params={"nregistro": nregistro}, timeout=15)

            if response.status_code == 200:
                med_detail = response.json()
                atcs = med_detail.get("atcs", [])

                # Filtrar y limpiar datos ATC
                clean_atcs = []
                for atc in atcs:
                    if isinstance(atc, dict) and atc.get("codigo"):
                        clean_atcs.append(
                            {
                                "codigo": atc.get("codigo", ""),
                                "nombre": atc.get("nombre", ""),
                                "nivel": atc.get("nivel", 0),
                            }
                        )

                return clean_atcs

        except requests.exceptions.Timeout:
            logger.warning(f"[CIMA] Timeout obteniendo ATC para nregistro {nregistro}")
        except requests.exceptions.RequestException as e:
            logger.warning(f"[CIMA] Error de conexión obteniendo ATC para nregistro {nregistro}: {str(e)}")
        except Exception as e:
            logger.warning(f"[CIMA] Error inesperado obteniendo ATC para nregistro {nregistro}: {str(e)}")

        return []

    def _get_therapeutic_group(self, atc_code: str) -> str:
        """
        Obtiene el grupo terapéutico basado en código ATC

        Args:
            atc_code: Código ATC completo

        Returns:
            Descripción del grupo terapéutico
        """
        if not atc_code or len(atc_code) < 1:
            return ""

        # Mapeo de grupos ATC nivel 1
        groups = {
            "A": "TRACTO ALIMENTARIO Y METABOLISMO",
            "B": "SANGRE Y ÓRGANOS HEMATOPOYÉTICOS",
            "C": "SISTEMA CARDIOVASCULAR",
            "D": "DERMATOLÓGICOS",
            "G": "SISTEMA GENITOURINARIO Y HORMONAS SEXUALES",
            "H": "PREPARADOS HORMONALES SISTÉMICOS",
            "J": "ANTIINFECCIOSOS PARA USO SISTÉMICO",
            "L": "AGENTES ANTINEOPLÁSICOS E INMUNOMODULADORES",
            "M": "SISTEMA MUSCULOESQUELÉTICO",
            "N": "SISTEMA NERVIOSO",
            "P": "PRODUCTOS ANTIPARASITARIOS",
            "R": "SISTEMA RESPIRATORIO",
            "S": "ÓRGANOS DE LOS SENTIDOS",
            "V": "VARIOS",
        }

        first_letter = atc_code[0].upper()
        return groups.get(first_letter, "OTROS")[:200]

    def sync_catalog_render_safe(
        self,
        db: Session,
        update_system_status_callback=None,
        max_products: int = 500,
        continue_from_checkpoint: bool = True,
    ) -> Dict[str, Any]:
        """
        Sincronización automática por chunks para Render.
        Procesa chunks de productos con auto-continuación hasta completar TODO CIMA.
        """
        return self.sync_catalog_auto_chunked(db, update_system_status_callback, chunk_size=max_products)

    async def sync_catalog_auto_chunked(
        self, db: Session, update_system_status_callback=None, chunk_size: int = 500
    ) -> Dict[str, Any]:
        """
        Sincronización automática por chunks para Render.
        Procesa chunks pequeños con auto-continuación automática hasta completar TODO CIMA.
        Cada chunk toma 2-3 minutos máximo para no matar workers.

        Args:
            db: Sesión de base de datos
            update_system_status_callback: Función para actualizar el estado del sistema
            chunk_size: Productos por chunk (500 = ~3min, 300 = ~2min)

        Returns:
            Estadísticas de sincronización
        """
        import gc
        import json
        import time

        from sqlalchemy.dialects.postgresql import insert

        from app.models import ProductCatalog, SystemStatus
        from app.models.system_status import SystemComponent, SystemStatusEnum

        stats = {
            "total_processed": 0,
            "created": 0,
            "updated": 0,
            "errors": 0,
            "chunks_completed": 0,
            "chunk_size": chunk_size,
        }

        # Detección de stall: rastrear progreso
        stall_detection = {
            "last_page": None,
            "last_page_time": time.time(),
            "consecutive_same_page": 0,
            "stall_threshold_seconds": 120,  # 2 minutos sin progreso = stall
            "min_consecutive_chunks": 3,  # Mínimo de chunks sin progreso para confirmar stall
        }

        # Tracking de chunks vacíos consecutivos
        stats["consecutive_empty_chunks"] = 0

        try:
            # Verificar checkpoint existente
            start_page = 1
            cima_status = db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()
            if cima_status and cima_status.checkpoint_page and cima_status.checkpoint_page > 0:
                start_page = cima_status.checkpoint_page
                if cima_status.checkpoint_data:
                    try:
                        checkpoint_data = json.loads(cima_status.checkpoint_data)
                        stats["total_processed"] = checkpoint_data.get("total_processed_global", 0)
                        stats["chunks_completed"] = checkpoint_data.get("chunks_completed", 0)
                        logger.info(
                            f"[CIMA AUTO] ✅ Continuando desde página {start_page} (procesados: {stats['total_processed']})"
                        )
                    except (json.JSONDecodeError, KeyError, TypeError) as e:
                        logger.debug(f"Could not restore checkpoint data: {e}")
                        pass

            logger.info(
                f"[CIMA AUTO] 🚀 Iniciando sync auto-chunked desde página {start_page}, chunk_size={chunk_size}"
            )

            if update_system_status_callback:
                update_system_status_callback(
                    "processing",  # status
                    0,  # progress
                    f'Iniciando sync automático por chunks (chunk {stats["chunks_completed"] + 1})',  # message
                    stats,  # stats
                )

            estimated_total = 67000  # Estimado total
            max_execution_time = 180  # 3 minutos máximo por chunk
            chunk_start_time = time.time()

            # Procesar chunks automáticamente
            current_page = start_page
            while True:
                chunk_number = stats["chunks_completed"] + 1
                logger.info(f"[CIMA AUTO] Procesando chunk #{chunk_number} desde página {current_page}")

                chunk_processed = 0
                chunk_start_time = time.time()

                # Procesar UN chunk (ahora async)
                async for batch, page_num in self.fetch_presentations_streaming(batch_size=50, start_page=current_page):
                    batch_size = len(batch)
                    if batch_size == 0:
                        logger.info(f"[CIMA AUTO] ✅ COMPLETADO: No más productos en página {page_num}")
                        # Limpiar checkpoint
                        if cima_status:
                            cima_status.checkpoint_page = None
                            cima_status.checkpoint_data = None
                            db.commit()

                        if update_system_status_callback:
                            update_system_status_callback(
                                "COMPLETED",
                                100,
                                f'✅ CIMA AUTO COMPLETO: {stats["total_processed"]} productos, {stats["chunks_completed"]} chunks',
                                stats,
                            )

                        # Fix: Establecer status='completed' explícitamente para que catalog_maintenance_service
                        # reconozca la finalización exitosa del sync. Sin este campo, el servicio de mantenimiento
                        # no puede distinguir entre un sync exitoso y uno fallido, marcándolo incorrectamente
                        # como error. Este campo se verifica en catalog_maintenance_service.py al evaluar el
                        # resultado del proceso de sincronización CIMA.
                        stats["status"] = "completed"
                        return stats

                    # Procesar batch en micro-chunks
                    MICRO_CHUNK = 10
                    for i in range(0, batch_size, MICRO_CHUNK):
                        micro_batch = batch[i : i + MICRO_CHUNK]

                        try:
                            products_to_upsert = []
                            for pres in micro_batch:
                                mapped = self.map_to_product_catalog(pres)
                                if mapped and mapped.get("national_code"):
                                    mapped["id"] = uuid.uuid4()
                                    mapped["data_sources"] = "cima"
                                    mapped["sync_status"] = "SINCRONIZADO"
                                    mapped["updated_at"] = utc_now()
                                    products_to_upsert.append(mapped)

                            if products_to_upsert:
                                # UPSERT rápido - Issue #330: Agregar TODOS los campos CIMA
                                stmt = insert(ProductCatalog).values(products_to_upsert)
                                stmt = stmt.on_conflict_do_update(
                                    index_elements=["national_code"],
                                    set_={
                                        "cima_nombre_comercial": stmt.excluded.cima_nombre_comercial,
                                        "cima_laboratorio_titular": stmt.excluded.cima_laboratorio_titular,
                                        "cima_estado_registro": stmt.excluded.cima_estado_registro,
                                        "cima_requiere_receta": stmt.excluded.cima_requiere_receta,  # ← BUG FIX
                                        "cima_principios_activos": stmt.excluded.cima_principios_activos,
                                        "cima_es_generico": stmt.excluded.cima_es_generico,
                                        "cima_atc_code": stmt.excluded.cima_atc_code,
                                        "cima_atc_codes": stmt.excluded.cima_atc_codes,
                                        "data_sources": stmt.excluded.data_sources,
                                        "sync_status": stmt.excluded.sync_status,
                                        "updated_at": stmt.excluded.updated_at,
                                    },
                                )

                                db.execute(stmt)
                                db.commit()

                                stats["total_processed"] += len(products_to_upsert)
                                chunk_processed += len(products_to_upsert)

                            # Pausa mínima para no sobrecargar
                            time.sleep(0.05)  # 50ms

                        except Exception as e:
                            logger.error(f"[CIMA AUTO] Error en micro-chunk: {str(e)}")
                            db.rollback()
                            stats["errors"] += len(micro_batch)

                    current_page = page_num

                    # CONTROL DE TIEMPO: Si llevamos mucho tiempo, parar este chunk
                    if time.time() - chunk_start_time > max_execution_time:
                        logger.info(
                            f"[CIMA AUTO] ⏰ Tiempo límite alcanzado para chunk #{chunk_number} ({max_execution_time}s)"
                        )
                        break

                    # CONTROL DE TAMAÑO: Si ya procesamos suficientes productos en este chunk
                    if chunk_processed >= chunk_size:
                        logger.info(f"[CIMA AUTO] ✅ Chunk #{chunk_number} completado: {chunk_processed} productos")
                        break

                # Finalizar este chunk
                stats["chunks_completed"] += 1
                progress = min(int((stats["total_processed"] / estimated_total) * 100), 100)

                # Guardar checkpoint para el próximo chunk
                try:
                    if cima_status:
                        cima_status.checkpoint_page = current_page + 1
                        cima_status.checkpoint_data = json.dumps(
                            {
                                "total_processed_global": stats["total_processed"],
                                "chunks_completed": stats["chunks_completed"],
                                "last_page": current_page,
                                "progress": progress,
                            }
                        )
                        db.commit()
                        logger.info(
                            f"[CIMA AUTO] 💾 Checkpoint chunk #{stats['chunks_completed']}: página {current_page + 1}"
                        )
                except Exception as e:
                    logger.warning(f"[CIMA AUTO] Error checkpoint: {e}")

                # DETECCIÓN DE STALL: Verificar si estamos atascados en la misma página
                current_time = time.time()
                if stall_detection["last_page"] == current_page:
                    stall_detection["consecutive_same_page"] += 1
                    time_on_same_page = current_time - stall_detection["last_page_time"]

                    # Actualizar métrica de chunks consecutivos (Prometheus)
                    try:
                        from app.metrics.cima_metrics import cima_consecutive_same_page_chunks

                        cima_consecutive_same_page_chunks.labels(page_number=str(current_page)).set(
                            stall_detection["consecutive_same_page"]
                        )
                    except Exception:
                        pass  # No fallar si métricas no están disponibles

                    # Validar chunks consecutivos Y tiempo para evitar falsos positivos
                    if (
                        time_on_same_page > stall_detection["stall_threshold_seconds"]
                        and stall_detection["consecutive_same_page"] >= stall_detection["min_consecutive_chunks"]
                    ):
                        # FIX: Detectar FIN DE DATOS antes de marcar como STALL
                        # Si estamos cerca del total estimado (>95%) y en página alta, es fin de datos
                        estimated_products_at_page = current_page * chunk_size
                        is_near_end = (
                            stats["total_processed"] > estimated_total * 0.95  # >95% procesado
                            or estimated_products_at_page > estimated_total  # Página supera estimado
                            or current_page >= 335  # Página alta conocida como fin de CIMA
                        )

                        if is_near_end:
                            logger.info(
                                f"[CIMA AUTO] ✅ FIN DE DATOS DETECTADO (no STALL): "
                                f"Página {current_page}, procesados {stats['total_processed']}, "
                                f"estimado {estimated_total}"
                            )
                            # Limpiar checkpoint - sync completado
                            if cima_status:
                                cima_status.checkpoint_page = None
                                cima_status.checkpoint_data = None
                                cima_status.status = SystemStatusEnum.READY
                                cima_status.message = f"✅ CIMA sync completado: {stats['total_processed']} productos"
                                cima_status.progress = 100
                                db.commit()

                            if update_system_status_callback:
                                update_system_status_callback(
                                    "COMPLETED",
                                    100,
                                    f"✅ CIMA completado: {stats['total_processed']} productos (fin de datos detectado)",
                                    stats,
                                )

                            stats["status"] = "completed"
                            return stats

                        recovery_start_time = time.time()

                        logger.error(
                            f"[CIMA AUTO] ⚠️ STALL DETECTADO: Página {current_page} sin progreso por {int(time_on_same_page)}s "
                            f"(threshold: {stall_detection['stall_threshold_seconds']}s, "
                            f"chunks consecutivos: {stall_detection['consecutive_same_page']})",
                            consecutive_same_page=stall_detection["consecutive_same_page"],
                            chunks_completed=stats["chunks_completed"],
                        )

                        # Métricas Prometheus: Registrar stall detectado
                        try:
                            from app.metrics.cima_metrics import cima_stalls_total

                            # Determinar rango de página para métrica
                            if current_page <= 100:
                                page_range = "1-100"
                            elif current_page <= 200:
                                page_range = "101-200"
                            elif current_page <= 300:
                                page_range = "201-300"
                            elif current_page <= 400:
                                page_range = "301-400"
                            else:
                                page_range = "400+"
                            cima_stalls_total.labels(page_range=page_range).inc()
                        except Exception:
                            pass

                        # Persistir evento de stall para análisis
                        recovery_success = False
                        try:
                            from app.models.cima_stall_event import CIMAStallEvent

                            stall_event = CIMAStallEvent(
                                page_number=current_page,
                                stall_duration=int(time_on_same_page),
                                chunks_attempted=stall_detection["consecutive_same_page"],
                            )
                            db.add(stall_event)
                            db.commit()
                            logger.info("[CIMA AUTO] 📝 Evento de stall registrado en BD")
                            recovery_success = True
                        except Exception as e:
                            logger.warning(f"[CIMA AUTO] No se pudo registrar evento de stall: {e}")

                        # Retroceso parcial: volver 5 páginas atrás en lugar de resetear completamente
                        rollback_page = max(1, current_page - 5)
                        if cima_status:
                            cima_status.checkpoint_page = rollback_page
                            cima_status.checkpoint_data = json.dumps(
                                {
                                    "total_processed_global": stats["total_processed"],
                                    "chunks_completed": stats["chunks_completed"],
                                    "stall_recovery": True,
                                    "original_stall_page": current_page,
                                }
                            )
                            db.commit()

                        logger.info(
                            f"[CIMA AUTO] 🔄 Retroceso parcial: de página {current_page} a página {rollback_page}"
                        )

                        # Métricas Prometheus: Registrar recovery y duración
                        try:
                            from app.metrics.cima_metrics import (
                                cima_recovery_attempts_total,
                                cima_recovery_duration_seconds,
                            )

                            cima_recovery_attempts_total.labels(success=str(recovery_success).lower()).inc()
                            recovery_duration = time.time() - recovery_start_time
                            cima_recovery_duration_seconds.observe(recovery_duration)
                        except Exception:
                            pass

                        if update_system_status_callback:
                            update_system_status_callback(
                                "ERROR",
                                progress,
                                f"⚠️ Stall detectado en página {current_page}. Retrocediendo a página {rollback_page}...",
                                stats,
                            )

                        return stats
                else:
                    # Progreso detectado, resetear contador de stall
                    stall_detection["last_page"] = current_page
                    stall_detection["last_page_time"] = current_time
                    stall_detection["consecutive_same_page"] = 0

                    # Actualizar heartbeat age (Prometheus)
                    try:
                        from app.metrics.cima_metrics import cima_heartbeat_age_seconds

                        cima_heartbeat_age_seconds.set(0)  # Heartbeat reciente
                    except Exception:
                        pass

                if update_system_status_callback:
                    update_system_status_callback(
                        "processing",  # status
                        progress,  # progress
                        f'Chunk #{stats["chunks_completed"]} completado. Total: {stats["total_processed"]}/~{estimated_total}',  # message
                        stats,  # stats
                    )

                # Limpiar memoria entre chunks
                gc.collect()

                # PAUSA ENTRE CHUNKS para dar respiro al worker
                time.sleep(2.0)  # 2 segundos entre chunks

                # Si no hay más productos, terminar (requiere 3 chunks vacíos consecutivos)
                if chunk_processed == 0:
                    stats["consecutive_empty_chunks"] += 1
                    logger.info(
                        f"[CIMA AUTO] Chunk vacío detectado " f"({stats['consecutive_empty_chunks']}/3 consecutivos)"
                    )

                    if stats["consecutive_empty_chunks"] >= 3:
                        logger.info("[CIMA AUTO] ✅ Finalización confirmada (3 chunks vacíos consecutivos)")
                        # Limpiar checkpoint
                        if cima_status:
                            cima_status.checkpoint_page = None
                            cima_status.checkpoint_data = None
                            db.commit()
                        break
                else:
                    # Reset contador si hay datos
                    stats["consecutive_empty_chunks"] = 0

            # Fix: Establecer status='completed' al salir del while loop principal. Esta ruta de código
            # se ejecuta cuando el sync finaliza correctamente después de procesar todos los chunks
            # (mediante el break de 3 chunks vacíos consecutivos). Sin este campo, catalog_maintenance_service
            # interpretaría la ausencia de 'status' como un error ("Error desconocido en CIMA chunked"),
            # aunque el sync se completó exitosamente. Este es un escenario de finalización exitosa alternativo
            # a la ruta de batch_size == 0 en línea 1022.
            logger.info(f"[CIMA AUTO] ✅ Proceso terminado después de {stats['chunks_completed']} chunks")
            stats["status"] = "completed"
            return stats

        except Exception as e:
            from app.utils.error_logging import log_structured_error

            error_msg = f"Error en sincronización CIMA auto-chunked: {str(e)}"
            error_type = type(e).__name__

            log_structured_error(logger=logger, exception=e, context_message="[CIMA AUTO] ❌ Error crítico")

            if update_system_status_callback:
                update_system_status_callback(
                    "ERROR", 0, f"{error_msg} ({error_type})", {**stats, "error_type": error_type}
                )

            return {"error": error_msg, "error_type": error_type, **stats}

    async def sync_catalog_full_auto(self, db: Session, update_system_status_callback=None) -> Dict[str, Any]:
        """
        Sincronización COMPLETA de CIMA en una sola ejecución pero con pausas agresivas.
        Procesa TODOS los productos de CIMA (~67,000) de forma segura para Render.
        Usa micro-chunks de 5 productos y pausas largas para no sobrecargar.

        Args:
            db: Sesión de base de datos
            update_system_status_callback: Función para actualizar el estado del sistema

        Returns:
            Estadísticas de sincronización completa
        """
        import json
        import time

        from sqlalchemy.dialects.postgresql import insert

        from app.models import ProductCatalog, SystemStatus
        from app.models.system_status import SystemComponent

        stats = {
            "total_processed": 0,
            "created": 0,
            "updated": 0,
            "errors": 0,
            "batch_size": 25,  # Ultra pequeño para Render
        }

        try:
            # Verificar checkpoint existente para continuar donde se quedó
            start_page = 1
            cima_status = db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()
            if cima_status and cima_status.checkpoint_page and cima_status.checkpoint_page > 0:
                start_page = cima_status.checkpoint_page
                if cima_status.checkpoint_data:
                    try:
                        checkpoint_data = json.loads(cima_status.checkpoint_data)
                        stats["total_processed"] = checkpoint_data.get("total_processed_global", 0)
                        logger.info(
                            f"[CIMA FULL] ✅ CONTINUANDO desde página {start_page} (ya procesados: {stats['total_processed']})"
                        )
                    except (json.JSONDecodeError, KeyError, TypeError) as e:
                        logger.debug(f"Could not restore checkpoint data: {e}")
                        pass

            logger.info(f"[CIMA FULL] 🚀 Iniciando sincronización COMPLETA de CIMA desde página {start_page}")

            if update_system_status_callback:
                update_system_status_callback(
                    "processing",  # status
                    0,  # progress
                    f"Iniciando sync COMPLETO de CIMA (desde página {start_page})",  # message
                    stats,  # stats
                )

            batch_num = 0
            current_page = start_page
            estimated_total = 67000  # Estimado total de productos CIMA

            # Procesar TODOS los productos con lotes muy pequeños (async)
            async for batch, page_num in self.fetch_presentations_streaming(batch_size=25, start_page=start_page):
                batch_num += 1
                batch_size = len(batch)

                if batch_size == 0:
                    logger.info(f"[CIMA FULL] ✅ Finalizado: no más productos en página {page_num}")
                    break

                logger.info(f"[CIMA FULL] Procesando lote {batch_num} (página {page_num}) con {batch_size} productos")

                # Procesar en micro-chunks ultra-pequeños
                MICRO_CHUNK = 5  # Solo 5 productos por transacción DB
                for i in range(0, batch_size, MICRO_CHUNK):
                    micro_batch = batch[i : i + MICRO_CHUNK]

                    try:
                        products_to_upsert = []

                        for pres in micro_batch:
                            mapped = self.map_to_product_catalog(pres)
                            if mapped and mapped.get("national_code"):
                                mapped["id"] = uuid.uuid4()
                                mapped["data_sources"] = "cima"
                                mapped["sync_status"] = "SINCRONIZADO"
                                mapped["updated_at"] = utc_now()
                                products_to_upsert.append(mapped)

                        if products_to_upsert:
                            # UPSERT en PostgreSQL - Issue #330: Agregar TODOS los campos CIMA
                            stmt = insert(ProductCatalog).values(products_to_upsert)
                            stmt = stmt.on_conflict_do_update(
                                index_elements=["national_code"],
                                set_={
                                    "cima_nombre_comercial": stmt.excluded.cima_nombre_comercial,
                                    "cima_laboratorio_titular": stmt.excluded.cima_laboratorio_titular,
                                    "cima_estado_registro": stmt.excluded.cima_estado_registro,
                                    "cima_requiere_receta": stmt.excluded.cima_requiere_receta,  # ← BUG FIX
                                    "cima_principios_activos": stmt.excluded.cima_principios_activos,
                                    "cima_es_generico": stmt.excluded.cima_es_generico,
                                    "cima_atc_code": stmt.excluded.cima_atc_code,
                                    "cima_atc_codes": stmt.excluded.cima_atc_codes,
                                    "data_sources": stmt.excluded.data_sources,
                                    "sync_status": stmt.excluded.sync_status,
                                    "updated_at": stmt.excluded.updated_at,
                                },
                            )

                            db.execute(stmt)
                            db.commit()

                            stats["total_processed"] += len(products_to_upsert)

                        # PAUSA AGRESIVA para no sobrecargar Render
                        time.sleep(0.2)  # 200ms entre cada micro-chunk

                    except Exception as e:
                        logger.error(f"[CIMA FULL] Error en micro-chunk {i//MICRO_CHUNK + 1}: {str(e)}")
                        db.rollback()
                        stats["errors"] += len(micro_batch)

                # Actualizar progreso
                progress = min(int((stats["total_processed"] / estimated_total) * 100), 100)

                # Guardar checkpoint cada 3 lotes (más frecuente)
                if batch_num % 3 == 0:
                    try:
                        if cima_status:
                            cima_status.checkpoint_page = page_num + 1
                            cima_status.checkpoint_data = json.dumps(
                                {
                                    "total_processed_global": stats["total_processed"],
                                    "last_page": page_num,
                                    "last_batch": batch_num,
                                    "progress": progress,
                                }
                            )
                            db.commit()
                            logger.info(
                                f"[CIMA FULL] 💾 Checkpoint: página {page_num + 1}, procesados: {stats['total_processed']}"
                            )
                    except Exception as e:
                        logger.warning(f"[CIMA FULL] Error checkpoint: {e}")

                if update_system_status_callback and batch_num % 10 == 0:  # Actualizar menos frecuentemente
                    update_system_status_callback(
                        "PROCESSING",
                        progress,
                        f'Procesados {stats["total_processed"]} de ~{estimated_total} productos (página {page_num})',
                        stats,
                    )

                # PAUSA LARGA entre lotes
                time.sleep(1.0)  # 1 segundo entre lotes

            # Limpiar checkpoints al completar
            try:
                if cima_status:
                    cima_status.checkpoint_page = None
                    cima_status.checkpoint_data = None
                    db.commit()
            except Exception as e:
                logger.debug(f"Could not clear checkpoint data: {e}")
                pass

            if update_system_status_callback:
                update_system_status_callback(
                    "COMPLETED",
                    100,
                    f'✅ Sync CIMA COMPLETO: {stats["total_processed"]} productos sincronizados',
                    stats,
                )

            logger.info(f"[CIMA FULL] ✅ SYNC COMPLETO: {stats['total_processed']} productos procesados")
            return stats

        except Exception as e:
            logger.error(f"[CIMA RENDER] ❌ Error en sync render-safe: {str(e)}")
            if update_system_status_callback:
                update_system_status_callback("ERROR", 0, f"Error en sync render-safe: {str(e)}", stats)
            return {"error": str(e), **stats}

    def update_product_catalog_from_cima(self, db: Session, force_update: bool = False) -> Dict[str, Any]:
        """
        Actualiza product_catalog con datos de CIMA

        Args:
            db: Sesión de base de datos
            force_update: Forzar actualización aunque no hayan pasado 10 días

        Returns:
            Estadísticas del proceso
        """
        from app.models.product_catalog import ProductCatalog

        stats = {
            "started_at": utc_now(),
            "total_processed": 0,
            "created": 0,
            "updated": 0,
            "errors": 0,
            "skipped": 0,
        }

        try:
            # Verificar última actualización (informativo, no bloquea)
            last_update = (
                db.query(func.max(ProductCatalog.updated_at))
                .filter(ProductCatalog.data_sources.like("%cima%"))
                .scalar()
            )

            if last_update:
                days_since_update = (utc_now() - last_update).days
                hours_since_update = (utc_now() - last_update).total_seconds() / 3600

                if hours_since_update < 1:
                    # Solo si fue actualizado hace menos de 1 hora, preguntar
                    logger.info(f"[CIMA] Última actualización hace {hours_since_update:.1f} horas")
                    if not force_update:
                        logger.warning("[CIMA] Actualización muy reciente. Use force=true para forzar")
                        stats["status"] = "recently_updated"
                        stats["message"] = f"CIMA fue actualizado hace {hours_since_update:.1f} horas"
                        stats["last_update"] = last_update.isoformat()
                        stats["hours_since_update"] = hours_since_update
                        return stats
                else:
                    logger.info(f"[CIMA] Última actualización hace {days_since_update} días. Procediendo...")

            logger.info("[CIMA] Iniciando actualización de product_catalog desde CIMA")

            # Descargar todas las presentaciones (contienen códigos nacionales)
            presentations = self.fetch_all_presentations_batch()

            if not presentations:
                logger.error("[CIMA] No se pudieron descargar presentaciones")
                stats["status"] = "error"
                stats["message"] = "No se pudieron descargar datos de CIMA"
                return stats

            logger.info(f"[CIMA] Procesando {len(presentations)} presentaciones")

            # BULK OPERATIONS: Procesar todos los productos primero
            logger.info(f"[CIMA] Preparando {len(presentations)} productos para BULK UPSERT...")

            # Mapear todos los productos válidos
            products_to_upsert = []
            for pres in presentations:
                try:
                    # Mapear datos de presentación
                    mapped_data = self.map_to_product_catalog(pres)

                    if not mapped_data.get("national_code"):
                        stats["skipped"] += 1
                        continue

                    # Agregar campos adicionales necesarios
                    mapped_data["data_sources"] = "cima"
                    mapped_data["sync_status"] = "SINCRONIZADO"
                    mapped_data["updated_at"] = utc_now()

                    products_to_upsert.append(mapped_data)

                except Exception as e:
                    stats["errors"] += 1
                    cn = pres.get("cn", "N/A")
                    nombre = pres.get("nombre", "N/A")[:50]
                    logger.error(f"[CIMA] Error mapeando presentación CN={cn} ({nombre}): {str(e)}")
                    if stats["errors"] <= 3:
                        logger.error(f"[CIMA] Datos de presentación que falló: {pres}")
                    continue

            # BULK UPSERT con PostgreSQL
            from sqlalchemy.dialects.postgresql import insert

            # REDUCIDO A 100 PARA EVITAR BLOQUEOS EN POSTGRESQL
            # Tamaño más pequeño = más iteraciones pero más estable
            CHUNK_SIZE = 100
            total_to_process = len(products_to_upsert)

            logger.info(f"[CIMA] Iniciando BULK UPSERT de {total_to_process} productos en chunks de {CHUNK_SIZE}...")

            for i in range(0, total_to_process, CHUNK_SIZE):
                chunk = products_to_upsert[i : i + CHUNK_SIZE]
                chunk_size = len(chunk)
                chunk_number = (i // CHUNK_SIZE) + 1
                total_chunks = (total_to_process + CHUNK_SIZE - 1) // CHUNK_SIZE

                logger.info(f"[CIMA BULK] Procesando chunk {chunk_number}/{total_chunks} ({chunk_size} productos)...")

                try:
                    # Preparar statement UPSERT
                    stmt = insert(ProductCatalog).values(chunk)

                    # En caso de conflicto, actualizar solo campos CIMA y metadatos
                    update_dict = {}
                    # Lista de campos CIMA válidos según DATA_CATALOG.md
                    valid_cima_fields = {
                        "cima_vmp",
                        "cima_vmpp",
                        "cima_nombre_comercial",
                        "cima_atc_codes",
                        "cima_atc_code",
                        "cima_grupo_terapeutico",
                        "cima_principios_activos",
                        "cima_forma_farmaceutica",
                        "cima_via_administracion",
                        "cima_dosis",
                        "cima_laboratorio_titular",
                        "cima_condiciones_prescripcion",
                        "cima_estado_registro",
                        "cima_requiere_receta",
                        "cima_es_generico",
                        "cima_uso_veterinario",
                    }

                    # Obtener todos los campos del primer elemento del chunk para saber qué actualizar
                    if chunk:
                        for col_name in chunk[0].keys():
                            # Actualizar solo campos CIMA válidos y algunos metadatos
                            if col_name in valid_cima_fields or col_name in [
                                "updated_at",
                                "sync_status",
                            ]:
                                update_dict[col_name] = stmt.excluded[col_name]
                            # Para data_sources, mantener fuentes existentes
                            elif col_name == "data_sources":
                                # Simplemente concatenar si no existe 'cima'
                                update_dict[col_name] = case(
                                    (ProductCatalog.data_sources.is_(None), "cima"),
                                    (
                                        ProductCatalog.data_sources.like("%cima%"),
                                        ProductCatalog.data_sources,
                                    ),
                                    else_=func.concat(ProductCatalog.data_sources, ",cima"),
                                )

                    # Ejecutar UPSERT masivo
                    stmt = stmt.on_conflict_do_update(index_elements=["national_code"], set_=update_dict)

                    # Ejecutar la operación bulk con manejo de errores mejorado
                    try:
                        # Añadir un tiempo de espera en la query
                        db.execute(text("SET statement_timeout = '30s'"))
                        db.execute(stmt)
                        db.commit()
                        # Resetear el timeout
                        db.execute(text("RESET statement_timeout"))
                    except Exception as e:
                        logger.error(f"[CIMA BULK] Error en chunk {chunk_number}: {str(e)[:200]}")
                        db.rollback()
                        stats["errors"] += chunk_size
                        # Continuar con el siguiente chunk
                        continue

                    stats["total_processed"] += chunk_size

                    # Log de progreso detallado
                    progress_pct = (stats["total_processed"] / total_to_process) * 100
                    logger.info(
                        f"[CIMA BULK] Procesados {stats['total_processed']}/{total_to_process} medicamentos ({progress_pct:.1f}%) - Chunk {chunk_number}/{total_chunks} completado"
                    )

                except Exception as e:
                    stats["errors"] += chunk_size
                    logger.error(f"[CIMA BULK] Error procesando chunk {i//CHUNK_SIZE + 1}: {str(e)}")
                    db.rollback()

                    # Fallback: procesar individualmente este chunk
                    for product_data in chunk:
                        try:
                            national_code = product_data["national_code"]
                            existing = (
                                db.query(ProductCatalog).filter(ProductCatalog.national_code == national_code).first()
                            )

                            if existing:
                                # Actualizar campos CIMA
                                for key, value in product_data.items():
                                    if key.startswith("cima_") or key in [
                                        "updated_at",
                                        "sync_status",
                                    ]:
                                        setattr(existing, key, value)

                                # Actualizar fuentes de datos
                                sources = existing.data_sources or ""
                                if "cima" not in sources:
                                    sources = "cima" if not sources else sources + ",cima"
                                    existing.data_sources = sources
                                stats["updated"] += 1
                            else:
                                # Crear nuevo producto
                                new_product = ProductCatalog(**product_data)
                                db.add(new_product)
                                stats["created"] += 1

                            db.commit()
                            stats["errors"] -= 1  # Reducir error si se procesa exitosamente

                        except Exception as inner_e:
                            logger.warning(
                                f"[CIMA] Error procesando producto individual {product_data.get('national_code', 'unknown')}: {str(inner_e)}"
                            )
                            db.rollback()

            stats["completed_at"] = utc_now()
            stats["duration_seconds"] = (stats["completed_at"] - stats["started_at"]).total_seconds()
            stats["status"] = "success"

            # Validar resultados y reportar problemas
            total_attempts = stats["total_processed"] + stats["skipped"] + stats["errors"]
            if total_attempts != len(presentations):
                logger.warning(
                    f"[CIMA] Discrepancia en conteo: intentos={total_attempts}, presentaciones={len(presentations)}"
                )

            # Advertir si hay demasiados errores
            if stats["errors"] > len(presentations) * 0.1:  # Más del 10% de errores
                logger.warning(
                    f"[CIMA] Alto número de errores: {stats['errors']}/{len(presentations)} ({stats['errors']/len(presentations)*100:.1f}%)"
                )
                stats["high_error_rate"] = True

            # Advertir si no se procesó nada
            if stats["total_processed"] == 0:
                logger.error(
                    "[CIMA] ¡NINGÚN producto procesado exitosamente! Verificar conectividad y formato de datos"
                )
                stats["status"] = "failed"
                stats["message"] = "No se procesó ningún producto"

            logger.info(
                f"[CIMA] Actualización completada: procesados={stats['total_processed']}, errores={stats['errors']}, omitidos={stats['skipped']}"
            )

        except Exception as e:
            logger.error(f"[CIMA] Error crítico en actualización: {str(e)}")
            db.rollback()
            stats["status"] = "error"
            stats["message"] = str(e)
            stats["critical_failure"] = True

        return stats

    def is_cima_data_recent(self, db: Session, max_days: int = 10) -> bool:
        """
        Verifica si los datos de CIMA están actualizados

        Args:
            db: Sesión de base de datos
            max_days: Días máximos para considerar datos frescos

        Returns:
            True si los datos son recientes
        """
        from app.models.product_catalog import ProductCatalog

        try:
            # Buscar última actualización de datos CIMA
            last_update = (
                db.query(func.max(ProductCatalog.updated_at))
                .filter(ProductCatalog.data_sources.like("%cima%"))
                .scalar()
            )

            if not last_update:
                return False

            days_old = (utc_now() - last_update).days
            return days_old <= max_days

        except Exception as e:
            logger.error(f"[CIMA] Error verificando frescura de datos: {str(e)}")
            return False

    async def sync_catalog_streaming(self, db: Session, update_system_status_callback=None) -> Dict[str, Any]:
        """
        Sincroniza el catálogo con datos de CIMA usando procesamiento por streaming.
        Más eficiente que el método original que carga todo en memoria.
        Soporta checkpoints para reanudar después de interrupciones.

        Args:
            db: Sesión de base de datos
            update_system_status_callback: Función para actualizar el estado del sistema

        Returns:
            Estadísticas de sincronización
        """
        import json

        from sqlalchemy import case, func, text
        from sqlalchemy.dialects.postgresql import insert

        from app.models import ProductCatalog, SystemStatus
        from app.models.system_status import SystemComponent

        stats = {
            "total_processed": 0,
            "created": 0,
            "updated": 0,
            "errors": 0,
            "batch_size": 200,
        }

        try:
            logger.info("[CIMA] === INICIANDO SINCRONIZACIÓN STREAMING CON CATÁLOGO ===")

            # Buscar checkpoint existente
            cima_status = db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()
            start_page = 1

            if cima_status and cima_status.checkpoint_page and cima_status.checkpoint_page > 0:
                # Reanudar desde checkpoint
                start_page = cima_status.checkpoint_page
                logger.info(f"[CIMA] ✅ REANUDANDO desde checkpoint en página {start_page}")

                # Restaurar estadísticas del checkpoint si existen
                if cima_status.checkpoint_data:
                    try:
                        checkpoint_data = json.loads(cima_status.checkpoint_data)
                        stats = checkpoint_data.get("stats", stats)
                        logger.info(
                            f"[CIMA] Estadísticas restauradas: {stats['total_processed']} productos procesados anteriormente"
                        )
                    except Exception as e:
                        logger.warning(f"[CIMA] No se pudieron restaurar estadísticas del checkpoint: {e}")

            # Actualizar estado inicial si hay callback
            if update_system_status_callback:
                message = "Iniciando sincronización streaming con CIMA..."
                if start_page > 1:
                    message = f"Reanudando sincronización desde página {start_page}..."
                update_system_status_callback("INITIALIZING", 0, message, stats)

            batch_num = start_page - 1  # Ajustar contador de lotes
            estimated_total = 67000  # Estimado conocido

            # Procesar por lotes usando el generador async
            async for batch, current_page in self.fetch_presentations_streaming(start_page=start_page):
                batch_num += 1
                batch_size = len(batch)

                if batch_size == 0:
                    continue

                logger.info(f"[CIMA STREAMING] Procesando lote {batch_num} con {batch_size} productos")

                # Mapear datos del lote
                products_to_upsert = []
                for pres in batch:
                    try:
                        # Usar el método correcto de mapeo
                        mapped = self.map_to_product_catalog(pres)

                        if mapped and mapped.get("national_code"):
                            # Agregar campos adicionales necesarios
                            mapped["id"] = uuid.uuid4()  # Generar UUID para nueva fila
                            mapped["data_sources"] = "cima"
                            mapped["sync_status"] = "SINCRONIZADO"
                            mapped["updated_at"] = utc_now()

                            # Filtrar campos con valor None y asegurar que solo sean valores simples (no SQL expressions)
                            mapped_filtered = {}
                            for k, v in mapped.items():
                                if v is not None:
                                    # Verificar que el valor sea un tipo simple de Python, no un SQL expression
                                    if isinstance(
                                        v,
                                        (
                                            str,
                                            int,
                                            float,
                                            bool,
                                            list,
                                            dict,
                                            datetime,
                                            uuid.UUID,
                                        ),
                                    ):
                                        # Para strings, no incluir cadenas vacías en campos opcionales
                                        if (
                                            isinstance(v, str)
                                            and v.strip() == ""
                                            and k.startswith("cima_")
                                            and k
                                            not in [
                                                "national_code",
                                                "data_sources",
                                                "sync_status",
                                            ]
                                        ):
                                            continue  # Omitir strings vacíos en campos opcionales
                                        mapped_filtered[k] = v
                                    else:
                                        # Log si encontramos algo inesperado, pero no incluirlo
                                        logger.warning(f"[CIMA] Ignorando campo {k} con tipo no esperado: {type(v)}")

                            products_to_upsert.append(mapped_filtered)
                        else:
                            stats["errors"] += 1
                    except Exception as e:
                        stats["errors"] += 1
                        cn = pres.get("cn", "N/A")
                        logger.error(f"[CIMA] Error mapeando presentación CN={cn}: {str(e)}")
                        continue

                # Si no hay productos válidos en este lote, continuar
                if not products_to_upsert:
                    logger.warning(f"[CIMA STREAMING] Lote {batch_num} sin productos válidos")
                    continue

                # BULK UPSERT del lote
                try:
                    # Preparar statement UPSERT
                    stmt = insert(ProductCatalog).values(products_to_upsert)

                    # En caso de conflicto, actualizar solo campos CIMA y metadatos
                    update_dict = {}
                    valid_cima_fields = {
                        "cima_vmp",
                        "cima_vmpp",
                        "cima_nombre_comercial",
                        "cima_atc_codes",
                        "cima_atc_code",
                        "cima_grupo_terapeutico",
                        "cima_principios_activos",
                        "cima_forma_farmaceutica",
                        "cima_via_administracion",
                        "cima_dosis",
                        "cima_laboratorio_titular",
                        "cima_condiciones_prescripcion",
                        "cima_estado_registro",
                        "cima_requiere_receta",
                        "cima_es_generico",
                        "cima_uso_veterinario",
                    }

                    # Obtener campos del primer elemento para saber qué actualizar
                    # IMPORTANTE: Solo incluir campos que NO sean None para evitar errores SQL
                    if products_to_upsert:
                        for col_name, col_value in products_to_upsert[0].items():
                            # Manejo especial para data_sources (requiere SQL expression)
                            if col_name == "data_sources":
                                # Mantener fuentes existentes y agregar CIMA si no está
                                update_dict[col_name] = case(
                                    (ProductCatalog.data_sources.is_(None), "cima"),
                                    (
                                        ProductCatalog.data_sources.like("%cima%"),
                                        ProductCatalog.data_sources,
                                    ),
                                    else_=func.concat(ProductCatalog.data_sources, ",cima"),
                                )
                            # Solo incluir campos con valores no-None y que sean campos válidos
                            elif col_value is not None and (
                                col_name in valid_cima_fields or col_name in ["updated_at", "sync_status"]
                            ):
                                update_dict[col_name] = stmt.excluded[col_name]

                    # Ejecutar UPSERT con timeout
                    stmt = stmt.on_conflict_do_update(index_elements=["national_code"], set_=update_dict)

                    db.execute(text("SET statement_timeout = '30s'"))
                    db.execute(stmt)
                    db.commit()
                    db.execute(text("RESET statement_timeout"))

                    # Actualizar estadísticas
                    # result.rowcount incluye tanto inserts como updates
                    batch_processed = len(products_to_upsert)
                    stats["total_processed"] += batch_processed

                    # Log de progreso
                    progress = min(int((stats["total_processed"] / estimated_total) * 100), 99)
                    logger.info(
                        f"[CIMA STREAMING] Lote {batch_num} completado | "
                        f"Procesados: {stats['total_processed']} | "
                        f"Progreso: {progress}%"
                    )

                    # Guardar checkpoint cada 5 lotes (1000 productos aprox)
                    if batch_num % 5 == 0:
                        self._save_checkpoint(db, current_page, stats)

                    # Actualizar estado si hay callback
                    if update_system_status_callback:
                        update_system_status_callback(
                            "PROCESSING",
                            progress,
                            f"Procesando lote {batch_num} - Total: {stats['total_processed']} productos",
                            stats,
                        )

                except Exception as e:
                    logger.error(f"[CIMA STREAMING] Error en lote {batch_num}: {str(e)}")
                    db.rollback()
                    stats["errors"] += batch_size

                    # Intentar procesar individualmente como fallback
                    for product_data in products_to_upsert:
                        try:
                            national_code = product_data["national_code"]
                            existing = (
                                db.query(ProductCatalog).filter(ProductCatalog.national_code == national_code).first()
                            )

                            if existing:
                                # Actualizar campos CIMA
                                for key, value in product_data.items():
                                    if key.startswith("cima_") or key in [
                                        "updated_at",
                                        "sync_status",
                                    ]:
                                        setattr(existing, key, value)

                                # Actualizar fuentes de datos
                                sources = existing.data_sources or ""
                                if "cima" not in sources.lower():
                                    existing.data_sources = f"{sources},cima" if sources else "cima"

                                stats["updated"] += 1
                            else:
                                # Crear nuevo
                                new_product = ProductCatalog(**product_data)
                                db.add(new_product)
                                stats["created"] += 1

                            db.commit()

                        except Exception as e2:
                            logger.error(f"[CIMA] Error individual en CN {national_code}: {str(e2)}")
                            db.rollback()
                            stats["errors"] += 1

            # Limpiar checkpoint al completar exitosamente
            self._clear_checkpoint(db)

            # Estado final
            if update_system_status_callback:
                update_system_status_callback(
                    "COMPLETED",
                    100,
                    f"Sincronización completada: {stats['total_processed']} productos procesados",
                    stats,
                )

            logger.info("[CIMA] === SINCRONIZACIÓN STREAMING COMPLETADA ===")
            logger.info(f"[CIMA] Total procesados: {stats['total_processed']}")
            logger.info(f"[CIMA] Creados: {stats.get('created', 'N/A')}")
            logger.info(f"[CIMA] Actualizados: {stats.get('updated', 'N/A')}")
            logger.info(f"[CIMA] Errores: {stats['errors']}")

            return stats

        except Exception as e:
            logger.error(f"[CIMA] Error en sincronización streaming: {str(e)}")

            if update_system_status_callback:
                update_system_status_callback(
                    "ERROR",
                    (int((stats["total_processed"] / 67000) * 100) if stats["total_processed"] > 0 else 0),
                    f"Error: {str(e)}",
                    stats,
                )

            import traceback

            traceback.print_exc()

            return {
                "error": str(e),
                "total_processed": stats.get("total_processed", 0),
                "created": stats.get("created", 0),
                "updated": stats.get("updated", 0),
                "errors": stats.get("errors", 0),
            }

    def _save_checkpoint(self, db: Session, current_page: int, stats: Dict[str, Any]):
        """
        Guarda checkpoint para poder reanudar sincronización.

        Args:
            db: Sesión de base de datos
            current_page: Página actual siendo procesada
            stats: Estadísticas actuales
        """
        import json

        from app.models.system_status import SystemComponent, SystemStatus

        try:
            cima_status = db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()

            if cima_status:
                # Guardar checkpoint
                cima_status.checkpoint_page = current_page
                cima_status.checkpoint_timestamp = utc_now()
                cima_status.checkpoint_data = json.dumps(
                    {
                        "stats": stats,
                        "timestamp": utc_now().isoformat(),
                        "page": current_page,
                    }
                )

                db.commit()
                logger.info(
                    f"[CIMA] 💾 Checkpoint guardado en página {current_page} ({stats['total_processed']} productos)"
                )
        except Exception as e:
            logger.warning(f"[CIMA] No se pudo guardar checkpoint: {e}")
            # No fallar si no se puede guardar checkpoint

    def _clear_checkpoint(self, db: Session):
        """
        Limpia checkpoint después de sincronización exitosa.

        Args:
            db: Sesión de base de datos
        """
        from app.models.system_status import SystemComponent, SystemStatus

        try:
            cima_status = db.query(SystemStatus).filter_by(component=SystemComponent.CIMA).first()

            if cima_status:
                cima_status.checkpoint_page = 0
                cima_status.checkpoint_timestamp = None
                cima_status.checkpoint_data = None

                db.commit()
                logger.info("[CIMA] ✅ Checkpoint limpiado tras sincronización exitosa")
        except Exception as e:
            logger.warning(f"[CIMA] No se pudo limpiar checkpoint: {e}")
            # No fallar si no se puede limpiar checkpoint

    async def enrich_nomenclator_only_products(
        self, db: Session, max_products: int = 10000, concurrency: int = 50
    ) -> Dict[str, Any]:
        """
        Enriquece productos que SOLO tienen datos de nomenclator (sin CIMA).

        Usa llamadas PARALELAS (50 concurrentes por defecto) para completar
        ~6,000 productos en ~2 minutos (vs ~45 min secuencial).

        Este método se ejecuta automáticamente al final del sync CIMA.

        Args:
            db: Sesión de base de datos
            max_products: Máximo de productos a enriquecer (default: todos)
            concurrency: Número de llamadas paralelas (default: 50)

        Returns:
            Estadísticas del proceso de enriquecimiento
        """
        import time
        from app.models import ProductCatalog

        start_time = time.time()
        stats = {
            "total_found": 0,
            "enriched": 0,
            "not_found": 0,
            "errors": 0,
            "started_at": utc_now().isoformat(),
        }

        try:
            # Buscar productos solo con nomenclator
            products_to_enrich = (
                db.query(ProductCatalog)
                .filter(ProductCatalog.data_sources == "nomenclator")
                .filter(ProductCatalog.nomen_estado == "ALTA")
                .limit(max_products)
                .all()
            )

            stats["total_found"] = len(products_to_enrich)

            if not products_to_enrich:
                logger.info("[CIMA FALLBACK] No hay productos solo-nomenclator para enriquecer")
                stats["status"] = "completed"
                stats["duration_seconds"] = 0
                return stats

            logger.info(
                f"[CIMA FALLBACK] 🔄 Enriqueciendo {len(products_to_enrich)} productos "
                f"con {concurrency} llamadas paralelas"
            )

            # Crear diccionario CN -> producto para actualización rápida
            products_by_cn = {p.national_code: p for p in products_to_enrich}
            all_cns = list(products_by_cn.keys())

            async def fetch_one(client: httpx.AsyncClient, cn: str) -> tuple:
                """Fetch un producto de CIMA."""
                try:
                    response = await client.get(
                        "https://cima.aemps.es/cima/rest/medicamento",
                        params={"cn": cn}
                    )
                    if response.status_code == 200:
                        return (cn, "success", response.json())
                    elif response.status_code in (204, 404):
                        # 204: No content (producto sanitario/parafarmacia, no medicamento)
                        # 404: No encontrado
                        return (cn, "not_found", None)
                    else:
                        return (cn, "error", None)
                except Exception as e:
                    return (cn, "error", str(e))

            # Procesar en chunks de `concurrency` llamadas paralelas
            # Límite de conexiones para evitar rate limiting de CIMA
            limits = httpx.Limits(max_connections=concurrency, max_keepalive_connections=10)
            async with httpx.AsyncClient(timeout=30.0, limits=limits) as client:
                for i in range(0, len(all_cns), concurrency):
                    chunk = all_cns[i:i + concurrency]

                    # Ejecutar chunk en paralelo
                    tasks = [fetch_one(client, cn) for cn in chunk]
                    results = await asyncio.gather(*tasks)

                    # Pequeña pausa entre batches para evitar rate limiting
                    await asyncio.sleep(0.5)

                    # Procesar resultados
                    for cn, status, data in results:
                        if status == "success" and data:
                            product = products_by_cn[cn]
                            product.cima_requiere_receta = data.get("receta")
                            product.cima_nombre_comercial = data.get("nombre", "")[:500]
                            product.cima_laboratorio_titular = data.get("labtitular", "")[:200]
                            product.cima_estado_registro = (
                                "COMERCIALIZADO" if data.get("comerc") else "NO_COMERCIALIZADO"
                            )

                            pactivos = data.get("pactivos")
                            if pactivos:
                                product.cima_principios_activos = [pactivos]

                            atcs = data.get("atcs", [])
                            if atcs:
                                for atc in atcs:
                                    if isinstance(atc, dict) and atc.get("codigo"):
                                        product.cima_atc_code = atc.get("codigo")

                            product.data_sources = "nomenclator,cima"
                            product.updated_at = utc_now()
                            stats["enriched"] += 1

                        elif status == "not_found":
                            stats["not_found"] += 1
                        else:
                            stats["errors"] += 1

                    # Commit cada chunk y log progreso
                    db.commit()
                    processed = min(i + concurrency, len(all_cns))
                    logger.info(
                        f"[CIMA FALLBACK] Progreso: {processed}/{len(all_cns)} "
                        f"({stats['enriched']} enriquecidos)"
                    )

            duration = time.time() - start_time
            stats["completed_at"] = utc_now().isoformat()
            stats["status"] = "completed"
            stats["duration_seconds"] = round(duration, 1)

            logger.info(
                f"[CIMA FALLBACK] ✅ Completado en {duration:.1f}s: "
                f"{stats['enriched']}/{stats['total_found']} productos enriquecidos"
            )

            return stats

        except Exception as e:
            logger.error(f"[CIMA FALLBACK] ❌ Error: {str(e)}")
            db.rollback()
            stats["error"] = str(e)
            stats["status"] = "error"
            return stats


# Singleton instance para usar en toda la aplicación
cima_integration_service = CIMAIntegrationService()
