﻿"""
Excepciones personalizadas para el sistema xFarma.
Define una jerarquía clara de excepciones del dominio farmacéutico.
"""


from app.utils.datetime_utils import utc_now


class PharmacyException(Exception):
    """Base para excepciones del dominio farmacéutico."""

    def __init__(self, message: str = None, code: str = None, details: dict = None):
        """
        Inicializa la excepción con información adicional.

        Args:
            message: Mensaje descriptivo del error
            code: Código único del error para debugging
            details: Diccionario con información adicional del contexto
        """
        super().__init__(message)
        self.message = message or "Error en operación farmacéutica"
        self.code = code
        self.details = details or {}


# Excepciones de Catálogo
class ProductNotFoundError(PharmacyException):
    """Producto no encontrado en catálogo."""

    def __init__(self, product_code: str, catalog_type: str = None, search_method: str = None):
        catalog_info = f" ({catalog_type})" if catalog_type else ""
        method_info = f" usando búsqueda por {search_method}" if search_method else ""
        super().__init__(
            message=f"Producto {product_code} no encontrado en catálogo{catalog_info}{method_info}",
            code="PRODUCT_NOT_FOUND",
            details={
                "product_code": product_code,
                "catalog_type": catalog_type,
                "search_method": search_method,
                "timestamp": utc_now().isoformat(),
            },
        )


class CatalogSyncError(PharmacyException):
    """Error en sincronización de catálogo CIMA/Nomenclator."""

    def __init__(self, source: str, reason: str = None, items_processed: int = 0, items_failed: int = 0):
        progress_info = ""
        if items_processed > 0:
            progress_info = f" (Procesados: {items_processed}, Fallidos: {items_failed})"
        super().__init__(
            message=f"Error sincronizando catálogo {source}: {reason}{progress_info}",
            code="CATALOG_SYNC_ERROR",
            details={
                "source": source,
                "reason": reason,
                "items_processed": items_processed,
                "items_failed": items_failed,
                "timestamp": utc_now().isoformat(),
            },
        )


# Excepciones de Enriquecimiento
class EnrichmentError(PharmacyException):
    """Error en proceso de enriquecimiento de datos."""

    def __init__(self, stage: str, product_count: int = None, reason: str = None, pharmacy_id: str = None):
        count_info = f" ({product_count} productos)" if product_count else ""
        pharmacy_info = f" para farmacia {pharmacy_id}" if pharmacy_id else ""
        super().__init__(
            message=f"Error en enriquecimiento - etapa {stage}{count_info}{pharmacy_info}: {reason}",
            code="ENRICHMENT_ERROR",
            details={
                "stage": stage,
                "product_count": product_count,
                "reason": reason,
                "pharmacy_id": pharmacy_id,
                "timestamp": utc_now().isoformat(),
            },
        )


class MissingCIMADataError(EnrichmentError):
    """Datos CIMA faltantes para enriquecimiento."""

    def __init__(self, product_codes: list):
        super().__init__(
            stage="CIMA",
            reason=f"Faltan datos CIMA para {len(product_codes)} productos",
        )
        self.details["missing_codes"] = product_codes[:10]  # Solo primeros 10


class MissingNomenclatorDataError(EnrichmentError):
    """Datos Nomenclator faltantes para enriquecimiento."""

    def __init__(self, product_codes: list):
        super().__init__(
            stage="Nomenclator",
            reason=f"Faltan datos Nomenclator para {len(product_codes)} productos",
        )
        self.details["missing_codes"] = product_codes[:10]


# Excepciones de Parseo ERP
class InvalidERPFormatError(PharmacyException):
    """Formato de archivo ERP no válido."""

    def __init__(self, file_name: str, parser_type: str, reason: str = None, expected_format: str = None):
        format_info = f" (Se esperaba: {expected_format})" if expected_format else ""
        super().__init__(
            message=f"Formato inválido para {parser_type} en archivo '{file_name}': {reason}{format_info}",
            code="INVALID_ERP_FORMAT",
            details={
                "file_name": file_name,
                "parser_type": parser_type,
                "reason": reason,
                "expected_format": expected_format,
                "timestamp": utc_now().isoformat(),
            },
        )


class MissingRequiredColumnsError(InvalidERPFormatError):
    """Columnas requeridas faltantes en archivo ERP."""

    def __init__(self, file_name: str, parser_type: str, missing_columns: list):
        super().__init__(
            file_name=file_name,
            parser_type=parser_type,
            reason=f"Faltan columnas: {', '.join(missing_columns)}",
        )
        self.details["missing_columns"] = missing_columns


class InvalidDataTypeError(InvalidERPFormatError):
    """Tipo de datos inválido en archivo ERP."""

    def __init__(self, file_name: str, parser_type: str, column: str, expected_type: str):
        super().__init__(
            file_name=file_name,
            parser_type=parser_type,
            reason=f"Columna {column} debe ser tipo {expected_type}",
        )
        self.details.update({"column": column, "expected_type": expected_type})


# Excepciones de Farmacia
class PharmacyNotFoundError(PharmacyException):
    """Farmacia no encontrada en sistema."""

    def __init__(self, pharmacy_id: str = None, cif: str = None):
        identifier = pharmacy_id or cif
        super().__init__(
            message=f"Farmacia {identifier} no encontrada",
            code="PHARMACY_NOT_FOUND",
            details={"pharmacy_id": pharmacy_id, "cif": cif},
        )


class PharmacyInactiveError(PharmacyException):
    """Operación sobre farmacia inactiva."""

    def __init__(self, pharmacy_id: str, operation: str):
        super().__init__(
            message=f"Farmacia {pharmacy_id} está inactiva para {operation}",
            code="PHARMACY_INACTIVE",
            details={"pharmacy_id": pharmacy_id, "operation": operation},
        )


# Excepciones de Procesamiento
class ProcessingError(PharmacyException):
    """Error general de procesamiento de datos."""

    def __init__(self, process: str, step: str = None, reason: str = None):
        super().__init__(
            message=f"Error en proceso {process}: {reason}",
            code="PROCESSING_ERROR",
            details={"process": process, "step": step, "reason": reason},
        )


class DataValidationError(ProcessingError):
    """Error en validación de datos."""

    def __init__(self, field: str, value: any, constraint: str):
        super().__init__(
            process="validation",
            reason=f"Campo {field} con valor {value} viola restricción: {constraint}",
        )
        self.details.update({"field": field, "value": value, "constraint": constraint})


class DuplicateDataError(ProcessingError):
    """Datos duplicados detectados."""

    def __init__(self, entity_type: str, identifier: str):
        super().__init__(
            process="deduplication",
            reason=f"{entity_type} duplicado: {identifier}",
        )
        self.details.update({"entity_type": entity_type, "identifier": identifier})


# Excepciones de Laboratorios/Partners
class PartnerNotFoundError(PharmacyException):
    """Partner/Laboratorio no encontrado."""

    def __init__(self, partner_id: str = None, lab_code: str = None):
        identifier = partner_id or lab_code
        super().__init__(
            message=f"Partner {identifier} no encontrado",
            code="PARTNER_NOT_FOUND",
            details={"partner_id": partner_id, "lab_code": lab_code},
        )


class PartnerConfigurationError(PharmacyException):
    """Error en configuración de partner."""

    def __init__(self, partner_id: str, config_issue: str):
        super().__init__(
            message=f"Configuración inválida para partner {partner_id}: {config_issue}",
            code="PARTNER_CONFIG_ERROR",
            details={"partner_id": partner_id, "config_issue": config_issue},
        )


# Excepciones de Base de Datos
class DatabaseError(PharmacyException):
    """Error en operación de base de datos."""

    def __init__(self, operation: str, table: str = None, reason: str = None):
        super().__init__(
            message=f"Error BD en {operation}: {reason}",
            code="DATABASE_ERROR",
            details={"operation": operation, "table": table, "reason": reason},
        )


class ConnectionError(DatabaseError):
    """Error de conexión a base de datos."""

    def __init__(self, database: str = "xfarma_db", reason: str = None):
        super().__init__(
            operation="connection",
            reason=f"No se pudo conectar a {database}: {reason}",
        )
        self.details["database"] = database


class TransactionError(DatabaseError):
    """Error en transacción de base de datos."""

    def __init__(self, transaction_type: str, reason: str = None):
        super().__init__(
            operation="transaction",
            reason=f"Fallo en transacción {transaction_type}: {reason}",
        )
        self.details["transaction_type"] = transaction_type


# Excepciones de Sistema/Notificaciones
class NotificationError(PharmacyException):
    """Error en sistema de notificaciones."""

    def __init__(self, notification_type: str, recipient: str = None, reason: str = None):
        super().__init__(
            message=f"Fallo enviando {notification_type}: {reason}",
            code="NOTIFICATION_ERROR",
            details={
                "notification_type": notification_type,
                "recipient": recipient,
                "reason": reason,
            },
        )


class SystemAlertError(PharmacyException):
    """Error en sistema de alertas."""

    def __init__(self, alert_id: str = None, severity: str = None, reason: str = None):
        super().__init__(
            message=f"Error en alerta del sistema: {reason}",
            code="SYSTEM_ALERT_ERROR",
            details={"alert_id": alert_id, "severity": severity, "reason": reason},
        )


# Excepciones de Medidas/Analytics
class MeasureCalculationError(PharmacyException):
    """Error calculando medida/métrica."""

    def __init__(self, measure_name: str, pharmacy_id: str = None, reason: str = None):
        super().__init__(
            message=f"Error calculando {measure_name}: {reason}",
            code="MEASURE_CALC_ERROR",
            details={
                "measure_name": measure_name,
                "pharmacy_id": pharmacy_id,
                "reason": reason,
            },
        )


class InsufficientDataError(MeasureCalculationError):
    """Datos insuficientes para cálculo."""

    def __init__(self, measure_name: str, required_months: int, available_months: int):
        super().__init__(
            measure_name=measure_name,
            reason=f"Se requieren {required_months} meses, solo hay {available_months}",
        )
        self.details.update(
            {
                "required_months": required_months,
                "available_months": available_months,
            }
        )


# Excepciones de API/HTTP
class APIError(PharmacyException):
    """Error en llamada API."""

    def __init__(self, endpoint: str, status_code: int = None, response: str = None):
        super().__init__(
            message=f"Error API en {endpoint}: {status_code}",
            code="API_ERROR",
            details={
                "endpoint": endpoint,
                "status_code": status_code,
                "response": response,
            },
        )


class RateLimitError(APIError):
    """Límite de tasa excedido."""

    def __init__(self, endpoint: str, retry_after: int = None):
        # First call parent with basic info
        super().__init__(endpoint=endpoint, status_code=429)

        # Then update the message to include retry_after
        if retry_after:
            self.message = f"Límite de tasa excedido en {endpoint}. Reintentar en {retry_after}s"
        else:
            self.message = f"Límite de tasa excedido en {endpoint}"

        # Store retry_after in details
        self.details["retry_after"] = retry_after

    def __str__(self):
        """Override to use our updated message"""
        return self.message


class AuthenticationError(APIError):
    """Error de autenticación."""

    def __init__(self, endpoint: str, auth_type: str = None):
        super().__init__(endpoint=endpoint, status_code=401)
        self.details["auth_type"] = auth_type
        self.message = f"Autenticación fallida en {endpoint}"


# Excepciones de Validación General
class ValidationError(PharmacyException):
    """Error general de validación."""

    def __init__(self, message: str, field: str = None, value: any = None):
        super().__init__(
            message=message,
            code="VALIDATION_ERROR",
            details={"field": field, "value": value, "timestamp": utc_now().isoformat()},
        )


# Excepciones de Configuración
class ConfigurationError(PharmacyException):
    """Error de configuración del sistema."""

    def __init__(self, component: str, missing_config: str = None, reason: str = None):
        super().__init__(
            message=f"Configuración inválida en {component}: {reason or missing_config}",
            code="CONFIG_ERROR",
            details={
                "component": component,
                "missing_config": missing_config,
                "reason": reason,
            },
        )


class EnvironmentError(ConfigurationError):
    """Variable de entorno faltante o inválida."""

    def __init__(self, env_var: str, required_for: str = None):
        super().__init__(
            component="environment",
            missing_config=env_var,
            reason=f"Variable {env_var} requerida para {required_for}",
        )
        self.details["env_var"] = env_var
        self.details["required_for"] = required_for
