"""Unified models for ERP data.

These models represent the semantic data layer (SDL) that abstracts
differences between various pharmacy ERP systems.

All adapters translate their ERP-specific schemas to these models.
"""

from dataclasses import dataclass, field
from datetime import date, datetime
from decimal import Decimal
from enum import Enum
from typing import Optional


class PrescriptionType(str, Enum):
    """Type of prescription/sale."""

    RECETA = "receta"  # Electronic prescription (eReceta)
    VENTA_LIBRE = "venta_libre"  # OTC sale
    SPD = "spd"  # Personalized dosage system
    MUTUA = "mutua"  # Private insurance
    VETERINARIA = "veterinaria"  # Veterinary
    UNKNOWN = "unknown"


class OrderStatus(str, Enum):
    """Order status."""

    PENDING = "pending"  # Not yet sent
    SENT = "sent"  # Transmitted to supplier
    PARTIAL = "partial"  # Partially received
    RECEIVED = "received"  # Fully received
    CANCELLED = "cancelled"


@dataclass
class UnifiedSale:
    """Unified sale record from any ERP.

    Represents a single product sale, regardless of the source ERP.
    Multiple UnifiedSale records may belong to the same transaction.
    """

    id: str  # kaiFarma internal ID (UUID)
    erp_id: str  # Original ID from ERP (e.g., idoperacion)
    timestamp: datetime  # Sale date/time
    product_code: str  # CN/EAN barcode (normalized)
    product_name: str  # Product description
    quantity: int  # Units sold
    pvp: Decimal  # Unit price (PVP)
    puc: Optional[Decimal] = None  # Unit cost (if available)
    discount: Optional[Decimal] = None  # Discount applied (amount or %)
    prescription_type: PrescriptionType = PrescriptionType.UNKNOWN
    employee_id: Optional[str] = None  # Employee who made the sale
    customer_id: Optional[str] = None  # Customer (if loyalty program)

    # Enrichment fields (populated by kaiFarma, not ERP)
    laboratory_name: Optional[str] = None
    is_generic: bool = False
    atc_code: Optional[str] = None

    @property
    def total(self) -> Decimal:
        """Calculate line total."""
        return self.pvp * self.quantity - (self.discount or Decimal("0"))


@dataclass
class UnifiedStock:
    """Unified stock record from any ERP.

    Represents current inventory state for a product.
    """

    product_code: str  # CN/EAN barcode (normalized)
    product_name: str  # Product description
    quantity: int  # Current stock units
    min_stock: Optional[int] = None  # Minimum stock configured
    max_stock: Optional[int] = None  # Maximum stock configured
    location: Optional[str] = None  # Physical location (shelf, etc.)
    expiry_date: Optional[date] = None  # Nearest expiry date
    last_purchase_date: Optional[date] = None
    last_purchase_price: Optional[Decimal] = None
    average_cost: Optional[Decimal] = None
    pvp: Optional[Decimal] = None

    # Enrichment fields
    laboratory_name: Optional[str] = None
    is_generic: bool = False
    family_name: Optional[str] = None
    subfamily_name: Optional[str] = None

    @property
    def is_low_stock(self) -> bool:
        """Check if stock is below minimum."""
        if self.min_stock is None:
            return False
        return self.quantity < self.min_stock

    @property
    def days_until_expiry(self) -> Optional[int]:
        """Days until nearest expiry."""
        if self.expiry_date is None:
            return None
        delta = self.expiry_date - date.today()
        return delta.days


@dataclass
class OrderLine:
    """Single line in an order."""

    product_code: str
    product_name: str
    quantity_ordered: int
    quantity_pending: int = 0
    quantity_received: int = 0
    unit_cost: Optional[Decimal] = None


@dataclass
class UnifiedOrder:
    """Unified order to supplier.

    Represents a purchase order sent to a distributor.
    """

    id: str  # kaiFarma internal ID
    erp_id: str  # Original ID from ERP
    supplier_code: str  # Supplier identifier
    supplier_name: str  # Supplier name
    order_date: datetime
    status: OrderStatus = OrderStatus.PENDING
    lines: list[OrderLine] = field(default_factory=list)

    @property
    def total_items(self) -> int:
        """Total items ordered."""
        return sum(line.quantity_ordered for line in self.lines)

    @property
    def pending_items(self) -> int:
        """Items still pending."""
        return sum(line.quantity_pending for line in self.lines)


@dataclass
class ReceiptLine:
    """Single line in a receipt/delivery note."""

    product_code: str
    product_name: str
    quantity: int
    unit_cost: Decimal
    pvp: Optional[Decimal] = None
    lot_number: Optional[str] = None
    expiry_date: Optional[date] = None


@dataclass
class UnifiedReceipt:
    """Unified receipt/delivery note from supplier.

    Represents goods received from a distributor.
    """

    id: str  # kaiFarma internal ID
    erp_id: str  # Original ID from ERP
    supplier_code: str
    supplier_name: str
    receipt_date: datetime
    lines: list[ReceiptLine] = field(default_factory=list)
    delivery_note_number: Optional[str] = None
    invoice_number: Optional[str] = None

    @property
    def total_cost(self) -> Decimal:
        """Total receipt cost."""
        return sum(
            line.unit_cost * line.quantity
            for line in self.lines
        )


@dataclass
class UnifiedCustomer:
    """Unified customer record.

    Represents a pharmacy customer (loyalty program member).
    """

    id: str  # kaiFarma internal ID
    erp_id: str  # Original ID from ERP
    name: str
    phone: Optional[str] = None
    email: Optional[str] = None
    loyalty_points: int = 0
    last_purchase: Optional[date] = None
    health_card_number: Optional[str] = None  # SIP/TSI
    birth_date: Optional[date] = None


@dataclass
class UnifiedEmployee:
    """Unified employee record."""

    id: str  # kaiFarma internal ID
    erp_id: str  # Original ID from ERP
    name: str
    first_name: str
    last_name: str
    tax_id: Optional[str] = None  # NIF
    college_number: Optional[str] = None  # Pharmacist number
    is_owner: bool = False
    is_pharmacist: bool = False


@dataclass
class UnifiedCustomerOrder:
    """Unified customer order (encargo) record.

    Represents a product reserved/ordered for a specific customer.
    These are products that have been ordered but not yet picked up.

    Used by: Modulo 1 - Cazador de Encargos
    """

    id: str  # kaiFarma internal ID
    erp_order_id: str  # Original ID from ERP (e.g., "ENC-17399-45123")
    created_at: datetime  # When the order was created
    customer_name: str  # Customer name (for display)
    customer_phone: Optional[str] = None  # Phone to call
    customer_email: Optional[str] = None  # Email contact
    product_code: str = ""  # CN/EAN
    product_name: str = ""  # Product description
    units: int = 1  # Quantity ordered
    pvp: Decimal = field(default_factory=lambda: Decimal("0"))  # PVP value
    days_pending: int = 0  # Days since order was created

    @property
    def economic_value(self) -> Decimal:
        """Total economic value locked in this order."""
        return self.pvp * self.units

    @property
    def severity(self) -> str:
        """Urgency level based on days pending.

        Returns:
            - "critical": 30+ days (should be returned to supplier)
            - "urgent": 15-29 days (call customer urgently)
            - "warning": 7-14 days (needs attention)
            - "info": 0-6 days (recent, OK)
        """
        if self.days_pending >= 30:
            return "critical"
        if self.days_pending >= 15:
            return "urgent"
        if self.days_pending >= 7:
            return "warning"
        return "info"
