#!/usr/bin/env python
"""Test script for READ-ONLY guards in Farmanager adapter.

Run from project root:
    python -m backend.app.erp_adapters.farmanager.test_readonly_guard

Tests that write operations are properly blocked.
"""

from .adapter import FarmanagerAdapter, FarmanagerReadOnlyError
from ..models import UnifiedOrder, OrderLine, OrderStatus
from datetime import datetime
from decimal import Decimal


def main():
    """Test READ-ONLY guards."""

    print("=" * 60)
    print("Farmanager READ-ONLY Guard Test")
    print("=" * 60)

    # Test 1: Verify default is read-only
    print("\n1. Testing default read_only=True...")
    adapter = FarmanagerAdapter(
        host="localhost",
        port=3306,
        user="administrador",
        password="nimda2009",
        database="gesql",
    )
    assert adapter.is_read_only, "Default should be read-only"
    print("   ✅ Default is read-only")

    # Test 2: Verify write operation is blocked (create_order_proposal)
    print("\n2. Testing create_order_proposal is BLOCKED...")
    try:
        test_order = UnifiedOrder(
            id="test-id",
            erp_id="",
            supplier_code="TEST",
            supplier_name="Test Supplier",
            order_date=datetime.now(),
            status=OrderStatus.PENDING,
            lines=[
                OrderLine(
                    product_code="123456",
                    product_name="Test Product",
                    quantity_ordered=10,
                    quantity_pending=10,
                    quantity_received=0,
                    unit_cost=Decimal("5.00"),
                )
            ],
        )
        adapter.create_order_proposal(test_order)
        print("   ❌ FAILED: Write should have been blocked!")
        return False
    except FarmanagerReadOnlyError as e:
        print(f"   ✅ Correctly blocked: {str(e)[:60]}...")

    # Test 3: Verify write operation is blocked (create_encargo)
    print("\n3. Testing create_encargo is BLOCKED...")
    try:
        adapter.create_encargo("123456", 5, "customer-1")
        print("   ❌ FAILED: Write should have been blocked!")
        return False
    except FarmanagerReadOnlyError as e:
        print(f"   ✅ Correctly blocked: {str(e)[:60]}...")

    # Test 4: Verify read operations still work
    print("\n4. Testing read operations still work...")
    success, message = adapter.test_connection()
    if success:
        print(f"   ✅ Read operations work: {message}")
    else:
        print(f"   ❌ Read failed: {message}")
        return False

    # Test 5: Test with read_only=False but no ALLOWED_WRITE_OPS
    print("\n5. Testing read_only=False but empty ALLOWED_WRITE_OPS...")
    adapter_rw = FarmanagerAdapter(
        host="localhost",
        port=3306,
        user="administrador",
        password="nimda2009",
        database="gesql",
        read_only=False,  # Allow writes...
    )
    assert not adapter_rw.is_read_only, "Should not be read-only"

    try:
        adapter_rw.create_order_proposal(test_order)
        print("   ❌ FAILED: Should block - operation not in ALLOWED_WRITE_OPS")
        return False
    except FarmanagerReadOnlyError as e:
        print(f"   ✅ Correctly blocked (not in allowed ops): {str(e)[:50]}...")

    # Test 6: Test with specific operation allowed
    print("\n6. Testing with 'propuesta_pedido' in ALLOWED_WRITE_OPS...")

    # Temporarily add allowed operation
    original_ops = FarmanagerAdapter.ALLOWED_WRITE_OPS.copy()
    FarmanagerAdapter.ALLOWED_WRITE_OPS = {"propuesta_pedido"}

    adapter_allowed = FarmanagerAdapter(
        host="localhost",
        port=3306,
        user="administrador",
        password="nimda2009",
        database="gesql",
        read_only=False,
    )

    try:
        adapter_allowed.create_order_proposal(test_order)
        print("   ❌ FAILED: NotImplementedError expected")
        return False
    except NotImplementedError as e:
        print(f"   ✅ Passed guard, got NotImplementedError (expected): {str(e)[:40]}...")
    except FarmanagerReadOnlyError:
        print("   ❌ FAILED: Should have passed the guard")
        return False
    finally:
        # Restore original
        FarmanagerAdapter.ALLOWED_WRITE_OPS = original_ops

    print("\n" + "=" * 60)
    print("✅ ALL READ-ONLY GUARD TESTS PASSED!")
    print("=" * 60)
    print("\nSummary:")
    print("- Default mode: READ-ONLY ✅")
    print("- create_order_proposal: BLOCKED ✅")
    print("- create_encargo: BLOCKED ✅")
    print("- Read operations: WORKING ✅")
    print("- ALLOWED_WRITE_OPS enforcement: WORKING ✅")
    print("=" * 60)
    return True


if __name__ == "__main__":
    main()
