#!/usr/bin/env python
"""Test script for Farmanager adapter.

Run from project root:
    python -m backend.app.erp_adapters.farmanager.test_connection
"""

from datetime import date, timedelta
from .adapter import FarmanagerAdapter


def main():
    """Test Farmanager adapter with local database."""

    print("=" * 60)
    print("Farmanager Adapter Test")
    print("=" * 60)

    # Connection parameters from user
    adapter = FarmanagerAdapter(
        host="localhost",
        port=3306,
        user="administrador",
        password="nimda2009",
        database="gesql",
    )

    # Test connection
    print("\n1. Testing connection...")
    success, message = adapter.test_connection()
    print(f"   Result: {'OK' if success else 'FAILED'}")
    print(f"   Message: {message}")

    if not success:
        return

    with adapter:
        # Test stock
        print("\n2. Testing stock retrieval...")
        stock_count = adapter.get_stock_count()
        print(f"   Products in stock: {stock_count:,}")

        print("   Sample stock items:")
        for i, stock in enumerate(adapter.get_current_stock()):
            if i >= 5:
                break
            print(f"   - {stock.product_code}: {stock.product_name[:40]} ({stock.quantity} units)")

        # Test sales
        print("\n3. Testing sales retrieval...")
        today = date.today()
        week_ago = today - timedelta(days=7)
        sales_count = adapter.get_sales_count(week_ago, today)
        print(f"   Sales in last 7 days: {sales_count:,}")

        print("   Recent sales:")
        for i, sale in enumerate(adapter.get_sales(today - timedelta(days=1), today)):
            if i >= 5:
                break
            print(f"   - {sale.timestamp}: {sale.product_name[:30]} x{sale.quantity} @ {sale.pvp}EUR")

        # Test employees
        print("\n4. Testing employees retrieval...")
        employees = list(adapter.get_employees())
        print(f"   Total employees: {len(employees)}")
        for emp in employees[:3]:
            role = "Owner" if emp.is_owner else ("Pharmacist" if emp.is_pharmacist else "Staff")
            print(f"   - {emp.name} ({role})")

        # Test customers
        print("\n5. Testing customers retrieval...")
        customers = list(adapter.get_customers())
        print(f"   Total customers: {len(customers)}")

        # Test pending orders
        print("\n6. Testing pending orders...")
        orders = list(adapter.get_pending_orders())
        print(f"   Pending orders: {len(orders)}")
        for order in orders[:3]:
            print(f"   - Order {order.erp_id}: {order.supplier_name} ({order.total_items} items)")

        # Test recent receipts
        print("\n7. Testing recent receipts...")
        receipts = list(adapter.get_recent_receipts(today - timedelta(days=7)))
        print(f"   Receipts in last 7 days: {len(receipts)}")
        for receipt in receipts[:3]:
            print(f"   - {receipt.receipt_date}: {receipt.supplier_name} ({len(receipt.lines)} lines)")

    print("\n" + "=" * 60)
    print("All tests completed successfully!")
    print("=" * 60)


if __name__ == "__main__":
    main()
