"""Farmanager ERP Adapter Implementation.

Connects to Farmanager MySQL database and translates to unified models.

Database Schema Reference: docs/proposals/FARMANAGER_SCHEMA_MAPPING.md

Key Tables:
- ges301: Sales operations (header)
- ges302: Sales operation lines
- ges101: Product catalog
- ges103: Product codes (CN/EAN)
- inventario: Current stock
- ges407: Purchases/receipts
- ges406: Orders to suppliers
- ges401: Suppliers
- ges122: Laboratories
- ges115/116: Families/subfamilies
- ges050/051: Customers
- empleado: Employees
"""

import logging
import uuid
from datetime import date, datetime
from decimal import Decimal
from typing import Iterator, Optional

from ..base import ERPAdapter
from ..models import (
    OrderLine,
    OrderStatus,
    PrescriptionType,
    ReceiptLine,
    UnifiedCustomer,
    UnifiedCustomerOrder,
    UnifiedEmployee,
    UnifiedOrder,
    UnifiedReceipt,
    UnifiedSale,
    UnifiedStock,
)

logger = logging.getLogger(__name__)

# Try to import mysql connector
try:
    import mysql.connector
    from mysql.connector import Error as MySQLError

    MYSQL_AVAILABLE = True
except ImportError:
    MYSQL_AVAILABLE = False
    MySQLError = Exception  # Fallback type


class FarmanagerReadOnlyError(Exception):
    """Raised when attempting to write in read-only mode."""
    pass


class FarmanagerAdapter(ERPAdapter):
    """Adapter for Farmanager ERP (MySQL database).

    Farmanager is a Spanish pharmacy ERP with ~15% market share.
    Database uses MySQL with tables prefixed 'ges' (e.g., ges301, ges101).

    IMPORTANT: This adapter is READ-ONLY by default.
    Write operations are blocked unless explicitly enabled.

    Future write operations (when enabled):
    - propuesta_pedido: Order proposals to suppliers
    - encargos: Special customer orders/requests

    Example:
        adapter = FarmanagerAdapter(
            host="localhost",
            user="your_username",
            password="your_password",  # Use environment variables in production
            database="gesql"
        )
        with adapter:
            for sale in adapter.get_sales(date(2026, 1, 1), date(2026, 1, 8)):
                print(f"{sale.product_code}: {sale.quantity} units")
    """

    # Allowed write operations (empty = read-only)
    # Future options: {"propuesta_pedido", "encargos"}
    ALLOWED_WRITE_OPS: set[str] = set()

    def __init__(
        self,
        host: str = "localhost",
        port: int = 3306,
        user: str = "administrador",
        password: str = "",
        database: str = "gesql",
        charset: str = "utf8",  # Use utf8 for older MySQL versions
        read_only: bool = True,  # SAFETY: Read-only by default
    ):
        """Initialize Farmanager adapter.

        Args:
            host: MySQL host
            port: MySQL port (default 3306)
            user: Database user
            password: Database password
            database: Database name (usually 'gesql')
            charset: Character encoding
            read_only: If True (default), block all write operations
        """
        if not MYSQL_AVAILABLE:
            raise ImportError(
                "mysql-connector-python is required for FarmanagerAdapter. "
                "Install with: pip install mysql-connector-python"
            )

        self._host = host
        self._port = port
        self._user = user
        self._password = password
        self._database = database
        self._charset = charset
        self._connection = None
        self._version: Optional[str] = None
        self._read_only = read_only

        # Log read-only status
        if read_only:
            logger.info("FarmanagerAdapter initialized in READ-ONLY mode")
        else:
            logger.warning(
                "FarmanagerAdapter initialized with WRITE access - "
                "allowed operations: %s",
                self.ALLOWED_WRITE_OPS or "NONE (still read-only)"
            )

    @property
    def erp_name(self) -> str:
        return "Farmanager"

    @property
    def erp_version(self) -> Optional[str]:
        return self._version

    @property
    def is_read_only(self) -> bool:
        """Check if adapter is in read-only mode."""
        return self._read_only

    def _assert_read_only(self) -> None:
        """Guard: Raise error if any write is attempted in read-only mode."""
        if self._read_only:
            raise FarmanagerReadOnlyError(
                "BLOCKED: Write operation attempted in READ-ONLY mode. "
                "The FarmanagerAdapter is configured for read-only access. "
                "To enable writes, set read_only=False and configure ALLOWED_WRITE_OPS."
            )

    def _assert_write_allowed(self, operation: str) -> None:
        """Guard: Check if a specific write operation is allowed.

        Args:
            operation: Name of the write operation (e.g., 'propuesta_pedido', 'encargos')

        Raises:
            FarmanagerReadOnlyError: If operation is not allowed
        """
        self._assert_read_only()  # First check read-only mode

        if operation not in self.ALLOWED_WRITE_OPS:
            raise FarmanagerReadOnlyError(
                f"BLOCKED: Write operation '{operation}' is not in ALLOWED_WRITE_OPS. "
                f"Currently allowed: {self.ALLOWED_WRITE_OPS or 'NONE'}. "
                "Modify ALLOWED_WRITE_OPS to enable specific write operations."
            )

    def connect(self) -> bool:
        """Connect to Farmanager MySQL database."""
        try:
            self._connection = mysql.connector.connect(
                host=self._host,
                port=self._port,
                user=self._user,
                password=self._password,
                database=self._database,
                charset=self._charset,
                use_unicode=True,
            )
            logger.info(f"Connected to Farmanager at {self._host}:{self._port}")

            # Try to detect version from config
            cursor = None
            try:
                cursor = self._connection.cursor(dictionary=True, buffered=True)
                cursor.execute("SELECT centerUrl FROM ges001 LIMIT 1")
                row = cursor.fetchone()
                if row and row.get("centerUrl"):
                    self._version = "Detected"
            except MySQLError:
                pass
            finally:
                if cursor:
                    cursor.close()

            return True

        except MySQLError as e:
            logger.error(f"Failed to connect to Farmanager: {e}")
            return False

    def disconnect(self) -> None:
        """Close database connection."""
        if self._connection:
            try:
                # Consume any unread results before closing
                if self._connection.is_connected():
                    self._connection.close()
                    logger.info("Disconnected from Farmanager")
            except MySQLError:
                pass  # Ignore errors during disconnect
        self._connection = None

    def is_connected(self) -> bool:
        """Check connection status."""
        if self._connection is None:
            return False
        try:
            return self._connection.is_connected()
        except MySQLError:
            return False

    def test_connection(self) -> tuple[bool, str]:
        """Test database connection."""
        cursor = None
        try:
            if not self.is_connected():
                if not self.connect():
                    return False, "Failed to establish connection"

            cursor = self._connection.cursor(buffered=True)
            cursor.execute("SELECT COUNT(*) FROM ges101")
            count = cursor.fetchone()[0]

            return True, f"Connected. Product catalog has {count:,} items."

        except MySQLError as e:
            return False, f"Connection test failed: {e}"
        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Sales Data
    # ==========================================================================

    def get_sales(
        self,
        from_date: date,
        to_date: date,
        include_prescriptions: bool = True,
    ) -> Iterator[UnifiedSale]:
        """Get sales from Farmanager.

        Queries ges301 (operations header) joined with ges302 (operation lines).
        Operation type 1 = sales.

        Note:
            Cost field uses inventario.coste which is PMC (Precio Medio Coste).
            For PUC (Precio Última Compra), would need subquery to ges407:
                SELECT bcoste FROM ges407 WHERE carticulo=X ORDER BY tfecha DESC LIMIT 1
            Current implementation prioritizes query performance over exact PUC.
            TODO: Add optional parameter to choose PMC vs PUC if needed.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        # Query includes nlinea for unique line identification within operation
        # Unique key: idoperacion + nlinea (operation + line number)
        query = """
            SELECT
                o.idoperacion,
                l.nlinea,
                o.tfechahora,
                l.carticulo,
                l.cnombre,
                l.iunidades,
                l.bpvp,
                l.bdtolinea,
                l.bimporte,
                l.ldoe,
                o.idempleado,
                o.idcliente,
                i.coste as puc
            FROM ges301 o
            JOIN ges302 l ON o.idoperacion = l.idoperacion
            LEFT JOIN inventario i ON l.carticulo = i.codigo
            WHERE o.ntipo = 1
              AND DATE(o.tfechahora) BETWEEN %s AND %s
            ORDER BY o.tfechahora DESC, l.nlinea ASC
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query, (from_date, to_date))

            for row in cursor:
                # Determine prescription type
                is_prescription = bool(row.get("ldoe", 0))
                prescription_type = (
                    PrescriptionType.RECETA if is_prescription
                    else PrescriptionType.VENTA_LIBRE
                )

                # Skip prescriptions if not requested
                if not include_prescriptions and is_prescription:
                    continue

                # Composite erp_id: operation_line for guaranteed uniqueness
                erp_id = f"{row['idoperacion']}_{row.get('nlinea', 0)}"

                yield UnifiedSale(
                    id=str(uuid.uuid4()),
                    erp_id=erp_id,
                    timestamp=row["tfechahora"],
                    product_code=row["carticulo"] or "",
                    product_name=row["cnombre"] or "",
                    quantity=row["iunidades"] or 0,
                    pvp=Decimal(str(row["bpvp"] or 0)),
                    puc=Decimal(str(row["puc"])) if row.get("puc") else None,
                    discount=Decimal(str(row["bdtolinea"] or 0)),
                    prescription_type=prescription_type,
                    employee_id=str(row["idempleado"]) if row.get("idempleado") else None,
                    customer_id=str(row["idcliente"]) if row.get("idcliente") else None,
                )
        finally:
            if cursor:
                cursor.close()

    def get_sales_count(self, from_date: date, to_date: date) -> int:
        """Optimized sales count using SQL COUNT."""
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        query = """
            SELECT COUNT(*) as cnt
            FROM ges301 o
            JOIN ges302 l ON o.idoperacion = l.idoperacion
            WHERE o.ntipo = 1
              AND DATE(o.tfechahora) BETWEEN %s AND %s
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query, (from_date, to_date))
            result = cursor.fetchone()
            return result["cnt"] if result else 0
        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Stock Data
    # ==========================================================================

    def get_current_stock(self) -> Iterator[UnifiedStock]:
        """Get current stock from Farmanager.

        Uses 'inventario' table (denormalized view) joined with ges101/ges103
        for product details.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        query = """
            SELECT
                i.codigo,
                i.stock,
                i.laboratorio,
                i.generico,
                i.familia,
                i.subfamilia,
                i.pvp,
                i.coste,
                a.cnombre,
                c.istockminimo,
                c.istockmaximo,
                c.dcaducidad
            FROM inventario i
            LEFT JOIN ges103 cd ON i.codigo = cd.ccodigo AND cd.lprincipal = 1
            LEFT JOIN ges101 a ON cd.idarticulo = a.idarticulo
            LEFT JOIN ges104 c ON a.idarticulo = c.idarticulo
            WHERE i.stock > 0
            ORDER BY i.stock DESC
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query)

            for row in cursor:
                yield UnifiedStock(
                    product_code=row["codigo"] or "",
                    product_name=row["cnombre"] or "",
                    quantity=int(row["stock"] or 0),
                    min_stock=row["istockminimo"],
                    max_stock=row["istockmaximo"],
                    expiry_date=row["dcaducidad"],
                    average_cost=Decimal(str(row["coste"] or 0)),
                    pvp=Decimal(str(row["pvp"] or 0)),
                    laboratory_name=row["laboratorio"],
                    is_generic=bool(row.get("generico", 0)),
                    family_name=row["familia"],
                    subfamily_name=row["subfamilia"],
                )
        finally:
            if cursor:
                cursor.close()

    def get_stock_count(self) -> int:
        """Optimized stock count using SQL COUNT."""
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        cursor = None
        try:
            cursor = self._connection.cursor(buffered=True)
            cursor.execute("SELECT COUNT(*) FROM inventario WHERE stock > 0")
            count = cursor.fetchone()[0]
            return count
        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Orders Data
    # ==========================================================================

    def get_pending_orders(self) -> Iterator[UnifiedOrder]:
        """Get pending orders to suppliers.

        Uses ges406 (order lines) joined with ges401 (suppliers).
        Filters for orders with pending items (ipendiente > 0).

        Note:
            Farmanager ges406 doesn't have an explicit order creation date.
            We use the earliest receipt date (tfecha from related ges407) as an
            approximation. If no receipts exist yet, falls back to current timestamp.

        Performance:
            Uses a single query with GROUP_CONCAT to avoid N+1 query pattern.
            All order lines are fetched in one query and grouped in Python.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        # Single query to get all pending order lines with order metadata
        # This eliminates the N+1 query pattern
        query = """
            SELECT
                o.idpedido,
                o.idmodelo,
                o.carticulo,
                o.cnombre,
                o.ipedir,
                o.ipendiente,
                o.irecepcion,
                o.bcoste,
                p.iproveedor,
                p.cnombre as proveedor_nombre,
                MIN(r.tfecha) as first_receipt_date
            FROM ges406 o
            LEFT JOIN ges402 m ON o.idmodelo = m.idmodelo
            LEFT JOIN ges401 p ON m.iproveedor = p.iproveedor
            LEFT JOIN ges407 r ON o.carticulo = r.carticulo AND r.iproveedor = p.iproveedor
            WHERE o.ipendiente > 0
            GROUP BY o.idpedido, o.idmodelo, o.carticulo, o.cnombre, o.ipedir,
                     o.ipendiente, o.irecepcion, o.bcoste, p.iproveedor, p.cnombre
            ORDER BY o.idpedido DESC
            LIMIT 5000
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query)

            # Group lines by order ID in Python (single pass)
            orders: dict[int, dict] = {}

            for row in cursor:
                order_id = row["idpedido"]

                if order_id not in orders:
                    # Use first receipt date or fallback to now
                    order_date = row.get("first_receipt_date")
                    if not order_date:
                        order_date = datetime.now()

                    orders[order_id] = {
                        "supplier_code": str(row["iproveedor"] or ""),
                        "supplier_name": row["proveedor_nombre"] or "Unknown",
                        "order_date": order_date,
                        "lines": [],
                    }

                orders[order_id]["lines"].append(OrderLine(
                    product_code=row["carticulo"] or "",
                    product_name=row["cnombre"] or "",
                    quantity_ordered=row["ipedir"] or 0,
                    quantity_pending=row["ipendiente"] or 0,
                    quantity_received=row["irecepcion"] or 0,
                    unit_cost=Decimal(str(row["bcoste"] or 0)),
                ))

            # Yield orders with calculated status
            for order_id, order_data in orders.items():
                lines = order_data["lines"]
                total_pending = sum(line.quantity_pending for line in lines)
                total_received = sum(line.quantity_received for line in lines)

                if total_pending == 0 and total_received > 0:
                    status = OrderStatus.RECEIVED
                elif total_received > 0:
                    status = OrderStatus.PARTIAL
                else:
                    status = OrderStatus.PENDING

                yield UnifiedOrder(
                    id=str(uuid.uuid4()),
                    erp_id=str(order_id),
                    supplier_code=order_data["supplier_code"],
                    supplier_name=order_data["supplier_name"],
                    order_date=order_data["order_date"],
                    status=status,
                    lines=lines,
                )

        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Receipts Data
    # ==========================================================================

    def get_recent_receipts(
        self,
        from_date: date,
        max_lines: int = 50000,
    ) -> Iterator[UnifiedReceipt]:
        """Get delivery notes received since date.

        Uses ges407 (purchases/receipts).

        Args:
            from_date: Start date (inclusive)
            max_lines: Maximum purchase lines to retrieve (prevents memory exhaustion).
                       Default 50000 lines. Set to 0 for no limit (use with caution).

        Note:
            This method groups receipt lines by delivery note (albarán) which requires
            loading all lines into memory. The max_lines parameter prevents OOM errors
            with large date ranges. For very large datasets, consider pagination.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        limit_clause = f"LIMIT {max_lines}" if max_lines > 0 else ""
        query = f"""
            SELECT
                c.idCompra,
                c.carticulo,
                a.cnombre,
                c.icantidad,
                c.bcoste,
                c.bpvp,
                c.tfecha,
                c.ialbaran,
                c.ifactura,
                c.clote,
                c.dcaducidad,
                p.iproveedor,
                p.cnombre as proveedor_nombre
            FROM ges407 c
            LEFT JOIN ges101 a ON c.idarticulo = a.idarticulo
            LEFT JOIN ges401 p ON c.iproveedor = p.iproveedor
            WHERE DATE(c.tfecha) >= %s
            ORDER BY c.tfecha DESC
            {limit_clause}
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query, (from_date,))

            # Group by delivery note (ialbaran)
            receipts: dict[str, dict] = {}
            row_count = 0

            for row in cursor:
                row_count += 1
                albaran_id = row["ialbaran"] or row["idCompra"]
                key = f"{row['iproveedor']}_{albaran_id}"

                if key not in receipts:
                    receipts[key] = {
                        "id": str(uuid.uuid4()),
                        "erp_id": str(row["idCompra"]),
                        "supplier_code": str(row["iproveedor"] or ""),
                        "supplier_name": row["proveedor_nombre"] or "Unknown",
                        "receipt_date": row["tfecha"],
                        "delivery_note_number": str(albaran_id) if albaran_id else None,
                        "invoice_number": str(row["ifactura"]) if row.get("ifactura") else None,
                        "lines": [],
                    }

                receipts[key]["lines"].append(ReceiptLine(
                    product_code=row["carticulo"] or "",
                    product_name=row["cnombre"] or "",
                    quantity=row["icantidad"] or 0,
                    unit_cost=Decimal(str(row["bcoste"] or 0)),
                    pvp=Decimal(str(row["bpvp"] or 0)),
                    lot_number=row["clote"],
                    expiry_date=row["dcaducidad"],
                ))

            # Warn if limit was hit
            if max_lines > 0 and row_count >= max_lines:
                logger.warning(
                    f"get_recent_receipts hit max_lines limit ({max_lines}). "
                    f"Some receipts may be missing. Consider using a narrower date range."
                )

            for receipt_data in receipts.values():
                yield UnifiedReceipt(**receipt_data)

        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Customers Data
    # ==========================================================================

    def get_customers(
        self,
        max_customers: int = 50000,
    ) -> Iterator[UnifiedCustomer]:
        """Get loyalty program customers.

        Uses ges050 (customers) and ges051 (patient details).

        Args:
            max_customers: Maximum customers to retrieve. Default 50000.
                          Large pharmacies may have 20k+ loyalty members.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        query = f"""
            SELECT
                c.idcliente,
                c.cnombre,
                c.ctelefono1,
                c.cmail,
                s.TSI,
                s.dfechanac,
                s.ipuntosfidelizacion,
                s.ddispensacion
            FROM ges050 c
            LEFT JOIN ges051 s ON c.idcliente = s.idcliente AND s.lprincipal = 1
            ORDER BY c.idcliente
            LIMIT {max_customers}
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query)

            for row in cursor:
                yield UnifiedCustomer(
                    id=str(uuid.uuid4()),
                    erp_id=str(row["idcliente"]),
                    name=row["cnombre"] or "",
                    phone=row["ctelefono1"],
                    email=row["cmail"],
                    loyalty_points=row.get("ipuntosfidelizacion") or 0,
                    last_purchase=row.get("ddispensacion"),
                    health_card_number=row.get("TSI"),
                    birth_date=row.get("dfechanac"),
                )
        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Employees Data
    # ==========================================================================

    def get_employees(self, max_employees: int = 100) -> Iterator[UnifiedEmployee]:
        """Get pharmacy employees.

        Uses 'empleado' table.

        Args:
            max_employees: Maximum employees to retrieve. Default 100.
                          Most pharmacies have <20 employees.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        query = f"""
            SELECT
                id,
                cnombre,
                cprimerapellido,
                csegundoapellido,
                cnif,
                cnumcole,
                ltitular,
                lfarmaceutico
            FROM empleado
            ORDER BY id
            LIMIT {max_employees}
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query)

            for row in cursor:
                full_name = " ".join(filter(None, [
                    row["cnombre"],
                    row["cprimerapellido"],
                    row.get("csegundoapellido"),
                ]))

                yield UnifiedEmployee(
                    id=str(uuid.uuid4()),
                    erp_id=str(row["id"]),
                    name=full_name,
                    first_name=row["cnombre"] or "",
                    last_name=row["cprimerapellido"] or "",
                    tax_id=row.get("cnif"),
                    college_number=row.get("cnumcole"),
                    is_owner=bool(row.get("ltitular", 0)),
                    is_pharmacist=bool(row.get("lfarmaceutico", 0)),
                )
        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # Customer Orders (Encargos) - Module 1: Cazador de Encargos
    # ==========================================================================

    def get_pending_customer_orders(
        self,
        max_orders: int = 5000,
    ) -> Iterator[UnifiedCustomerOrder]:
        """Get pending customer orders (encargos) from Farmanager.

        Uses ges331 (encargos header) + ges332 (encargos lines) + ges050 (customers).
        Filters by ges332.iestado = 0 (pending/not picked up).

        Schema discovery: docs/FARMANAGER_ENCARGOS_SCHEMA.md

        Args:
            max_orders: Maximum orders to retrieve. Default 5000.
                       Prevents memory issues with very old pharmacies.

        Yields:
            UnifiedCustomerOrder for each pending line.

        Note:
            Returns one UnifiedCustomerOrder per LINE, not per encargo header.
            This allows tracking individual products separately.
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        # SECURITY: Validate max_orders to prevent SQL injection (Issue #1 from code review)
        if not isinstance(max_orders, int) or max_orders < 0:
            raise ValueError("max_orders must be a non-negative integer")

        query = f"""
            SELECT
                CONCAT('ENC-', e.idencargo, '-', l.idlinea) as erp_order_id,
                e.idencargo,
                l.idlinea,
                e.tfechahora as fecha_encargo,
                DATEDIFF(NOW(), e.tfechahora) as dias_pendiente,
                c.cnombre as customer_name,
                COALESCE(e.ctelefono1, c.ctelefono1) as customer_phone,
                COALESCE(e.cmail, c.cmail) as customer_email,
                l.carticulo as product_cn,
                l.cnombre as product_description,
                l.iunidades as units,
                l.bimporte as total_amount
            FROM ges331 e
            JOIN ges332 l ON e.idencargo = l.idencargo
            LEFT JOIN ges050 c ON e.idsubcliente = c.idcliente
            WHERE l.iestado = 0
            ORDER BY e.tfechahora ASC
            LIMIT {max_orders}
        """

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)
            cursor.execute(query)

            for row in cursor:
                # Handle NULL timestamps (Issue #3 from code review)
                created_at = row.get("fecha_encargo")
                if created_at is None:
                    logger.warning(
                        f"Customer order {row['erp_order_id']} has NULL fecha_encargo. "
                        f"Check ges331.tfechahora for idencargo={row.get('idencargo')}. "
                        "Skipping record."
                    )
                    continue

                # Use total_amount directly (Issue #2: bimporte is already total, not unit price)
                units = row.get("units") or 1
                total_amount = Decimal(str(row.get("total_amount") or 0))
                # Calculate unit price from total for consistency
                pvp_unit = total_amount / units if units > 0 else Decimal("0")

                # Handle bytes from MySQL CONCAT (Issue: MySQL returns bytes for CONCAT)
                erp_order_id = row["erp_order_id"]
                if isinstance(erp_order_id, bytes):
                    erp_order_id = erp_order_id.decode("utf-8")

                yield UnifiedCustomerOrder(
                    id=str(uuid.uuid4()),
                    erp_order_id=erp_order_id,
                    created_at=created_at,
                    customer_name=row.get("customer_name") or "Sin cliente",
                    customer_phone=row.get("customer_phone"),
                    customer_email=row.get("customer_email"),
                    product_code=row.get("product_cn") or "",
                    product_name=row.get("product_description") or "",
                    units=units,
                    pvp=pvp_unit,  # Now correctly stores unit price
                    days_pending=row.get("dias_pendiente") or 0,
                )
        finally:
            if cursor:
                cursor.close()

    def get_pending_customer_orders_summary(self) -> dict:
        """Get summary statistics for pending customer orders.

        Returns:
            Dict with:
            - total_orders: Number of pending encargos
            - total_lines: Number of pending lines
            - total_units: Total units pending
            - total_value_pvp: Total PVP value locked
            - by_severity: Breakdown by severity level
        """
        if not self.is_connected():
            raise RuntimeError("Not connected to database")

        cursor = None
        try:
            cursor = self._connection.cursor(dictionary=True, buffered=True)

            # Overall summary
            cursor.execute("""
                SELECT
                    COUNT(DISTINCT e.idencargo) as total_orders,
                    COUNT(l.idlinea) as total_lines,
                    COALESCE(SUM(l.iunidades), 0) as total_units,
                    COALESCE(SUM(l.bimporte), 0) as total_value_pvp
                FROM ges331 e
                JOIN ges332 l ON e.idencargo = l.idencargo
                WHERE l.iestado = 0
            """)
            summary = cursor.fetchone()

            # By severity
            cursor.execute("""
                SELECT
                    CASE
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 30 THEN 'critical'
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 15 THEN 'urgent'
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 7 THEN 'warning'
                        ELSE 'info'
                    END as severity,
                    COUNT(*) as count,
                    COALESCE(SUM(l.bimporte), 0) as value_pvp
                FROM ges331 e
                JOIN ges332 l ON e.idencargo = l.idencargo
                WHERE l.iestado = 0
                GROUP BY
                    CASE
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 30 THEN 'critical'
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 15 THEN 'urgent'
                        WHEN DATEDIFF(NOW(), e.tfechahora) >= 7 THEN 'warning'
                        ELSE 'info'
                    END
            """)
            by_severity = {
                "critical": {"count": 0, "value_pvp": 0},
                "urgent": {"count": 0, "value_pvp": 0},
                "warning": {"count": 0, "value_pvp": 0},
                "info": {"count": 0, "value_pvp": 0},
            }
            for row in cursor.fetchall():
                sev = row["severity"]
                if sev in by_severity:
                    by_severity[sev]["count"] = row["count"]
                    by_severity[sev]["value_pvp"] = float(row["value_pvp"])

            return {
                "total_orders": summary["total_orders"],
                "total_lines": summary["total_lines"],
                "total_units": summary["total_units"],
                "total_value_pvp": float(summary["total_value_pvp"]),
                "by_severity": by_severity,
            }

        finally:
            if cursor:
                cursor.close()

    # ==========================================================================
    # FUTURE WRITE OPERATIONS (Currently blocked by READ-ONLY guard)
    # ==========================================================================
    # These methods are placeholders for future write functionality.
    # To enable, set read_only=False and add operation to ALLOWED_WRITE_OPS.

    def create_order_proposal(self, order: UnifiedOrder) -> str:
        """[FUTURE] Create an order proposal in Farmanager.

        Creates a new order proposal (propuesta de pedido) that can be
        reviewed and confirmed by the pharmacist.

        Args:
            order: UnifiedOrder with supplier and lines

        Returns:
            ERP order ID if successful

        Raises:
            FarmanagerReadOnlyError: If write not allowed
        """
        self._assert_write_allowed("propuesta_pedido")

        # TODO: Implement when write access is enabled
        # Tables: ges404 (order lines), ges406 (confirmed orders)
        raise NotImplementedError(
            "create_order_proposal is not yet implemented. "
            "This is a placeholder for future functionality."
        )

    def create_encargo(self, product_code: str, quantity: int, customer_id: str) -> str:
        """[FUTURE] Create a special order (encargo) for a customer.

        Creates a reservation/special order for a product that needs
        to be ordered specifically for a customer.

        Args:
            product_code: CN/EAN of product to order
            quantity: Units to reserve
            customer_id: Customer requesting the order

        Returns:
            ERP encargo ID if successful

        Raises:
            FarmanagerReadOnlyError: If write not allowed
        """
        self._assert_write_allowed("encargos")

        # TODO: Implement when write access is enabled
        # Tables: Need to investigate encargos table structure
        raise NotImplementedError(
            "create_encargo is not yet implemented. "
            "This is a placeholder for future functionality."
        )
