"""Base ERP Adapter interface.

All ERP-specific adapters must implement this interface to provide
unified access to pharmacy data.
"""

from abc import ABC, abstractmethod
from datetime import date
from typing import Iterator, Optional

from .models import (
    UnifiedCustomer,
    UnifiedEmployee,
    UnifiedOrder,
    UnifiedReceipt,
    UnifiedSale,
    UnifiedStock,
)


class ERPAdapter(ABC):
    """Abstract base class for ERP database adapters.

    Each pharmacy ERP (Farmatic, Nixfarma, Farmanager, etc.) has its own
    database schema. Adapters translate ERP-specific schemas to unified
    models that kaiFarma can process consistently.

    Usage:
        adapter = FarmanagerAdapter(connection_string="mysql://...")
        if adapter.connect():
            for sale in adapter.get_sales(from_date, to_date):
                process_sale(sale)
    """

    @property
    @abstractmethod
    def erp_name(self) -> str:
        """Name of the ERP system (e.g., 'Farmanager', 'Farmatic')."""
        pass

    @property
    @abstractmethod
    def erp_version(self) -> Optional[str]:
        """ERP version if detectable."""
        pass

    @abstractmethod
    def connect(self) -> bool:
        """Establish connection to the ERP database.

        Returns:
            True if connection successful, False otherwise.
        """
        pass

    @abstractmethod
    def disconnect(self) -> None:
        """Close connection to the ERP database."""
        pass

    @abstractmethod
    def is_connected(self) -> bool:
        """Check if currently connected."""
        pass

    @abstractmethod
    def test_connection(self) -> tuple[bool, str]:
        """Test database connection.

        Returns:
            Tuple of (success, message).
            Message contains error details if failed.
        """
        pass

    # ==========================================================================
    # Core Data Methods
    # ==========================================================================

    @abstractmethod
    def get_sales(
        self,
        from_date: date,
        to_date: date,
        include_prescriptions: bool = True,
    ) -> Iterator[UnifiedSale]:
        """Get sales in date range.

        Args:
            from_date: Start date (inclusive)
            to_date: End date (inclusive)
            include_prescriptions: Whether to include prescription sales

        Yields:
            UnifiedSale objects
        """
        pass

    @abstractmethod
    def get_current_stock(self) -> Iterator[UnifiedStock]:
        """Get current inventory snapshot.

        Yields:
            UnifiedStock objects for all products with stock > 0
        """
        pass

    @abstractmethod
    def get_pending_orders(self) -> Iterator[UnifiedOrder]:
        """Get orders pending to suppliers.

        Yields:
            UnifiedOrder objects for pending/partial orders
        """
        pass

    @abstractmethod
    def get_recent_receipts(
        self,
        from_date: date,
    ) -> Iterator[UnifiedReceipt]:
        """Get delivery notes received since date.

        Args:
            from_date: Start date (inclusive)

        Yields:
            UnifiedReceipt objects
        """
        pass

    # ==========================================================================
    # Optional Methods (Not all ERPs support these)
    # ==========================================================================

    def get_customers(self) -> Iterator[UnifiedCustomer]:
        """Get loyalty program customers.

        Returns:
            Iterator of UnifiedCustomer, or empty if not supported.
        """
        return iter([])

    def get_employees(self) -> Iterator[UnifiedEmployee]:
        """Get pharmacy employees.

        Returns:
            Iterator of UnifiedEmployee, or empty if not supported.
        """
        return iter([])

    # ==========================================================================
    # Utility Methods
    # ==========================================================================

    def get_sales_count(
        self,
        from_date: date,
        to_date: date,
    ) -> int:
        """Get count of sales in date range.

        Override for efficiency if ERP supports COUNT queries.
        """
        return sum(1 for _ in self.get_sales(from_date, to_date))

    def get_stock_count(self) -> int:
        """Get count of products in stock.

        Override for efficiency if ERP supports COUNT queries.
        """
        return sum(1 for _ in self.get_current_stock())

    # ==========================================================================
    # Context Manager Support
    # ==========================================================================

    def __enter__(self) -> "ERPAdapter":
        """Context manager entry."""
        self.connect()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb) -> None:
        """Context manager exit."""
        self.disconnect()
