# backend/app/database.py
import os

from dotenv import load_dotenv
from sqlalchemy import create_engine, text
from sqlalchemy.orm import declarative_base, sessionmaker

# Cargar variables de entorno
# IMPORTANTE: override=False para NO sobreescribir variables ya configuradas
# por main.py (ej: KAIFARMA_LOCAL, DATABASE_URL para SQLite en modo local)
load_dotenv(override=False)

# URL de la base de datos
DATABASE_URL = os.getenv("DATABASE_URL", "postgresql://xfarma_user:xfarma_dev_2024@localhost:5432/xfarma_db")

# Convertir postgresql:// a postgresql+psycopg2:// si es necesario
if DATABASE_URL.startswith("postgresql://"):
    DATABASE_URL = DATABASE_URL.replace("postgresql://", "postgresql+psycopg2://", 1)

# Logging de configuración solo si no es producción
import logging

logger = logging.getLogger(__name__)

ENVIRONMENT = os.getenv("ENVIRONMENT", "development")
if ENVIRONMENT != "production":
    logger.info(f"DATABASE_URL del entorno: {os.getenv('DATABASE_URL', 'NO_SET')}")
    logger.info(f"ENVIRONMENT: {ENVIRONMENT}")

    # Debug: Mostrar configuración de DB (sin password)
    if "@" in DATABASE_URL:
        db_url_debug = DATABASE_URL.split("@")[0].split("//")[1].split(":")[0] + "@" + DATABASE_URL.split("@")[1]
    else:
        db_url_debug = DATABASE_URL
    logger.info(f"DATABASE_URL final: {db_url_debug}")

# Crear engine con configuración optimizada para producción
# ADR-004: Pool reducido para 2 workers en Render Starter (512MB)
# Fórmula: pool_size = (workers × 2) + 1 = 5
# Techo: 10 conexiones (5 + 5 overflow), margen de 10 para admin/scripts
engine = create_engine(
    DATABASE_URL,
    echo=False,  # Deshabilitado para mejor rendimiento
    pool_pre_ping=True,  # Verificar conexión antes de usar
    pool_size=5,  # ADR-004: Reducido de 10 a 5 (2 workers)
    max_overflow=5,  # ADR-004: Reducido de 20 a 5 (techo 10 conexiones)
    pool_recycle=3600,  # Reciclar conexiones cada hora (1h)
    pool_timeout=30,  # Timeout para obtener conexión (30s)
    connect_args=(
        {
            "connect_timeout": 10,  # Timeout de conexión inicial
            "application_name": "xfarma-backend",  # Identificar en logs PostgreSQL
            "client_encoding": "utf8",  # ✅ FIX: Forzar UTF-8 para caracteres españoles (Ñ, tildes)
        }
        if DATABASE_URL.startswith("postgresql")
        else {}
    ),
)

# Crear sesión
SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)

# Base para los modelos
Base = declarative_base()


# Dependency para obtener DB session en FastAPI
def get_db() -> SessionLocal:
    """
    Dependency para obtener una sesión de base de datos.
    Se usa en los endpoints de FastAPI.

    Yields:
        SessionLocal: Sesión de base de datos SQLAlchemy
    """
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()


# Función para verificar conexión
def check_database_connection() -> bool:
    """
    Verifica que la conexión a la base de datos funcione.
    Útil para health checks.

    Returns:
        bool: True si la conexión es exitosa, False en caso contrario
    """
    try:
        with engine.connect() as conn:
            conn.execute(text("SELECT 1"))
        return True
    except Exception as e:
        print(f"[ERROR] Error conectando a la base de datos: {e}")
        return False


# Función para crear todas las tablas
def init_db() -> bool:
    """
    Crea todas las tablas en la base de datos.
    Importa todos los modelos para asegurar que estén registrados.

    Returns:
        bool: True si las tablas se crearon exitosamente, False en caso contrario
    """
    try:
        # Importar todos los modelos desde el módulo centralizado

        # Crear tablas usando Base.metadata
        Base.metadata.create_all(bind=engine)
        logger.info("Tablas creadas exitosamente")
        return True
    except Exception as e:
        logger.error(f"Error creando tablas: {e}")
        return False
