# backend/app/core/taxonomy_tier1.py
"""
Taxonomía de 3 Niveles para Clasificación de Parafarmacia.

Arquitectura de Prioridad:
1. BLACKLIST - Productos a excluir (regalo, servicio, etc.)
2. TIER_1_SPECIFIC - Reglas específicas (aftas > colutorio)
3. TIER_1_GENERIC - Reglas genéricas (colutorio, pasta dental)

Las reglas de ingredient_rules.py actúan como TIER_2.
"""

import re
import unicodedata
from dataclasses import dataclass
from typing import Dict, FrozenSet, List, Optional, Tuple

# -----------------------------------------------------------------------------
# NIVEL 0: BLACKLIST (Lo que NO se debe clasificar como producto comercial)
# -----------------------------------------------------------------------------
BLACKLIST_KEYWORDS: FrozenSet[str] = frozenset({
    "regalo",
    "obsequio",
    "gratis",
    "tester",
    "probador",
    "muestras",
    "consulta",
})
# NOTA: Removido "servicio" y "bolsa" - demasiado genéricos
# "servicio" captura productos legítimos
# "bolsa" aparece en potitos/alimentación bebé

# SPD capturado como regex en TIER_1_SPECIFIC para mejor control

# -----------------------------------------------------------------------------
# NIVEL 1: REGLAS ESPECÍFICAS (Prioridad Alta)
# Se ejecutan PRIMERO. Si hay match aquí, paramos.
# Formato: (regex_pattern, necesidad, confidence)
# IMPORTANTE: Los nombres de categoría deben coincidir con symptom_taxonomy.py
# -----------------------------------------------------------------------------
TIER_1_SPECIFIC: List[Tuple[str, str, float]] = [
    # === SERVICIOS FARMACIA (SPD = Sistema Personalizado Dosificación) ===
    # Issue #457: 6 correcciones interno_no_venta → servicios_farmacia
    (r"^spd\b", "servicios_farmacia", 0.98),  # SPD ANA, SPD DANIEL, etc.
    (r"\bspd\s+\w+", "servicios_farmacia", 0.95),  # SPD seguido de nombre
    (r"\bspd\s+semanal\b", "servicios_farmacia", 0.98),  # SPD semanal

    # === REGALOS/BAZAR (Issue #457: decoraciones navideñas, juguetes) ===
    (r"\badorno\b", "regalos_bazar", 0.95),
    (r"\belfo\b", "regalos_bazar", 0.95),
    (r"\bpapa\s*noel\b", "regalos_bazar", 0.98),
    (r"\bgaspar\b", "regalos_bazar", 0.95),  # Reyes Magos
    (r"\bmelchor\b", "regalos_bazar", 0.95),
    (r"\bbaltasar\b", "regalos_bazar", 0.95),
    (r"\bnavide[nñ]", "regalos_bazar", 0.95),
    (r"\bjuguete\b", "regalos_bazar", 0.90),

    # === CEPILLOS ELÉCTRICOS (TIER1_SPECIFIC para prioridad) ===
    (r"\bcepillo.*electrico", "cepillo_electrico", 0.95),
    (r"\bcepillo.*recargable", "cepillo_electrico", 0.95),
    (r"\brecambio.*oral[- ]?b", "cepillo_electrico", 0.95),
    (r"\boral[- ]?b.*recambio", "cepillo_electrico", 0.95),
    (r"\boral[- ]?b\s+io\b", "cepillo_electrico", 0.95),  # iO Series (standalone word)
    (r"\boral[- ]?b.*vitality", "cepillo_electrico", 0.95),
    (r"\bsonicare\b", "cepillo_electrico", 0.95),

    # === EXFOLIACIÓN (TIER1_SPECIFIC para prioridad) ===
    (r"\blocion.*exfolia", "exfoliacion", 0.95),
    (r"\bexfolia.*locion", "exfoliacion", 0.95),
    (r"\bpate.*exfolia", "exfoliacion", 0.95),
    (r"\bexfoliante\b", "exfoliacion", 0.93),

    # === DENTAL ESPECÍFICO (Fix: Subcategorías dentales) ===
    # Anticaries: usuario busca TRATAR caries, no higiene genérica
    (r"\banticaries\b", "caries", 0.95),
    (r"\banti[- ]?caries", "caries", 0.95),
    # Sensibilidad dental: usuarios pagan extra por el tratamiento
    (r"\bdesensin\b", "sensibilidad_dental", 0.95),
    (r"\bsensibilidad.*dental", "sensibilidad_dental", 0.92),
    (r"\bafta", "aftas_llagas", 0.95),
    (r"\bllaga", "aftas_llagas", 0.95),
    (r"\baftex\b", "aftas_llagas", 0.95),
    (r"\baloclair\b", "aftas_llagas", 0.95),
    (r"\binterdental\b", "interdental", 0.92),  # INTERPROX, cepillos interdentales
    (r"\binterprox\b", "interdental", 0.95),
    (r"\bortodoncia", "higiene_dental", 0.90),
    (r"\bblanquea", "blanqueamiento", 0.85),
    (r"\benc[ií]a", "encias", 0.90),
    (r"\bgingiv", "encias", 0.95),
    (r"\bgingi", "encias", 0.92),  # GINGILACER, GINGIDEX, etc.
    (r"\bparogencyl\b", "encias", 0.95),
    (r"\bgingilacer\b", "encias", 0.95),
    (r"\bparoex\b", "encias", 0.95),  # GUM PAROEX
    (r"\bxerostom", "halitosis", 0.90),  # Boca seca -> halitosis
    (r"\bsensibil.*dental", "sensibilidad_dental", 0.90),
    (r"\bsensodyne\b", "sensibilidad_dental", 0.95),
    (r"\bsensivital\b", "sensibilidad_dental", 0.95),  # GUM SENSIVITAL
    (r"\bdientes sensibles", "sensibilidad_dental", 0.90),
    (r"\bpro[- ]?esmalte", "sensibilidad_dental", 0.85),

    # === PIEL Y HERIDAS (Fix: Cicatrización) ===
    (r"\bcicatri[zc]", "cicatrizacion", 0.95),
    (r"\bregeneran", "cicatrizacion", 0.90),
    (r"\brosa mosqueta", "cicatrizacion", 0.90),
    (r"\bagrietad[ao]s?\b", "cicatrizacion", 0.95),  # Manos agrietadas
    (r"\bagredid[ao]s?\b", "cicatrizacion", 0.90),  # Manos agredidas
    (r"\bbio[- ]?oil\b", "cicatrizacion", 0.95),
    (r"\bbepanthol\b", "cicatrizacion", 0.95),
    (r"\bblastoestimulina\b", "cicatrizacion", 0.95),
    (r"\bcontractubex\b", "cicatrizacion", 0.95),
    (r"\bcicatricure\b", "cicatrizacion", 0.95),
    (r"\bsuavinex.*cicatri", "cicatrizacion", 0.90),
    (r"\bcicabio\b", "cicatrizacion", 0.95),  # BIODERMA CICABIO
    (r"\baquaphor\b", "cicatrizacion", 0.95),  # Pomada reparadora oclusiva
    (r"\bnutracel\b", "cicatrices_estrias", 0.95),  # NUTRACEL antiestrías
    (r"\bestr[ií]a", "cicatrices_estrias", 0.90),
    (r"\btrofolastin\b", "cicatrices_estrias", 0.95),
    (r"\bquemadura", "quemaduras", 0.95),
    (r"\b[uú]lcer", "apositos_curas", 0.90),
    # BE+ MED protector/parche piel = apósitos protectores, no piel sensible
    (r"\bprotector.*piel\b", "apositos_curas", 0.92),
    (r"\bparche.*protector", "apositos_curas", 0.92),
    # Picaduras: productos específicos para aliviar picor
    (r"\bcalmatopic\b", "picaduras", 0.95),  # Roll-on/stick post-picaduras
    (r"\bpost[- ]?picadura", "picaduras", 0.95),
    (r"\bpicadura", "picaduras", 0.90),
    # Dermatitis seborreica (DS): rojeces y descamación (PIEL, no cabello)
    (r"\bdscontrol\b", "dermatitis_seborreica", 0.95),
    (r"\bds\s*control\b", "dermatitis_seborreica", 0.95),
    (r"\bdermatitis.*seborr", "dermatitis_seborreica", 0.95),
    # Piel atópica
    (r"\bat[oó]pi", "piel_atopica", 0.95),
    (r"\bpsorias", "psoriasis", 0.95),
    (r"\bpsorisdin\b", "psoriasis", 0.95),
    (r"\burea\s*(40|50)%?", "verrugas_callos", 0.90),  # Urea alta = queratolítico
    (r"\bverruga", "verrugas_callos", 0.95),
    (r"\bcallo\b", "verrugas_callos", 0.95),

    # === PROTESIS DENTAL (Issue #457: antes de limpieza_facial) ===
    (r"\blimpia.*pr[oó]tesis", "protesis_dental", 0.98),
    (r"\bpr[oó]tesis.*limpia", "protesis_dental", 0.98),
    (r"\bkin\s*oro\b", "protesis_dental", 0.98),  # Marca específica prótesis
    (r"\bcorega\b", "protesis_dental", 0.95),  # Marca específica prótesis

    # === CANDIDIASIS VAGINAL (Issue #457: antes de higiene_intima) ===
    (r"\bgine[- ]?canesten\b", "hongos_vaginales", 0.98),
    (r"\bcandidiasis", "hongos_vaginales", 0.95),
    (r"\bcandida\b", "hongos_vaginales", 0.90),
    (r"\bhongos?\s*vaginal", "hongos_vaginales", 0.95),

    # === CLORHEXIDINA BUCAL (Issue #457: antes de desinfeccion_heridas) ===
    # Productos bucales con clorhexidina van a encías/garganta, NO desinfección
    (r"\blizipaina\b", "dolor_garganta", 0.98),  # Garganta, tiene clorhexidina
    (r"\blacer.*clorhexidina", "encias", 0.98),  # Encías
    (r"\bclorhexidina.*enc[ií]a", "encias", 0.98),
    (r"\bcolutorio.*clorhexidina", "encias", 0.95),
    (r"\bclorhexidina.*bucal", "encias", 0.95),
    (r"\bclorhexidina.*dental", "encias", 0.95),
    (r"\bperio[- ]?aid\b", "encias", 0.95),  # PerioAid clorhexidina

    # === INGREDIENTES MÉDICOS ESPECÍFICOS (Migración de ingredient_rules) ===
    (r"\bclorhexidina\b", "desinfeccion_heridas", 0.95),  # Fallback: desinfección
    (r"\bminoxidil\b", "caida_cabello", 0.98),
    (r"\bpermetrina\b", "piojos", 0.98),
    (r"\bbenzoca[ií]na\b", "dolor_garganta", 0.90),
    # Bucomax ANTES de lidocaína (Bucomax tiene lidocaína pero es para garganta)
    (r"\bbucomax\b", "dolor_garganta", 0.95),
    (r"\blidoca[ií]na\b", "dolor_dental", 0.90),
    (r"\bketoprofeno\b", "dolor_muscular", 0.90),
    (r"\bdiclofenaco\b", "dolor_muscular", 0.90),
    (r"\bclotrimazol\b", "hongos_piel", 0.95),
    (r"\bmiconazol\b", "hongos_piel", 0.95),
    (r"\bterbinafina\b", "hongos_piel", 0.95),
    (r"\baciclovir\b", "herpes", 0.98),
    (r"\bhidrocortisona\b", "dermatitis", 0.90),

    # === ESTRÉS Y ANSIEDAD ===
    (r"\brelax\b", "estres_ansiedad", 0.85),
    (r"\bansiedad", "estres_ansiedad", 0.90),
    # Removed: \bcalma - too broad, matches "crema calmante" (skin products)
    (r"\bequirelax\b", "estres_ansiedad", 0.95),  # NS EQUIRELAX
    (r"\bcalmarelax\b", "estres_ansiedad", 0.95),  # NS CALMARELAX
    (r"\bvaleriana", "estres_ansiedad", 0.90),
    (r"\bpassiflora", "estres_ansiedad", 0.90),

    # === CONTORNO DE OJOS ===
    (r"\bcontorno.*ojos", "contorno_ojos", 0.95),
    (r"\bcontorno.*ojo\b", "contorno_ojos", 0.95),
    (r"\beye.*contour", "contorno_ojos", 0.90),

    # === FORMATO: CORPORALES POR VOLUMEN (>200ml = corporal, no facial) ===
    # Eucerin PH-5 LOCION 1L, lociones 400ml, etc. son corporales
    (r"\blocion.*1\s*l\b", "hidratacion_corporal", 0.95),
    (r"\blocion.*1000\s*ml", "hidratacion_corporal", 0.95),
    (r"\blocion.*750\s*ml", "hidratacion_corporal", 0.95),
    (r"\blocion.*500\s*ml", "hidratacion_corporal", 0.92),
    (r"\blocion.*400\s*ml", "hidratacion_corporal", 0.92),
    (r"\blocion.*300\s*ml", "hidratacion_corporal", 0.90),
    (r"\bph[- ]?5\b.*locion", "hidratacion_corporal", 0.95),  # Eucerin PH-5 siempre corporal

    # === SUPLEMENTOS CAPILARES ===
    (r"\baminohair\b", "caida_cabello", 0.95),
    (r"\bcomplidermol\b", "caida_cabello", 0.95),
    (r"\bhairpil\b", "caida_cabello", 0.95),
    (r"\bpantoteno.*cabello", "caida_cabello", 0.90),

    # === NUTRICIÓN CLÍNICA (productos médicos, no trivializar) ===
    (r"\bbarimix\b", "nutricion_clinica", 0.98),  # Bypass gástrico - no es vitaminas genéricas
    (r"\bbariatr", "nutricion_clinica", 0.95),  # Cirugía bariátrica
    (r"\bfresubin\b", "nutricion_clinica", 0.95),
    (r"\bensure\b", "nutricion_clinica", 0.95),
    (r"\bfortimel\b", "nutricion_clinica", 0.95),
    (r"\bresource\b", "nutricion_clinica", 0.90),

    # === NUTRICIÓN DEPORTIVA ===
    (r"\bprote[ií]na\b", "nutricion_deportiva", 0.85),
    (r"\bprotein\b", "nutricion_deportiva", 0.90),
    (r"\bwhey\b", "nutricion_deportiva", 0.95),
    (r"\bcreatin", "nutricion_deportiva", 0.95),
    (r"\bbcaa\b", "nutricion_deportiva", 0.95),
    (r"\bpre[- ]?workout", "nutricion_deportiva", 0.95),
    (r"\biso\s*100\b", "nutricion_deportiva", 0.95),
    (r"\bcorny.*protein", "nutricion_deportiva", 0.95),
    (r"\bbarrita.*protein", "nutricion_deportiva", 0.90),

    # === ARTICULAR VS MUSCULAR (Fix: Colágeno) ===
    (r"\bcol[aá]geno", "colageno", 0.90),  # Usar categoria existente
    (r"\bcondroitina", "colageno", 0.95),
    (r"\bglucosamina", "colageno", 0.95),
    (r"\bepaplus\b", "colageno", 0.95),
    (r"\bcolnatur\b", "colageno", 0.95),
    (r"\bfisiocrem\b", "dolor_muscular", 0.95),
    (r"\breflex\b", "dolor_muscular", 0.90),
    (r"\bradiosalil\b", "dolor_muscular", 0.90),
    (r"\b[aá]rnica\b", "hematomas", 0.95),
    (r"\barnidol\b", "hematomas", 0.95),

    # === CAPILAR ESPECÍFICO ===
    # INSIGHT: En capilar de farmacia, la GAMA define la necesidad, no la marca madre
    # René Furterer es genérico, pero Triphasic=caída, Astera=sensible, Neopur=caspa

    # --- EXCEPCIONES DS/SEBORREA (ANTES de marcas anticaída) ---
    # Iraltone/Pilopeptan son famosas por caída, pero tienen productos DS
    (r"\bds\s+emulsion", "caspa", 0.99),        # IRALTONE DS EMULSION → caspa, no caída
    (r"\bseb\s+ds\b", "caspa", 0.99),           # PILOPEPTAN SEB DS → caspa, no caída
    (r"\banti[- ]?escamas", "caspa", 0.95),     # Anti-escamas = caspa
    (r"\bnormalizante.*capilar", "caspa", 0.92), # Normalizante capilar = seborrea
    (r"\buso\s+frecuente", "capilar_frecuente", 0.95),  # Champú uso frecuente
    (r"\bextra\s+suave", "capilar_frecuente", 0.90),    # Extra suave
    # Issue #457: Champú en seco (dry shampoo) = cuidado_cabello, no accesorios
    (r"\bchampu.*seco\b", "cuidado_cabello", 0.95),
    (r"\bchampu.*en\s*seco\b", "cuidado_cabello", 0.95),
    (r"\bdry\s*shampoo\b", "cuidado_cabello", 0.95),

    # --- GAMAS RENÉ FURTERER (categorías actualizadas Issue #457) ---
    (r"\b5\s*sens\b", "capilar_hidratacion", 0.95),       # Aceite sublimador
    (r"\babsolue\s*keratine", "capilar_hidratacion", 0.95), # Reparación keratina
    (r"\bnaturia\b", "capilar_frecuente", 0.95),        # Champú micelar suave
    (r"\bkarite\b", "capilar_hidratacion", 0.95),       # Hidratación/disciplina
    (r"\btonucia\b", "capilar_densidad", 0.95),         # Redensificante (pelo fino)
    (r"\bsublime\s*curl", "capilar_hidratacion", 0.95), # Rizos
    (r"\bokara\b", "capilar_color", 0.95),              # Color (silver/blond)
    (r"\bcolor\s*glow", "capilar_color", 0.95),         # Protección térmica color
    (r"\bstyle\s*velo", "capilar_frecuente", 0.95),     # Styling
    (r"\bsolaire\b.*furterer", "proteccion_solar", 0.95),  # Solar capilar

    # --- PIOJOS (Pediculosis) ---
    (r"\bpiojo", "piojos", 0.98),
    (r"\bliendre", "piojos", 0.95),
    (r"\blendrera", "piojos", 0.95),
    (r"\bpedicul", "piojos", 0.95),
    (r"\bparanix\b", "piojos", 0.98),
    (r"\bfullmarks\b", "piojos", 0.98),
    (r"\bfilvit\b", "piojos", 0.98),
    (r"\botc.*antipiojos", "piojos", 0.95),
    (r"\bgoibi.*piojo", "piojos", 0.98),  # Goibi piojos (no confundir con picaduras)
    # Issue #457: NOSA desenredante con árbol del té = preventivo piojos
    (r"\bnosa\b.*desenredante", "piojos", 0.95),
    (r"\bnosa\b.*arbol.*te", "piojos", 0.95),

    # --- ANTICAÍDA (El Rey de la categoría) ---
    (r"\bc[aá][ií]da.*cabell", "caida_cabello", 0.95),
    (r"\banticaida", "caida_cabello", 0.95),
    (r"\balopecia", "caida_cabello", 0.95),
    (r"\bdensidad.*capilar", "caida_cabello", 0.90),
    (r"\bronquina\b", "caida_cabello", 0.90),
    (r"\baminexil\b", "caida_cabello", 0.95),
    (r"\bserenoa\b", "caida_cabello", 0.90),
    # Gamas Anticaída Famosas
    (r"\btriphasic\b", "caida_cabello", 0.98),   # René Furterer
    (r"\bforticea\b", "caida_cabello", 0.98),    # René Furterer
    (r"\bpilexil\b", "caida_cabello", 0.98),     # Lacer
    (r"\blambdapil\b", "caida_cabello", 0.98),   # Isdin
    (r"\biraltone\b", "caida_cabello", 0.98),    # Cantabria
    (r"\bpilopeptan\b", "caida_cabello", 0.98),
    (r"\bpriorin\b", "caida_cabello", 0.98),
    (r"\bkaidax\b", "caida_cabello", 0.98),
    (r"\banacaps\b", "caida_cabello", 0.98),     # Ducray oral
    (r"\banaphase\b", "caida_cabello", 0.98),    # Ducray champú/acondicionador
    (r"\bcreastim\b", "caida_cabello", 0.98),    # Ducray loción
    (r"\bneoptide\b", "caida_cabello", 0.98),    # Ducray loción
    (r"\bolistic\b", "caida_cabello", 0.95),
    # Minoxidil (Medicamentos OTC anticaída)
    (r"\bminoxidil\b", "caida_cabello", 0.98),
    (r"\balocutan\b", "caida_cabello", 0.98),    # Almirall
    (r"\bregaxidil\b", "caida_cabello", 0.98),
    (r"\blacovin\b", "caida_cabello", 0.98),
    # Vichy Dercos anticaída
    (r"\bdercos.*aminexil", "caida_cabello", 0.98),

    # --- CASPA Y DESCAMACIÓN ---
    (r"\bcaspa", "caspa", 0.95),
    (r"\banticaspa", "caspa", 0.95),
    (r"\bseborr.*capilar", "caspa", 0.92),       # Seborrea capilar -> caspa
    (r"\bseborr.*champu", "caspa", 0.92),
    (r"\bcostra.*lactea", "caspa", 0.90),        # Bebés
    (r"\bexfoliante.*capilar", "caspa", 0.90),
    # Gamas Caspa/DS Famosas
    (r"\bkertyol\b", "caspa", 0.98),             # Ducray
    (r"\bkelual\b", "caspa", 0.98),              # Ducray
    (r"\bsquanorm\b", "caspa", 0.98),            # Ducray
    (r"\bneopur\b", "caspa", 0.98),              # René Furterer
    (r"\bcurbicia\b", "caspa", 0.98),            # René Furterer
    (r"\bt[- ]?gel\b", "caspa", 0.95),           # Neutrogena T-Gel
    (r"\bnode.*ds\b", "caspa", 0.95),            # Bioderma Node DS
    # Isdin caspa/psoriasis
    (r"\bpsorisdin\b", "caspa", 0.98),           # Isdin - psoriasis capilar
    (r"\bnutradeica\b", "caspa", 0.98),          # Isdin - dermatitis seborreica
    # Vichy Dercos caspa
    (r"\bdercos.*anticaspa", "caspa", 0.98),
    (r"\bdercos.*psolution", "caspa", 0.98),     # Psoriasis capilar

    # --- CUERO CABELLUDO SENSIBLE (Issue #457 - categoría actualizada) ---
    (r"\bcuero.*cabelludo.*sensible", "capilar_cuero_sensible", 0.95),
    (r"\bpicor.*cuero", "capilar_cuero_sensible", 0.92),
    (r"\bantipruriginoso.*capilar", "capilar_cuero_sensible", 0.92),
    # Gamas Calmantes Famosas
    (r"\bastera\b", "capilar_cuero_sensible", 0.98),    # René Furterer - CALMANTE, no caída!
    (r"\bsensinol\b", "capilar_cuero_sensible", 0.98),  # Ducray
    (r"\belution\b", "capilar_cuero_sensible", 0.95),   # Ducray (Dermoprotector)
    (r"\bsensirine\b", "capilar_cuero_sensible", 0.95), # Dercos

    # === INFANTIL ESPECÍFICO ===
    # Esponjas bebé NO son alimentación (fallo grave detectado)
    (r"\besponja.*infantil", "bebes_infantil", 0.98),
    (r"\besponja.*beb[eé]", "bebes_infantil", 0.98),
    (r"\besponja.*ba[nñ]o.*infantil", "bebes_infantil", 0.98),
    # Accesorios bebé: cepillos limpiabiberones, cubiertos -> bebes_infantil
    (r"\bcepillo.*limpia.*biberon", "bebes_infantil", 0.95),
    (r"\blimpia.*biberon", "bebes_infantil", 0.95),
    (r"\bcubierto.*infantil", "bebes_infantil", 0.95),
    (r"\bcuchara.*beb[eé]", "bebes_infantil", 0.90),
    # Alimentación bebé (comida/fórmula)
    (r"\bblemil\b", "alimentacion_bebe", 0.95),
    (r"\bnidina\b", "alimentacion_bebe", 0.95),
    (r"\bnutrib[eé]n\b", "alimentacion_bebe", 0.95),
    (r"\bhero\s*baby", "alimentacion_bebe", 0.95),
    (r"\bpotito", "alimentacion_bebe", 0.95),
    (r"\bpapilla", "alimentacion_bebe", 0.95),
    # Issue #457: Biberón/tetina = accesorios bebé, NO alimentación
    (r"\bbiber[oó]n", "bebes_infantil", 0.95),
    (r"\btetina\b", "bebes_infantil", 0.90),
    (r"\bdetergente.*biberon", "bebes_infantil", 0.95),
    (r"\bcepillo.*biberon", "bebes_infantil", 0.95),
    # Higiene bebé
    (r"\bpa[nñ]al", "higiene_bebe", 0.95),
    (r"\bdodot\b", "higiene_bebe", 0.95),
    (r"\btoallita.*beb[eé]", "higiene_bebe", 0.90),
    (r"\bmustela\b", "higiene_bebe", 0.88),
    (r"\bpediatrics\b", "higiene_bebe", 0.85),
    (r"\bdermatitis.*pa[nñ]al", "dermatitis_panal", 0.95),
    (r"\bcambio.*pa[nñ]al", "dermatitis_panal", 0.90),
    # Confort bebé
    (r"\bc[oó]lico", "colicos_bebe", 0.95),
    (r"\banticolic", "colicos_bebe", 0.95),
    (r"\bchupete", "confort_bebe", 0.90),
    # Cuidado nasal bebé (NO es alimentación)
    (r"\baspirador.*nasal", "bebes_infantil", 0.95),
    (r"\bsuavinex.*aspirador", "bebes_infantil", 0.95),
    # Accesorios bebé (broches, cadenas, etc.) - NO es alimentación
    (r"\bbroche.*chupete", "bebes_infantil", 0.95),
    (r"\bbroche.*silicona.*suavinex", "bebes_infantil", 0.95),
    (r"\bbroche.*cadena", "bebes_infantil", 0.95),
    (r"\bcadenita.*chupete", "bebes_infantil", 0.95),
    (r"\bbroche.*suavinex", "bebes_infantil", 0.95),  # Cualquier broche Suavinex
    (r"\bsuavinex.*broche", "bebes_infantil", 0.95),
    # Issue #457: Lima/cortauñas bebé = bebes_infantil, no cuidado_cabello
    (r"\blima.*beb[eé]", "bebes_infantil", 0.98),
    (r"\blima.*infantil", "bebes_infantil", 0.98),
    (r"\bminicure.*beb[eé]", "bebes_infantil", 0.98),
    (r"\bcortau[nñ]as.*beb[eé]", "bebes_infantil", 0.98),
    (r"\bkit.*u[nñ]as.*beb[eé]", "bebes_infantil", 0.95),
    (r"\bbeter.*beb[eé]", "bebes_infantil", 0.95),  # Beter bebé = accesorios infantil

    # === HERPES (Antes de labios para prioridad) ===
    (r"\bherpes\b", "herpes_labial", 0.95),
    (r"\bcompeed.*herpes", "herpes_labial", 0.98),
    (r"\bzovirax\b", "herpes_labial", 0.95),
    (r"\bzovicrem\b", "herpes_labial", 0.98),  # Issue #457: Zovicrem antes de labial

    # === LABIOS (Prioridad alta para evitar falsos positivos con SPF) ===
    (r"\bb[aá]lsamo.*labi", "cuidado_labios", 0.92),
    (r"\blabial\b", "cuidado_labios", 0.88),
    (r"\blabios\b", "cuidado_labios", 0.85),
    (r"\bstick.*labi", "cuidado_labios", 0.92),
    (r"\bcarmex\b", "cuidado_labios", 0.95),
    (r"\bneutrogena.*lip", "cuidado_labios", 0.92),
    (r"\blip\s*balm", "cuidado_labios", 0.90),

    # === TESTS ESPECÍFICOS (Issue #457: desglose por tipo) ===
    # Test embarazo
    (r"\btest.*embarazo", "test_embarazo", 0.98),
    (r"\bpredictor\b", "test_embarazo", 0.98),
    (r"\bclearblue.*embarazo", "test_embarazo", 0.98),
    (r"\bprueba.*embarazo", "test_embarazo", 0.98),
    # Test ovulación
    (r"\btest.*ovulaci[oó]n", "test_ovulacion", 0.98),
    (r"\bclearblue.*ovulaci[oó]n", "test_ovulacion", 0.98),
    (r"\bclearblue\b", "test_embarazo", 0.90),  # Clearblue genérico → embarazo (más común)
    # Test COVID/gripe
    (r"\btest.*covid", "test_covid_gripe", 0.98),
    (r"\btest.*gripe", "test_covid_gripe", 0.98),
    (r"\btest.*ant[ií]geno", "test_covid_gripe", 0.95),
    (r"\bsars[- ]?cov", "test_covid_gripe", 0.95),
    (r"\bautodiagn[oó]stico.*covid", "test_covid_gripe", 0.95),
    # Issue #457: Tests diagnósticos (vitamina D, etc.) - no son suplementos
    (r"\bqassay\b.*test", "test_diagnostico", 0.98),
    (r"\btest.*vitamina\s*d\b", "test_diagnostico", 0.98),
    (r"\bautotest\b", "test_diagnostico", 0.90),
    # Fertilidad (suplementos, no tests)
    (r"\bvitalnatal\s+man\b", "fertilidad", 0.98),  # IVB Vitalnatal Man
    (r"\bivb\s+vitalnatal", "fertilidad", 0.95),    # IVB Vitalnatal (marca fertilidad)
    (r"\bgluc[oó]metro", "diabetes", 0.95),
    (r"\btira.*reactiva", "diabetes", 0.90),

    # === MATERIAL SANITARIO ESPECÍFICO ===
    (r"\bmedia.*compresi[oó]n", "varices", 0.95),
    (r"\bvarices", "varices", 0.95),
    # Farmalastic -> material_ortopedico (son productos ortopédicos)
    (r"\bfarmalastic\b", "material_ortopedico", 0.90),
    (r"\bmu[nñ]equera", "material_ortopedico", 0.90),
    (r"\btobillera", "material_ortopedico", 0.90),
    (r"\brodillera", "material_ortopedico", 0.90),
    (r"\bplantilla", "ortopedia_pie", 0.90),
    (r"\bterm[oó]metro", "termometros", 0.95),
    (r"\btensi[oó]metro", "tension_arterial", 0.95),
    (r"\bjeringuilla", "material_sanitario", 0.90),
    (r"\bap[oó]sito", "apositos_curas", 0.95),
    (r"\bvenda", "compresion_vendajes", 0.90),
    (r"\bgasa\b", "apositos_curas", 0.90),

    # === VETERINARIA (Mining: 33 fallbacks) ===
    (r"\bperros?\b", "veterinaria", 0.95),
    (r"\bgatos?\b", "veterinaria", 0.95),
    (r"\bcanino", "veterinaria", 0.95),
    (r"\bfelino", "veterinaria", 0.95),
    (r"\bapoquel\b", "veterinaria", 0.98),  # Antihistamínico veterinario
    (r"\badtab\b", "veterinaria", 0.98),    # Antiparasitario veterinario
    (r"\bsimparica\b", "veterinaria", 0.98),
    (r"\bnexgard\b", "veterinaria", 0.98),
    (r"\bbravecto\b", "veterinaria", 0.98),
    (r"\bfrontline\b", "veterinaria", 0.95),
    (r"\badvantix\b", "veterinaria", 0.95),
    (r"\bseresto\b", "veterinaria", 0.95),

    # === INCONTINENCIA (Mining: productos pérdidas leves) ===
    (r"\bindasec\b", "incontinencia", 0.95),  # Marca de incontinencia
    (r"\bp[eé]rdidas\s*leves", "incontinencia", 0.95),
    (r"\bincontinencia", "incontinencia", 0.95),
    (r"\bdermoseda\b", "incontinencia", 0.95),  # Línea Indasec Dermoseda

    # === SALUD SEXUAL (antes de higiene íntima para prioridad) ===
    (r"\bdurex\b", "salud_sexual", 0.95),  # Preservativos y lubricantes
    (r"\bpreservativ", "salud_sexual", 0.95),
    (r"\bcond[oó]n", "salud_sexual", 0.95),
    (r"\bcontrol\s*sex", "salud_sexual", 0.95),
    (r"\blubricante.*[ií]ntimo", "salud_sexual", 0.90),

    # === HIGIENE ÍNTIMA (Mining: 19 fallbacks) ===
    (r"\bcompresa", "higiene_intima", 0.95),
    (r"\bevax\b", "higiene_intima", 0.95),
    (r"\bsalvaslip", "higiene_intima", 0.95),
    (r"\btamp[oó]n", "higiene_intima", 0.95),
    (r"\bfemconfort\b", "higiene_intima", 0.95),
    (r"\bcottonlike\b", "higiene_intima", 0.90),

    # === DOLOR MUSCULAR MARCAS (Mining: 17 fallbacks) ===
    (r"\bdiclokern\b", "dolor_muscular", 0.95),
    (r"\bcinfadol\b", "dolor_muscular", 0.95),
    (r"\bcannaben\b", "dolor_muscular", 0.95),
    (r"\bactron\b", "dolor", 0.90),  # Analgésico genérico

    # === CONGESTIÓN NASAL (Mining: 16 fallbacks) ===
    (r"\baluneb\b", "congestion_nasal", 0.95),
    (r"\bbreathe\s*right", "congestion_nasal", 0.95),
    (r"\brinorub\b", "congestion_nasal", 0.95),

    # === GRIPE/RESFRIADO (Issue #457 verified) ===
    (r"\bcouldina\b", "gripe_resfriado", 0.98),
    (r"\bdesenfrio[rl]\b", "gripe_resfriado", 0.98),
    (r"\bilvico\b", "gripe_resfriado", 0.98),
    (r"\bnormogrip\b", "gripe_resfriado", 0.98),
    (r"\bpharmafren\b", "gripe_resfriado", 0.98),
    (r"\bpharmagrip\b", "gripe_resfriado", 0.98),

    # === ALERGIA/ANTIHISTAMÍNICOS (Issue #457 verified) ===
    (r"\babrilia\b", "alergia", 0.98),
    (r"\bbactil\b", "alergia", 0.98),
    (r"\breactine\b", "alergia", 0.98),
    (r"\brino[- ]?ebastel\b", "alergia", 0.98),
    (r"\btelfast\b", "alergia", 0.98),
    (r"\bclarityne\b", "alergia", 0.98),
    (r"\balercina\b", "alergia", 0.98),

    # === HEMORROIDES (Issue #457 verified) ===
    (r"\bhemoal\b", "hemorroides", 0.98),
    (r"\bhemoliv\b", "hemorroides", 0.98),
    (r"\banso\b.*pomada", "hemorroides", 0.98),
    (r"\bruscus\b", "hemorroides", 0.95),
    (r"\bdaflon\b", "hemorroides", 0.95),

    # === ACIDEZ/REFLUJO (Issue #457 verified) ===
    (r"\balmax\b", "acidez_reflujo", 0.98),
    (r"\barcid\b", "acidez_reflujo", 0.98),
    (r"\bomeprazol\b", "acidez_reflujo", 0.95),
    (r"\bomekaste\b", "acidez_reflujo", 0.95),

    # === MAGNESIO (Issue #457 verified) ===
    (r"\bmagnesioboi\b", "magnesio", 0.98),
    (r"\bmagnogene\b", "magnesio", 0.98),
    (r"\bnhco\s*magnesium\b", "magnesio", 0.98),

    # === VITAMINA D (Issue #457: separada de huesos) ===
    (r"\bd3\s*\+\s*k2\b", "vitamina_d", 0.98),  # D3+K2 combinaciones
    (r"\bvitamina\s*d3\b", "vitamina_d", 0.98),
    (r"\bdeltius\b", "vitamina_d", 0.98),  # Marca popular vitamina D
    (r"\bvigantol\b", "vitamina_d", 0.98),

    # === PIEL SENSIBLE (Mining: 14 fallbacks) ===
    (r"\bcetaphil\b", "piel_sensible", 0.95),
    (r"\bdermalex\b", "piel_sensible", 0.90),

    # === HERIDAS/DESINFECCIÓN (Mining: 10 fallbacks) ===
    (r"\bcristalmina\b", "desinfeccion_heridas", 0.95),
    (r"\bhidracalm\b", "cicatrizacion", 0.90),

    # === BEBÉS INFANTIL (Mining: 11 fallbacks) ===
    (r"\bbabynaturals\b", "bebes_infantil", 0.95),
    (r"\bpediasure\b", "alimentacion_bebe", 0.95),
    (r"\bblevit\b", "alimentacion_bebe", 0.95),
    (r"\bcapricare\b", "alimentacion_bebe", 0.95),

    # === CONTROL PESO (Issue #457: 5 correcciones vitaminas_general → control_peso) ===
    (r"\bwelness\s*lab\b", "control_peso", 0.98),  # Welness Lab capsulas
    (r"\bquemagras", "control_peso", 0.95),
    (r"\badelgaz", "control_peso", 0.90),
    (r"\bdrena\w*\b", "control_peso", 0.85),  # Drenante, drenaligne, etc.

    # === ENERGÍA/VITALIDAD (Issue #457: 3 correcciones) ===
    (r"\bblackbeepharmacy\b", "energia_vitalidad", 0.98),  # Marca energética
    (r"\bblackbee\b", "energia_vitalidad", 0.95),

    # === ÓPTICA (Mining: 73 fallbacks) ===
    (r"\bgafas\b", "optica", 0.95),
    (r"\bcadena.*gafas", "optica", 0.95),
    (r"\bmr\.?\s*boho\b", "optica", 0.95),
    (r"\blentilla", "lentillas", 0.95),

    # === ARRUGAS/ANTIEDAD MARCAS (Mining: 33 fallbacks) ===
    (r"\baminoskin\b", "arrugas_antiedad", 0.95),
    (r"\benergifique\b", "arrugas_antiedad", 0.95),

    # === HIGIENE CORPORAL MARCAS (Mining: 27 fallbacks) ===
    (r"\bacnaid\b", "acne", 0.95),
    # Ducray dermocosmetica
    (r"\bkeracnyl\b", "acne", 0.98),             # Ducray - acné
    (r"\bictyane\b", "hidratacion_corporal", 0.95),  # Ducray - pieles secas
    (r"\bdexyane\b", "piel_atopica", 0.98),      # Ducray - eczemas/atopía
    (r"\bmelascreen\b", "manchas", 0.98),        # Ducray - manchas
    # Isdin dermocosmetica
    (r"\bisdinceutics\b", "arrugas_antiedad", 0.98),  # Isdin antiedad
    (r"\bglicoisdin\b", "limpieza_facial", 0.95),     # Isdin peeling/exfoliante
    (r"\bureadin\b", "hidratacion_corporal", 0.98),   # Isdin urea (pieles secas)
    (r"\bnutratopic\b", "piel_atopica", 0.98),   # Isdin atopia
    # Nesira (Acofarma) - tiene múltiples sublíneas, necesita scraping
    # Por ahora dejamos sin regla específica para clasificar por producto
    (r"\bvivera\b", "higiene_corporal", 0.90),

    # === LIMPIEZA FACIAL (marcas específicas) ===
    # Biretix: línea limpieza vs tratamiento acné
    (r"\bbiretix.*cleanser", "limpieza_facial", 0.95),
    (r"\bbiretix.*gel\s*limpiador", "limpieza_facial", 0.95),
    (r"\bbiretix.*gel\s*reconfortante", "limpieza_facial", 0.95),
    # Esthederm Osmoclean: línea limpieza
    (r"\besthederm.*osmoclean", "limpieza_facial", 0.95),
    (r"\bosmoclean\b", "limpieza_facial", 0.95),

    # === PIEL SENSIBLE MARCAS (Mining: 17 fallbacks) ===
    # BE+ es una marca multiproducto - más específico abajo
    (r"\bbe\+\s*med\s*pediatric", "bebes_infantil", 0.95),
    (r"\bbe\+\s*med\s*acnicontrol", "acne", 0.95),
    (r"\bbe\+\s*med\s*hidracalm", "cicatrizacion", 0.95),
    (r"\bbe\+\s*med\s*escocedura", "dermatitis_panal", 0.95),
    (r"\bbe\+\s*med\s*capilar", "caida_cabello", 0.95),
    (r"\bbe\+\s*capilar", "caida_cabello", 0.95),
    # BE+ Solar/Skin Protect
    (r"\bbe\+\s*skin\s*protect.*spf", "proteccion_solar", 0.95),
    (r"\bbe\+\s*skinprotect.*spf", "proteccion_solar", 0.95),
    # BE+ Antiedad
    (r"\bbe\+\s*energifique", "arrugas_antiedad", 0.95),
    (r"\bbe\+\s*booster", "arrugas_antiedad", 0.95),  # BE+ Booster Antioxidante/Hidratante
    # BE+ Limpieza
    (r"\bbe\+\s*agua\s*micelar", "limpieza_facial", 0.95),
    (r"\bbe\+\s*gel\s*espuma", "limpieza_facial", 0.95),
    # BE+ Fallback genérico (baja confianza)
    (r"\bbe\+", "piel_sensible", 0.85),

    # === VITAMINAS MARCAS (Mining: 13 fallbacks + Issue #457 verified) ===
    (r"\bauxina\b", "vitaminas_general", 0.90),
    (r"\bbenadon\b", "vitaminas_general", 0.95),
    (r"\bbenerva\b", "vitaminas_general", 0.95),
    # Issue #457: Multivitamínicos verificados
    (r"\bmulticentrum\b", "vitaminas_general", 0.98),
    (r"\bpharmaton\b", "vitaminas_general", 0.98),
    (r"\bsupradyn\b", "vitaminas_general", 0.98),
    (r"\bbecozyme\b", "vitaminas_general", 0.98),
    (r"\baminobalance\b", "vitaminas_general", 0.95),
    (r"\baminocomplex\b", "vitaminas_general", 0.95),
    (r"\baminovital\b", "vitaminas_general", 0.95),
    (r"\bbonusan\b", "vitaminas_general", 0.95),
    (r"\bbuonavit\b", "vitamina_d", 0.98),  # Issue #457: Buonavit D3 = vitamina D
    (r"\bns\s*vitans\b", "vitaminas_general", 0.95),
    (r"\bvitans\b", "vitaminas_general", 0.90),

    # === DOLOR MARCAS (Mining: 23 fallbacks + Issue #457 verified) ===
    (r"\balgifast\b", "dolor", 0.95),
    (r"\bdifenadol\b", "dolor", 0.95),
    (r"\befferaldol\b", "dolor", 0.95),
    (r"\benandol\b", "dolor", 0.95),
    (r"\bespididol\b", "dolor", 0.95),
    # Issue #457: Productos verificados como dolor (analgésicos)
    (r"\bactromadol\b", "dolor", 0.98),
    (r"\bantidol\b", "dolor", 0.98),
    (r"\bapiretal\b", "dolor", 0.98),
    (r"\bdalsy\b", "dolor", 0.98),
    (r"\bdifenatil\b", "dolor", 0.98),
    (r"\bdifenadex\b", "dolor", 0.98),
    (r"\bdolostop\b", "dolor", 0.98),
    (r"\bdolovanz\b", "dolor", 0.98),
    (r"\bendolex\b", "dolor", 0.98),
    (r"\bgelocatil\b", "dolor", 0.98),
    (r"\bgeloprofen\b", "dolor", 0.98),
    (r"\bvoltaduo\b", "dolor", 0.98),

    # === TOS/MUCOSIDAD MARCAS (Mining: 10 fallbacks + Issue #457 verified) ===
    (r"\bangituss\b", "mucosidad_respiratoria", 0.95),
    (r"\bcinfatos\b", "mucosidad_respiratoria", 0.95),
    (r"\biniston\b", "mucosidad_respiratoria", 0.95),
    # Issue #457: Mucolíticos verificados
    (r"\bbisolvon\b", "mucosidad_respiratoria", 0.98),
    (r"\bfluimucil\b", "mucosidad_respiratoria", 0.98),
    (r"\bmucibron\b", "mucosidad_respiratoria", 0.98),
    (r"\bmucosan\b", "mucosidad_respiratoria", 0.98),
    (r"\bprospantus\b", "mucosidad_respiratoria", 0.98),
    (r"\bilvigrip\s*expe", "mucosidad_respiratoria", 0.98),

    # === PRÓTESIS DENTAL (Issue #457 verified) ===
    (r"\balgasiv\b", "protesis_dental", 0.98),
    (r"\bkukident\b", "protesis_dental", 0.98),

    # === MAREO/VIAJE ===
    (r"\bbiodramina\b", "mareo", 0.95),
    (r"\bcinfamar\b", "mareo", 0.95),
    (r"\bsweetsin.*stop.*mareo", "mareo", 0.98),  # Issue #457: Sweetsin Stop Mareo es medicinal
    (r"\bstop.*mareo.*sweetsin", "mareo", 0.98),

    # === DULCERÍA (Issue #457: caramelos sin propiedades medicinales) ===
    # ANTES de reglas genéricas de caramelos para capturar dulcería pura
    (r"\bfarline\s*sweetsin\b", "dulceria", 0.98),  # Farline Sweetsin = dulcería
    (r"\bsweetsin\s*caramelos?\b", "dulceria", 0.95),

    # === CERAVE (marca multiproducto) ===
    # Limpieza
    (r"\bcerave.*gel\s*limpiador", "limpieza_facial", 0.95),
    (r"\bcerave.*limpiador", "limpieza_facial", 0.95),
    # Acné (Issue #457 - 7 correcciones piel_atopica→acne)
    (r"\bcerave.*control.*imperfecciones", "acne", 0.98),
    (r"\bcerave.*imperfecciones", "acne", 0.95),
    (r"\bcerave.*aha", "acne", 0.95),
    (r"\bcerave.*bha", "acne", 0.95),
    (r"\bcerave.*acn[eé]", "acne", 0.95),
    # Hidratación piel seca (no atópica)
    (r"\bcerave.*crema\s*hidratante\s*piel\s*seca", "piel_seca", 0.95),
    (r"\bcerave.*hidratante", "piel_seca", 0.90),
    # Solar
    (r"\bcerave.*spf", "proteccion_solar", 0.95),
    # Fallback genérico (baja confianza)
    (r"\bcerave\b", "piel_atopica", 0.80),

    # === MATERIAL SANITARIO (Mining: 12 fallbacks) ===
    (r"\baguja\b", "material_sanitario", 0.90),
    (r"\bicogamma\b", "material_sanitario", 0.95),

    # === DOLOR MUSCULAR (Mining: 12 fallbacks + Issue #457 verified) ===
    (r"\bbolsa.*agua.*caliente", "termoterapia", 0.95),
    (r"\bcannabix\b", "dolor_muscular", 0.95),
    (r"\bfibroal\b", "dolor_muscular", 0.95),
    # Issue #457: Productos musculares verificados
    (r"\bcalbidol\b", "dolor_muscular", 0.98),
    (r"\bflogoprofen\b", "dolor_muscular", 0.98),
    (r"\bhidroxil\b", "dolor_muscular", 0.98),
    (r"\bthermacare\b", "dolor_muscular", 0.98),
    (r"\bvitagobens\b", "dolor_muscular", 0.98),
    (r"\bvoltadol\b", "dolor_muscular", 0.98),

    # === CICATRIZACIÓN (Mining: 13 fallbacks) ===
    (r"\bamniolina\b", "cicatrizacion", 0.95),
    (r"\bescocedura", "dermatitis_panal", 0.90),

    # === VITAMINAS (Mining: 33 fallbacks) ===
    (r"\bivb\b", "vitaminas_general", 0.90),

    # === DOLOR (Mining: 25 fallbacks) ===
    (r"\bibudol\b", "dolor", 0.95),
    (r"\bnurofen\b", "dolor", 0.95),
    (r"\bibufen\b", "dolor", 0.95),
    (r"\bfebrectal\b", "dolor_fiebre", 0.95),

    # === MATERIAL SANITARIO (Mining: 21 fallbacks) ===
    (r"\bjeringa\b", "material_sanitario", 0.95),

    # === OJO SECO (Mining: 20 fallbacks + Issue #457 verified) ===
    # Issue #457: Colirios hidratantes verificados
    (r"\baquoral\b", "ojo_seco", 0.98),
    (r"\bacuolens\b", "ojo_seco", 0.98),
    (r"\blagroben\b", "ojo_seco", 0.98),
    (r"\blipolasic\b", "ojo_seco", 0.98),
    (r"\blubristil\b", "ojo_seco", 0.98),
    (r"\boptiben\b", "ojo_seco", 0.95),
    (r"\bthealoz\b", "ojo_seco", 0.95),
    (r"\beyestil\b", "ojo_seco", 0.95),
    (r"\bomnitears\b", "ojo_seco", 0.95),
    (r"\bnurane\b", "ojo_seco", 0.95),
    (r"\blagrima.*artificial", "ojo_seco", 0.90),

    # === CONGESTIÓN NASAL (Mining: 19 fallbacks + Issue #457 verified) ===
    (r"\brhinovin\b", "congestion_nasal", 0.95),
    (r"\brinocusi\b", "congestion_nasal", 0.95),
    (r"\bdisneumon\b", "congestion_nasal", 0.95),
    (r"\bneilmed\b", "congestion_nasal", 0.95),
    (r"\barkorespira\b", "congestion_nasal", 0.95),
    # Issue #457: Descongestivos verificados
    (r"\bflonase\b", "congestion_nasal", 0.98),
    (r"\binhalvicks\b", "congestion_nasal", 0.98),
    (r"\bnasalvicks\b", "congestion_nasal", 0.98),
    (r"\bnormomar\b", "congestion_nasal", 0.98),
    (r"\bnormonasal\b", "congestion_nasal", 0.98),
    (r"\brespibien\b", "congestion_nasal", 0.98),
    (r"\brespidina\b", "congestion_nasal", 0.98),
    (r"\brespir\b", "congestion_nasal", 0.95),
    (r"\butabon\b", "congestion_nasal", 0.98),

    # === HIGIENE OÍDOS (Mining: 15 fallbacks) ===
    (r"\botifaes\b", "higiene_oidos", 0.95),
    (r"\baud[ií]spray\b", "higiene_oidos", 0.95),

    # === DESODORANTE (Mining: 14 fallbacks) ===
    (r"\bperspirex\b", "desodorante", 0.95),
    (r"\bisdindeo\b", "desodorante", 0.95),
    (r"\blambda\s*control", "desodorante", 0.95),

    # === ESTREÑIMIENTO (Mining: 13 fallbacks + Issue #457 verified) ===
    (r"\bemuliquen\b", "estrenimiento", 0.95),
    (r"\bmanasul\b", "estrenimiento", 0.95),
    (r"\bdulcolax\b", "estrenimiento", 0.95),
    # Issue #457: Laxantes verificados
    (r"\bfisioenema\b", "estrenimiento", 0.98),
    (r"\bplantaben\b", "estrenimiento", 0.98),
    (r"\bvilardell\b", "estrenimiento", 0.98),
    (r"\bsupositorio.*glicerina", "estrenimiento", 0.98),
    (r"\bglicerina.*supositorio", "estrenimiento", 0.98),
    (r"\bglycilax\b", "estrenimiento", 0.98),
    (r"\bmelilax\b", "estrenimiento", 0.98),

    # === FLORA INTESTINAL (Mining: 21 fallbacks + Issue #457 verified) ===
    (r"\bprofaes\b", "flora_intestinal", 0.95),
    (r"\bregenintest\b", "flora_intestinal", 0.95),
    (r"\bcolilen\b", "flora_intestinal", 0.95),
    # Issue #457: Probióticos verificados
    (r"\bultra[- ]?levura\b", "flora_intestinal", 0.98),
    (r"\bprodefen\b", "flora_intestinal", 0.98),
    (r"\binfloran\b", "flora_intestinal", 0.98),
    (r"\breuteri\b", "flora_intestinal", 0.95),

    # === HIGIENE CORPORAL MARCAS ===
    (r"\broger.*gallet\b", "higiene_corporal", 0.90),

    # === PIEL SECA (Mining round 3: 23 fallbacks) ===
    (r"\babs\s*skincare\b", "piel_seca", 0.90),
    (r"\banagras\b", "piel_seca", 0.95),
    (r"\bdexeryl\b", "piel_seca", 0.95),
    (r"\bneusc\b", "piel_seca", 0.90),

    # === HIGIENE ÍNTIMA (Mining round 3: 17 fallbacks) ===
    (r"\bactifemme\b", "higiene_intima", 0.95),
    (r"\bclogin\b", "higiene_intima", 0.95),
    (r"\bginecanesbalance\b", "higiene_intima", 0.95),

    # === ORTOPEDIA (Mining round 3: 16 fallbacks) ===
    # Nota: "ayudas técnicas" es jerga industrial, usuario busca "ortopedia"
    (r"\bflexipro\b", "ortopedia", 0.95),
    (r"\bcaminador\b", "ortopedia", 0.95),
    (r"\bandador\b", "ortopedia", 0.95),
    (r"\bmuleta\b", "ortopedia", 0.95),
    (r"\bsilla.*ruedas", "ortopedia", 0.95),
    (r"\bbast[oó]n\b", "ortopedia", 0.95),  # No "ayudas técnicas"
    (r"\bf[eé]rula\b", "ortopedia", 0.95),  # No "ayudas técnicas"
    (r"\btobillera\b", "ortopedia", 0.95),
    (r"\brodillera\b", "ortopedia", 0.95),
    (r"\bmu[nñ]equera\b", "ortopedia", 0.95),
    (r"\bcodera\b", "ortopedia", 0.95),

    # === BEBÉS (Mining round 3: 16 fallbacks) ===
    (r"\bbabe\s*pediatric", "bebes_infantil", 0.95),
    (r"\bbolso.*paseo", "bebes_infantil", 0.90),
    (r"\bcanastilla\b", "bebes_infantil", 0.90),

    # === PIEL SENSIBLE (Mining round 3: 16 fallbacks) ===
    (r"\bcutiderm\b", "piel_sensible", 0.95),
    (r"\bemolienta\b", "piel_sensible", 0.95),

    # === FLORA INTESTINAL (Mining round 3: 15 fallbacks) ===
    (r"\badomelle\b", "flora_intestinal", 0.95),
    (r"\bbifiselle\b", "flora_intestinal", 0.95),
    (r"\bbutirato\b", "flora_intestinal", 0.95),
    (r"\bcandi\s*defens\b", "flora_intestinal", 0.95),

    # === OJO SECO (Mining round 3: 14 fallbacks) ===
    (r"\balaon\b", "ojo_seco", 0.95),
    (r"\bcristalregen\b", "ojo_seco", 0.95),

    # === ARRUGAS (Mining round 3: 13 fallbacks) ===
    (r"\besthederm\b", "arrugas_antiedad", 0.95),
    (r"\bcosmeclinik\b", "arrugas_antiedad", 0.90),
    (r"\bdr\s*arthouros\b", "arrugas_antiedad", 0.95),

    # === GASES (Mining round 3: 12 fallbacks) ===
    (r"\baero[- ]?net\b", "gases_flatulencia", 0.95),
    (r"\baminodigest\b", "digestivo", 0.90),
    (r"\bbetaindigest\b", "digestivo", 0.90),

    # === CAÍDA CABELLO (Mining round 3: 12 fallbacks) ===
    (r"\balocutan\b", "caida_cabello", 0.95),  # Minoxidil
    (r"\bcistina\b", "caida_cabello", 0.90),
    (r"\bhairgen\b", "caida_cabello", 0.95),

    # === HIGIENE OÍDOS (Mining round 3: 12 fallbacks) ===
    (r"\bliade\b", "higiene_oidos", 0.95),
    (r"\baurizon\b", "higiene_oidos", 0.95),
    (r"\bcerumenol\b", "higiene_oidos", 0.95),
    (r"\bcleanotix\b", "higiene_oidos", 0.95),

    # === SUEÑO (Mining round 4: high-signal words) ===
    (r"\bzzzquil\b", "sueno_insomnio", 0.98),
    (r"\bgummies.*sue[nñ]o", "sueno_insomnio", 0.90),
    (r"\bsue[nñ]o.*gummies", "sueno_insomnio", 0.90),

    # === CALCIO/HUESOS (Mining round 4) ===
    (r"\bmastical\b", "calcio_huesos", 0.95),
    (r"\bcalcio.*masticable", "calcio_huesos", 0.90),

    # === NUTRICIÓN DEPORTIVA (Mining round 4) ===
    (r"\bbarrita\b", "nutricion_deportiva", 0.85),
    (r"\bbarebell", "nutricion_deportiva", 0.95),
    (r"\bbarebells\b", "nutricion_deportiva", 0.95),

    # === HOMEOPATÍA (Mining round 4) ===
    (r"\biberica\b", "homeopatia", 0.90),
    (r"\bglobulos\b", "homeopatia", 0.90),
    (r"\bhomeopat", "homeopatia", 0.95),
    (r"\bboiron\b", "homeopatia", 0.95),

    # === PIEL SECA (Mining round 4) ===
    (r"\bvaselina\b", "piel_seca", 0.90),
    (r"\bvaselix\b", "piel_seca", 0.95),

    # === OSTOMÍA (Mining round 4) ===
    (r"\bostom[ií]a", "ostomia", 0.95),
    (r"\bcolostom", "ostomia", 0.95),

    # === MATERIAL SANITARIO (Mining round 4) ===
    (r"\bcorysan\b", "material_sanitario", 0.95),

    # === HALITOSIS (Mining round 4) ===
    (r"\bbexident.*fresh", "halitosis", 0.95),
    (r"\bbreath\b", "halitosis", 0.85),

    # === ÓPTICA (Mining round 4) ===
    (r"\btwins\b", "optica", 0.90),

    # === ACCESORIOS BELLEZA (Mining round 4) ===
    (r"\bvitry\b", "accesorios_belleza", 0.95),

    # === UÑAS (Issue #457: productos de manicura) ===
    (r"\bquitaesmalte\b", "unas", 0.98),
    (r"\bquita[- ]?esmalte\b", "unas", 0.98),
    (r"\besmalte.*u[nñ]as", "unas", 0.95),
    (r"\baceite.*cut[ií]cula", "unas", 0.95),

    # === HIGIENE OÍDOS (Issue #457: bastoncillos) ===
    (r"\bbastoncillo", "higiene_oidos", 0.95),
    (r"\bbastones.*o[ií]do", "higiene_oidos", 0.95),
    (r"\bhisopos?\b", "higiene_oidos", 0.90),

    # === CUIDADO DEPENDIENTES (Issue #457: esponjas enjabonadas) ===
    (r"\besponja.*enjabonada", "cuidado_dependientes", 0.98),
    (r"\bmanopla.*enjabonada", "cuidado_dependientes", 0.98),
    (r"\besponja.*desechable.*jabon", "cuidado_dependientes", 0.95),

    # === TATUAJES (Issue #457: aftercare tatuajes) ===
    (r"\btattoo\b", "tatuajes", 0.95),
    (r"\btatuaje", "tatuajes", 0.95),
    (r"\baftercare.*tattoo", "tatuajes", 0.98),

    # =========================================================================
    # CAPILAR ESPECIALIZADO (Issue #457 - Feedback Loop)
    # Klorane, René Furterer, Lazartigue - Reglas por gama/ingrediente
    # =========================================================================

    # --- KLORANE (Fitoterapia por ingrediente botánico) ---
    # Anticaída
    (r"klorane.*quinina", "caida_cabello", 0.95),
    (r"klorane.*edelweiss", "caida_cabello", 0.95),
    (r"klorane.*keratincaps", "caida_cabello", 0.95),  # Nutricosmética
    # Caspa/Grasa (seborregulador)
    (r"klorane.*ortiga", "cabello_graso", 0.95),
    (r"klorane.*cidra", "cabello_graso", 0.95),
    (r"klorane.*mirto", "caspa", 0.95),  # Caspa grasa
    # Cuero cabelludo sensible
    (r"klorane.*peonia", "capilar_cuero_sensible", 0.95),
    # Uso frecuente
    (r"klorane.*avena", "capilar_frecuente", 0.95),
    (r"klorane.*leche.*almendra", "capilar_frecuente", 0.95),
    # Hidratación/Nutrición (pelo seco/dañado)
    (r"klorane.*cupuacu", "capilar_hidratacion", 0.95),
    (r"klorane.*mango", "capilar_hidratacion", 0.95),
    (r"klorane.*datil", "capilar_hidratacion", 0.95),
    # Otros
    (r"klorane.*menta", "capilar_frecuente", 0.90),  # Detox
    (r"klorane.*centaura", "canas", 0.95),  # Pelo blanco/gris

    # --- RENÉ FURTERER (Premium / Tratamiento - Nombres de fantasía) ---
    # Anticaída
    (r"triphasic", "caida_cabello", 0.95),
    (r"vitalfan", "caida_cabello", 0.95),  # Suplemento oral
    (r"forticea", "caida_cabello", 0.95),
    (r"complexe\s*5", "caida_cabello", 0.95),  # Estimulante pre-champú
    # Caspa/Grasa
    (r"neopur", "caspa", 0.95),
    (r"curbicia", "cabello_graso", 0.95),
    (r"melaleuca", "caspa", 0.95),
    # Cuero cabelludo sensible
    (r"astera", "capilar_cuero_sensible", 0.95),
    # Hidratación/Nutrición
    (r"karite", "capilar_hidratacion", 0.95),
    (r"absolue\s*k", "capilar_hidratacion", 0.95),  # Absolue Keratine
    # Densidad (pelo fino/envejecido)
    (r"tonucia", "capilar_densidad", 0.95),
    # Color
    (r"okara", "capilar_color", 0.95),
    # Uso frecuente
    (r"naturia", "capilar_frecuente", 0.95),

    # --- LAZARTIGUE (Botánica / Vegana) ---
    # Anticaída
    (r"lazartigue.*stronger", "caida_cabello", 0.95),
    (r"lazartigue.*thicker", "caida_cabello", 0.95),
    (r"lazartigue.*boost", "caida_cabello", 0.95),
    (r"lazartigue.*fortify", "caida_cabello", 0.95),
    # Caspa/Grasa
    (r"lazartigue.*clear", "caspa", 0.95),
    (r"lazartigue.*rebalance", "cabello_graso", 0.95),  # Raíz grasa/punta seca
    # Hidratación/Nutrición
    (r"lazartigue.*repair", "capilar_hidratacion", 0.95),
    (r"lazartigue.*nourish", "capilar_hidratacion", 0.95),
    (r"lazartigue.*huile.*reves", "capilar_hidratacion", 0.95),  # Aceite de los sueños
    # Color
    (r"lazartigue.*colour", "capilar_color", 0.95),
]

# -----------------------------------------------------------------------------
# NIVEL 2: REGLAS GENÉRICAS (Prioridad Baja)
# Se ejecutan SOLO si no hubo match en Nivel 1 ni en ingredient_rules.
# Formato: (regex_pattern, necesidad, confidence)
# IMPORTANTE: Los nombres de categoría deben coincidir con symptom_taxonomy.py
# -----------------------------------------------------------------------------
TIER_1_GENERIC: List[Tuple[str, str, float]] = [
    # === DOLOR Y FIEBRE ===
    (r"\bibuprofeno", "dolor_fiebre", 0.90),
    (r"\bparacetamol", "dolor_fiebre", 0.90),
    (r"\baspirina", "dolor_fiebre", 0.90),
    (r"\bdexketoprofeno", "dolor_fiebre", 0.90),
    (r"\benantyum\b", "dolor_fiebre", 0.90),
    (r"\bnolotil\b", "dolor_fiebre", 0.90),

    # === RESPIRATORIO ===
    (r"\btos\s*seca", "tos_seca", 0.95),
    (r"\bantitus[ií]geno", "tos_seca", 0.90),
    (r"\bflutox\b", "tos_seca", 0.95),
    (r"\bromilar\b", "tos_seca", 0.95),
    (r"\btos\s*mucos", "mucosidad_respiratoria", 0.95),
    (r"\bmucol[ií]tico", "mucosidad_respiratoria", 0.95),
    (r"\bmucosan\b", "mucosidad_respiratoria", 0.95),
    (r"\bjarabe.*tos", "mucosidad_respiratoria", 0.85),
    (r"\bfrenadol\b", "gripe_resfriado", 0.95),
    (r"\bfarmacatarro\b", "gripe_resfriado", 0.95),
    (r"\bspray.*nasal", "congestion_nasal", 0.85),
    (r"\bagua.*mar", "congestion_nasal", 0.80),
    (r"\brinomer\b", "congestion_nasal", 0.95),
    (r"\brespir[ae]", "congestion_nasal", 0.80),

    # === GARGANTA ===
    (r"\bgarganta", "dolor_garganta", 0.90),
    (r"\bstrepsils\b", "dolor_garganta", 0.95),
    (r"\blizipaina\b", "dolor_garganta", 0.95),
    (r"\bpharysol\b", "dolor_garganta", 0.95),

    # === DIGESTIVO ===
    (r"\balmax\b", "acidez_reflujo", 0.95),
    (r"\bgaviscon\b", "acidez_reflujo", 0.95),
    (r"\bomeprazol", "acidez_reflujo", 0.90),
    (r"\bprobi[oó]tico", "probioticos", 0.90),
    (r"\blactobacillus", "probioticos", 0.90),
    (r"\blaxante", "estrenimiento", 0.90),
    (r"\bdulcolax\b", "estrenimiento", 0.95),

    # === CEPILLOS DENTALES (específicos antes de genérico) ===
    # Nota: Reglas de cepillos eléctricos están en TIER1_SPECIFIC (arriba)
    # Cepillos interdentales
    (r"\bcepillo.*interdental", "interdental", 0.95),
    (r"\binterproximal\b", "interdental", 0.95),

    # === HIGIENE BUCAL INFANTIL (Issue #457) ===
    (r"\bcepillo.*infantil", "higiene_bucal_infantil", 0.95),
    (r"\bcepillo.*ni[nñ]o", "higiene_bucal_infantil", 0.95),
    (r"\bcepillo.*junior", "higiene_bucal_infantil", 0.95),
    (r"\bcepillo.*kids", "higiene_bucal_infantil", 0.95),
    (r"\bpasta.*infantil", "higiene_bucal_infantil", 0.95),
    (r"\bpasta.*ni[nñ]o", "higiene_bucal_infantil", 0.95),
    (r"\bdent[ií]frico.*infantil", "higiene_bucal_infantil", 0.95),
    (r"\bfluocaril.*junior", "higiene_bucal_infantil", 0.95),
    (r"\bphb.*junior", "higiene_bucal_infantil", 0.95),
    (r"\blacer.*junior", "higiene_bucal_infantil", 0.95),
    (r"\blacer.*infantil", "higiene_bucal_infantil", 0.95),
    (r"\bvitis.*junior", "higiene_bucal_infantil", 0.95),
    (r"\bvitis.*kids", "higiene_bucal_infantil", 0.95),
    (r"\bgum.*junior", "higiene_bucal_infantil", 0.95),
    (r"\bgum.*kids", "higiene_bucal_infantil", 0.95),

    # === DENTAL SUBCATEGORÍAS (Issue #457: desglose de higiene_dental) ===
    (r"\bcolutorio", "enjuague_bucal", 0.90),
    (r"\benjuague\s*bucal", "enjuague_bucal", 0.95),
    (r"\bpasta\s*dent", "pasta_dental", 0.90),
    (r"\bdent[ií]frico", "pasta_dental", 0.90),
    (r"\bcepillo\s*dent", "cepillo_manual", 0.90),  # Cepillos manuales
    (r"\bcepillo\s*de\s*dientes", "cepillo_manual", 0.90),
    (r"\bseda\s*dental", "seda_dental", 0.90),
    (r"\bhilo\s*dental", "seda_dental", 0.90),
    (r"\boral[- ]?b\b", "cepillo_manual", 0.80),  # Fallback Oral-B sin especificar

    # === LIMPIEZA FACIAL ===
    (r"\blimpiador", "limpieza_facial", 0.88),
    (r"\bmoussant\b", "limpieza_facial", 0.90),
    (r"\bgel.*limpia", "limpieza_facial", 0.85),
    (r"\bmicelar", "limpieza_facial", 0.88),
    (r"\bdesmaquill", "limpieza_facial", 0.90),
    (r"\bsensibio.*gel", "limpieza_facial", 0.92),  # BIODERMA SENSIBIO GEL
    (r"\blocion.*exfolia", "exfoliacion", 0.92),  # Loción exfoliante
    (r"\bexfolia.*locion", "exfoliacion", 0.92),  # Exfoliante loción
    (r"\bpate.*exfolia", "exfoliacion", 0.92),  # Pate exfoliante
    (r"\bexfoliante\b", "exfoliacion", 0.90),  # Productos exfoliantes
    (r"\bexfolia", "limpieza_facial", 0.85),  # Fallback genérico
    (r"\btonico\b", "limpieza_facial", 0.80),

    # === HIGIENE ÍNTIMA ===
    (r"\b[ií]ntim[ao]", "higiene_intima", 0.90),
    (r"\bzelesse\b", "higiene_intima", 0.95),
    (r"\bvaginal", "higiene_intima", 0.90),
    (r"\bginecol", "higiene_intima", 0.90),
    (r"\blactacyd\b", "higiene_intima", 0.95),
    (r"\bchilly\b", "higiene_intima", 0.95),
    (r"\bsaforelle\b", "higiene_intima", 0.95),
    (r"\bcumlaude.*higiene.*intim", "higiene_intima", 0.98),
    (r"\bgermisdin.*intim", "higiene_intima", 0.98),  # Isdin íntimo

    # === EMBARAZO Y LACTANCIA ===
    (r"\bfemibion\b", "embarazo_lactancia", 0.95),
    (r"\bnatalben\b", "embarazo_lactancia", 0.95),
    (r"\bgestagyn\b", "embarazo_lactancia", 0.95),
    (r"\bprena?tal", "embarazo_lactancia", 0.90),
    (r"\blactancia", "lactancia", 0.90),
    (r"\bpurelan\b", "lactancia", 0.95),  # Crema pezones

    # === PIEL SENSIBLE Y ATÓPICA ===
    (r"\bsensibio\b", "piel_sensible", 0.90),
    (r"\btolerance\b", "piel_sensible", 0.85),
    (r"\bcalm", "piel_sensible", 0.75),
    (r"\bleti\s*at4\b", "piel_atopica", 0.95),
    (r"\bexomega\b", "piel_atopica", 0.95),
    (r"\bxemose\b", "piel_atopica", 0.95),

    # === ROJECES Y ROSÁCEA ===
    (r"\brojacea", "rojeces_rosacea", 0.95),
    (r"\brosaliac\b", "rojeces_rosacea", 0.95),
    (r"\banti[- ]?roj", "rojeces_rosacea", 0.90),

    # === SUEÑO ===
    (r"\bsu[eé][nñ]o\b", "sueno_insomnio", 0.85),
    (r"\binsomnio", "sueno_insomnio", 0.95),
    (r"\bdormidina\b", "sueno_insomnio", 0.95),
    (r"\bso[nñ]adol\b", "sueno_insomnio", 0.95),

    # === DERMOCOSMÉTICA BÁSICA ===
    (r"\bsolar\b", "proteccion_solar", 0.80),
    (r"\bfotoprotec", "proteccion_solar", 0.95),
    (r"\bfusion\s*water", "proteccion_solar", 0.98),   # Isdin
    (r"\beryfotona\b", "proteccion_solar", 0.98),      # Isdin prevención
    (r"\bsunbrush\b", "proteccion_solar", 0.98),       # Isdin
    (r"\buveblock\b", "proteccion_solar", 0.98),       # Isdin
    # Issue #457: Heliocare siempre es protección solar (no apositos)
    (r"\bheliocare\b", "proteccion_solar", 0.98),
    (r"\bspf\s*\d+", "proteccion_solar", 0.75),  # Bajado para no ganar a labios
    (r"\burea\b", "hidratacion_corporal", 0.65),  # Urea baja = hidratación
    # Champú genérico -> capilar_frecuente (Issue #457 - fallback, NO caída)
    # Las gamas específicas (Triphasic, Astera, etc.) se capturan en TIER_1_SPECIFIC
    (r"\bchampu", "capilar_frecuente", 0.65),
    (r"\bchamp[uú]", "capilar_frecuente", 0.65),

    # === VITAMINAS Y SUPLEMENTOS (Genérico - Baja prioridad) ===
    (r"\bvitamina\s*c\b", "vitaminas_minerales", 0.75),
    (r"\bvitamina\s*d\b", "vitamina_d", 0.85),  # Issue #457: Vitamina D separada de huesos
    (r"\bomega\s*3", "omega3", 0.85),
    (r"\bmelatonina", "sueno_insomnio", 0.95),
    (r"\bmagnesio", "magnesio", 0.80),
    (r"\bhierro\b", "hierro", 0.80),

    # =========================================================================
    # MINING P3 (2025-12-20): Reglas extraídas de 4286 productos auto-aprobados
    # =========================================================================

    # --- ALIMENTACIÓN BEBÉ ---
    (r"\bsmileat\b", "alimentacion_bebe", 0.98),  # 29 productos P3
    (r"\btarrito\b", "alimentacion_bebe", 0.94),  # 18 productos P3
    (r"\bhero\s*baby\b", "alimentacion_bebe", 0.98),  # 33 productos P3 (bigram)
    (r"\banticolico\b", "alimentacion_bebe", 0.98),  # 13 productos P3

    # --- CONFORT/HIGIENE BEBÉ ---
    (r"\bchupete\s*silicona\b", "confort_bebe", 0.98),  # 42 productos P3 (bigram)
    (r"\bpa[ñn]al\b", "higiene_bebe", 0.91),  # 23 productos P3

    # --- MATERIAL ORTOPÉDICO ---
    (r"\bmu[ñn]equera\b", "material_ortopedico", 0.98),  # 24 productos P3
    (r"\btech\b", "material_ortopedico", 0.90),  # 22 productos P3 (Farmalastic Tech)
    (r"\bcompression\b", "material_ortopedico", 0.95),  # 15 productos P3
    (r"\bactius\s*orliman\b", "material_ortopedico", 0.89),  # 28 productos P3 (bigram)

    # --- PERFUMERÍA ---
    (r"\biap\s*pharma\b", "perfumeria", 0.98),  # 22 productos P3 (bigram)
    (r"\bpour\s*femme\b", "perfumeria", 0.98),  # 15 productos P3 (bigram)
    (r"\bpour\s*homme\b", "perfumeria", 0.98),  # Complemento lógico

    # --- DOLOR GARGANTA ---
    (r"\bricola\b", "dolor_garganta", 0.98),  # 20 productos P3
    (r"\bazucar\b", "dolor_garganta", 0.85),  # 16 productos P3 (caramelos sin azúcar)

    # --- INCONTINENCIA ---
    (r"\btena\b", "incontinencia", 0.94),  # 18 productos P3
    (r"\borina\b", "incontinencia", 0.90),  # 17 productos P3

    # --- HIGIENE CABELLO ---
    (r"\bmoncho\s*moreno\b", "higiene_cabello", 0.94),  # 18 productos P3 (bigram)

    # --- INTERDENTAL ---
    (r"\btrav-ler\b", "interdental", 0.98),  # 16 productos P3
    (r"\bcepillo\s*interdental\b", "interdental", 0.98),  # 28 productos P3 (bigram)
    (r"\bespacio\s*interdental\b", "interdental", 0.98),  # Mejora de "espacio" solo

    # --- CEPILLO ELÉCTRICO ---
    (r"\bdental\s*electrico\b", "cepillo_electrico", 0.98),  # 32 productos P3 (bigram)

    # --- ARRUGAS/ANTIEDAD ---
    (r"\bmedik8\b", "arrugas_antiedad", 0.93),  # 14 productos P3

    # --- HOMEOPATÍA ---
    (r"\bc15\b", "homeopatia", 0.90),  # 14 productos P3 (dilución homeopática)

    # --- APÓSITOS ---
    (r"\baposito\s*adhesivo\b", "apositos_curas", 0.93),  # 45 productos P3 (bigram)
    (r"\baposito\s*esteril\b", "apositos_curas", 0.93),  # 27 productos P3 (bigram)

    # --- ENCÍAS ---
    (r"\bbexident\s*encias\b", "encias", 0.98),  # 21 productos P3 (bigram)

    # =========================================================================
    # MINING P1 CORRECTIONS (2025-12-20): Reglas de correcciones manuales
    # =========================================================================

    # --- IVB WELLNESS (marca con líneas específicas) ---
    (r"\bivb\s*memory", "concentracion_memoria", 0.98),  # IVB Memory-On
    (r"\bivb\s*omega", "omega3", 0.98),  # IVB Omega3
    (r"\bivb\s*somni", "sueno_insomnio", 0.98),  # IVB Somnilove
    (r"\bivb\s*fiber", "estrenimiento", 0.98),  # IVB Fiber Total

    # --- BEBÉ/INFANTIL ---
    (r"\bmordedor\b", "bebes_infantil", 0.95),  # Mordedores bebé
    (r"\btaza\s*aprendizaje\b", "bebes_infantil", 0.95),  # Tazas aprendizaje

    # --- MATERIAL SANITARIO ---
    (r"\bglucoje[c]?t\b", "material_sanitario", 0.98),  # Glucoject (punción)
    (r"\besparadrapo\b", "material_sanitario", 0.95),  # Esparadrapos

    # --- MATERIAL ORTOPÉDICO ---
    (r"\bmuslera\b", "material_ortopedico", 0.98),  # Musleras

    # --- DENTAL ---
    (r"\belmex\s*sensitive\b", "sensibilidad_dental", 0.98),  # Elmex Sensitive
    (r"\bfittydent\b", "protesis_dental", 0.98),  # Fittydent adhesivo
    (r"\btwisted\s*floss\b", "seda_dental", 0.98),  # GUM Twisted Floss

    # --- OFTÁLMICO ---
    (r"\blubristil\b", "ojo_seco", 0.98),  # Lubristil

    # --- ENERGÍA/VITALIDAD ---
    (r"\benergia\s*[&y]\s*vitalidad\b", "energia_vitalidad", 0.95),  # Multicentrum Energía & Vitalidad

    # =========================================================================
    # MINING P2 CORRECTIONS (2025-12-20): 130 correcciones analizadas
    # =========================================================================

    # --- IVB WELLNESS (líneas adicionales) ---
    (r"\bivb\s*tyroenergy\b", "energia_vitalidad", 0.98),
    (r"\bivb\s*zerodol\b", "dolor_muscular", 0.98),
    (r"\bivb\s*stress\s*off\b", "estres_ansiedad", 0.98),
    (r"\bivb\s*vegan\s*omega\b", "omega3", 0.98),
    (r"\bivb\s*magnesio\b", "magnesio", 0.98),
    (r"\bivb\s*vital\s*natal\b", "embarazo_lactancia", 0.98),
    (r"\bivb\s*vitalnatal\b", "embarazo_lactancia", 0.98),
    (r"\bivb\s*metabolic\b", "control_peso", 0.98),
    (r"\bivb\s*satisens\b", "control_peso", 0.98),
    (r"\bivb\s*menomaster\b", "menopausia", 0.98),
    (r"\bivb\s*ferti\s*up\b", "fertilidad", 0.98),
    (r"\bivb\s*femmebalance\b", "ciclo_menstrual", 0.98),
    (r"\bivb\s*ferroplus\b", "hierro", 0.98),
    (r"\bivb\s*articare\b", "dolor_articular", 0.98),

    # --- SUPRADYN (líneas específicas) ---
    (r"\bsupradyn\s*sue[ñn]o\b", "sueno_insomnio", 0.98),
    (r"\bsupradyn\s*memory\b", "concentracion_memoria", 0.98),
    (r"\bsupradyn\s*anti\s*stress\b", "estres_ansiedad", 0.98),

    # --- NS VITANS (líneas específicas) ---
    (r"\bns\s*vitans\s*recovery\b", "energia_vitalidad", 0.98),
    (r"\bns\s*vitans\s*magnesio\b", "magnesio", 0.98),
    (r"\bns\s*vitans\s*cogni\b", "concentracion_memoria", 0.98),
    (r"\bns\s*vitans\s*estres\b", "estres_ansiedad", 0.98),
    (r"\bns\s*vitans\s*articulaciones\b", "dolor_articular", 0.98),
    (r"\bns\s*vitans\s*vitamina\s*d\b", "vitamina_d", 0.98),
    (r"\bns\s*so[ñn]aben\b", "melatonina", 0.98),

    # --- MELATONINA (productos específicos) ---
    (r"\bmelamil\b", "melatonina", 0.98),
    (r"\bzzzquil\b", "melatonina", 0.98),

    # --- GRIPE (no solo dolor) ---
    (r"\bgelocatil\s*gripe\b", "gripe_resfriado", 0.98),

    # --- DESINFECCIÓN ---
    (r"\balcohol\s*\d+", "desinfeccion_heridas", 0.95),
    (r"\bagua\s*oxigenada\b", "desinfeccion_heridas", 0.95),
    (r"\bpovidona\b", "desinfeccion_heridas", 0.95),

    # --- HIGIENE NASAL ---
    (r"\bnormomar\b", "higiene_nasal", 0.98),
    (r"\bagua\s*marina\b", "higiene_nasal", 0.90),

    # --- BÁLSAMOS RESPIRATORIOS ---
    (r"\bvaporub\b", "balsamos_respiratorios", 0.98),
    (r"\bbabyrub\b", "balsamos_respiratorios", 0.98),

    # --- CEPILLO MANUAL ---
    (r"\bgum\s*pro\b.*cepillo|cepillo.*\bgum\s*pro\b", "cepillo_manual", 0.95),

    # --- CANDIDIASIS ---
    (r"\bcumlaude.*clx\b", "candidiasis_vaginal", 0.98),

    # --- DIGESTIVO ---
    (r"\bpankreoflat\b", "digestion_pesada", 0.98),
    (r"\biberogast\b", "digestion_pesada", 0.98),

    # --- DOLOR GARGANTA (productos específicos) ---
    (r"\bjuanola\b", "dolor_garganta", 0.95),
    (r"\bhialix\b", "dolor_garganta", 0.98),
    (r"\bzarbees\b", "dolor_garganta", 0.95),
    (r"\bpropolis\b", "dolor_garganta", 0.90),

    # --- PIOJOS ---
    (r"\bkitapic\s*antipiojos\b", "piojos", 0.98),

    # --- CASPA ---
    (r"\bpirotex\b", "caspa", 0.98),

    # --- DOLOR MUSCULAR ---
    (r"\bthermacare\b", "dolor_muscular", 0.95),

    # =========================================================================
    # MARCAS SIN DETECTAR (2025-12-20): Ataque quirúrgico a P2 sin marca
    # =========================================================================

    # --- OPTICA (gafas graduadas/sol) ---
    (r"\btwins\s+(gold|silver|platinum|sole)\b", "optica", 0.98),
    (r"\bgafas\s+twins\b", "optica", 0.98),
    (r"\bfarmamoda\b", "optica", 0.95),
    (r"\bmr\.?\s*boho\b", "optica", 0.95),

    # --- ALIMENTACION BEBE ---
    # HERO: Pattern genérico (solo|kids|puffs|cereales) + pattern numérico \d+ cereales
    (r"\bhero\s+(solo|kids|puffs|cereales)\b", "alimentacion_bebe", 0.98),
    (r"\bhero\s+\d+\s+cereales\b", "alimentacion_bebe", 0.98),
    (r"\bpediasure\b", "alimentacion_bebe", 0.95),

    # --- SALUD SEXUAL ---
    (r"\bdurex\b", "salud_sexual", 0.98),
    (r"\bcontrol\s+nature\b", "salud_sexual", 0.98),
    (r"\bcontrol\s+preservativ", "salud_sexual", 0.98),
    (r"\blubets\b", "salud_sexual", 0.95),

    # --- HIGIENE INTIMA ---
    (r"\bevax\b", "higiene_intima", 0.95),
    (r"\bfarmaconfort\b", "higiene_intima", 0.98),
    (r"\btampax\b", "higiene_intima", 0.98),
    (r"\benna\s+cycle\b", "higiene_intima", 0.98),

    # --- INCONTINENCIA ---
    (r"\bindasec\b", "incontinencia", 0.98),

    # --- PIEL SENSIBLE / ARRUGAS ---
    (r"\bsensilis\b", "piel_sensible", 0.90),
    (r"\bcetaphil\b", "piel_sensible", 0.95),

    # --- HIGIENE CORPORAL ---
    (r"\broger\s*&?\s*gallet\b", "higiene_corporal", 0.95),

    # --- SUEÑO ---
    (r"\bzzzquil\b", "sueno_insomnio", 0.98),
    (r"\bso[ñn]odina\b", "sueno_insomnio", 0.95),

    # --- NUTRICION DEPORTIVA ---
    (r"\bbarebells?\b", "nutricion_deportiva", 0.98),
    (r"\bpaleobull\b", "nutricion_deportiva", 0.95),

    # --- BEBES INFANTIL ---
    (r"\bneo\s+peques\b", "bebes_infantil", 0.95),
    (r"\bbabynaturals\b", "bebes_infantil", 0.95),

    # --- VETERINARIA ---
    (r"\bfrontline\b", "veterinaria", 0.98),
    (r"\badvantix\b", "veterinaria", 0.98),
    (r"\badtab\b", "veterinaria", 0.98),
    (r"\bvectra\s+3d\b", "veterinaria", 0.98),

    # --- HERIDAS/APOSITOS ---
    (r"\bhansaplast\b", "heridas_apositos", 0.95),
    (r"\bleukopor\b", "heridas_apositos", 0.95),
    (r"\bomnipor\b", "heridas_apositos", 0.95),
    (r"\bomnifilm\b", "heridas_apositos", 0.95),

    # --- HIGIENE OIDOS ---
    (r"\baudispray\b", "higiene_oidos", 0.98),
    (r"\bcleanotix\b", "higiene_oidos", 0.95),
    (r"\bdryotix\b", "higiene_oidos", 0.95),

    # --- OJO SECO ---
    (r"\beyestil\b", "ojo_seco", 0.95),
    (r"\bba[ñn]oftal\b", "ojo_seco", 0.95),

    # --- MUCOSIDAD RESPIRATORIA ---
    (r"\bisla\s+medic\b", "mucosidad_respiratoria", 0.95),
    (r"\bpropol\s*2\b", "mucosidad_respiratoria", 0.95),
    (r"\bprobactis\s+strep\b", "mucosidad_respiratoria", 0.95),

    # --- CONGESTION NASAL ---
    (r"\baluneb\b", "congestion_nasal", 0.95),
    (r"\bneilmed\b", "congestion_nasal", 0.95),
    (r"\brinastel\b", "congestion_nasal", 0.95),

    # --- MATERIAL SANITARIO ---
    (r"\bcorysan\b", "material_sanitario", 0.95),
    (r"\bvygon\b", "material_sanitario", 0.95),

    # --- PROTESIS DENTAL ---
    (r"\bcorega\b", "protesis_dental", 0.95),
    (r"\bkukident\b", "protesis_dental", 0.95),

    # --- ARRUGAS (marcas premium) ---
    (r"\besthederm\b", "arrugas_antiedad", 0.95),

    # --- ALIMENTACION BEBE ---
    (r"\bnutriben\b", "alimentacion_bebe", 0.95),

    # --- DOLOR MUSCULAR ---
    (r"\bfisiocrem\b", "dolor_muscular", 0.95),

    # --- CAPILAR ---
    (r"\bducray\b", "capilar_frecuente", 0.95),
]

# -----------------------------------------------------------------------------
# SUPRESIÓN DE TÓPICOS (Fix: Vitaminas en cremas)
# Si es crema/gel, PROHIBIDO asignar estas categorías
# -----------------------------------------------------------------------------
SUPPRESS_IF_TOPICAL: FrozenSet[str] = frozenset({
    # Suplementos orales - nunca para tópicos
    "vitaminas_minerales",
    "vitaminas_general",
    "omega3",
    "nutricion_clinica",
    "hierro",
    "magnesio",
    "calcio_vitamina_d",
    "vitamina_d",  # Issue #457
    "probioticos",
    "sueno_insomnio",
    "control_peso",
    "alimentacion_general",
    "energia_vitalidad",
    "defensas_inmunidad",
    "colageno",
    "nutricion_deportiva",
    "estres_ansiedad",  # Suplementos calmantes, no cremas calmantes
    # Digestivo - nunca para tópicos
    "digestivo",
    "acidez_reflujo",
    "estrenimiento",
    "gases_flatulencia",
    "diarrea",
})

# -----------------------------------------------------------------------------
# RESCUE MAPPINGS: Cuando suprimimos un tópico, usamos el propósito
# Categorías deben coincidir con symptom_taxonomy.py
# -----------------------------------------------------------------------------
TOPICAL_RESCUE_MAPPINGS: Dict[str, str] = {
    "herida": "cicatrizacion",
    "cicatriz": "cicatrizacion",
    "muscul": "dolor_muscular",
    "dolor": "dolor_muscular",
    "hongo": "hongos_piel",
    "pie": "hongos_piel",
    "quemad": "quemaduras",
    "picadura": "picaduras",
    "acne": "acne",
    "arruga": "arrugas_antiedad",
    "antiedad": "arrugas_antiedad",
}

TOPICAL_FORMAT_FALLBACKS: Dict[str, str] = {
    "topico_facial": "hidratacion_facial",
    "topico_corporal": "hidratacion_corporal",
    "topico_capilar": "caida_cabello",
    "topico_labial": "cicatrizacion",  # Labios = cuidado/cicatrización
    "oftalmico": "ojo_seco",
    "bucal": "enjuague_bucal",  # Issue #457: Subcategoría específica
}


# -----------------------------------------------------------------------------
# FUNCIONES DE UTILIDAD
# -----------------------------------------------------------------------------

def normalize_text(text: str) -> str:
    """Normaliza texto: lowercase + sin acentos."""
    if not text:
        return ""
    text = text.lower()
    # Remove accents
    text = unicodedata.normalize('NFD', text)
    text = ''.join(c for c in text if unicodedata.category(c) != 'Mn')
    return text


@dataclass
class Tier1Match:
    """Resultado de match en Tier 1."""
    necesidad: str
    confidence: float
    matched_term: str
    tier: str  # "blacklist", "specific", "generic"


def match_tier1(product_name: str) -> Optional[Tier1Match]:
    """
    Motor de reglas determinista Tier 1.

    Orden de prioridad:
    1. BLACKLIST - Excluir productos no comerciales
    2. SPECIFIC - Reglas específicas (aftas, cicatriz, etc.)
    3. GENERIC - Reglas genéricas (colutorio, pasta dental, etc.)

    Returns:
        Tier1Match si hay match, None si no hay match
    """
    clean_name = normalize_text(product_name)

    # 1. CHECK BLACKLIST
    for keyword in BLACKLIST_KEYWORDS:
        if keyword in clean_name:
            return Tier1Match(
                necesidad="interno_no_venta",
                confidence=1.0,
                matched_term=keyword,
                tier="blacklist"
            )

    # 2. CHECK SPECIFIC (Prioridad Alta)
    for pattern, necesidad, confidence in TIER_1_SPECIFIC:
        match = re.search(pattern, clean_name, re.IGNORECASE)
        if match:
            return Tier1Match(
                necesidad=necesidad,
                confidence=confidence,
                matched_term=match.group(),
                tier="specific"
            )

    # 3. CHECK GENERIC (Prioridad Baja)
    for pattern, necesidad, confidence in TIER_1_GENERIC:
        match = re.search(pattern, clean_name, re.IGNORECASE)
        if match:
            return Tier1Match(
                necesidad=necesidad,
                confidence=confidence,
                matched_term=match.group(),
                tier="generic"
            )

    return None


def rescue_suppressed_topical(
    proposito_principal: Optional[str],
    formato: str
) -> str:
    """
    Rescata categoría cuando se suprime un tópico.

    Args:
        proposito_principal: El propósito del producto según LLM
        formato: El formato galénico (topico_facial, etc.)

    Returns:
        Categoría de rescate apropiada
    """
    if proposito_principal:
        prop_lower = proposito_principal.lower()
        for keyword, category in TOPICAL_RESCUE_MAPPINGS.items():
            if keyword in prop_lower:
                return category

    # Fallback por formato
    return TOPICAL_FORMAT_FALLBACKS.get(formato, "hidratacion_corporal")


# -----------------------------------------------------------------------------
# ALL_CATEGORIES - Conjunto de todas las categorías definidas
# Issue #462: TaxonomyLabelerService - Validación de cobertura TIER1_MAPPING
# -----------------------------------------------------------------------------

def _extract_all_categories() -> FrozenSet[str]:
    """
    Extrae todas las categorías únicas de TIER_1_SPECIFIC y TIER_1_GENERIC.

    Se ejecuta una sola vez al importar el módulo.
    """
    categories = set()

    # Extraer de TIER_1_SPECIFIC
    for _, category, _ in TIER_1_SPECIFIC:
        categories.add(category)

    # Extraer de TIER_1_GENERIC
    for _, category, _ in TIER_1_GENERIC:
        categories.add(category)

    # Añadir categorías de mappings auxiliares
    categories.update(TOPICAL_FORMAT_FALLBACKS.values())
    categories.update(TOPICAL_RESCUE_MAPPINGS.values())

    # Añadir blacklist category
    categories.add("interno_no_venta")

    return frozenset(categories)


# Conjunto inmutable de todas las categorías (para validación de TIER1_MAPPING)
ALL_CATEGORIES: FrozenSet[str] = _extract_all_categories()
