# backend/app/core/subscription_limits.py
"""
Subscription limits and RBAC permissions for xFarma Admin Panel.

Issue #348: Admin Panel Unificado - RBAC & Subscription Management

Este módulo define:
- Permisos del sistema (6 permissions × 2 roles)
- Límites de suscripción (free, pro, max)
- Mapping de roles a permisos
"""

from enum import Enum
from typing import Any, Dict, Tuple

# =============================================================================
# RBAC PERMISSIONS (Issue #348)
# =============================================================================

class Permission(str, Enum):
    """
    Permisos granulares para Admin Panel y Control de Acceso por Rol.

    Issue #348: Admin Panel - 10 permisos administrativos
    Issue #541: Roles Titular vs Operativo - 6 permisos de negocio

    Permisos Administrativos (admin only):
    - manage_users: Crear/editar/eliminar usuarios y farmacias
    - view_users: Ver listado de usuarios
    - delete_sales_data: Eliminar datos de ventas (GDPR Article 17)
    - delete_accounts: Eliminar cuentas completas (GDPR Article 17)
    - manage_database: Gestionar backups y vistas materializadas
    - view_system_stats: Ver estadísticas del sistema (storage, metrics)
    - view_audit_logs: Ver registros de auditoría del sistema
    - clear_cache: Limpiar cachés del sistema (CIMA, nomenclator, labs)
    - admin_catalog_manage: Gestionar sincronización de catálogo CIMA/nomenclator (Issue #349)

    Permisos de Negocio (Issue #541: Titular vs Operativo):
    - view_financial_data: Ver márgenes, costes, rentabilidad (SOLO TITULAR)
    - view_employee_data: Ver datos de empleados y facturación (SOLO TITULAR)
    - manage_partners: Gestionar partners/laboratorios (SOLO TITULAR)
    - view_operational_data: Ver stock, ventas, productos (TITULAR + OPERATIVO)
    - upload_sales: Subir archivos de ventas (TITULAR + OPERATIVO)
    - view_dashboards: Acceso a dashboards operativos (TITULAR + OPERATIVO)
    """
    # User Management (Admin)
    MANAGE_USERS = "manage_users"
    VIEW_USERS = "view_users"

    # GDPR Compliance (Article 17 - Right to Erasure)
    DELETE_SALES_DATA = "delete_sales_data"
    DELETE_ACCOUNTS = "delete_accounts"

    # System Administration (Admin)
    MANAGE_DATABASE = "manage_database"  # Issue #348 FASE 2
    VIEW_SYSTEM_STATS = "view_system_stats"  # Issue #348 FASE 2
    VIEW_AUDIT_LOGS = "view_audit_logs"
    CLEAR_CACHE = "clear_cache"
    ADMIN_CATALOG_MANAGE = "admin_catalog_manage"  # Issue #349

    # =========================================================================
    # Issue #541: Permisos de Negocio - Titular vs Operativo
    # =========================================================================

    # Permisos EXCLUSIVOS del Titular (datos financieros sensibles)
    VIEW_FINANCIAL_DATA = "view_financial_data"      # Márgenes, PMC, rentabilidad
    VIEW_EMPLOYEE_DATA = "view_employee_data"        # Facturación por empleado
    MANAGE_PARTNERS = "manage_partners"              # Config partners/laboratorios

    # Permisos compartidos (Titular + Operativo)
    VIEW_OPERATIONAL_DATA = "view_operational_data"  # Stock, ventas (sin margen)
    UPLOAD_SALES = "upload_sales"                    # Subir archivos ERP
    VIEW_DASHBOARDS = "view_dashboards"              # Dashboards operativos


# Mapping de roles a permisos
# CRITICAL: Use tuples (immutable) to prevent accidental modification
# Issue #541: Añadidos roles "titular" y "operativo" para control de acceso a datos sensibles
ROLE_PERMISSIONS: Dict[str, Tuple[str, ...]] = {
    # =========================================================================
    # Rol: ADMIN (superusuario del sistema)
    # Uso: Solo para administradores de kaiFarma (no farmacias)
    # =========================================================================
    "admin": (
        # Admin permissions
        Permission.MANAGE_USERS,
        Permission.VIEW_USERS,
        Permission.DELETE_SALES_DATA,
        Permission.DELETE_ACCOUNTS,
        Permission.MANAGE_DATABASE,
        Permission.VIEW_SYSTEM_STATS,
        Permission.VIEW_AUDIT_LOGS,
        Permission.CLEAR_CACHE,
        Permission.ADMIN_CATALOG_MANAGE,
        # Business permissions (admin tiene todos)
        Permission.VIEW_FINANCIAL_DATA,
        Permission.VIEW_EMPLOYEE_DATA,
        Permission.MANAGE_PARTNERS,
        Permission.VIEW_OPERATIONAL_DATA,
        Permission.UPLOAD_SALES,
        Permission.VIEW_DASHBOARDS,
    ),

    # =========================================================================
    # Rol: TITULAR (propietario de farmacia)
    # Uso: El dueño de la farmacia - ve TODO incluyendo datos financieros
    # Issue #541: Nuevo rol para kaiFarma Local-First
    # =========================================================================
    "titular": (
        # Business permissions - TODOS los datos
        Permission.VIEW_FINANCIAL_DATA,      # ✅ Márgenes, PMC, rentabilidad
        Permission.VIEW_EMPLOYEE_DATA,       # ✅ Facturación por empleado
        Permission.MANAGE_PARTNERS,          # ✅ Config partners/laboratorios
        Permission.VIEW_OPERATIONAL_DATA,    # ✅ Stock, ventas
        Permission.UPLOAD_SALES,             # ✅ Subir archivos ERP
        Permission.VIEW_DASHBOARDS,          # ✅ Dashboards
    ),

    # =========================================================================
    # Rol: OPERATIVO (empleado de farmacia)
    # Uso: Farmacéuticos y personal - NO ve datos financieros sensibles
    # Issue #541: Nuevo rol para kaiFarma Local-First
    # =========================================================================
    "operativo": (
        # Business permissions - SOLO datos operativos (sin financieros)
        # ❌ VIEW_FINANCIAL_DATA - NO puede ver márgenes ni costes
        # ❌ VIEW_EMPLOYEE_DATA - NO puede ver datos de otros empleados
        # ❌ MANAGE_PARTNERS - NO puede configurar partners
        Permission.VIEW_OPERATIONAL_DATA,    # ✅ Stock, ventas (sin margen)
        Permission.UPLOAD_SALES,             # ✅ Subir archivos ERP
        Permission.VIEW_DASHBOARDS,          # ✅ Dashboards operativos
    ),

    # =========================================================================
    # Rol: USER (legacy - mantener compatibilidad)
    # Uso: Usuarios existentes sin rol específico
    # =========================================================================
    "user": (
        # Default: mismos permisos que operativo (conservative)
        Permission.VIEW_OPERATIONAL_DATA,
        Permission.UPLOAD_SALES,
        Permission.VIEW_DASHBOARDS,
    ),
}


# =============================================================================
# SUBSCRIPTION PLANS (Issue #348)
# =============================================================================

class SubscriptionPlan(str, Enum):
    """
    Planes de suscripción para farmacias.

    3 planes con límites crecientes de almacenamiento:
    - free: 100 MB (prueba/startups)
    - pro: 1 GB (farmacias pequeñas/medianas)
    - max: 10 GB (farmacias grandes/cadenas)
    """
    FREE = "free"
    PRO = "pro"
    MAX = "max"


# Límites de almacenamiento por plan (en MB)
STORAGE_LIMITS_MB: Dict[str, int] = {
    SubscriptionPlan.FREE: 100,    # 100 MB
    SubscriptionPlan.PRO: 1024,    # 1 GB
    SubscriptionPlan.MAX: 10240,   # 10 GB
}


# Límites de almacenamiento por plan (en bytes, para queries)
STORAGE_LIMITS_BYTES: Dict[str, int] = {
    SubscriptionPlan.FREE: 100 * 1024 * 1024,      # 100 MB
    SubscriptionPlan.PRO: 1024 * 1024 * 1024,      # 1 GB
    SubscriptionPlan.MAX: 10240 * 1024 * 1024,     # 10 GB
}


# Características por plan
PLAN_FEATURES: Dict[str, Dict[str, Any]] = {
    SubscriptionPlan.FREE: {
        "storage_mb": 100,
        "max_sales_records": 10000,
        "max_file_uploads": 5,
        "backup_retention_days": 7,
        "api_calls_per_day": 1000,
        "support_level": "community",
        "description": "Plan gratuito para pruebas y startups",
    },
    SubscriptionPlan.PRO: {
        "storage_mb": 1024,
        "max_sales_records": 100000,
        "max_file_uploads": 50,
        "backup_retention_days": 30,
        "api_calls_per_day": 10000,
        "support_level": "email",
        "description": "Plan profesional para farmacias pequeñas y medianas",
    },
    SubscriptionPlan.MAX: {
        "storage_mb": 10240,
        "max_sales_records": 1000000,
        "max_file_uploads": 500,
        "backup_retention_days": 90,
        "api_calls_per_day": 100000,
        "support_level": "priority",
        "description": "Plan máximo para farmacias grandes y cadenas",
    },
}


# =============================================================================
# HELPER FUNCTIONS
# =============================================================================

def get_storage_limit_mb(plan: str) -> int:
    """
    Obtiene el límite de almacenamiento en MB para un plan.

    Args:
        plan: Subscription plan ('free', 'pro', 'max')

    Returns:
        Límite de almacenamiento en MB

    Example:
        >>> get_storage_limit_mb('pro')
        1024
    """
    return STORAGE_LIMITS_MB.get(plan, STORAGE_LIMITS_MB[SubscriptionPlan.FREE])


def get_storage_limit_bytes(plan: str) -> int:
    """
    Obtiene el límite de almacenamiento en bytes para un plan.

    Args:
        plan: Subscription plan ('free', 'pro', 'max')

    Returns:
        Límite de almacenamiento en bytes

    Example:
        >>> get_storage_limit_bytes('pro')
        1073741824  # 1 GB in bytes
    """
    return STORAGE_LIMITS_BYTES.get(plan, STORAGE_LIMITS_BYTES[SubscriptionPlan.FREE])


def get_plan_features(plan: str) -> Dict[str, Any]:
    """
    Obtiene todas las características de un plan.

    Args:
        plan: Subscription plan ('free', 'pro', 'max')

    Returns:
        Diccionario con características del plan

    Example:
        >>> features = get_plan_features('pro')
        >>> features['max_sales_records']
        100000
    """
    return PLAN_FEATURES.get(plan, PLAN_FEATURES[SubscriptionPlan.FREE])


def has_permission(role_name: str, permission: str) -> bool:
    """
    Verifica si un rol tiene un permiso específico.

    Args:
        role_name: Nombre del rol ('admin', 'user')
        permission: Permission constant (e.g., Permission.MANAGE_USERS)

    Returns:
        True si el rol tiene el permiso, False otherwise

    Example:
        >>> has_permission('admin', Permission.MANAGE_USERS)
        True
        >>> has_permission('user', Permission.MANAGE_USERS)
        False
    """
    role_perms = ROLE_PERMISSIONS.get(role_name, ())
    return permission in role_perms


def get_role_permissions(role_name: str) -> Tuple[str, ...]:
    """
    Obtiene todos los permisos de un rol.

    Args:
        role_name: Nombre del rol ('admin', 'user')

    Returns:
        Tuple de permisos del rol (immutable)

    Example:
        >>> perms = get_role_permissions('admin')
        >>> len(perms)
        6
    """
    return ROLE_PERMISSIONS.get(role_name, ())


def validate_subscription_plan(plan: str) -> bool:
    """
    Valida si un plan de suscripción es válido.

    Args:
        plan: Subscription plan string

    Returns:
        True si el plan es válido ('free', 'pro', 'max')

    Example:
        >>> validate_subscription_plan('pro')
        True
        >>> validate_subscription_plan('invalid')
        False
    """
    return plan in [SubscriptionPlan.FREE, SubscriptionPlan.PRO, SubscriptionPlan.MAX]


# =============================================================================
# ROLE UUIDs (Issue #348, #541)
# =============================================================================

# Fixed UUIDs para roles (insertados en migración)
ADMIN_ROLE_UUID = "22222222-2222-2222-2222-222222222222"
USER_ROLE_UUID = "33333333-3333-3333-3333-333333333333"
# Issue #541: Nuevos roles para kaiFarma Local-First
TITULAR_ROLE_UUID = "44444444-4444-4444-4444-444444444444"
OPERATIVO_ROLE_UUID = "55555555-5555-5555-5555-555555555555"


# Mapping de nombre a UUID
ROLE_UUID_MAPPING: Dict[str, str] = {
    "admin": ADMIN_ROLE_UUID,
    "user": USER_ROLE_UUID,
    "titular": TITULAR_ROLE_UUID,      # Issue #541
    "operativo": OPERATIVO_ROLE_UUID,  # Issue #541
}


def get_role_uuid(role_name: str) -> str:
    """
    Obtiene el UUID de un rol por su nombre.

    Args:
        role_name: Nombre del rol ('admin', 'user')

    Returns:
        UUID del rol como string

    Example:
        >>> get_role_uuid('admin')
        '22222222-2222-2222-2222-222222222222'
    """
    return ROLE_UUID_MAPPING.get(role_name, USER_ROLE_UUID)


# =============================================================================
# FEATURE ACCESS FUNCTIONS (Issue #402)
# =============================================================================

def can_use_employee_filtering(plan: str) -> bool:
    """
    Verifica si un plan tiene acceso a employee filtering.

    Issue #402: Employee filtering es una feature PRO+.

    Args:
        plan: Subscription plan ('free', 'pro', 'max')

    Returns:
        True si el plan es PRO o MAX, False para FREE

    Example:
        >>> can_use_employee_filtering('free')
        False
        >>> can_use_employee_filtering('pro')
        True
        >>> can_use_employee_filtering('max')
        True
    """
    return plan in (SubscriptionPlan.PRO, SubscriptionPlan.MAX)
