# backend/app/core/security_local/local_state.py
"""
Local Security Manager for kaiFarma Desktop (Pivot 2026).

Session-based authentication for local Windows installation.
State resides in process memory, not in JWT tokens.

Architecture:
- Single pharmacy per installation (1:1)
- PIN-based unlock (like phone lock screen)
- Auto-lock after inactivity
- License key validated against Hub on startup (when Hub API is ready)

Security:
- PIN hashed with bcrypt (salted, slow hash)
- Roles: titular (full access) vs operativo (no financial data)
"""

import logging
import os
from datetime import datetime
from typing import Literal, Optional

from passlib.context import CryptContext
from pydantic import BaseModel

from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

# Bcrypt context for PIN hashing (secure, salted, slow)
_pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")


class PharmacySession(BaseModel):
    """Current session state for the local pharmacy terminal."""

    is_unlocked: bool = False
    pharmacy_name: str = "Farmacia Local"
    license_key: str = ""
    license_valid: bool = True  # Assume valid until Hub validates
    license_tier: Literal["free", "pro", "enterprise"] = "pro"
    role: Literal["titular", "operativo"] = "operativo"
    last_activity: datetime = None  # Will be set on creation
    auto_lock_minutes: int = 5

    class Config:
        # Allow datetime mutation
        validate_assignment = True

    def __init__(self, **data):
        # Set default last_activity with utc_now() if not provided
        if "last_activity" not in data or data["last_activity"] is None:
            data["last_activity"] = utc_now()
        super().__init__(**data)


class LicenseValidationResult(BaseModel):
    """Result of license validation against Hub."""

    valid: bool
    tier: Literal["free", "pro", "enterprise"] = "free"
    expires_at: Optional[datetime] = None
    message: str = ""

    # Grace period info (Pivot 2026)
    in_grace_period: bool = False
    days_remaining: Optional[int] = None

    # Kill-switch status
    kill_switch_active: bool = False


class LocalSecurityManager:
    """
    Singleton manager for local terminal security.

    Replaces JWT authentication for desktop installations.
    The session state lives in RAM - no tokens to validate.

    Usage:
        # At startup (in main.py)
        security_manager.initialize("LICENSE-KEY", "Farmacia Example")

        # In endpoints (via deps.py)
        session = security_manager.get_session()
        if not session.is_unlocked:
            raise HTTPException(401, "Terminal bloqueado")

        # From frontend
        POST /api/v1/auth/local/unlock {"pin": "1234"}
        POST /api/v1/auth/local/lock
        GET /api/v1/auth/local/status
    """

    _instance: Optional["LocalSecurityManager"] = None
    _session: Optional[PharmacySession] = None
    _pin_hash: Optional[str] = None

    def __new__(cls) -> "LocalSecurityManager":
        """Singleton pattern - only one instance per process."""
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance

    def initialize(
        self,
        license_key: str,
        pharmacy_name: str,
        pin_hash: Optional[str] = None,
        role: Literal["titular", "operativo"] = "operativo",
        auto_lock_minutes: int = 5,
        license_result: Optional[LicenseValidationResult] = None,
    ) -> None:
        """
        Initialize the security manager at application startup.

        Called by the Smart Loader after validating the license key.

        Args:
            license_key: The pharmacy's license key
            pharmacy_name: Display name for the pharmacy
            pin_hash: Pre-hashed PIN (if None, defaults to "0000")
            role: User role for permission checks
            auto_lock_minutes: Minutes of inactivity before auto-lock
            license_result: Pre-computed license validation result (from LicenseClientService)
        """
        # Use pre-computed result or fall back to stub validation
        if license_result is None:
            license_result = self._validate_license(license_key)

        self._session = PharmacySession(
            is_unlocked=False,  # Always start locked
            license_key=license_key,
            license_valid=license_result.valid,
            license_tier=license_result.tier,
            pharmacy_name=pharmacy_name,
            role=role,
            auto_lock_minutes=auto_lock_minutes,
            last_activity=utc_now(),
        )

        # Default PIN "0000" for first-time setup
        if pin_hash:
            self._pin_hash = pin_hash
        else:
            self._pin_hash = self._hash_pin("0000")

        logger.info(
            "[SECURITY_LOCAL] Manager initialized",
            extra={
                "pharmacy_name": pharmacy_name,
                "role": role,
                "license_valid": license_result.valid,
                "license_tier": license_result.tier,
            },
        )

    def _validate_license(self, license_key: str) -> LicenseValidationResult:
        """
        Validate license key (synchronous stub).

        IMPORTANT: This is a fallback stub. The actual license validation
        should be done asynchronously via LicenseClientService in main.py,
        and the result passed to initialize().

        This stub is kept for:
        1. Backward compatibility
        2. Development/testing without Hub
        3. Emergency fallback

        For production, use:
            from app.services.license_client import LicenseClientService
            result = await license_client.validate_license(key)
            security_manager.initialize(..., license_result=result)
        """
        hub_url = os.getenv("HUB_URL", "")

        if not hub_url:
            # Hub not configured - accept license with warning
            logger.warning(
                "[SECURITY_LOCAL] License validation skipped - HUB_URL not configured. "
                "License accepted in offline mode.",
                extra={"license_key": license_key[:8] + "..."},
            )
            return LicenseValidationResult(
                valid=True,
                tier="pro",  # Default to PRO for MVP
                message="Offline mode - license validation skipped",
            )

        logger.warning(
            "[SECURITY_LOCAL] Using stub validation - LicenseClientService not used",
            extra={"license_key": license_key[:8] + "...", "hub_url": hub_url},
        )
        return LicenseValidationResult(
            valid=True,
            tier="pro",
            message="Stub validation - use LicenseClientService for production",
        )

    def _hash_pin(self, pin: str) -> str:
        """
        Hash a PIN for secure storage.

        Uses bcrypt with automatic salt for security.
        Even though PINs are short (4-6 digits), bcrypt's
        slow hashing makes brute-force attacks impractical.
        """
        return _pwd_context.hash(pin)

    def _verify_pin(self, pin: str, pin_hash: str) -> bool:
        """Verify a PIN against its bcrypt hash."""
        try:
            return _pwd_context.verify(pin, pin_hash)
        except Exception as e:
            logger.error(f"[SECURITY_LOCAL] PIN verification error: {e}")
            return False

    def unlock(self, pin: str) -> bool:
        """
        Attempt to unlock the terminal with the given PIN.

        Args:
            pin: The PIN entered by the user

        Returns:
            True if unlock successful, False if PIN incorrect
        """
        if self._pin_hash is None:
            logger.warning("[SECURITY_LOCAL] Unlock attempt with no PIN configured")
            return False

        if self._verify_pin(pin, self._pin_hash):
            if self._session:
                self._session.is_unlocked = True
                self._session.last_activity = utc_now()
                logger.info("[SECURITY_LOCAL] Terminal unlocked")
            return True

        logger.warning("[SECURITY_LOCAL] Invalid PIN attempt")
        return False

    def lock(self) -> None:
        """Lock the terminal (manual lock or auto-lock)."""
        if self._session:
            self._session.is_unlocked = False
            logger.info("[SECURITY_LOCAL] Terminal locked")

    def refresh_activity(self) -> None:
        """Update last_activity timestamp (called on user interaction)."""
        if self._session and self._session.is_unlocked:
            self._session.last_activity = utc_now()

    def check_auto_lock(self) -> bool:
        """
        Check if terminal should auto-lock due to inactivity.

        Returns:
            True if terminal was locked, False otherwise
        """
        if self._session and self._session.is_unlocked:
            inactive_seconds = (utc_now() - self._session.last_activity).total_seconds()
            timeout_seconds = self._session.auto_lock_minutes * 60

            if inactive_seconds > timeout_seconds:
                logger.info(
                    "[SECURITY_LOCAL] Auto-lock triggered",
                    extra={"inactive_seconds": inactive_seconds},
                )
                self.lock()
                return True
        return False

    def get_session(self) -> Optional[PharmacySession]:
        """Get the current session (may be None if not initialized)."""
        return self._session

    def is_initialized(self) -> bool:
        """Check if the manager has been initialized."""
        return self._session is not None

    def set_pin(self, new_pin: str) -> None:
        """
        Change the terminal PIN.

        Note: Authorization check (titular only) is done at the API layer.
        """
        self._pin_hash = self._hash_pin(new_pin)
        logger.info("[SECURITY_LOCAL] PIN changed")

    def is_titular(self) -> bool:
        """Check if current session has titular role."""
        return self._session is not None and self._session.role == "titular"

    @staticmethod
    def is_local_mode() -> bool:
        """Check if running in local mode (vs cloud/Hub mode)."""
        return os.getenv("KAIFARMA_LOCAL", "").lower() == "true"


# Global singleton instance
security_manager = LocalSecurityManager()
