# backend/app/core/security.py
"""
Security utilities for OAuth2 + JWT authentication
Supports traditional login and social OAuth providers

Pivot 2026: Conditional imports for local mode (no authlib/jose needed)
"""

import os
from datetime import timedelta
from typing import Any, Dict, Optional

from dotenv import load_dotenv
from fastapi import Depends, HTTPException, status
from fastapi.security import HTTPAuthorizationCredentials, HTTPBearer
from passlib.context import CryptContext
from sqlalchemy.orm import Session

from app.database import get_db
from app.models.user import User
from app.utils.datetime_utils import utc_now

# Load environment variables
load_dotenv()

# Pivot 2026: Detect local mode BEFORE importing OAuth/JWT modules
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"

if IS_LOCAL_MODE:
    # Local mode: Provide stubs (OAuth/JWT not used - PIN-based auth)
    OAuth = None
    oauth = None
    JWTError = Exception
    jwt = None  # type: ignore
    GOOGLE_CLIENT_ID = ""
    GOOGLE_CLIENT_SECRET = ""
    MICROSOFT_CLIENT_ID = ""
    MICROSOFT_CLIENT_SECRET = ""
    # Use a simple secret for local session tokens
    SECRET_KEY = os.getenv("SECRET_KEY", "local-development-key-not-for-production")
else:
    # Cloud mode: Full OAuth/JWT support
    from authlib.integrations.starlette_client import OAuth
    from jose import JWTError, jwt

    # OAuth configuration
    GOOGLE_CLIENT_ID = os.getenv("GOOGLE_CLIENT_ID", "")
    GOOGLE_CLIENT_SECRET = os.getenv("GOOGLE_CLIENT_SECRET", "")
    MICROSOFT_CLIENT_ID = os.getenv("MICROSOFT_CLIENT_ID", "")
    MICROSOFT_CLIENT_SECRET = os.getenv("MICROSOFT_CLIENT_SECRET", "")

    # Security configuration
    SECRET_KEY = os.getenv("SECRET_KEY")
    if not SECRET_KEY:
        raise ValueError("SECRET_KEY must be set in environment variables for production")

    # OAuth client
    oauth = OAuth()

# Common configuration
ALGORITHM = "HS256"
ACCESS_TOKEN_EXPIRE_MINUTES = int(os.getenv("ACCESS_TOKEN_EXPIRE_MINUTES", "15"))
REFRESH_TOKEN_EXPIRE_DAYS = int(os.getenv("REFRESH_TOKEN_EXPIRE_DAYS", "7"))

# Base URL for OAuth callbacks
BASE_URL = os.getenv("BASE_URL", "http://localhost:8000")

# Password hashing - Modern argon2 setup (recommended for 2025)
pwd_context = CryptContext(
    schemes=["argon2"],
    deprecated="auto",
    # Argon2 configuration - secure defaults
    argon2__memory_cost=65536,  # 64 MB
    argon2__time_cost=3,  # 3 iterations
    argon2__parallelism=1,  # Single thread
)

# HTTP Bearer for JWT tokens
security = HTTPBearer()


def configure_oauth():
    """Configure OAuth providers"""

    # Configure Google OAuth
    if GOOGLE_CLIENT_ID and GOOGLE_CLIENT_SECRET:
        oauth.register(
            name="google",
            client_id=GOOGLE_CLIENT_ID,
            client_secret=GOOGLE_CLIENT_SECRET,
            server_metadata_url="https://accounts.google.com/.well-known/openid-configuration",
            client_kwargs={
                "scope": "openid email profile",
                "prompt": "select_account",
            },
        )

    # Configure Microsoft OAuth
    if MICROSOFT_CLIENT_ID and MICROSOFT_CLIENT_SECRET:
        oauth.register(
            name="microsoft",
            client_id=MICROSOFT_CLIENT_ID,
            client_secret=MICROSOFT_CLIENT_SECRET,
            authorize_url="https://login.microsoftonline.com/common/oauth2/v2.0/authorize",
            access_token_url="https://login.microsoftonline.com/common/oauth2/v2.0/token",
            client_kwargs={
                "scope": "openid email profile",
            },
        )


# Password utilities
def verify_password(plain_password: str, hashed_password: str) -> bool:
    """Verify a password against its hash"""
    return pwd_context.verify(plain_password, hashed_password)


def get_password_hash(password: str) -> str:
    """Hash a password"""
    return pwd_context.hash(password)


# JWT utilities
def create_access_token(data: dict, expires_delta: Optional[timedelta] = None) -> str:
    """Create a JWT access token"""
    to_encode = data.copy()

    if expires_delta:
        expire = utc_now() + expires_delta
    else:
        expire = utc_now() + timedelta(minutes=ACCESS_TOKEN_EXPIRE_MINUTES)

    to_encode.update({"exp": expire, "type": "access"})
    encoded_jwt = jwt.encode(to_encode, SECRET_KEY, algorithm=ALGORITHM)
    return encoded_jwt


def create_refresh_token(data: dict) -> str:
    """Create a JWT refresh token"""
    to_encode = data.copy()
    expire = utc_now() + timedelta(days=REFRESH_TOKEN_EXPIRE_DAYS)
    to_encode.update({"exp": expire, "type": "refresh"})
    encoded_jwt = jwt.encode(to_encode, SECRET_KEY, algorithm=ALGORITHM)
    return encoded_jwt


def verify_token(token: str, token_type: str = "access") -> Dict[str, Any]:
    """Verify and decode a JWT token"""
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])

        # Check token type
        if payload.get("type") != token_type:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail=f"Invalid token type. Expected {token_type}",
            )

        return payload

    except JWTError:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Could not validate credentials",
        )


# Authentication dependencies
async def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security), db: Session = Depends(get_db)
) -> User:
    """Get current authenticated user from JWT token"""

    token = credentials.credentials

    try:
        payload = verify_token(token, "access")
        user_id = payload.get("sub")

        if user_id is None:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Invalid authentication credentials",
            )

    except JWTError:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid authentication credentials",
        )

    user = db.query(User).filter(User.id == user_id).first()

    if user is None:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="User not found",
        )

    return user


async def get_current_active_user(current_user: User = Depends(get_current_user)) -> User:
    """Get current active user"""

    if not current_user.is_active:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Inactive user")

    return current_user


def require_permissions(*permissions: str):
    """
    DEPRECATED: Use app.api.deps.require_permission() instead.

    This version lacks:
    - Audit logging (CRITICAL for compliance)
    - Soft-delete checks (deleted_at validation)
    - Structured error messages

    The version in deps.py is more complete and should be used for all new code.
    This function is kept for backwards compatibility only.

    Migration guide:
        # OLD (security.py)
        @router.get("/endpoint", dependencies=[Depends(require_permissions("MANAGE_USERS"))])

        # NEW (deps.py)
        from app.api.deps import require_permission
        @router.get("/endpoint", dependencies=[Depends(require_permission("MANAGE_USERS"))])
    """
    import warnings
    warnings.warn(
        "security.require_permissions() is deprecated. Use api.deps.require_permission() instead.",
        DeprecationWarning,
        stacklevel=2
    )

    async def permission_checker(current_user: User = Depends(get_current_active_user)) -> User:
        for permission in permissions:
            if not current_user.has_permission(permission):
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN, detail=f"Not enough permissions. Required: {permission}"
                )
        return current_user

    return permission_checker


def pharmacy_access_required(pharmacy_id: str, current_user: User = Depends(get_current_active_user)) -> User:
    """Check if user has access to a specific pharmacy"""

    # Admins have access to all pharmacies
    if current_user.is_superuser or current_user.role == "admin":
        return current_user

    # Check if user belongs to the pharmacy
    if str(current_user.pharmacy_id) != pharmacy_id:
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="You don't have access to this pharmacy")

    return current_user


# Initialize OAuth on module load (skip in local mode)
if not IS_LOCAL_MODE:
    configure_oauth()
