# backend/app/core/schema_validator.py
"""
Database Schema Validator
Valida que el esquema de base de datos coincida con el estado esperado de Alembic
Previene el arranque del backend si hay inconsistencias críticas
"""

from typing import Dict

from sqlalchemy import inspect, text
from sqlalchemy.orm import Session

from app.core.logging_config import configure_advanced_logging

logger = configure_advanced_logging()


def validate_schema_on_startup(db: Session, strict: bool = True) -> Dict[str, any]:
    """
    Valida que el esquema de base de datos esté sincronizado con Alembic.

    Args:
        db: Sesión de base de datos
        strict: Si True, lanza excepción en caso de error crítico

    Returns:
        Dict con resultado de validación

    Raises:
        RuntimeError: Si strict=True y hay errores críticos
    """
    try:
        # 1. Verificar que exista alembic_version
        alembic_version = db.execute(text("SELECT version_num FROM alembic_version")).fetchone()

        if not alembic_version:
            error_msg = "CRITICAL: alembic_version table exists but is empty"
            logger.critical("schema.validation.critical_failure", error=error_msg)
            if strict:
                raise RuntimeError(error_msg)
            return {"status": "error", "message": error_msg}

        current_version = alembic_version[0]
        logger.info("schema.validation.alembic_version", version=current_version)

        # 2. Verificar tablas esperadas existen
        inspector = inspect(db.bind)
        existing_tables = set(inspector.get_table_names())

        # Tablas críticas que DEBEN existir
        required_tables = {
            "users",
            "pharmacies",
            "sales_data",
            "product_catalog",
            "sales_enrichment",
            "file_uploads",
            "invitations",
            "audit_logs",
            "system_status",
            "system_health_metrics",
            "homogeneous_groups",
            "homogeneous_groups_master",
            "pharmacy_homogeneous_metrics",
            "nomenclator_local",
            "pharmacy_partners",
        }

        missing_tables = required_tables - existing_tables

        if missing_tables:
            error_msg = f"CRITICAL: Missing required tables: {missing_tables}"
            logger.critical(
                "schema.validation.missing_tables",
                missing=list(missing_tables),
                alembic_version=current_version,
                existing_count=len(existing_tables),
            )

            if strict:
                raise RuntimeError(
                    f"Database schema inconsistency detected.\n"
                    f"Alembic version: {current_version}\n"
                    f"Missing tables: {missing_tables}\n"
                    f"This usually indicates alembic_version is out of sync.\n"
                    f'Run: docker-compose exec postgres psql -U xfarma_user -d xfarma_db -c "DELETE FROM alembic_version;"\n'
                    f"Then: docker-compose exec backend alembic upgrade head"
                )

            return {
                "status": "error",
                "message": error_msg,
                "alembic_version": current_version,
                "missing_tables": list(missing_tables),
            }

        # 3. Todo OK
        logger.info(
            "schema.validation.success",
            alembic_version=current_version,
            tables_count=len(existing_tables),
            required_tables_found=len(required_tables),
        )

        return {
            "status": "ok",
            "alembic_version": current_version,
            "tables_count": len(existing_tables),
            "validation": "passed",
        }

    except RuntimeError:
        raise  # Re-lanzar RuntimeError para bloquear startup
    except Exception as e:
        logger.error("schema.validation.unexpected_error", error=str(e))
        if strict:
            raise RuntimeError(f"Schema validation failed: {e}")
        return {"status": "error", "message": str(e)}


def get_schema_status(db: Session) -> Dict[str, any]:
    """
    Obtiene estado actual del esquema para endpoints de salud.
    Version no-strict de validate_schema_on_startup.

    Args:
        db: Sesión de base de datos

    Returns:
        Dict con estado del esquema
    """
    return validate_schema_on_startup(db, strict=False)
