"""
Rate Limiting Configuration for Admin Operations.
Protege endpoints destructivos contra abuso.
Integrado con el sistema de rate limiting existente.
"""

import logging

from fastapi import Request, Response
from fastapi.responses import JSONResponse
from slowapi import Limiter
from slowapi.errors import RateLimitExceeded
from slowapi.util import get_remote_address

logger = logging.getLogger(__name__)

# Crear instancia del limiter usando IP remota como key
limiter = Limiter(key_func=get_remote_address)


# Rate limits específicos para operaciones admin
ADMIN_RATE_LIMITS = {
    # Operaciones destructivas - muy restrictivas
    "delete_operations": "1/minute",  # Solo 1 borrado por minuto
    "clean_operations": "1/hour",  # Solo 1 limpieza por hora
    # Operaciones de sincronización - moderadas
    "sync_operations": "3/hour",  # Máximo 3 syncs por hora
    "force_sync": "1/hour",  # Solo 1 force sync por hora
    # Operaciones de mantenimiento - moderadas
    "vacuum_operations": "2/hour",  # 2 VACUUM por hora
    "reindex_operations": "2/hour",  # 2 reindex por hora
    # Operaciones generales admin - permisivas
    "admin_read": "300/minute",  # Lecturas frecuentes permitidas
    "admin_write": "60/minute",  # Escrituras moderadas
    "admin_general": "100/minute",  # Operaciones generales
}


def create_rate_limit_error_handler(request: Request, exc: RateLimitExceeded) -> Response:
    """
    Manejador personalizado para errores de rate limit.
    """
    response = JSONResponse(
        status_code=429,
        content={
            "detail": f"Rate limit excedido: {exc.detail}",
            "message": "Demasiadas solicitudes. Por favor espere antes de intentar nuevamente.",
            "retry_after": exc.retry_after if hasattr(exc, "retry_after") else 60,
        },
    )

    # Agregar headers estándar de rate limiting
    response.headers["Retry-After"] = str(exc.retry_after if hasattr(exc, "retry_after") else 60)
    response.headers["X-RateLimit-Limit"] = str(exc.limit if hasattr(exc, "limit") else "N/A")

    # Log del rate limit excedido
    logger.warning(f"Rate limit exceeded for {request.client.host}: {request.url.path}")

    return response


def get_admin_limiter(operation_type: str):
    """
    Obtiene el decorator de rate limiting para un tipo de operación admin.

    Args:
        operation_type: Tipo de operación (delete, sync, etc.)

    Returns:
        Decorator de rate limiting configurado
    """
    rate = ADMIN_RATE_LIMITS.get(f"{operation_type}_operations", ADMIN_RATE_LIMITS["admin_general"])
    return limiter.limit(rate)


# Decorators pre-configurados para uso común
admin_delete_limit = limiter.limit(ADMIN_RATE_LIMITS["delete_operations"])
admin_sync_limit = limiter.limit(ADMIN_RATE_LIMITS["sync_operations"])
admin_vacuum_limit = limiter.limit(ADMIN_RATE_LIMITS["vacuum_operations"])
admin_general_limit = limiter.limit(ADMIN_RATE_LIMITS["admin_general"])


# Función para verificar si un usuario ha excedido el rate limit
def check_rate_limit(request: Request, operation_type: str) -> bool:
    """
    Verifica si una request excedería el rate limit sin consumir el límite.

    Args:
        request: FastAPI Request object
        operation_type: Tipo de operación a verificar

    Returns:
        True si la operación está permitida, False si excedería el límite
    """
    # Esta es una verificación simplificada
    # En producción, implementarías verificación contra Redis/cache
    return True  # Por ahora siempre permite, implementar lógica real según necesidad


# Middleware para aplicar rate limiting global a rutas admin
async def admin_rate_limit_middleware(request: Request, call_next):
    """
    Middleware para aplicar rate limiting a todas las rutas /api/v1/admin/*.
    """
    if request.url.path.startswith("/api/v1/admin/"):
        # Aplicar rate limiting general para rutas admin
        # Esto es adicional a los límites específicos por endpoint
        pass  # Implementar según necesidad

    response = await call_next(request)
    return response
