# backend/app/core/rate_limiter.py
"""
Rate limiting middleware para xFarma
Implementa límites de request por IP y endpoint según Issue #23 Fase 1.2
"""

import logging
import time
from collections import defaultdict, deque
from typing import Dict, Optional

from fastapi import HTTPException, Request
from fastapi.responses import JSONResponse

logger = logging.getLogger(__name__)


class RateLimitConfig:
    """Configuración de rate limiting"""

    # Límites globales por IP
    GLOBAL_REQUESTS_PER_MINUTE = 100
    GLOBAL_REQUESTS_PER_HOUR = 1000

    # Límites específicos para laboratory endpoints
    # Aumentado de 500 a 1000 (Issue rate limiting /generics - 2025-12-01)
    LABORATORY_REQUESTS_PER_MINUTE = 60
    LABORATORY_REQUESTS_PER_HOUR = 1000

    # Límites de payload
    MAX_REQUEST_SIZE_BYTES = 50 * 1024 * 1024  # 50MB - suficiente para archivos CSV de farmacias
    REQUEST_TIMEOUT_SECONDS = 5

    # Ventanas de tiempo
    MINUTE_WINDOW = 60
    HOUR_WINDOW = 3600

    # Endpoints específicos con límites especiales
    LABORATORY_ENDPOINTS = [
        "/api/v1/laboratory-mapping/codes-to-names",
        "/api/v1/laboratory-mapping/names-to-codes",
        "/api/v1/laboratory-mapping/generic-laboratories",
    ]

    # Endpoints excluidos del rate limiting (Dash internos)
    EXCLUDED_ENDPOINTS = ["/_dash-update-component", "/_dash-layout", "/_dash-dependencies", "/_favicon.ico"]


class RateLimitStore:
    """Store en memoria para tracking de rate limits por IP"""

    def __init__(self):
        # {ip: deque of request timestamps}
        self.requests_per_minute: Dict[str, deque] = defaultdict(deque)
        self.requests_per_hour: Dict[str, deque] = defaultdict(deque)

        # Cleanup timestamp
        self.last_cleanup = time.time()
        self.cleanup_interval = 300  # 5 minutos

    def _cleanup_old_requests(self):
        """Limpieza periódica de requests antiguos para evitar memory leak"""
        now = time.time()

        if now - self.last_cleanup < self.cleanup_interval:
            return

        # Limpiar requests de más de 1 hora
        cutoff_time = now - RateLimitConfig.HOUR_WINDOW

        for ip in list(self.requests_per_minute.keys()):
            # Limpiar minute window
            minute_queue = self.requests_per_minute[ip]
            while minute_queue and minute_queue[0] < now - RateLimitConfig.MINUTE_WINDOW:
                minute_queue.popleft()

            # Limpiar hour window
            hour_queue = self.requests_per_hour[ip]
            while hour_queue and hour_queue[0] < cutoff_time:
                hour_queue.popleft()

            # Eliminar IP si no tiene requests recientes
            if not minute_queue and not hour_queue:
                del self.requests_per_minute[ip]
                del self.requests_per_hour[ip]

        self.last_cleanup = now
        logger.debug(f"[RATE_LIMITER] Cleanup completado. IPs activos: {len(self.requests_per_minute)}")

    def add_request(self, ip: str) -> None:
        """Agregar nuevo request para una IP"""
        now = time.time()
        self.requests_per_minute[ip].append(now)
        self.requests_per_hour[ip].append(now)
        self._cleanup_old_requests()

    def get_request_counts(self, ip: str) -> tuple[int, int]:
        """Obtener conteo de requests en ventana de minuto y hora"""
        now = time.time()

        # Limpiar requests antiguos para esta IP
        minute_queue = self.requests_per_minute[ip]
        while minute_queue and minute_queue[0] < now - RateLimitConfig.MINUTE_WINDOW:
            minute_queue.popleft()

        hour_queue = self.requests_per_hour[ip]
        while hour_queue and hour_queue[0] < now - RateLimitConfig.HOUR_WINDOW:
            hour_queue.popleft()

        return len(minute_queue), len(hour_queue)

    def check_rate_limit(self, ip: str, is_laboratory_endpoint: bool = False) -> Optional[dict]:
        """
        Verificar si la IP ha excedido el rate limit.

        Returns:
            None si no hay límite excedido
            Dict con información del error si hay límite excedido
        """
        minute_count, hour_count = self.get_request_counts(ip)

        # Determinar límites según el tipo de endpoint
        if is_laboratory_endpoint:
            minute_limit = RateLimitConfig.LABORATORY_REQUESTS_PER_MINUTE
            hour_limit = RateLimitConfig.LABORATORY_REQUESTS_PER_HOUR
        else:
            minute_limit = RateLimitConfig.GLOBAL_REQUESTS_PER_MINUTE
            hour_limit = RateLimitConfig.GLOBAL_REQUESTS_PER_HOUR

        # Verificar límite por minuto
        if minute_count >= minute_limit:
            return {
                "error_type": "rate_limit_exceeded",
                "message": f"Demasiadas requests por minuto. Límite: {minute_limit}/min",
                "retry_after": 60,
                "limit_type": "per_minute",
                "current_count": minute_count,
                "limit": minute_limit,
            }

        # Verificar límite por hora
        if hour_count >= hour_limit:
            return {
                "error_type": "rate_limit_exceeded",
                "message": f"Demasiadas requests por hora. Límite: {hour_limit}/hour",
                "retry_after": 3600,
                "limit_type": "per_hour",
                "current_count": hour_count,
                "limit": hour_limit,
            }

        return None


# Store global para rate limiting
rate_limit_store = RateLimitStore()


def get_client_ip(request: Request) -> str:
    """Extraer IP del cliente, considerando proxies"""
    # Verificar headers de proxy
    forwarded_for = request.headers.get("X-Forwarded-For")
    if forwarded_for:
        # Tomar la primera IP (cliente original)
        return forwarded_for.split(",")[0].strip()

    real_ip = request.headers.get("X-Real-IP")
    if real_ip:
        return real_ip.strip()

    # Fallback a IP directa
    return request.client.host if request.client else "unknown"


async def rate_limit_middleware(request: Request, call_next):
    """
    Middleware de rate limiting para FastAPI.

    Aplica límites diferenciados:
    - Endpoints laboratory: límites más restrictivos
    - Otros endpoints: límites globales
    """
    client_ip = get_client_ip(request)
    path = request.url.path

    # Verificar si el endpoint está excluido del rate limiting
    is_excluded = any(excluded_path in path for excluded_path in RateLimitConfig.EXCLUDED_ENDPOINTS)

    # Si está excluido, continuar sin rate limiting
    if is_excluded:
        return await call_next(request)

    # Determinar si es endpoint de laboratory
    is_laboratory_endpoint = any(laboratory_path in path for laboratory_path in RateLimitConfig.LABORATORY_ENDPOINTS)

    # Verificar rate limit ANTES de procesar request
    rate_limit_error = rate_limit_store.check_rate_limit(ip=client_ip, is_laboratory_endpoint=is_laboratory_endpoint)

    if rate_limit_error:
        logger.warning(
            f"[RATE_LIMITER] Rate limit exceeded: IP={client_ip}, "
            f"path={path}, error={rate_limit_error['limit_type']}"
        )

        return JSONResponse(
            status_code=429,
            content=rate_limit_error,
            headers={
                "Retry-After": str(rate_limit_error["retry_after"]),
                "X-RateLimit-Limit": str(rate_limit_error["limit"]),
                "X-RateLimit-Remaining": "0",
                "X-RateLimit-Reset": str(int(time.time() + rate_limit_error["retry_after"])),
            },
        )

    # Verificar tamaño de payload si es POST/PUT/PATCH
    if request.method in ["POST", "PUT", "PATCH"]:
        content_length = request.headers.get("Content-Length")
        if content_length and int(content_length) > RateLimitConfig.MAX_REQUEST_SIZE_BYTES:
            logger.warning(
                f"[RATE_LIMITER] Request too large: IP={client_ip}, "
                f"size={content_length}, limit={RateLimitConfig.MAX_REQUEST_SIZE_BYTES}"
            )

            return JSONResponse(
                status_code=413,
                content={
                    "error_type": "payload_too_large",
                    "message": f"Request size exceeds {RateLimitConfig.MAX_REQUEST_SIZE_BYTES} bytes",
                    "max_size_bytes": RateLimitConfig.MAX_REQUEST_SIZE_BYTES,
                },
            )

    # Registrar request exitoso
    rate_limit_store.add_request(client_ip)

    # Continuar con el request
    try:
        response = await call_next(request)

        # Agregar headers informativos
        minute_count, hour_count = rate_limit_store.get_request_counts(client_ip)

        if is_laboratory_endpoint:
            minute_limit = RateLimitConfig.LABORATORY_REQUESTS_PER_MINUTE
            hour_limit = RateLimitConfig.LABORATORY_REQUESTS_PER_HOUR
        else:
            minute_limit = RateLimitConfig.GLOBAL_REQUESTS_PER_MINUTE
            hour_limit = RateLimitConfig.GLOBAL_REQUESTS_PER_HOUR

        response.headers["X-RateLimit-Limit-Minute"] = str(minute_limit)
        response.headers["X-RateLimit-Remaining-Minute"] = str(max(0, minute_limit - minute_count))
        response.headers["X-RateLimit-Limit-Hour"] = str(hour_limit)
        response.headers["X-RateLimit-Remaining-Hour"] = str(max(0, hour_limit - hour_count))

        return response

    except Exception as e:
        logger.error(f"[RATE_LIMITER] Error processing request: {e}")
        raise


# Decorador para aplicar rate limiting específico a endpoints individuales
def rate_limit(requests_per_minute: Optional[int] = None, requests_per_hour: Optional[int] = None):
    """
    Decorador para aplicar rate limiting específico a un endpoint.

    Args:
        requests_per_minute: Límite personalizado por minuto
        requests_per_hour: Límite personalizado por hora
    """

    def decorator(func):
        async def wrapper(request: Request, *args, **kwargs):
            client_ip = get_client_ip(request)

            # Usar límites personalizados si están especificados
            minute_limit = requests_per_minute or RateLimitConfig.GLOBAL_REQUESTS_PER_MINUTE
            hour_limit = requests_per_hour or RateLimitConfig.GLOBAL_REQUESTS_PER_HOUR

            minute_count, hour_count = rate_limit_store.get_request_counts(client_ip)

            if minute_count >= minute_limit:
                raise HTTPException(
                    status_code=429,
                    detail={
                        "error_type": "rate_limit_exceeded",
                        "message": f"Demasiadas requests por minuto. Límite: {minute_limit}/min",
                        "retry_after": 60,
                        "limit_type": "per_minute",
                    },
                )

            if hour_count >= hour_limit:
                raise HTTPException(
                    status_code=429,
                    detail={
                        "error_type": "rate_limit_exceeded",
                        "message": f"Demasiadas requests por hora. Límite: {hour_limit}/hour",
                        "retry_after": 3600,
                        "limit_type": "per_hour",
                    },
                )

            return await func(request, *args, **kwargs)

        return wrapper

    return decorator
