# backend/app/core/permissions.py
"""
Permission constants for role-based access control.

Defines all available permissions in the system and their assignments to roles.
This centralizes permission management and prevents string duplication.
"""

from typing import Dict, List

# ============================================================================
# PERMISSION CONSTANTS
# ============================================================================

# Admin-specific permissions
MANAGE_CATALOG = "manage_catalog"
VIEW_SYSTEM_STATUS = "view_system_status"
MANAGE_USERS = "manage_users"
DELETE_ANY = "delete_any"
TRIGGER_EXTERNAL_SYNC = "trigger_external_sync"
VIEW_ALL_DATA = "view_all_data"

# User permissions
VIEW = "view"
UPLOAD = "upload"
EDIT_OWN = "edit_own"
VIEW_OWN_REPORTS = "view_own_reports"
TRIGGER_ENRICHMENT = "trigger_enrichment"
VIEW_DATA_FRESHNESS = "view_data_freshness"

# ============================================================================
# ROLE PERMISSION MAPPINGS
# ============================================================================

# Simplified permission map (only admin and user roles - Issue #85)
ROLE_PERMISSIONS: Dict[str, List[str]] = {
    "admin": [
        # Admin-specific permissions
        VIEW_SYSTEM_STATUS,
        TRIGGER_EXTERNAL_SYNC,
        MANAGE_USERS,
        MANAGE_CATALOG,
        VIEW_ALL_DATA,
        DELETE_ANY,
        # User permissions (admins inherit all user permissions)
        VIEW,
        UPLOAD,
        EDIT_OWN,
        VIEW_OWN_REPORTS,
        TRIGGER_ENRICHMENT,
        VIEW_DATA_FRESHNESS,
    ],
    "user": [
        VIEW,
        UPLOAD,
        EDIT_OWN,
        VIEW_OWN_REPORTS,
        TRIGGER_ENRICHMENT,
        VIEW_DATA_FRESHNESS,
    ],
}

# ============================================================================
# PERMISSION HELPERS
# ============================================================================


def get_permissions_for_role(role: str) -> List[str]:
    """
    Get all permissions for a given role.

    Args:
        role: Role name ('admin' or 'user')

    Returns:
        List of permission strings for the role
    """
    return ROLE_PERMISSIONS.get(role, [])


def has_permission(role: str, permission: str, is_superuser: bool = False) -> bool:
    """
    Check if a role has a specific permission.

    Args:
        role: Role name ('admin' or 'user')
        permission: Permission to check
        is_superuser: Whether the user is a superuser (has all permissions)

    Returns:
        True if role has permission, False otherwise
    """
    # Superusers have all permissions
    if is_superuser:
        return True

    role_permissions = get_permissions_for_role(role)
    return permission in role_permissions


# ============================================================================
# PERMISSION DESCRIPTIONS
# ============================================================================

PERMISSION_DESCRIPTIONS: Dict[str, str] = {
    # Admin permissions
    MANAGE_CATALOG: "Gestionar catálogo de productos (limpieza, reindexación, duplicados)",
    VIEW_SYSTEM_STATUS: "Ver estado del sistema y métricas de salud",
    MANAGE_USERS: "Gestionar usuarios y asignaciones de farmacias",
    DELETE_ANY: "Eliminar cualquier registro del sistema",
    TRIGGER_EXTERNAL_SYNC: "Sincronizar con CIMA/nomenclator manualmente",
    VIEW_ALL_DATA: "Ver datos de todas las farmacias (no solo la propia)",
    # User permissions
    VIEW: "Ver datos de su propia farmacia",
    UPLOAD: "Subir archivos de ventas ERP",
    EDIT_OWN: "Editar sus propios datos y configuración",
    VIEW_OWN_REPORTS: "Ver informes y análisis de su farmacia",
    TRIGGER_ENRICHMENT: "Activar enriquecimiento de datos de ventas",
    VIEW_DATA_FRESHNESS: "Ver frescura de datos del catálogo",
}


def get_permission_description(permission: str) -> str:
    """
    Get human-readable description of a permission.

    Args:
        permission: Permission constant

    Returns:
        Spanish description of the permission
    """
    return PERMISSION_DESCRIPTIONS.get(permission, f"Permiso: {permission}")
