# backend/app/core/logging_config.py
"""
Configuración avanzada de logging para xFarma
Sistema diferenciado por entorno con niveles granulares
"""

import logging
import os
import sys
from datetime import datetime
from typing import Dict, Optional

import pytz
import structlog


class LoggingLevels:
    """Niveles de logging personalizados para xFarma"""

    # Niveles estándar
    DEBUG = logging.DEBUG  # 10
    INFO = logging.INFO  # 20
    WARNING = logging.WARNING  # 30
    ERROR = logging.ERROR  # 40
    CRITICAL = logging.CRITICAL  # 50

    # Niveles personalizados
    HEALTH_CHECK = 15  # Entre DEBUG e INFO para health checks
    DASH_INTERNAL = 12  # Para callbacks internos de Dash
    SYSTEM_STATS = 25  # Entre INFO y WARNING para estadísticas


class EnvironmentLoggingConfig:
    """Configuración de logging específica por entorno"""

    @staticmethod
    def get_config(environment: str) -> Dict:
        """Obtener configuración de logging según entorno"""

        base_config = {"timezone": "Europe/Madrid", "json_logs": False, "include_extra_fields": True}

        if environment == "production":
            return {
                **base_config,
                "level": logging.WARNING,  # Solo WARNING y superiores
                "json_logs": True,  # JSON para parsing
                "filter_health_checks": True,
                "filter_dash_internals": True,
                "include_request_id": True,
                "stats_interval_minutes": 5,
                "max_log_size_mb": 100,
                "performance_logging": True,
            }

        elif environment == "staging":
            return {
                **base_config,
                "level": logging.INFO,
                "json_logs": True,
                "filter_health_checks": True,
                "filter_dash_internals": False,  # Mantener para debugging
                "include_request_id": True,
                "stats_interval_minutes": 15,
                "performance_logging": True,
            }

        else:  # development
            return {
                **base_config,
                "level": logging.DEBUG,
                "json_logs": False,  # Human readable para desarrollo
                "filter_health_checks": False,
                "filter_dash_internals": False,
                "include_request_id": False,
                "stats_interval_minutes": 30,
                "performance_logging": False,
                "include_stack_traces": True,
            }


def add_custom_log_levels():
    """Agregar niveles de logging personalizados"""

    # HEALTH_CHECK level
    logging.addLevelName(LoggingLevels.HEALTH_CHECK, "HEALTH_CHECK")

    def health_check(self, message, *args, **kwargs):
        if self.isEnabledFor(LoggingLevels.HEALTH_CHECK):
            self._log(LoggingLevels.HEALTH_CHECK, message, args, **kwargs)

    logging.Logger.health_check = health_check

    # DASH_INTERNAL level
    logging.addLevelName(LoggingLevels.DASH_INTERNAL, "DASH_INTERNAL")

    def dash_internal(self, message, *args, **kwargs):
        if self.isEnabledFor(LoggingLevels.DASH_INTERNAL):
            self._log(LoggingLevels.DASH_INTERNAL, message, args, **kwargs)

    logging.Logger.dash_internal = dash_internal

    # SYSTEM_STATS level
    logging.addLevelName(LoggingLevels.SYSTEM_STATS, "SYSTEM_STATS")

    def system_stats(self, message, *args, **kwargs):
        if self.isEnabledFor(LoggingLevels.SYSTEM_STATS):
            self._log(LoggingLevels.SYSTEM_STATS, message, args, **kwargs)

    logging.Logger.system_stats = system_stats


def create_madrid_timestamper():
    """Crear timestamper personalizado para zona horaria Madrid"""
    madrid_tz = pytz.timezone("Europe/Madrid")

    def madrid_timestamper(_, __, event_dict):
        madrid_time = datetime.now(madrid_tz)
        event_dict["timestamp"] = madrid_time.isoformat()
        event_dict["timezone"] = "Europe/Madrid"
        return event_dict

    return madrid_timestamper


class EnvironmentFilter(logging.Filter):
    """Filtro de logging basado en configuración de entorno"""

    def __init__(self, config: Dict):
        super().__init__()
        self.config = config
        self.environment = os.getenv("ENVIRONMENT", "development")

    def filter(self, record: logging.LogRecord) -> bool:
        """Filtrar logs basándose en configuración de entorno"""

        # Siempre permitir errores críticos
        if record.levelno >= logging.ERROR:
            return True

        # Filtros específicos por entorno
        if self.config.get("filter_health_checks") and "health" in getattr(record, "pathname", "").lower():
            return record.levelno >= logging.WARNING

        if self.config.get("filter_dash_internals") and record.levelno == LoggingLevels.DASH_INTERNAL:
            return False

        # Aplicar nivel mínimo configurado
        return record.levelno >= self.config.get("level", logging.INFO)


def configure_advanced_logging(environment: Optional[str] = None) -> structlog.BoundLogger:
    """
    Configurar sistema de logging avanzado para xFarma

    Args:
        environment: Entorno específico (development, staging, production)

    Returns:
        structlog.BoundLogger: Logger configurado
    """
    environment = environment or os.getenv("ENVIRONMENT", "development")
    config = EnvironmentLoggingConfig.get_config(environment)

    # Agregar niveles personalizados
    add_custom_log_levels()

    # Procesadores base
    processors = [
        structlog.stdlib.filter_by_level,
        structlog.stdlib.add_logger_name,
        structlog.stdlib.add_log_level,
        structlog.stdlib.PositionalArgumentsFormatter(),
        create_madrid_timestamper(),
        structlog.processors.StackInfoRenderer(),
        structlog.processors.format_exc_info,
    ]

    # Procesadores específicos por entorno
    if config.get("include_extra_fields"):

        def add_environment_info(_, __, event_dict):
            event_dict["environment"] = environment
            event_dict["service"] = "xfarma-backend"
            return event_dict

        processors.append(add_environment_info)

    if config.get("include_request_id"):

        def add_request_context(_, __, event_dict):
            # Agregar contexto de request si está disponible
            # Esto se puede mejorar con contextvars en el futuro
            return event_dict

        processors.append(add_request_context)

    # Renderer final
    if config["json_logs"]:
        processors.append(structlog.processors.JSONRenderer())
    else:
        processors.append(structlog.dev.ConsoleRenderer())

    # Configurar structlog
    structlog.configure(
        processors=processors,
        context_class=dict,
        logger_factory=structlog.stdlib.LoggerFactory(),
        wrapper_class=structlog.stdlib.BoundLogger,
        cache_logger_on_first_use=True,
    )

    # Configurar logging básico con filtros
    env_filter = EnvironmentFilter(config)

    # Handler para stdout con filtros
    handler = logging.StreamHandler(sys.stdout)
    handler.addFilter(env_filter)
    handler.setLevel(config["level"])

    # Configurar root logger
    logging.basicConfig(
        format="%(message)s",
        handlers=[handler],
        level=config["level"],
    )

    # Logger específico para terceros con niveles más altos
    for logger_name in ["uvicorn", "httpx", "fastapi"]:
        third_party_logger = logging.getLogger(logger_name)
        third_party_logger.setLevel(max(logging.WARNING, config["level"]))

    # Log inicial de configuración
    logger = structlog.get_logger("logging_config")
    logger.info(
        "advanced_logging.configured",
        environment=environment,
        level=logging.getLevelName(config["level"]),
        json_logs=config["json_logs"],
        filters_enabled=config.get("filter_health_checks", False),
    )

    return logger


def get_logging_stats() -> Dict:
    """Obtener estadísticas del sistema de logging actual"""
    environment = os.getenv("ENVIRONMENT", "development")
    config = EnvironmentLoggingConfig.get_config(environment)

    return {
        "environment": environment,
        "current_level": logging.getLevelName(config["level"]),
        "json_output": config["json_logs"],
        "health_check_filtered": config.get("filter_health_checks", False),
        "dash_internals_filtered": config.get("filter_dash_internals", False),
        "custom_levels_available": ["HEALTH_CHECK", "DASH_INTERNAL", "SYSTEM_STATS"],
        "madrid_timezone": True,
        "performance_monitoring": config.get("performance_logging", False),
    }


# Logger preconfigurado para uso inmediato
def get_logger(name: str) -> structlog.BoundLogger:
    """
    Obtener logger preconfigurado

    Args:
        name: Nombre del logger (usualmente __name__)

    Returns:
        structlog.BoundLogger: Logger listo para uso
    """
    return structlog.get_logger(name)
