# backend/app/core/env_integrity.py
"""
Verificación de integridad de variables de entorno críticas.

Implementado para Issue #[INVESTIGACIÓN]: Pérdida de datos usuario dgruiz@gmail.com
Fecha: 2025-10-09

Este módulo verifica que todas las variables de entorno críticas estén
configuradas correctamente al iniciar la aplicación, detectando configuraciones
faltantes o incorrectas que puedan causar pérdida de datos o fallos de seguridad.
"""
import math
import os
import sys
from pathlib import Path
from typing import Dict, List, Optional, Tuple

import structlog

logger = structlog.get_logger(__name__)


class EnvIntegrityError(Exception):
    """Error cuando la integridad de .env es crítica y no se puede continuar"""

    pass


class EnvIntegrityWarning:
    """Clase para representar warnings de integridad .env"""

    def __init__(self, variable: str, message: str):
        self.variable = variable
        self.message = message


def validate_secret_key_entropy(key: str, min_entropy: float = 3.5) -> Tuple[bool, float]:
    """
    Validar que SECRET_KEY tiene suficiente entropía (no predecible).

    Args:
        key: SECRET_KEY a validar
        min_entropy: Entropía mínima requerida (Shannon entropy en bits)

    Returns:
        Tuple (is_valid, entropy_value)
    """
    if len(key) < 32:
        return False, 0.0

    # Calcular entropía de Shannon
    byte_counts = {}
    for char in key:
        byte_counts[char] = byte_counts.get(char, 0) + 1

    entropy = 0.0
    key_length = len(key)

    for count in byte_counts.values():
        probability = count / key_length
        if probability > 0:
            entropy -= probability * math.log2(probability)

    # Entropía normalizada (0-8 bits por carácter)
    # >3.5 indica buena distribución de caracteres
    return entropy >= min_entropy, entropy


# Patrones débiles comunes en SECRET_KEY
WEAK_SECRET_PATTERNS = [
    "dev",
    "development",
    "test",
    "testing",
    "example",
    "changeme",
    "change_me",
    "secret",
    "password",
    "default",
    "sample",
    "demo",
    "local",
    "localhost",
]


# Variables de entorno CRÍTICAS (sistema no puede arrancar sin ellas)
CRITICAL_ENV_VARS = {
    "DATABASE_URL": {
        "description": "URL de conexión a PostgreSQL",
        "example": "postgresql://xfarma_user:password@postgres:5432/xfarma_db",
        "required_in": ["development", "production"],
    },
    "SECRET_KEY": {
        "description": "Secret key para JWT tokens",
        "example": "dev_secret_key_cambiar_en_produccion_2024",
        "required_in": ["development", "production"],
        "min_length": 32,
    },
    "ENVIRONMENT": {
        "description": "Entorno de ejecución",
        "example": "development",
        "valid_values": ["development", "production"],
        "required_in": ["development", "production"],
    },
}

# Variables de entorno IMPORTANTES (warnings si faltan, pero no críticas)
IMPORTANT_ENV_VARS = {
    "DEV_ADMIN_EMAIL": {
        "description": "Email del usuario admin para desarrollo",
        "example": "admin@example.com",
        "required_in": ["development"],
        "warning_if_missing": True,
    },
    "REDIS_URL": {
        "description": "URL de conexión a Redis",
        "example": "redis://redis:6379/0",
        "required_in": ["development", "production"],
        "warning_if_missing": True,
    },
    "CIMA_API_KEY": {
        "description": "API key para CIMA",
        "example": "cima_api_key_here",
        "required_in": ["production"],
        "warning_if_missing": False,  # Solo warning en producción
    },
}


def check_env_integrity(
    strict: bool = False, environment: Optional[str] = None
) -> Tuple[bool, List[str], List[EnvIntegrityWarning]]:
    """
    Verificar integridad de variables de entorno.

    Args:
        strict: Si True, warnings también causan fallo
        environment: Environment específico a validar (None = detectar de ENVIRONMENT var)

    Returns:
        Tuple (is_valid, errors, warnings)
        - is_valid: True si no hay errores críticos
        - errors: Lista de errores críticos encontrados
        - warnings: Lista de warnings no críticos

    Raises:
        EnvIntegrityError: Si hay errores críticos y strict=True
    """
    errors: List[str] = []
    warnings: List[EnvIntegrityWarning] = []

    # Detectar entorno actual
    current_env = environment or os.getenv("ENVIRONMENT", "development")

    logger.info("env_integrity.check.started", environment=current_env, strict=strict)

    # 1. Verificar variables CRÍTICAS
    for var_name, config in CRITICAL_ENV_VARS.items():
        if current_env not in config["required_in"]:
            continue  # No requerida en este entorno

        var_value = os.getenv(var_name)

        # Variable faltante
        if not var_value:
            error_msg = (
                f"❌ Variable crítica faltante: {var_name}\n"
                f"   Descripción: {config['description']}\n"
                f"   Ejemplo: {config['example']}\n"
                f"   Requerida en: {', '.join(config['required_in'])}"
            )
            errors.append(error_msg)
            logger.critical(
                "env_integrity.critical_missing",
                variable=var_name,
                environment=current_env,
                description=config["description"],
            )
            continue

        # Validar valores permitidos
        if "valid_values" in config:
            if var_value not in config["valid_values"]:
                error_msg = (
                    f"❌ Variable con valor inválido: {var_name}={var_value}\n"
                    f"   Valores válidos: {', '.join(config['valid_values'])}"
                )
                errors.append(error_msg)
                logger.critical(
                    "env_integrity.invalid_value",
                    variable=var_name,
                    value=var_value,
                    valid_values=config["valid_values"],
                )
                continue

        # Validar longitud mínima (para SECRET_KEY)
        if "min_length" in config:
            if len(var_value) < config["min_length"]:
                error_msg = (
                    f"❌ Variable muy corta: {var_name} (longitud: {len(var_value)})\n"
                    f"   Longitud mínima: {config['min_length']}\n"
                    f"   CRÍTICO para seguridad de tokens JWT"
                )
                errors.append(error_msg)
                logger.critical(
                    "env_integrity.insufficient_length",
                    variable=var_name,
                    actual_length=len(var_value),
                    min_length=config["min_length"],
                )

    # 2. Verificar variables IMPORTANTES
    for var_name, config in IMPORTANT_ENV_VARS.items():
        if current_env not in config["required_in"]:
            continue  # No requerida en este entorno

        var_value = os.getenv(var_name)

        if not var_value and config.get("warning_if_missing", False):
            warning = EnvIntegrityWarning(
                variable=var_name,
                message=(
                    f"⚠️  Variable importante faltante: {var_name}\n"
                    f"   Descripción: {config['description']}\n"
                    f"   Ejemplo: {config['example']}\n"
                    f"   Recomendado en: {', '.join(config['required_in'])}"
                ),
            )
            warnings.append(warning)
            logger.warning(
                "env_integrity.important_missing",
                variable=var_name,
                environment=current_env,
                description=config["description"],
            )

    # 3. Validaciones especiales según entorno
    secret_key = os.getenv("SECRET_KEY", "")

    # Validación extendida de SECRET_KEY (todos los entornos)
    if secret_key:
        # 3.1. Detectar patrones débiles comunes
        weak_patterns_found = [pattern for pattern in WEAK_SECRET_PATTERNS if pattern in secret_key.lower()]

        if weak_patterns_found:
            error_msg = (
                f"❌ SECRET_KEY contiene patrones débiles: {', '.join(weak_patterns_found)}\n"
                f"   CRÍTICO: Usar secret key generado criptográficamente seguro\n"
                f"   Generar con: python -c 'import secrets; print(secrets.token_urlsafe(32))'"
            )
            errors.append(error_msg)
            logger.critical(
                "env_integrity.weak_secret_pattern", patterns_found=weak_patterns_found, environment=current_env
            )

        # 3.2. Validar entropía (distribución de caracteres)
        is_valid_entropy, entropy_value = validate_secret_key_entropy(secret_key)

        if not is_valid_entropy:
            error_msg = (
                f"❌ SECRET_KEY tiene entropía insuficiente: {entropy_value:.2f} bits/carácter\n"
                f"   Mínimo requerido: 3.5 bits/carácter\n"
                f"   Esto indica que la key es predecible o tiene baja aleatoriedad\n"
                f"   Ejemplo de key DÉBIL: 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa' (todos iguales)\n"
                f"   CRÍTICO: Generar key con buena distribución aleatoria\n"
                f"   Generar con: python -c 'import secrets; print(secrets.token_urlsafe(32))'"
            )
            errors.append(error_msg)
            logger.critical(
                "env_integrity.low_entropy_secret", entropy=entropy_value, min_required=3.5, environment=current_env
            )

        # 3.3. Validación adicional para producción
        if current_env == "production":
            # Advertir si entropy es baja (aunque pase el mínimo)
            if is_valid_entropy and entropy_value < 4.0:
                warning = EnvIntegrityWarning(
                    variable="SECRET_KEY",
                    message=(
                        f"⚠️  SECRET_KEY tiene entropía baja para producción: {entropy_value:.2f} bits/carácter\n"
                        f"   Recomendado: >4.0 bits/carácter para mayor seguridad\n"
                        f"   Considerar regenerar con: python -c 'import secrets; print(secrets.token_urlsafe(48))'"
                    ),
                )
                warnings.append(warning)
                logger.warning("env_integrity.suboptimal_entropy_production", entropy=entropy_value, recommended=4.0)

    # 4. FIX Issue #5: Validar conexión PostgreSQL
    database_url = os.getenv("DATABASE_URL")
    if database_url and not any("DATABASE_URL" in err for err in errors):
        # Solo intentar conexión si DATABASE_URL está configurada y no tiene errores previos
        try:
            from sqlalchemy import create_engine, text

            # Crear engine con timeout corto
            engine = create_engine(database_url, pool_pre_ping=True, connect_args={"connect_timeout": 5})

            # Probar conexión con query simple
            with engine.connect() as conn:
                result = conn.execute(text("SELECT 1"))
                result.fetchone()

            logger.info("env_integrity.database_connection_ok")

        except Exception as e:
            error_msg = (
                f"❌ DATABASE_URL configurada pero conexión falla\n"
                f"   URL: {database_url[:30]}... (truncado por seguridad)\n"
                f"   Error: {str(e)[:200]}\n"
                f"   Verificar:\n"
                f"     • PostgreSQL está ejecutándose\n"
                f"     • Credenciales son correctas\n"
                f"     • Host/puerto accesibles\n"
                f"     • Base de datos existe"
            )
            errors.append(error_msg)
            logger.critical("env_integrity.database_connection_failed", error=str(e), url_prefix=database_url[:30])

    # 5. Verificar archivo .env existe
    env_file = Path(".env")
    if not env_file.exists():
        warning = EnvIntegrityWarning(
            variable=".env",
            message=(
                "⚠️  Archivo .env no encontrado\n"
                "   El sistema usará variables de entorno del sistema\n"
                "   Recomendado: copiar .env.example a .env y configurar"
            ),
        )
        warnings.append(warning)
        logger.warning("env_integrity.env_file_missing")

    # Resultado final
    is_valid = len(errors) == 0 and (not strict or len(warnings) == 0)

    if errors:
        logger.critical(
            "env_integrity.check.failed", error_count=len(errors), warning_count=len(warnings), environment=current_env
        )
    elif warnings:
        logger.warning("env_integrity.check.warnings", warning_count=len(warnings), environment=current_env)
    else:
        logger.info("env_integrity.check.passed", environment=current_env)

    return is_valid, errors, warnings


def print_integrity_report(errors: List[str], warnings: List[EnvIntegrityWarning], exit_on_error: bool = True) -> None:
    """
    Imprimir reporte de integridad de .env en consola.

    Args:
        errors: Lista de errores críticos
        warnings: Lista de warnings
        exit_on_error: Si True, sale con exit(1) si hay errores
    """
    print("\n" + "=" * 70)
    print("🔍 VERIFICACIÓN DE INTEGRIDAD .ENV")
    print("=" * 70)

    if errors:
        print("\n❌ ERRORES CRÍTICOS DETECTADOS:")
        print("-" * 70)
        for i, error in enumerate(errors, 1):
            print(f"\n{i}. {error}")

        print("\n" + "=" * 70)
        print("❌ El sistema NO PUEDE ARRANCAR con estos errores")
        print("=" * 70)
        print("\n📖 Solución:")
        print("   1. Copiar .env.example a .env si no existe:")
        print("      cp .env.example .env")
        print("   2. Configurar variables faltantes en .env")
        print("   3. Reintentar startup")
        print()

        if exit_on_error:
            sys.exit(1)

    if warnings:
        print("\n⚠️  WARNINGS DETECTADOS:")
        print("-" * 70)
        for i, warning in enumerate(warnings, 1):
            print(f"\n{i}. {warning.message}")

        print("\n" + "=" * 70)
        print("⚠️  El sistema puede arrancar, pero se recomienda configurar estas variables")
        print("=" * 70)
        print()

    if not errors and not warnings:
        print("\n✅ Todas las variables de entorno críticas están configuradas correctamente")
        print("=" * 70)
        print()


def verify_env_on_startup(strict: bool = False) -> bool:
    """
    Verificar .env al startup de la aplicación.

    Args:
        strict: Si True, warnings también causan fallo

    Returns:
        True si integridad OK, False si hay problemas

    Raises:
        EnvIntegrityError: Si hay errores críticos
    """
    is_valid, errors, warnings = check_env_integrity(strict=strict)

    # Siempre imprimir reporte en consola
    print_integrity_report(errors, warnings, exit_on_error=True)

    if not is_valid:
        raise EnvIntegrityError(f"Integridad de .env FALLÓ: {len(errors)} errores, {len(warnings)} warnings")

    return is_valid


def get_env_vars_summary() -> Dict[str, any]:
    """
    Obtener resumen de variables de entorno configuradas.

    Returns:
        Dict con estadísticas de variables configuradas
    """
    current_env = os.getenv("ENVIRONMENT", "development")

    critical_configured = sum(
        1
        for var_name, config in CRITICAL_ENV_VARS.items()
        if current_env in config["required_in"] and os.getenv(var_name)
    )
    critical_total = sum(1 for config in CRITICAL_ENV_VARS.values() if current_env in config["required_in"])

    important_configured = sum(
        1
        for var_name, config in IMPORTANT_ENV_VARS.items()
        if current_env in config["required_in"] and os.getenv(var_name)
    )
    important_total = sum(1 for config in IMPORTANT_ENV_VARS.values() if current_env in config["required_in"])

    return {
        "environment": current_env,
        "critical_vars": {
            "configured": critical_configured,
            "total": critical_total,
            "percentage": (critical_configured / critical_total * 100) if critical_total > 0 else 0,
        },
        "important_vars": {
            "configured": important_configured,
            "total": important_total,
            "percentage": (important_configured / important_total * 100) if important_total > 0 else 0,
        },
        "env_file_exists": Path(".env").exists(),
    }
