# backend/app/core/cache_config.py
"""
Configuración de Redis optimizada para laboratory mapping caching en Render
FASE 1.3 - Issue #23: Laboratory Mapping Cache Layer

Características:
- Connection pooling optimizado para multi-worker Render
- Configuración TTL diferenciada por tipo de datos
- Memory optimization con compresión automática
- Fallback graceful cuando Redis no disponible
- Cache eviction policy LRU
"""

import json
import logging
import os
import time
from typing import Any, List, Optional, Union

logger = logging.getLogger(__name__)

# Redis es opcional - no disponible en modo local (Windows desktop)
try:
    import redis as redis_module
    from redis.connection import ConnectionPool
    REDIS_MODULE_AVAILABLE = True
    # Type alias for redis client
    RedisClient = redis_module.Redis
except ImportError:
    redis_module = None  # type: ignore
    ConnectionPool = None  # type: ignore
    REDIS_MODULE_AVAILABLE = False
    RedisClient = Any  # type: ignore
    logger.info("[CACHE] Redis module not available - cache disabled (local mode)")


class CacheConfig:
    """Configuración de cache optimizada para Render production"""

    # Redis Enable/Disable Flag
    # Set REDIS_ENABLED=false to skip Redis initialization entirely (faster health checks)
    REDIS_ENABLED = os.getenv("REDIS_ENABLED", "true").lower() in ("true", "1", "yes")

    # TTL Configuration (en segundos)
    TTL_LABORATORY_MAPPINGS = 24 * 60 * 60  # 24 horas - datos estables
    TTL_GENERIC_LABORATORIES = 12 * 60 * 60  # 12 horas - queries dinámicas
    TTL_CACHE_STATS = 1 * 60 * 60  # 1 hora - métricas
    TTL_CATALOG_VERSION = 30 * 24 * 60 * 60  # 30 días - versión de catálogo

    # Cache Key Patterns
    KEY_CODES_TO_NAMES = "lab:codes:{hash}"
    KEY_NAMES_TO_CODES = "lab:names:{hash}"
    KEY_GENERIC_LABS = "lab:generics:{query_hash}"
    KEY_CACHE_STATS = "lab:stats:{type}"
    KEY_CATALOG_VERSION = "lab:catalog_version"
    KEY_WARMING_STATUS = "lab:warming_status"

    # Performance Configuration
    MAX_CACHE_SIZE_MB = 50  # Límite memoria para laboratory cache en Render
    BATCH_SIZE = 100  # Batch size para operaciones masivas
    COMPRESSION_THRESHOLD = 1024  # Comprimir payloads > 1KB

    # Redis Connection Configuration (optimizado Render)
    REDIS_POOL_SIZE = 10  # Pool size para multi-worker
    REDIS_POOL_TIMEOUT = 5  # Timeout obtener conexión
    REDIS_SOCKET_TIMEOUT = 3  # Timeout operaciones Redis
    REDIS_SOCKET_CONNECT_TIMEOUT = 2  # Timeout conexión inicial

    # Fallback Configuration
    ENABLE_GRACEFUL_DEGRADATION = True
    CACHE_FAIL_OPEN = True  # Si Redis falla, usar BD directamente


class RedisManager:
    """Manager de conexiones Redis optimizado para Render multi-worker"""

    def __init__(self):
        self._pool: Optional[ConnectionPool] = None
        self._redis: Optional[RedisClient] = None
        self._is_available = False
        self._last_health_check = 0
        self._health_check_interval = 30  # 30 segundos

    def initialize(self) -> bool:
        """Inicializar conexión Redis con configuración optimizada"""
        # Check if Redis module is available (not in local Windows mode)
        if not REDIS_MODULE_AVAILABLE:
            logger.info("[CACHE] Redis module not available, usando fallback a SQLite/PostgreSQL")
            self._is_available = False
            return False

        # Check if Redis is disabled via environment variable
        if not CacheConfig.REDIS_ENABLED:
            logger.info("[CACHE] Redis deshabilitado via REDIS_ENABLED=false, usando fallback a PostgreSQL")
            self._is_available = False
            return False

        try:
            redis_url = os.getenv("REDIS_URL") or "redis://localhost:6379/0"
            if not redis_url or redis_url.strip() == "":
                logger.info("[CACHE] REDIS_URL vacía, usando fallback a PostgreSQL")
                self._is_available = False
                return False

            # Detectar entorno Render
            is_render = os.getenv("RENDER") == "true" or "onrender.com" in os.getenv("RENDER_EXTERNAL_URL", "")

            # Configuración específica para Render
            pool_config = {
                "max_connections": CacheConfig.REDIS_POOL_SIZE,
                "retry_on_timeout": True,
                "socket_timeout": CacheConfig.REDIS_SOCKET_TIMEOUT,
                "socket_connect_timeout": CacheConfig.REDIS_SOCKET_CONNECT_TIMEOUT,
                "health_check_interval": 30,
            }

            if is_render:
                # Configuración adicional para Render
                pool_config.update(
                    {
                        "socket_keepalive": True,
                        "socket_keepalive_options": {},
                        "retry_on_error": [redis_module.ConnectionError, redis_module.TimeoutError],
                    }
                )

            # Crear pool de conexiones
            self._pool = ConnectionPool.from_url(redis_url, **pool_config)

            # Crear cliente Redis
            self._redis = redis_module.Redis(connection_pool=self._pool, decode_responses=True, health_check_interval=30)

            # Verificar conexión y configurar optimizaciones
            self._configure_redis_optimizations()

            self._is_available = True
            logger.info(f"[CACHE] Redis inicializado correctamente (Render: {is_render})")
            return True

        except Exception as e:
            logger.error(f"[CACHE] Error inicializando Redis: {str(e)[:200]}")
            self._is_available = False

            if not CacheConfig.ENABLE_GRACEFUL_DEGRADATION:
                raise
            return False

    def _configure_redis_optimizations(self) -> None:
        """Configurar optimizaciones Redis para laboratory cache"""
        try:
            # Ping de conectividad
            self._redis.ping()

            # CRITICAL FIX: Detect Render managed Redis to avoid config errors
            is_managed = self._is_managed_redis()

            if not is_managed:
                # Solo configurar en Redis self-hosted
                try:
                    self._redis.config_set("maxmemory-policy", "allkeys-lru")
                    self._redis.config_set("timeout", "300")  # 5 minutos
                    logger.info("[CACHE] Optimizaciones Redis aplicadas (self-hosted)")
                except (redis_module.ResponseError if REDIS_MODULE_AVAILABLE else Exception) as e:
                    logger.warning(f"[CACHE] Config no permitida: {e}")
            else:
                logger.info("[CACHE] Redis managed detectado, saltando configuración")

        except Exception as e:
            logger.warning(f"[CACHE] Error aplicando optimizaciones: {str(e)[:100]}")

    def _is_managed_redis(self) -> bool:
        """Detectar si es Redis managed (Render, AWS, etc.)"""
        try:
            # Try to get Redis info to detect managed service
            info = self._redis.info()
            # Check for managed service indicators
            # Render Redis and other managed services often have specific patterns
            server_info = str(info.get("redis_version", "")).lower()

            # Also check if we're on Render environment
            is_render = os.getenv("RENDER") == "true" or "onrender.com" in os.getenv("RENDER_EXTERNAL_URL", "")

            # If on Render, assume managed Redis
            if is_render:
                return True

            # Try a config command to see if it's restricted
            try:
                self._redis.config_get("maxmemory-policy")
                return False  # If we can read config, likely not managed
            except (redis_module.ResponseError if REDIS_MODULE_AVAILABLE else Exception):
                return True  # Config restricted, likely managed

        except Exception:
            # If we can't determine, assume managed to be safe
            return True

    def is_available(self) -> bool:
        """Verificar disponibilidad de Redis con health check periódico"""
        # Skip if Redis module not available (local mode)
        if not REDIS_MODULE_AVAILABLE:
            return False

        # Skip health checks if Redis is disabled
        if not CacheConfig.REDIS_ENABLED:
            return False

        current_time = time.time()

        if current_time - self._last_health_check > self._health_check_interval:
            self._health_check()
            self._last_health_check = current_time

        return self._is_available

    def _health_check(self) -> None:
        """
        Health check interno de Redis con retry logic y exponential backoff.
        Issue #116: Mejora de robustez con retries automáticos.
        """
        max_retries = 3
        backoff_delays = [1, 2, 4]  # segundos para exponential backoff

        for attempt in range(1, max_retries + 1):
            try:
                if self._redis:
                    self._redis.ping()
                    self._is_available = True
                    if attempt > 1:
                        logger.info(f"[CACHE] Redis reconnected after {attempt} attempts")
                    return
                else:
                    self._is_available = False
                    return
            except Exception as e:
                if attempt < max_retries:
                    delay = backoff_delays[attempt - 1]
                    logger.warning(
                        f"[CACHE] Redis health check failed (attempt {attempt}/{max_retries}), "
                        f"retrying in {delay}s... Error: {str(e)[:100]}"
                    )
                    time.sleep(delay)
                else:
                    self._is_available = False
                    if CacheConfig.ENABLE_GRACEFUL_DEGRADATION:
                        logger.error(f"[CACHE] Redis unavailable after {max_retries} attempts, " "usando fallback")

    def get_client(self) -> Optional[RedisClient]:
        """Obtener cliente Redis si está disponible"""
        if self.is_available():
            return self._redis
        return None

    def close(self) -> None:
        """Cerrar conexiones Redis"""
        try:
            if self._pool:
                self._pool.disconnect()
            logger.info("[CACHE] Conexiones Redis cerradas")
        except Exception as e:
            logger.error(f"[CACHE] Error cerrando Redis: {str(e)[:100]}")


class CacheKeyGenerator:
    """Generador de cache keys eficientes para laboratory mappings"""

    @staticmethod
    def hash_codes(codes: List[str]) -> str:
        """Generar hash para lista de códigos"""
        if not codes:
            return "all"

        # Ordenar para hash consistente
        sorted_codes = sorted(set(codes))
        content = "|".join(sorted_codes)

        # Simple hash para evitar dependencias externas
        hash_value = hash(content) % (10**8)  # Limitar tamaño
        return f"{abs(hash_value):08d}"

    @staticmethod
    def hash_names(names: List[str]) -> str:
        """Generar hash para lista de nombres"""
        if not names:
            return "all"

        # Normalizar nombres y ordenar
        normalized_names = [name.upper().strip() for name in names]
        sorted_names = sorted(set(normalized_names))
        content = "|".join(sorted_names)

        hash_value = hash(content) % (10**8)
        return f"{abs(hash_value):08d}"

    @staticmethod
    def hash_generic_query(page: int, per_page: int, search: Optional[str] = None) -> str:
        """Generar hash para query de laboratorios genéricos"""
        components = [f"p{page}", f"pp{per_page}", f"s{search.upper().strip() if search else 'none'}"]
        content = "|".join(components)
        hash_value = hash(content) % (10**6)  # Hash más corto para queries
        return f"{abs(hash_value):06d}"


class CacheSerializer:
    """Serialización optimizada para payloads de laboratory cache"""

    @staticmethod
    def serialize(data: Any) -> str:
        """Serializar datos para cache con compresión opcional"""
        try:
            json_str = json.dumps(data, ensure_ascii=False, separators=(",", ":"))

            # Comprimir si supera threshold
            if len(json_str) > CacheConfig.COMPRESSION_THRESHOLD:
                try:
                    import gzip

                    compressed = gzip.compress(json_str.encode("utf-8"))
                    import base64

                    encoded = base64.b64encode(compressed).decode("ascii")
                    return f"gzip:{encoded}"
                except ImportError:
                    # Fallback sin compresión
                    pass

            return json_str

        except Exception as e:
            logger.error(f"[CACHE] Error serializando datos: {str(e)[:100]}")
            raise

    @staticmethod
    def deserialize(data: Union[str, bytes]) -> Any:
        """Deserializar datos desde cache con descompresión

        Maneja tanto datos string (decode_responses=True) como bytes (decode_responses=False)
        para compatibilidad con diferentes configuraciones de Redis.
        """
        try:
            # Normalizar a string si viene como bytes
            if isinstance(data, bytes):
                data = data.decode("utf-8")

            # Ahora data es definitivamente un string
            if data.startswith("gzip:"):
                try:
                    import base64
                    import gzip

                    encoded = data[5:]  # Remove 'gzip:' prefix
                    compressed = base64.b64decode(encoded.encode("ascii"))
                    json_str = gzip.decompress(compressed).decode("utf-8")
                except ImportError:
                    raise ValueError("Compresión no disponible")
            else:
                json_str = data

            return json.loads(json_str)

        except Exception as e:
            logger.error(f"[CACHE] Error deserializando datos: {str(e)[:100]}")
            raise


# Instancia global del Redis Manager
redis_manager = RedisManager()


def init_redis() -> bool:
    """Inicializar Redis al startup de la aplicación"""
    return redis_manager.initialize()


def get_redis() -> Optional[RedisClient]:
    """Obtener cliente Redis para uso en servicios"""
    return redis_manager.get_client()


def is_cache_available() -> bool:
    """Verificar si el cache está disponible"""
    return redis_manager.is_available()


def close_redis() -> None:
    """Cerrar conexiones Redis al shutdown"""
    redis_manager.close()
