# backend/app/constants/taxonomy.py
"""
Mapeo de categorías granulares a macro-categorías para el Treemap.

Issue #462: TaxonomyLabelerService

Este mapeo convierte las 100+ categorías granulares de taxonomy_tier1.py
en ~15 macro-categorías visualizables en el Treemap de W14.

Reglas:
1. Cada categoría granular DEBE tener un mapeo (validación al startup)
2. Las macro-categorías siguen la lógica de negocio farmacéutico
3. "excluir" marca productos que no deben aparecer en analytics
"""

from typing import Dict, FrozenSet

# Macro-categorías para el Treemap (Tier 1)
TIER1_MACRO_CATEGORIES: FrozenSet[str] = frozenset({
    "higiene_bucal",
    "dermocosmetica",
    "cuidado_capilar",
    "proteccion_solar",
    "salud_digestiva",
    "dolor_muscular",
    "nutricion_vitaminas",
    "infantil_bebe",
    "salud_respiratoria",
    "higiene_corporal",
    "salud_ocular",
    "salud_sexual_intima",
    "material_sanitario",
    "bienestar_mental",
    "control_peso_deporte",
    "otros_parafarmacia",
    "excluir",  # No mostrar en analytics
})

# Mapeo: categoría granular → macro-categoría
TIER1_MAPPING: Dict[str, str] = {
    # === HIGIENE BUCAL (dental, encías, prótesis) ===
    "aftas_llagas": "higiene_bucal",
    "sensibilidad_dental": "higiene_bucal",
    "encias": "higiene_bucal",
    "caries": "higiene_bucal",
    "blanqueamiento": "higiene_bucal",
    "halitosis": "higiene_bucal",
    "interdental": "higiene_bucal",
    "higiene_dental": "higiene_bucal",
    "protesis_dental": "higiene_bucal",
    "cepillo_electrico": "higiene_bucal",
    "cepillo_manual": "higiene_bucal",
    "dolor_dental": "higiene_bucal",
    "enjuague_bucal": "higiene_bucal",
    "pasta_dental": "higiene_bucal",
    "seda_dental": "higiene_bucal",
    "higiene_bucal_infantil": "higiene_bucal",

    # === DERMOCOSMÉTICA (facial, antiedad, acné, piel) ===
    "hidratacion_facial": "dermocosmetica",
    "antiedad": "dermocosmetica",
    "acne": "dermocosmetica",
    "limpieza_facial": "dermocosmetica",
    "contorno_ojos": "dermocosmetica",
    "exfoliacion": "dermocosmetica",
    "piel_sensible": "dermocosmetica",
    "piel_atopica": "dermocosmetica",
    "dermatitis": "dermocosmetica",
    "psoriasis": "dermocosmetica",
    "manchas": "dermocosmetica",
    "arrugas_antiedad": "dermocosmetica",
    "piel_seca": "dermocosmetica",
    "rojeces_rosacea": "dermocosmetica",
    "cicatrizacion": "dermocosmetica",
    "cicatrices_estrias": "dermocosmetica",
    "quemaduras": "dermocosmetica",
    "picaduras": "dermocosmetica",
    "dermatitis_seborreica": "dermocosmetica",
    "verrugas_callos": "dermocosmetica",
    "herpes": "dermocosmetica",
    "herpes_labial": "dermocosmetica",
    "cuidado_labios": "dermocosmetica",
    "tatuajes": "dermocosmetica",
    "hongos_piel": "dermocosmetica",

    # === CUIDADO CAPILAR ===
    "caida_cabello": "cuidado_capilar",
    "champus": "cuidado_capilar",
    "acondicionadores": "cuidado_capilar",
    "tratamientos_capilares": "cuidado_capilar",
    "piojos": "cuidado_capilar",
    "caspa": "cuidado_capilar",
    "cabello_graso": "cuidado_capilar",
    "capilar_frecuente": "cuidado_capilar",
    "capilar_hidratacion": "cuidado_capilar",
    "capilar_densidad": "cuidado_capilar",
    "capilar_color": "cuidado_capilar",
    "capilar_cuero_sensible": "cuidado_capilar",
    "cuidado_cabello": "cuidado_capilar",
    "higiene_cabello": "cuidado_capilar",
    "canas": "cuidado_capilar",

    # === PROTECCIÓN SOLAR ===
    "proteccion_solar": "proteccion_solar",
    "aftersun": "proteccion_solar",
    "autobronceadores": "proteccion_solar",

    # === SALUD DIGESTIVA ===
    "probioticos": "salud_digestiva",
    "laxantes": "salud_digestiva",
    "antiacidos": "salud_digestiva",
    "gases_digestion": "salud_digestiva",
    "gases_flatulencia": "salud_digestiva",
    "digestivo": "salud_digestiva",
    "digestion_pesada": "salud_digestiva",
    "estrenimiento": "salud_digestiva",
    "flora_intestinal": "salud_digestiva",
    "acidez_reflujo": "salud_digestiva",
    "hemorroides": "salud_digestiva",

    # === DOLOR Y MUSCULAR ===
    "dolor_muscular": "dolor_muscular",
    "dolor_articular": "dolor_muscular",
    "contracturas": "dolor_muscular",
    "antiinflamatorios": "dolor_muscular",
    "dolor": "dolor_muscular",
    "dolor_fiebre": "dolor_muscular",
    "hematomas": "dolor_muscular",
    "colageno": "dolor_muscular",
    "termoterapia": "dolor_muscular",

    # === NUTRICIÓN Y VITAMINAS ===
    "vitaminas": "nutricion_vitaminas",
    "vitaminas_general": "nutricion_vitaminas",
    "vitaminas_minerales": "nutricion_vitaminas",
    "minerales": "nutricion_vitaminas",
    "energia_vitalidad": "nutricion_vitaminas",
    "dietetica": "nutricion_vitaminas",
    "vitamina_d": "nutricion_vitaminas",
    "magnesio": "nutricion_vitaminas",
    "hierro": "nutricion_vitaminas",
    "calcio_huesos": "nutricion_vitaminas",
    "nutricion_clinica": "nutricion_vitaminas",
    "concentracion_memoria": "nutricion_vitaminas",
    "melatonina": "nutricion_vitaminas",
    "homeopatia": "nutricion_vitaminas",
    "omega3": "nutricion_vitaminas",

    # === INFANTIL Y BEBÉ ===
    "alimentacion_infantil": "infantil_bebe",
    "alimentacion_bebe": "infantil_bebe",
    "cuidado_bebe": "infantil_bebe",
    "higiene_infantil": "infantil_bebe",
    "higiene_bebe": "infantil_bebe",
    "bebes_infantil": "infantil_bebe",
    "dermatitis_panal": "infantil_bebe",
    "colicos_bebe": "infantil_bebe",
    "confort_bebe": "infantil_bebe",

    # === SALUD RESPIRATORIA ===
    "gripe_resfriado": "salud_respiratoria",
    "congestion_nasal": "salud_respiratoria",
    "higiene_nasal": "salud_respiratoria",
    "dolor_garganta": "salud_respiratoria",
    "mucosidad_respiratoria": "salud_respiratoria",
    "tos_seca": "salud_respiratoria",
    "alergia": "salud_respiratoria",
    "balsamos_respiratorios": "salud_respiratoria",

    # === HIGIENE CORPORAL ===
    "hidratacion_corporal": "higiene_corporal",
    "higiene_corporal": "higiene_corporal",
    "desodorante": "higiene_corporal",
    "higiene_oidos": "higiene_corporal",
    "cuidado_dependientes": "higiene_corporal",
    "unas": "higiene_corporal",
    "accesorios_belleza": "higiene_corporal",
    "perfumeria": "higiene_corporal",

    # === SALUD OCULAR ===
    "ojo_seco": "salud_ocular",
    "lentillas": "salud_ocular",
    "optica": "salud_ocular",

    # === SALUD SEXUAL E ÍNTIMA ===
    "salud_sexual": "salud_sexual_intima",
    "higiene_intima": "salud_sexual_intima",
    "hongos_vaginales": "salud_sexual_intima",
    "candidiasis_vaginal": "salud_sexual_intima",
    "embarazo_lactancia": "salud_sexual_intima",
    "lactancia": "salud_sexual_intima",
    "test_embarazo": "salud_sexual_intima",
    "test_ovulacion": "salud_sexual_intima",
    "fertilidad": "salud_sexual_intima",
    "menopausia": "salud_sexual_intima",
    "ciclo_menstrual": "salud_sexual_intima",
    "incontinencia": "salud_sexual_intima",

    # === MATERIAL SANITARIO Y ORTOPEDIA ===
    "material_sanitario": "material_sanitario",
    "apositos_curas": "material_sanitario",
    "heridas_apositos": "material_sanitario",
    "desinfeccion_heridas": "material_sanitario",
    "compresion_vendajes": "material_sanitario",
    "material_ortopedico": "material_sanitario",
    "ortopedia": "material_sanitario",
    "ortopedia_pie": "material_sanitario",
    "varices": "material_sanitario",
    "termometros": "material_sanitario",
    "tension_arterial": "material_sanitario",
    "diabetes": "material_sanitario",
    "test_covid_gripe": "material_sanitario",
    "test_diagnostico": "material_sanitario",
    "ostomia": "material_sanitario",

    # === BIENESTAR MENTAL ===
    "estres_ansiedad": "bienestar_mental",
    "sueno_insomnio": "bienestar_mental",
    "mareo": "bienestar_mental",

    # === CONTROL PESO Y DEPORTE ===
    "control_peso": "control_peso_deporte",
    "nutricion_deportiva": "control_peso_deporte",

    # === OTROS PARAFARMACIA ===
    "veterinaria": "otros_parafarmacia",
    "otros": "otros_parafarmacia",
    "dulceria": "otros_parafarmacia",

    # === EXCLUIR DE ANALYTICS ===
    "interno_no_venta": "excluir",
    "regalos_bazar": "excluir",
    "servicios_farmacia": "excluir",
}


def get_tier1_for_category(category: str) -> str:
    """
    Obtiene la macro-categoría (Tier 1) para una categoría granular.

    Args:
        category: Categoría granular (ej: "sensibilidad_dental")

    Returns:
        Macro-categoría (ej: "higiene_bucal") o "otros_parafarmacia" si no existe
    """
    return TIER1_MAPPING.get(category, "otros_parafarmacia")


def validate_tier1_coverage(all_categories: frozenset) -> set:
    """
    Valida que todas las categorías tengan mapeo a Tier 1.

    Args:
        all_categories: Conjunto de todas las categorías de taxonomy_tier1.py

    Returns:
        Conjunto de categorías sin mapear (vacío si todo OK)

    Raises:
        RuntimeError: Si hay categorías sin mapear (en modo estricto)
    """
    return all_categories - set(TIER1_MAPPING.keys())
