# backend/app/config/laboratory_config.py
"""
Configuración centralizada para laboratory mapping system
Contiene constantes configurables para validación, rate limiting y seguridad
"""

import os
from typing import List


class LaboratoryConfig:
    """Configuración principal para laboratory mapping"""

    # Validación de requests
    MAX_CODES_PER_REQUEST: int = int(os.getenv("LAB_MAX_CODES_PER_REQUEST", "20"))
    MAX_NAMES_PER_REQUEST: int = int(os.getenv("LAB_MAX_NAMES_PER_REQUEST", "20"))
    MAX_NAME_LENGTH: int = int(os.getenv("LAB_MAX_NAME_LENGTH", "200"))
    MAX_SEARCH_LENGTH: int = int(os.getenv("LAB_MAX_SEARCH_LENGTH", "100"))

    # Paginación
    MAX_PAGE: int = int(os.getenv("LAB_MAX_PAGE", "1000"))
    MAX_PER_PAGE: int = int(os.getenv("LAB_MAX_PER_PAGE", "100"))
    DEFAULT_PER_PAGE: int = int(os.getenv("LAB_DEFAULT_PER_PAGE", "50"))

    # Rate limiting específico para laboratory
    REQUESTS_PER_MINUTE: int = int(os.getenv("LAB_REQUESTS_PER_MINUTE", "60"))
    REQUESTS_PER_HOUR: int = int(os.getenv("LAB_REQUESTS_PER_HOUR", "500"))

    # Limits de payload
    MAX_REQUEST_SIZE_KB: int = int(os.getenv("LAB_MAX_REQUEST_SIZE_KB", "10"))
    REQUEST_TIMEOUT_SECONDS: int = int(os.getenv("LAB_REQUEST_TIMEOUT", "5"))

    # Patrones de validación
    LABORATORY_CODE_PATTERN: str = os.getenv("LAB_CODE_PATTERN", r"^\d{1,4}$")

    # Seguridad
    ENABLE_XSS_PROTECTION: bool = os.getenv("LAB_ENABLE_XSS_PROTECTION", "true").lower() == "true"
    ENABLE_SQL_INJECTION_PROTECTION: bool = os.getenv("LAB_ENABLE_SQL_PROTECTION", "true").lower() == "true"
    ENABLE_REQUEST_LOGGING: bool = os.getenv("LAB_ENABLE_REQUEST_LOGGING", "true").lower() == "true"

    # Logging
    LOG_SUSPICIOUS_REQUESTS: bool = os.getenv("LAB_LOG_SUSPICIOUS", "true").lower() == "true"
    LOG_RATE_LIMIT_VIOLATIONS: bool = os.getenv("LAB_LOG_RATE_LIMITS", "true").lower() == "true"

    @classmethod
    def get_dangerous_chars(cls) -> List[str]:
        """Obtener lista de caracteres peligrosos para XSS"""
        default_chars = ["<", ">", "'", '"', "&", "javascript:", "script", "onload=", "onerror="]
        custom_chars = os.getenv("LAB_DANGEROUS_CHARS", "").split(",")
        return default_chars + [char.strip() for char in custom_chars if char.strip()]

    @classmethod
    def get_generic_laboratory_patterns(cls) -> List[str]:
        """Obtener patrones para identificar laboratorios genéricos"""
        default_patterns = [
            "CINFA",
            "NORMON",
            "TEVA",
            "SANDOZ",
            "KERN",
            "RATIOPHARM",
            "STADA",
            "MYLAN",
            "ACCORD",
            "AUROBINDO",
            "ACTAVIS",
            "APOTEX",
            "BLUEFISH",
            "ZENTIVA",
        ]

        custom_patterns = os.getenv("LAB_GENERIC_PATTERNS", "").split(",")
        all_patterns = default_patterns + [p.strip() for p in custom_patterns if p.strip()]

        return list(set(all_patterns))  # Eliminar duplicados

    @classmethod
    def is_development_mode(cls) -> bool:
        """Verificar si estamos en modo desarrollo"""
        return os.getenv("ENVIRONMENT", "development").lower() == "development"

    @classmethod
    def is_production_mode(cls) -> bool:
        """Verificar si estamos en modo producción"""
        return os.getenv("ENVIRONMENT", "development").lower() == "production"

    @classmethod
    def get_database_query_limit(cls) -> int:
        """Obtener límite de queries de BD según el entorno"""
        if cls.is_production_mode():
            return int(os.getenv("LAB_PROD_QUERY_LIMIT", "200"))
        else:
            return int(os.getenv("LAB_DEV_QUERY_LIMIT", "500"))

    # === FASE 2.1: SQL QUERY CONFIGURATION CONSTANTS ===

    # Query limits y timeouts
    LABORATORY_MAPPING_LIMIT: int = int(os.getenv("LAB_MAPPING_LIMIT", "200"))
    DATABASE_TIMEOUT_SECONDS: int = int(os.getenv("LAB_DB_TIMEOUT_SECONDS", "30"))
    MAX_CODES_IN_QUERY: int = int(os.getenv("LAB_MAX_CODES_IN_QUERY", "50"))

    # Base table y columnas (evitar hardcoding en queries)
    PRODUCT_CATALOG_TABLE: str = "product_catalog"
    NOMEN_CODE_COLUMN: str = "nomen_codigo_laboratorio"
    NOMEN_NAME_COLUMN: str = "nomen_laboratorio"
    NOMEN_STATUS_COLUMN: str = "nomen_estado"

    # Estados válidos para laboratorios
    ACTIVE_LAB_STATUS: str = "ALTA"
    INACTIVE_LAB_STATUS: str = "BAJA"

    # Validación columnas de laboratorio
    MIN_LAB_CODE_LENGTH: int = 1
    MAX_LAB_CODE_LENGTH: int = 4

    @classmethod
    def get_base_laboratory_columns(cls) -> str:
        """Obtener columnas base para queries de laboratorio"""
        return f"DISTINCT {cls.NOMEN_CODE_COLUMN}, {cls.NOMEN_NAME_COLUMN}"

    @classmethod
    def get_laboratory_base_conditions(cls) -> List[str]:
        """Obtener condiciones WHERE base para queries de laboratorio"""
        return [
            f"{cls.NOMEN_CODE_COLUMN} IS NOT NULL",
            f"{cls.NOMEN_NAME_COLUMN} IS NOT NULL",
            f"LENGTH({cls.NOMEN_CODE_COLUMN}) BETWEEN {cls.MIN_LAB_CODE_LENGTH} AND {cls.MAX_LAB_CODE_LENGTH}",
        ]

    @classmethod
    def get_active_laboratory_condition(cls) -> str:
        """Obtener condición WHERE para laboratorios activos"""
        return f"{cls.NOMEN_STATUS_COLUMN} = '{cls.ACTIVE_LAB_STATUS}'"


class LaboratorySecurityConfig:
    """Configuración específica de seguridad"""

    # Headers de seguridad obligatorios
    REQUIRED_SECURITY_HEADERS = {
        "X-Content-Type-Options": "nosniff",
        "X-Frame-Options": "DENY",
        "X-XSS-Protection": "1; mode=block",
        "Referrer-Policy": "strict-origin-when-cross-origin",
    }

    # IPs en whitelist (desarrollo/testing)
    @classmethod
    def get_whitelisted_ips(cls) -> List[str]:
        """Obtener IPs en whitelist que no tienen rate limiting"""
        whitelist_env = os.getenv("LAB_IP_WHITELIST", "")
        if not whitelist_env:
            return []

        ips = [ip.strip() for ip in whitelist_env.split(",") if ip.strip()]

        # En desarrollo, agregar localhost automáticamente
        if LaboratoryConfig.is_development_mode():
            dev_ips = ["127.0.0.1", "localhost", "::1"]
            ips.extend(dev_ips)

        return list(set(ips))

    # IPs bloqueadas
    @classmethod
    def get_blocked_ips(cls) -> List[str]:
        """Obtener IPs bloqueadas que reciben 403 inmediatamente"""
        blocked_env = os.getenv("LAB_IP_BLOCKLIST", "")
        if not blocked_env:
            return []

        return [ip.strip() for ip in blocked_env.split(",") if ip.strip()]

    @classmethod
    def is_suspicious_request(cls, user_agent: str, referer: str) -> bool:
        """Detectar requests sospechosos basado en User-Agent y Referer"""
        if not user_agent:
            return True

        suspicious_agents = [
            "bot",
            "crawler",
            "spider",
            "scraper",
            "curl",
            "wget",
            "python-requests",
            "httpie",
            "postman",
        ]

        user_agent_lower = user_agent.lower()
        for suspicious in suspicious_agents:
            if suspicious in user_agent_lower:
                return True

        return False


class LaboratoryPerformanceConfig:
    """Configuración de performance y optimización"""

    # Cache settings
    ENABLE_RESPONSE_CACHING: bool = os.getenv("LAB_ENABLE_CACHING", "true").lower() == "true"
    CACHE_TTL_SECONDS: int = int(os.getenv("LAB_CACHE_TTL", "300"))  # 5 minutos

    # Database optimization
    MAX_DB_CONNECTIONS: int = int(os.getenv("LAB_MAX_DB_CONNECTIONS", "5"))
    DB_QUERY_TIMEOUT: int = int(os.getenv("LAB_DB_TIMEOUT", "30"))

    # Batch processing
    ENABLE_BATCH_PROCESSING: bool = os.getenv("LAB_ENABLE_BATCH", "true").lower() == "true"
    BATCH_SIZE: int = int(os.getenv("LAB_BATCH_SIZE", "50"))

    @classmethod
    def get_cache_key_prefix(cls) -> str:
        """Obtener prefijo para keys de cache"""
        return f"lab_mapping_{os.getenv('ENVIRONMENT', 'dev')}"

    @classmethod
    def should_use_cache(cls, request_size: int) -> bool:
        """Determinar si usar cache basado en el tamaño del request"""
        # Usar cache solo para requests pequeños (mejor performance)
        return cls.ENABLE_RESPONSE_CACHING and request_size <= 5


# Configuración de monitoreo y métricas
class LaboratoryMonitoringConfig:
    """Configuración de monitoreo y métricas"""

    ENABLE_METRICS: bool = os.getenv("LAB_ENABLE_METRICS", "true").lower() == "true"
    ENABLE_DETAILED_LOGGING: bool = os.getenv("LAB_DETAILED_LOGGING", "false").lower() == "true"

    # Métricas a trackear
    TRACK_RESPONSE_TIMES: bool = True
    TRACK_ERROR_RATES: bool = True
    TRACK_RATE_LIMIT_HITS: bool = True
    TRACK_VALIDATION_FAILURES: bool = True

    @classmethod
    def get_log_level(cls) -> str:
        """Obtener nivel de log según el entorno"""
        if LaboratoryConfig.is_development_mode():
            return os.getenv("LAB_LOG_LEVEL", "DEBUG")
        else:
            return os.getenv("LAB_LOG_LEVEL", "INFO")


# Validación de configuración al importar
def validate_config():
    """Validar que la configuración es consistente"""
    config = LaboratoryConfig()

    # Validar límites lógicos
    assert config.MAX_CODES_PER_REQUEST > 0, "MAX_CODES_PER_REQUEST debe ser > 0"
    assert config.MAX_CODES_PER_REQUEST <= 100, "MAX_CODES_PER_REQUEST muy alto (>100)"
    assert config.REQUESTS_PER_MINUTE <= config.REQUESTS_PER_HOUR, "Rate limit inconsistente"
    assert config.MAX_REQUEST_SIZE_KB > 0, "MAX_REQUEST_SIZE_KB debe ser > 0"

    return True


# Ejecutar validación al importar
if __name__ != "__main__":
    validate_config()
