# backend/app/api/version_example.py
"""
Ejemplo de endpoints versionados para demostrar el sistema de versionado.
Este archivo muestra cómo implementar diferentes versiones de la misma funcionalidad.
"""

from typing import Any, Dict

import structlog
from fastapi import APIRouter, Request

from app.middleware import get_api_version, requires_version

logger = structlog.get_logger(__name__)
router = APIRouter()


@router.get("/info")
async def api_info(request: Request) -> Dict[str, Any]:
    """
    Endpoint que devuelve información diferente según la versión de API.

    v1: Información básica
    v2: Información detallada con estadísticas
    """
    version = get_api_version(request)

    base_info = {
        "api_version": version,
        "service": "xfarma",
        "status": "active",
        "timestamp": "2025-09-17T10:00:00Z",
    }

    if version == "v1":
        return base_info

    elif version == "v2":
        # v2 incluye información adicional
        return {
            **base_info,
            "features": {
                "advanced_analytics": True,
                "batch_processing": True,
                "real_time_sync": True,
                "catalog_versioning": True,
            },
            "statistics": {
                "total_pharmacies": 150,
                "processed_files": 1250,
                "catalog_entries": 25000,
            },
            "performance": {
                "avg_response_time": "120ms",
                "uptime": "99.9%",
                "cache_hit_rate": "85%",
            },
        }

    return base_info


@router.get("/v1/compatibility")
async def v1_compatibility_endpoint(request: Request) -> Dict[str, Any]:
    """
    Endpoint específico de v1 - funciona solo con v1.
    Útil para mantener compatibilidad hacia atrás.
    """
    return {
        "version": "v1",
        "message": "Este endpoint solo funciona en v1",
        "deprecated": False,
        "migration_path": None,
    }


@router.get("/v2-features")
@requires_version("v2")
async def v2_advanced_endpoint(request: Request) -> Dict[str, Any]:
    """
    Endpoint que requiere v2 mínimo.
    Demuestra el uso del decorador @requires_version.
    """
    return {
        "version": "v2",
        "message": "Funcionalidades avanzadas solo disponibles en v2",
        "features": [
            "Procesamiento en lote",
            "Análisis predictivo",
            "Sincronización en tiempo real",
            "APIs de catálogo avanzadas",
        ],
        "beta_features": [
            "Machine Learning para clasificación",
            "Reportes automatizados",
            "Integración con sistemas externos",
        ],
    }


@router.get("/versioning/test")
async def versioning_test(request: Request) -> Dict[str, Any]:
    """
    Endpoint de testing para verificar que el sistema de versiones funciona.
    Muestra todos los métodos de detección de versión.
    """
    version = get_api_version(request)

    # Información de debugging sobre cómo se detectó la versión
    detection_info = {
        "header": request.headers.get("API-Version"),
        "query": request.query_params.get("version"),
        "path_version": None,  # Se detectaría automáticamente si el path tiene /api/v1/
        "content_type": request.headers.get("content-type", ""),
        "detected_version": version,
    }

    # Extraer versión del path si existe
    import re

    path_match = re.match(r"^/api/(v\d+)/", request.url.path)
    if path_match:
        detection_info["path_version"] = path_match.group(1)

    return {
        "current_version": version,
        "detection_methods": detection_info,
        "test_examples": {
            "header": "API-Version: v2",
            "query": "?version=v2",
            "path": "/api/v2/versioning/test",
            "content_type": "Content-Type: application/vnd.xfarma.v2+json",
        },
        "available_endpoints": {
            "v1": ["/api/info", "/api/v1/compatibility"],
            "v2": ["/api/info", "/api/v2/v2-features"],
        },
    }
