"""
API Router para Dashboard Venta Libre (Issue #461)

Endpoints para análisis de ventas de productos OTC por categoría NECESIDAD.
"""

from datetime import date
from typing import List, Optional
from uuid import UUID

import structlog
from fastapi import APIRouter, Depends, HTTPException, Query
from pydantic import BaseModel, Field
from sqlalchemy.orm import Session

from app.api.deps import get_current_admin_user, get_current_user, get_db
from app.models.pharmacy_targets import DEFAULT_TARGETS, PharmacyTarget
from app.models.user import User
from app.schemas.cluster_management import (
    CategoryValidation,
    ClusterMergeRequest,
    ClusterMergeResponse,
    ClusterSplitRequest,
    ClusterSplitResponse,
    ClusterStats,
    MergePreviewResponse,
    SplitPreviewResponse,
)
from app.schemas.symptom_taxonomy import L1WithL2Category
from app.services.cluster_management_service import get_cluster_management_service
from app.services.feedback_service_v2 import FeedbackServiceV2
from app.services.ventalibre_service import ventalibre_service

struct_logger = structlog.get_logger(__name__)


VALID_TARGET_TYPES = {
    "margin_min", "margin_target", "margin_excellent",
    "hhi_max", "sales_target", "price_avg"
}


def _get_margin_targets_for_chart(
    db: Session, pharmacy_id: UUID, target_types: List[str] = None
) -> List[dict]:
    """
    Obtiene targets de margen de una farmacia en formato chart.

    Si no hay targets configurados, devuelve los valores por defecto.

    Args:
        db: Sesión de base de datos
        pharmacy_id: ID de la farmacia
        target_types: Lista de tipos a obtener. Valores válidos:
            margin_min, margin_target, margin_excellent, hhi_max, sales_target, price_avg

    Returns:
        Lista de targets en formato chart: [{value, color, line_style, label}, ...]
    """
    if target_types is None:
        target_types = ["margin_min", "margin_target", "margin_excellent"]

    # Validar tipos de target
    invalid_types = set(target_types) - VALID_TARGET_TYPES
    if invalid_types:
        struct_logger.warning(
            "ventalibre.invalid_target_types_requested",
            invalid=list(invalid_types),
            valid=list(VALID_TARGET_TYPES),
        )

    # Buscar targets configurados de la farmacia
    targets = (
        db.query(PharmacyTarget)
        .filter(
            PharmacyTarget.pharmacy_id == pharmacy_id,
            PharmacyTarget.target_type.in_(target_types),
            PharmacyTarget.category.is_(None),  # Solo targets globales
        )
        .all()
    )

    if targets:
        return [t.to_chart_dict() for t in targets]

    # Si no hay targets configurados, usar defaults
    return [
        {
            "value": float(d["value"]),
            "color": d["color"],
            "line_style": d["line_style"],
            "label": d["label"],
        }
        for d in DEFAULT_TARGETS
        if d["target_type"] in target_types
    ]


# Request/Response schemas for feedback
class VentaLibreCorrectionRequest(BaseModel):
    """Request para corregir clasificación de producto venta libre."""
    product_id: UUID = Field(..., description="ID del producto en ProductCatalogVentaLibre")
    corrected_category: str = Field(..., min_length=1, max_length=100, description="Nueva categoría NECESIDAD")
    reviewer_notes: Optional[str] = Field(None, max_length=500, description="Notas del revisor")


class VentaLibreCorrectionResponse(BaseModel):
    """Response de corrección exitosa."""
    success: bool
    product_id: UUID
    old_category: Optional[str]
    new_category: str
    message: str

router = APIRouter(prefix="/ventalibre", tags=["ventalibre"])


@router.get("/date-range")
async def get_ventalibre_date_range(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener rango de fechas de ventas venta libre disponibles.

    Retorna las fechas mínima y máxima de ventas de productos venta libre
    para la farmacia del usuario autenticado.

    Returns:
        Dict con min_date y max_date en formato ISO:
        {
            "min_date": "2023-01-15",
            "max_date": "2025-12-31"
        }
    """
    from datetime import timedelta
    from sqlalchemy import func
    from app.models.sales_data import SalesData
    from app.models.sales_enrichment import SalesEnrichment

    pharmacy_id = current_user.pharmacy_id

    try:
        # Obtener rango de fechas de productos venta libre
        date_range = (
            db.query(
                func.min(SalesData.sale_date).label("min_date"),
                func.max(SalesData.sale_date).label("max_date"),
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesEnrichment.product_type == "venta_libre",
                SalesData.quantity > 0,
            )
            .first()
        )

        if date_range and date_range.min_date and date_range.max_date:
            min_date = date_range.min_date
            max_date = date_range.max_date
        else:
            # Fallback si no hay datos venta libre
            today = date.today()
            min_date = today - timedelta(days=365)
            max_date = today

        struct_logger.info(
            "ventalibre_date_range",
            pharmacy_id=str(pharmacy_id),
            min_date=min_date.isoformat(),
            max_date=max_date.isoformat(),
        )

        return {
            "min_date": min_date.isoformat(),
            "max_date": max_date.isoformat(),
        }

    except Exception as e:
        struct_logger.error(
            "ventalibre_date_range_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
        )
        # Fallback en caso de error
        today = date.today()
        return {
            "min_date": (today - timedelta(days=365)).isoformat(),
            "max_date": today.isoformat(),
        }


@router.get("/sales-by-necesidad")
async def get_sales_by_necesidad(
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Datos jerárquicos para treemap de ventas por NECESIDAD.

    Agrega ventas de productos venta libre por categoría ml_category.
    Usado para el treemap interactivo del dashboard.

    Args:
        date_from: Fecha inicio del rango de análisis
        date_to: Fecha fin del rango de análisis
        employee_ids: Lista de IDs de empleados para filtrar (opcional)
        db: Sesión de base de datos
        current_user: Usuario autenticado (para obtener pharmacy_id)

    Returns:
        Dict con nodes para treemap:
        {
            "nodes": [
                {"category": "proteccion_solar", "sales": 15000.0, "count": 234, "percentage": 25.3},
                ...
            ],
            "total_sales": 60000.0,
            "total_products": 1247
        }
    """
    return ventalibre_service.get_sales_by_necesidad(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids
    )


@router.get("/products")
async def get_products(
    necesidad: Optional[str] = Query(None, description="Filtrar por categoría NECESIDAD"),
    search: Optional[str] = Query(None, description="Buscar en nombre de producto"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    limit: int = Query(50, le=100, description="Límite de resultados"),
    offset: int = Query(0, description="Offset para paginación"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista paginada de productos venta libre con filtros.

    Devuelve productos con ventas agregadas, categoría NECESIDAD, confianza ML,
    marca detectada y totales de ventas.

    Args:
        necesidad: Filtrar por categoría NECESIDAD específica
        search: Texto de búsqueda en nombre de producto
        employee_ids: Lista de IDs de empleados para filtrar
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango
        limit: Número máximo de resultados (max 100)
        offset: Offset para paginación
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con productos paginados:
        {
            "products": [...],
            "total": 1247,
            "page": 1,
            "pages": 25
        }
    """
    return ventalibre_service.get_products_list(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        necesidad=necesidad,
        search=search,
        employee_ids=employee_ids,
        date_from=date_from,
        date_to=date_to,
        limit=limit,
        offset=offset
    )


@router.get("/kpis")
async def get_kpis(
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    KPIs del dashboard venta libre.

    Proporciona métricas agregadas para la cabecera del dashboard:
    - Total de ventas en euros
    - Número de productos únicos
    - Número de categorías con ventas
    - Porcentaje de cobertura de clasificación
    - Crecimiento interanual (YoY)

    Args:
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango
        employee_ids: Lista de IDs de empleados
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con KPIs:
        {
            "total_sales": 45230.50,
            "total_products": 1247,
            "categories_count": 32,
            "coverage_percent": 94.2,
            "yoy_growth": 12.3
        }
    """
    return ventalibre_service.get_kpis(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids
    )


@router.get("/categories")
async def get_categories(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista de categorías NECESIDAD disponibles con conteos.

    Devuelve todas las categorías NECESIDAD que tienen ventas registradas
    para la farmacia, ordenadas por número de productos.

    Args:
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con categorías:
        {
            "categories": [
                {"value": "proteccion_solar", "label": "Proteccion Solar", "count": 234},
                ...
            ]
        }
    """
    return ventalibre_service.get_categories(
        db=db,
        pharmacy_id=current_user.pharmacy_id
    )


class VentaLibreValidationRequest(BaseModel):
    """Request para validación Tinder-style de producto venta libre."""
    action: str = Field(..., pattern="^(approve|move)$", description="Acción: approve o move")
    target_necesidad: Optional[str] = Field(None, description="Categoría destino (solo para action=move)")
    human_verified: bool = Field(True, description="Marcar como verificado humanamente")


class VentaLibreValidationResponse(BaseModel):
    """Response de validación exitosa."""
    success: bool
    product_id: UUID
    action: str
    old_category: Optional[str]
    new_category: Optional[str]
    message: str


@router.post("/validate/{product_id}", response_model=VentaLibreValidationResponse)
async def validate_product(
    product_id: UUID,
    request: VentaLibreValidationRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Validación Tinder-style de producto venta libre.

    Permite aprobar la categoría actual o mover a otra categoría.
    Usado por el flujo de validación del Tab 2 Gestión.

    Actions:
    - approve: Marca human_verified=True sin cambiar categoría
    - move: Cambia categoría y marca human_verified=True

    Args:
        product_id: ID del producto en ProductCatalogVentaLibre
        request: Datos de la validación (action, target_necesidad, human_verified)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        VentaLibreValidationResponse con resultado de la validación
    """
    from app.models.product_catalog_venta_libre import ProductCatalogVentaLibre

    # Buscar producto
    product = db.query(ProductCatalogVentaLibre).filter(
        ProductCatalogVentaLibre.id == product_id
    ).first()

    if not product:
        raise HTTPException(status_code=404, detail="Producto no encontrado")

    old_category = product.ml_category

    try:
        if request.action == "approve":
            # Solo marcar como verificado
            product.human_verified = request.human_verified
            db.commit()

            return VentaLibreValidationResponse(
                success=True,
                product_id=product_id,
                action="approve",
                old_category=old_category,
                new_category=old_category,
                message=f"Producto aprobado con categoría '{old_category}'"
            )

        elif request.action == "move":
            if not request.target_necesidad:
                raise HTTPException(
                    status_code=400,
                    detail="target_necesidad es requerido para action=move"
                )

            from app.api.feedback import NECESIDAD_CATEGORIES

            # Validar categoría destino
            if request.target_necesidad not in NECESIDAD_CATEGORIES:
                valid_cats = ", ".join(sorted(list(NECESIDAD_CATEGORIES.keys())[:10])) + "..."
                raise HTTPException(
                    status_code=400,
                    detail=f"Categoría '{request.target_necesidad}' no válida. Usar: {valid_cats}"
                )

            # Actualizar categoría
            product.human_corrected_category = request.target_necesidad
            product.human_verified = request.human_verified
            product.correction_type = "MANUAL_VALIDATION"
            product.prediction_source = "HUMAN"
            db.commit()

            return VentaLibreValidationResponse(
                success=True,
                product_id=product_id,
                action="move",
                old_category=old_category,
                new_category=request.target_necesidad,
                message=f"Producto movido de '{old_category}' a '{request.target_necesidad}'"
            )

    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Error de validación: {str(e)}")


@router.post("/correct", response_model=VentaLibreCorrectionResponse)
async def correct_product_category(
    request: VentaLibreCorrectionRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Corrige la categoría NECESIDAD de un producto venta libre.

    Usa FeedbackServiceV2 para actualizar ProductCatalogVentaLibre.
    La corrección afecta a TODAS las ventas de este producto.

    Args:
        request: Datos de la corrección (product_id, corrected_category, reviewer_notes)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        VentaLibreCorrectionResponse con resultado de la corrección
    """
    from app.api.feedback import NECESIDAD_CATEGORIES

    # Validar categoría (solo NecesidadEspecifica)
    if request.corrected_category not in NECESIDAD_CATEGORIES:
        valid_cats = ", ".join(sorted(list(NECESIDAD_CATEGORIES.keys())[:10])) + "..."
        raise HTTPException(
            status_code=400,
            detail=f"Categoría '{request.corrected_category}' no válida. Usar: {valid_cats}"
        )

    try:
        service = FeedbackServiceV2(db)
        product = service.correct_classification(
            product_id=request.product_id,
            corrected_category=request.corrected_category,
            reviewer_notes=request.reviewer_notes,
        )

        return VentaLibreCorrectionResponse(
            success=True,
            product_id=product.id,
            old_category=product.ml_category,  # Categoría ML original
            new_category=product.verified_category or request.corrected_category,
            message=f"Categoría actualizada correctamente a '{request.corrected_category}'"
        )

    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error al corregir: {str(e)}")


# === Cluster Management Endpoints (Issue #464) ===


class ClusterSplitRequestAPI(BaseModel):
    """Request body for cluster split (source_category comes from path)."""
    product_ids: List[UUID] = Field(
        ...,
        min_length=1,
        description="Lista de IDs de productos a mover a la nueva categoria"
    )
    new_category: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Categoria destino (debe ser un valor valido de NecesidadEspecifica)"
    )
    notes: Optional[str] = Field(
        default=None,
        max_length=500,
        description="Notas explicando la razon del split"
    )


@router.post("/clusters/{category}/split", response_model=ClusterSplitResponse)
async def split_cluster(
    category: str,
    request: ClusterSplitRequestAPI,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
):
    """
    Divide un cluster moviendo productos seleccionados a una nueva categoria.

    Esta operacion es atomica: todos los productos se mueven o ninguno.
    Despues de mover, sincroniza automaticamente los registros de SalesEnrichment.

    **Requiere permisos de administrador.**

    Args:
        category: Categoria origen de los productos
        request: Datos del split (product_ids, new_category, notes)
        db: Sesion de base de datos
        current_user: Usuario admin autenticado

    Returns:
        ClusterSplitResponse con detalles de la operacion

    Raises:
        400: Si la categoria destino es invalida
        403: Si el usuario no es admin
        404: Si no se encuentran productos
        500: Error interno
    """
    struct_logger.info(
        "cluster_split_request",
        source_category=category,
        new_category=request.new_category,
        product_count=len(request.product_ids),
        user_email=current_user.email,
    )

    try:
        service = get_cluster_management_service(db)

        # Construir request completo con categoria del path
        full_request = ClusterSplitRequest(
            source_category=category,
            product_ids=request.product_ids,
            new_category=request.new_category,
            notes=request.notes,
        )

        result = service.split_cluster(request=full_request, user=current_user)

        struct_logger.info(
            "cluster_split_completed",
            source_category=category,
            new_category=request.new_category,
            products_moved=result.products_moved,
            enrichments_synced=result.enrichments_synced,
        )

        return result

    except ValueError as e:
        struct_logger.warning(
            "cluster_split_validation_error",
            error=str(e),
            source_category=category,
        )
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        struct_logger.error(
            "cluster_split_error",
            error=str(e),
            source_category=category,
        )
        raise HTTPException(status_code=500, detail=f"Error en split: {str(e)}")


@router.post("/clusters/merge", response_model=ClusterMergeResponse)
async def merge_clusters(
    request: ClusterMergeRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
):
    """
    Fusiona multiples categorias en una categoria destino.

    Mueve TODOS los productos de las categorias origen a la categoria destino.
    Esta operacion es atomica: todos los productos se mueven o ninguno.

    **Requiere permisos de administrador.**

    Args:
        request: Datos del merge (source_categories, destination_category, notes)
        db: Sesion de base de datos
        current_user: Usuario admin autenticado

    Returns:
        ClusterMergeResponse con detalles de la operacion

    Raises:
        400: Si la categoria destino es invalida o esta en las categorias origen
        403: Si el usuario no es admin
        404: Si no se encuentran productos en las categorias origen
        500: Error interno
    """
    struct_logger.info(
        "cluster_merge_request",
        source_categories=request.source_categories,
        destination_category=request.destination_category,
        user_email=current_user.email,
    )

    try:
        service = get_cluster_management_service(db)
        result = service.merge_clusters(request=request, user=current_user)

        struct_logger.info(
            "cluster_merge_completed",
            source_categories=request.source_categories,
            destination_category=request.destination_category,
            total_products_merged=result.total_products_merged,
            enrichments_synced=result.enrichments_synced,
        )

        return result

    except ValueError as e:
        struct_logger.warning(
            "cluster_merge_validation_error",
            error=str(e),
            destination_category=request.destination_category,
        )
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        struct_logger.error(
            "cluster_merge_error",
            error=str(e),
            destination_category=request.destination_category,
        )
        raise HTTPException(status_code=500, detail=f"Error en merge: {str(e)}")


@router.get("/clusters/{category}/preview-split", response_model=SplitPreviewResponse)
async def preview_split(
    category: str,
    product_ids: str = Query(
        ...,
        description="IDs de productos separados por coma (UUIDs)"
    ),
    new_category: str = Query(
        ...,
        description="Categoria destino propuesta"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Previsualiza una operacion de split sin ejecutarla (dry run).

    Muestra que productos se moverian, el impacto en ambas categorias
    y cualquier advertencia relevante.

    Args:
        category: Categoria origen
        product_ids: IDs de productos separados por coma
        new_category: Categoria destino propuesta
        db: Sesion de base de datos
        current_user: Usuario autenticado

    Returns:
        SplitPreviewResponse con detalles de la operacion propuesta
    """
    struct_logger.info(
        "cluster_split_preview",
        source_category=category,
        new_category=new_category,
        user_email=current_user.email,
    )

    try:
        # Parsear UUIDs de la cadena separada por comas
        uuid_list: List[UUID] = []
        for pid in product_ids.split(","):
            pid = pid.strip()
            if pid:
                try:
                    uuid_list.append(UUID(pid))
                except ValueError:
                    raise HTTPException(
                        status_code=400,
                        detail=f"UUID invalido: {pid}"
                    )

        if not uuid_list:
            raise HTTPException(
                status_code=400,
                detail="Se requiere al menos un product_id"
            )

        service = get_cluster_management_service(db)
        result = service.preview_split(
            source_category=category,
            product_ids=uuid_list,
            new_category=new_category,
        )

        return result

    except HTTPException:
        raise
    except Exception as e:
        struct_logger.error(
            "cluster_split_preview_error",
            error=str(e),
            source_category=category,
        )
        raise HTTPException(status_code=500, detail=f"Error en preview: {str(e)}")


@router.get("/clusters/preview-merge", response_model=MergePreviewResponse)
async def preview_merge(
    source_categories: str = Query(
        ...,
        description="Categorias origen separadas por coma"
    ),
    destination: str = Query(
        ...,
        description="Categoria destino"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Previsualiza una operacion de merge sin ejecutarla (dry run).

    Muestra que categorias se fusionarian, el impacto en productos
    y cualquier advertencia relevante.

    Args:
        source_categories: Categorias origen separadas por coma
        destination: Categoria destino
        db: Sesion de base de datos
        current_user: Usuario autenticado

    Returns:
        MergePreviewResponse con detalles de la operacion propuesta
    """
    struct_logger.info(
        "cluster_merge_preview",
        source_categories=source_categories,
        destination=destination,
        user_email=current_user.email,
    )

    try:
        # Parsear categorias de la cadena separada por comas
        categories_list = [cat.strip() for cat in source_categories.split(",") if cat.strip()]

        if not categories_list:
            raise HTTPException(
                status_code=400,
                detail="Se requiere al menos una categoria origen"
            )

        service = get_cluster_management_service(db)
        result = service.preview_merge(
            source_categories=categories_list,
            destination_category=destination,
        )

        return result

    except HTTPException:
        raise
    except Exception as e:
        struct_logger.error(
            "cluster_merge_preview_error",
            error=str(e),
            destination=destination,
        )
        raise HTTPException(status_code=500, detail=f"Error en preview: {str(e)}")


@router.get("/clusters/{category}/stats", response_model=ClusterStats)
async def get_cluster_stats(
    category: str,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene estadisticas detalladas de un cluster (categoria).

    Incluye conteos de productos, ventas, marcas principales,
    metricas de calidad y tasa de verificacion.

    Args:
        category: Nombre de la categoria
        db: Sesion de base de datos
        current_user: Usuario autenticado

    Returns:
        ClusterStats con estadisticas del cluster
    """
    struct_logger.info(
        "cluster_stats_request",
        category=category,
        user_email=current_user.email,
    )

    try:
        service = get_cluster_management_service(db)
        result = service.get_cluster_stats(category=category)

        return result

    except Exception as e:
        struct_logger.error(
            "cluster_stats_error",
            error=str(e),
            category=category,
        )
        raise HTTPException(status_code=500, detail=f"Error obteniendo stats: {str(e)}")


@router.get("/clusters/validate-name", response_model=CategoryValidation)
async def validate_category_name(
    name: str = Query(..., description="Nombre de categoria a validar"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Valida si un nombre de categoria es valido.

    Una categoria es valida si existe en el enum NecesidadEspecifica (~130 valores).
    Tambien detecta aliases comunes y sugiere el valor correcto.

    Args:
        name: Nombre de la categoria a validar
        db: Sesion de base de datos
        current_user: Usuario autenticado

    Returns:
        CategoryValidation con resultado de la validacion
    """
    struct_logger.info(
        "category_validation_request",
        name=name,
        user_email=current_user.email,
    )

    try:
        service = get_cluster_management_service(db)
        result = service.validate_category_name(name=name)

        return result

    except Exception as e:
        struct_logger.error(
            "category_validation_error",
            error=str(e),
            name=name,
        )
        raise HTTPException(status_code=500, detail=f"Error validando categoria: {str(e)}")


# === Evolution & Analytics Endpoints (Issue #491) ===


@router.get("/time-series")
async def get_time_series(
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    period_months: int = Query(12, description="Meses hacia atrás si no se especifica fecha"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Serie temporal mensual de ventas por categoría NECESIDAD (Issue #491).

    Devuelve datos en formato long/flat para gráfico de evolución temporal.
    Agrupa ventas por mes (YYYY-MM) y categoría ml_category.

    Args:
        date_from: Fecha inicio del rango (opcional, default: period_months atrás)
        date_to: Fecha fin del rango (opcional, default: hoy)
        employee_ids: Lista de IDs de empleados para filtrar
        period_months: Meses hacia atrás si no se especifica date_from
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con time_series, category_summary y totales:
        {
            "time_series": [{"period": "2024-01", "category": "proteccion_solar", "sales": 1500, "units": 234}, ...],
            "category_summary": [{"category": "proteccion_solar", "total_sales": 18000, "total_units": 2800}, ...],
            "total_sales": 60000.0,
            "date_from": "2024-01-01",
            "date_to": "2024-12-31"
        }
    """
    return ventalibre_service.get_time_series(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids,
        period_months=period_months
    )


@router.get("/yoy-comparison")
async def get_yoy_comparison(
    date_from: Optional[date] = Query(None, description="Fecha inicio período actual"),
    date_to: Optional[date] = Query(None, description="Fecha fin período actual"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Comparación YoY (Year-over-Year) por categoría NECESIDAD (Issue #491).

    Compara ventas del período actual vs mismo período del año anterior.
    Útil para análisis de tendencias y variaciones interanuales.

    Args:
        date_from: Fecha inicio período actual (default: hace 12 meses)
        date_to: Fecha fin período actual (default: hoy)
        employee_ids: Lista de IDs de empleados para filtrar
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con comparación YoY por categoría:
        {
            "categories": [
                {"category": "proteccion_solar", "current_sales": 15000, "previous_sales": 12000,
                 "variation_euros": 3000, "variation_percent": 25.0, "trend": "up"},
                ...
            ],
            "total_current": 60000.0,
            "total_previous": 50000.0,
            "total_variation_percent": 20.0,
            "current_period": {"from": "2024-01-01", "to": "2024-12-31"},
            "previous_period": {"from": "2023-01-01", "to": "2023-12-31"}
        }
    """
    return ventalibre_service.get_yoy_comparison(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids
    )


@router.get("/top-contributors")
async def get_top_contributors(
    date_from: Optional[date] = Query(None, description="Fecha inicio período actual"),
    date_to: Optional[date] = Query(None, description="Fecha fin período actual"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    limit: int = Query(10, le=50, description="Número de productos a devolver"),
    direction: str = Query("all", description="Filtrar: 'up' (crecimiento), 'down' (decrecimiento), 'all'"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Top productos que más contribuyen al crecimiento/decrecimiento YoY (Issue #491).

    Identifica los productos con mayor impacto en la variación de ventas
    respecto al año anterior.

    Args:
        date_from: Fecha inicio período actual
        date_to: Fecha fin período actual
        employee_ids: Lista de IDs de empleados para filtrar
        limit: Número máximo de productos (max 50)
        direction: "up" (solo crecimiento), "down" (solo decrecimiento), "all" (ambos)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con top contributors:
        {
            "contributors": [
                {"product_name": "ISDIN FOTOPROTECTOR SPF50+", "category": "proteccion_solar",
                 "current_sales": 2500, "previous_sales": 1500, "variation_euros": 1000,
                 "variation_percent": 66.7, "impact_percent": 3.3},
                ...
            ],
            "total_change": 3000.0,
            "direction": "all"
        }
    """
    # Validar direction
    if direction not in ("up", "down", "all"):
        raise HTTPException(status_code=400, detail="direction debe ser 'up', 'down' o 'all'")

    return ventalibre_service.get_top_contributors(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids,
        limit=limit,
        direction=direction
    )


# ============================================================================
# Issue #493: Endpoints para Tab "Categorías y Marcas"
# ============================================================================


@router.get("/brands/{necesidad}")
async def get_brands_by_necesidad(
    necesidad: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    employee_name: Optional[str] = Query(None, description="Nombre de empleado"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener marcas por categoría NECESIDAD con índice HHI.

    Retorna lista de marcas ordenadas por ventas, con:
    - Cuota de mercado (%)
    - Margen promedio (%)
    - Unidades vendidas
    - Índice HHI de concentración

    HHI Interpretación:
    - < 1500: Mercado atomizado (muy competitivo)
    - 1500-2500: Concentración moderada
    - > 2500: Oligopolio (dominado por líderes)

    Args:
        necesidad: Categoría NECESIDAD a analizar (ej: "proteccion_solar")
        date_from: Fecha inicio del rango de análisis
        date_to: Fecha fin del rango de análisis
        employee_name: Filtrar por nombre de empleado

    Returns:
        {
            "brands": [{"brand": "isdin", "sales": 5000, "share": 35.5, ...}, ...],
            "total_sales": 14000.0,
            "hhi": 1850.5,
            "hhi_interpretation": {"level": "medium", "title": "Concentración moderada", ...},
            "coverage_percent": 87.4,
            "brand_duel_available": true
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    return ventalibre_service.get_brands_by_necesidad(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        necesidad=necesidad,
        date_from=date_from,
        date_to=date_to,
        employee_name=employee_name,
    )


@router.get("/market-share-evolution/{necesidad}")
async def get_market_share_evolution(
    necesidad: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    top_n: int = Query(5, ge=1, le=10, description="Número de marcas top a mostrar"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Evolución temporal de cuota de mercado por marca.

    Retorna serie temporal mensual con porcentaje de cada marca,
    ideal para gráfico de áreas apiladas 100%.

    Args:
        necesidad: Categoría NECESIDAD a analizar
        date_from: Fecha inicio (default: últimos 12 meses)
        date_to: Fecha fin (default: hoy)
        top_n: Número de marcas top a mostrar (resto agrupado en "Otras")

    Returns:
        {
            "time_series": [
                {"month": "2024-01", "isdin": 35.5, "lacer": 22.1, "Otras": 42.4},
                {"month": "2024-02", "isdin": 33.2, "lacer": 24.5, "Otras": 42.3},
                ...
            ],
            "top_brands": ["isdin", "lacer", "gum", ...]
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    return ventalibre_service.get_brand_market_share_evolution(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        necesidad=necesidad,
        date_from=date_from,
        date_to=date_to,
        top_n=top_n,
    )


@router.get("/value-quadrant/{necesidad}")
async def get_value_quadrant(
    necesidad: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Datos para scatter plot Margen(%) vs Volumen(€) - Cuadrante de Valor.

    Los thresholds son dinámicos (medianas de la categoría), garantizando
    que siempre haya marcas en los 4 cuadrantes.

    Cuadrantes:
    - star: Alto volumen, alto margen (Estrellas - proteger)
    - traffic: Alto volumen, bajo margen (Generadores de Tráfico)
    - opportunity: Bajo volumen, alto margen (Oportunidades - potenciar)
    - review: Bajo volumen, bajo margen (Revisar - evaluar)

    Args:
        necesidad: Categoría NECESIDAD a analizar
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango

    Returns:
        {
            "brands": [
                {"brand": "isdin", "sales": 5000, "margin_pct": 35.5,
                 "quadrant": "star", "quadrant_label": "Estrellas"},
                ...
            ],
            "thresholds": {"median_sales": 2500.0, "median_margin": 28.5},
            "targets": [{"value": 18.0, "color": "#dc3545", "line_style": "dot", "label": "Mínimo 18%"}, ...]
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    result = ventalibre_service.get_brand_value_quadrant(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        necesidad=necesidad,
        date_from=date_from,
        date_to=date_to,
    )

    # Issue #510: Añadir targets de margen para líneas de referencia
    result["targets"] = _get_margin_targets_for_chart(db, current_user.pharmacy_id)

    return result


@router.get("/brand-duel")
async def get_brand_duel(
    brand_a: str = Query(..., description="Primera marca a comparar"),
    brand_b: str = Query(..., description="Segunda marca a comparar"),
    necesidad: str = Query(..., description="Categoría NECESIDAD"),
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Comparación lado a lado de 2 marcas (Brand Duel Mode).

    Compara métricas clave entre dos marcas seleccionadas,
    indicando el ganador en cada métrica.

    Métricas comparadas:
    - share: Cuota de mercado (%) - Mayor es mejor
    - margin_pct: Margen promedio (%) - Mayor es mejor
    - avg_ticket: Ticket promedio (€) - Mayor es mejor
    - units: Unidades vendidas - Mayor es mejor
    - total_margin: Margen total (€) - Mayor es mejor

    Args:
        brand_a: Primera marca a comparar (ej: "sensodyne")
        brand_b: Segunda marca a comparar (ej: "lacer")
        necesidad: Categoría NECESIDAD donde comparar
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango

    Returns:
        {
            "brand_a": {"brand": "sensodyne", "sales": 3500, "share": 22.5, ...},
            "brand_b": {"brand": "lacer", "sales": 4200, "share": 27.1, ...},
            "winners": {"share": "b", "margin_pct": "a", "avg_ticket": "a", ...},
            "total_category_sales": 15500.0
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    return ventalibre_service.get_brand_duel(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        brand_a=brand_a,
        brand_b=brand_b,
        necesidad=necesidad,
        date_from=date_from,
        date_to=date_to,
    )


@router.get("/price-distribution/{necesidad}")
async def get_price_distribution(
    necesidad: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    top_n: int = Query(10, ge=1, le=20, description="Número de marcas a analizar"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Distribución de precios por marca para boxplot.

    Útil para detectar canibalización de precios entre marcas:
    - Solapamiento de IQR indica competencia directa
    - Cajas muy estiradas indican catálogo de precios inconsistente
    - Outliers pueden indicar productos mal categorizados

    Args:
        necesidad: Categoría NECESIDAD a analizar
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango
        top_n: Número de marcas top a incluir

    Returns:
        {
            "brands": [
                {
                    "brand": "isdin",
                    "count": 45,
                    "min": 8.50,
                    "q1": 15.00,
                    "median": 22.50,
                    "q3": 35.00,
                    "max": 65.00,
                    "iqr": 20.00,
                    "sample_products": [{"code": "123456", "name": "...", "price": 22.50}, ...]
                },
                ...
            ]
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    return ventalibre_service.get_price_distribution(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        necesidad=necesidad,
        date_from=date_from,
        date_to=date_to,
        top_n=top_n,
    )


# ============================================================================
# Issue #494: Endpoints para Tab "Producto y Surtido" - Market Basket Analysis
# ============================================================================

from app.schemas.product_analysis import (
    AlternativesResponse,
    ComplementaryResponse,
    CompetitiveAnalysisResponse,
    FullProductAnalysisResponse,
    ProductFichaResponse,
    ProductKPIs,
    ProductSearchResponse,
    RecommendationResponse,
)
from app.services.basket_analysis_service import BasketAnalysisService


@router.get("/search", response_model=ProductSearchResponse)
async def search_products(
    q: str = Query(..., min_length=3, description="Texto de búsqueda (mínimo 3 caracteres)"),
    limit: int = Query(20, ge=1, le=50, description="Máximo de resultados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Búsqueda de productos con autocompletado (Issue #494).

    Busca productos venta libre por CN, nombre o marca.
    Prioriza: CN exacto > Nombre starts_with > Nombre contains.
    Optimizado para autocompletado en buscador.

    Args:
        q: Texto de búsqueda (mínimo 3 caracteres)
        limit: Número máximo de resultados (max 50)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        ProductSearchResponse con lista de productos encontrados
    """
    from sqlalchemy import case, distinct, func, literal, or_
    from app.models.sales_data import SalesData
    from app.models.sales_enrichment import SalesEnrichment

    pharmacy_id = current_user.pharmacy_id
    search_term = q.strip().lower()

    # Priorización de resultados para autocompletado:
    # 1 = CN exacto (mejor match)
    # 2 = CN empieza con el término
    # 3 = Nombre empieza con el término (utiliza índice)
    # 4 = Nombre contiene el término
    # 5 = Marca contiene el término
    priority_order = case(
        (SalesData.codigo_nacional == search_term, literal(1)),
        (SalesData.codigo_nacional.ilike(f"{search_term}%"), literal(2)),
        (SalesData.product_name.ilike(f"{search_term}%"), literal(3)),
        (SalesData.product_name.ilike(f"%{search_term}%"), literal(4)),
        else_=literal(5),
    )

    # Query base: productos venta libre con ventas
    query = (
        db.query(
            SalesData.codigo_nacional,
            func.max(SalesData.product_name).label("product_name"),
            func.max(SalesEnrichment.detected_brand).label("detected_brand"),
            func.max(SalesEnrichment.ml_category).label("ml_category"),
            func.sum(SalesData.total_amount).label("total_sales"),
        )
        .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
        .filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesEnrichment.product_type == "venta_libre",
            SalesData.quantity > 0,
            or_(
                SalesData.codigo_nacional.ilike(f"%{search_term}%"),
                SalesData.product_name.ilike(f"%{search_term}%"),
                SalesEnrichment.detected_brand.ilike(f"%{search_term}%"),
            ),
        )
        .group_by(SalesData.codigo_nacional)
        # Ordenar por prioridad (mejor match primero), luego por ventas
        .order_by(func.min(priority_order), func.sum(SalesData.total_amount).desc())
        .limit(limit)
    )

    results = []
    for r in query.all():
        results.append({
            "codigo_nacional": r.codigo_nacional or "",
            "product_name": r.product_name or "Desconocido",
            "detected_brand": r.detected_brand,
            "ml_category": r.ml_category,
            "total_sales": float(r.total_sales or 0),
        })

    # Contar total (sin limit)
    count_query = (
        db.query(func.count(distinct(SalesData.codigo_nacional)))
        .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
        .filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesEnrichment.product_type == "venta_libre",
            or_(
                SalesData.codigo_nacional.ilike(f"%{search_term}%"),
                SalesData.product_name.ilike(f"%{search_term}%"),
                SalesEnrichment.detected_brand.ilike(f"%{search_term}%"),
            ),
        )
    )
    total_count = count_query.scalar() or 0

    struct_logger.info(
        "product_search",
        pharmacy_id=str(pharmacy_id),
        query=q,
        results_count=len(results),
        total_count=total_count,
    )

    return ProductSearchResponse(
        results=results,
        total_count=total_count,
        query=q,
    )


@router.get("/products/{product_code}/ficha", response_model=ProductFichaResponse)
async def get_product_ficha(
    product_code: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Ficha completa de producto con KPIs (Issue #494).

    Devuelve información detallada del producto incluyendo:
    - Datos básicos (nombre, marca, categoría)
    - KPIs: MAT, GMROI, días cobertura, margen vs categoría

    Args:
        product_code: Código Nacional del producto
        date_from: Fecha inicio para cálculo de KPIs
        date_to: Fecha fin para cálculo de KPIs
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        ProductFichaResponse con ficha completa y KPIs
    """
    from datetime import timedelta
    from sqlalchemy import func
    from app.models.sales_data import SalesData
    from app.models.sales_enrichment import SalesEnrichment

    pharmacy_id = current_user.pharmacy_id

    # Defaults de fecha
    if not date_to:
        date_to = date.today()
    if not date_from:
        date_from = date_to - timedelta(days=365)

    # Obtener datos del producto
    product_query = (
        db.query(
            SalesData.codigo_nacional,
            func.max(SalesData.product_name).label("product_name"),
            func.max(SalesEnrichment.detected_brand).label("detected_brand"),
            func.max(SalesEnrichment.brand_line).label("brand_line"),
            func.max(SalesEnrichment.ml_category).label("ml_category"),
            func.max(SalesEnrichment.ml_subcategory).label("ml_subcategory"),
            func.sum(SalesData.total_amount).label("mat"),
            func.avg(SalesData.margin_percentage).label("avg_margin"),
        )
        .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
        .filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesData.codigo_nacional == product_code,
            SalesData.sale_date >= date_from,
            SalesData.sale_date <= date_to,
            SalesData.quantity > 0,
        )
        .group_by(SalesData.codigo_nacional)
        .first()
    )

    if not product_query:
        raise HTTPException(status_code=404, detail=f"Producto {product_code} no encontrado")

    # Calcular KPIs usando el servicio
    basket_service = BasketAnalysisService()

    # Días de cobertura
    coverage_days = basket_service._calculate_coverage_days(db, pharmacy_id, product_code)
    coverage_alert = "info"
    if coverage_days:
        if coverage_days < 7:
            coverage_alert = "danger"
        elif coverage_days < 14:
            coverage_alert = "warning"
        elif coverage_days < 30:
            coverage_alert = "success"

    # MAT YoY
    mat_yoy_change = basket_service._calculate_mat_trend(db, pharmacy_id, product_code)

    # GMROI
    gmroi = basket_service._calculate_simple_gmroi(db, pharmacy_id, product_code)
    gmroi_interpretation = "Sin datos"
    if gmroi:
        if gmroi >= 3.0:
            gmroi_interpretation = "ROI excelente"
        elif gmroi >= 2.0:
            gmroi_interpretation = "ROI bueno"
        elif gmroi >= 1.0:
            gmroi_interpretation = "ROI moderado"
        else:
            gmroi_interpretation = "ROI bajo"

    # Margen vs categoría
    margin_vs_cat, _ = basket_service._calculate_margin_vs_category(db, pharmacy_id, product_code)

    struct_logger.info(
        "product_ficha",
        pharmacy_id=str(pharmacy_id),
        product_code=product_code,
        mat=float(product_query.mat or 0),
        coverage_days=coverage_days,
    )

    return ProductFichaResponse(
        product_code=product_code,
        product_name=product_query.product_name or "Desconocido",
        detected_brand=product_query.detected_brand,
        brand_line=product_query.brand_line,
        ml_category=product_query.ml_category,
        ml_subcategory=product_query.ml_subcategory,
        kpis=ProductKPIs(
            mat=round(float(product_query.mat or 0), 2),
            mat_yoy_change=mat_yoy_change,
            gmroi=round(gmroi, 2) if gmroi else 0.0,
            gmroi_interpretation=gmroi_interpretation,
            coverage_days=coverage_days,
            coverage_alert_level=coverage_alert,
            margin_vs_category=margin_vs_cat or 0.0,
        ),
    )


@router.get("/products/{product_code}/alternatives", response_model=AlternativesResponse)
async def get_product_alternatives(
    product_code: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    limit: int = Query(10, ge=1, le=50, description="Máximo de alternativas"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Productos alternativos por categoría NECESIDAD (Issue #494).

    Devuelve productos del mismo NECESIDAD pero diferente marca,
    ordenados por ventas. Útil para sugerir switching.

    Args:
        product_code: Código Nacional del producto base
        date_from: Fecha inicio
        date_to: Fecha fin
        limit: Máximo de alternativas a retornar
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        AlternativesResponse con lista de alternativas y diferencial de margen
    """
    from datetime import timedelta

    pharmacy_id = current_user.pharmacy_id

    # Defaults de fecha
    if not date_to:
        date_to = date.today()
    if not date_from:
        date_from = date_to - timedelta(days=365)

    basket_service = BasketAnalysisService()
    result = basket_service.get_alternatives_by_necesidad(
        db=db,
        pharmacy_id=pharmacy_id,
        product_code=product_code,
        date_from=date_from,
        date_to=date_to,
        limit=limit,
    )

    return AlternativesResponse(**result)


@router.get("/products/{product_code}/complementary", response_model=ComplementaryResponse)
async def get_complementary_products(
    product_code: str,
    days: int = Query(90, ge=30, le=365, description="Días de análisis (default: 90 = 3 meses)"),
    min_support: float = Query(0.01, ge=0.001, le=0.5, description="Support mínimo (default: 1%)"),
    min_confidence: float = Query(0.1, ge=0.01, le=1.0, description="Confidence mínimo (default: 10%)"),
    limit: int = Query(10, ge=1, le=50, description="Máximo de productos"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Productos complementarios por Market Basket Analysis (Issue #494).

    Analiza cestas/tickets para encontrar productos frecuentemente
    comprados junto con el producto base.

    Métricas:
    - Support: % de cestas con esta combinación
    - Confidence: P(complementario | base)
    - Lift: Fuerza de asociación (>1 = relevante)

    Args:
        product_code: Código Nacional del producto base
        days: Días de análisis histórico (default: 90)
        min_support: Support mínimo para incluir producto
        min_confidence: Confidence mínimo para incluir producto
        limit: Máximo de productos complementarios
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        ComplementaryResponse con productos complementarios y métricas MBA
    """
    pharmacy_id = current_user.pharmacy_id

    basket_service = BasketAnalysisService()
    result = basket_service.get_complementary_products(
        db=db,
        pharmacy_id=pharmacy_id,
        product_code=product_code,
        days=days,
        min_support=min_support,
        min_confidence=min_confidence,
        limit=limit,
    )

    return ComplementaryResponse(**result)


@router.get("/products/{product_code}/competitive-analysis", response_model=CompetitiveAnalysisResponse)
async def get_competitive_analysis(
    product_code: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Análisis competitivo para scatter plot (Issue #494).

    Devuelve datos de todos los productos de la misma categoría NECESIDAD
    para visualizar en scatter plot (Ventas vs Margen).
    El producto seleccionado se marca con is_selected=True.

    Args:
        product_code: Código Nacional del producto base
        date_from: Fecha inicio
        date_to: Fecha fin
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        CompetitiveAnalysisResponse con puntos para scatter y promedios de categoría
    """
    from datetime import timedelta
    from sqlalchemy import func
    from app.models.sales_data import SalesData
    from app.models.sales_enrichment import SalesEnrichment
    from app.models.inventory_snapshot import InventorySnapshot

    pharmacy_id = current_user.pharmacy_id

    # Defaults de fecha
    if not date_to:
        date_to = date.today()
    if not date_from:
        date_from = date_to - timedelta(days=365)

    # Obtener categoría del producto base
    base_category = (
        db.query(SalesEnrichment.ml_category)
        .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
        .filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesData.codigo_nacional == product_code,
        )
        .first()
    )

    if not base_category or not base_category.ml_category:
        raise HTTPException(
            status_code=404,
            detail=f"Producto {product_code} no encontrado o sin categoría NECESIDAD"
        )

    category = base_category.ml_category

    # Obtener todos los productos de la categoría
    products_query = (
        db.query(
            SalesData.codigo_nacional,
            func.max(SalesData.product_name).label("product_name"),
            func.max(SalesEnrichment.detected_brand).label("detected_brand"),
            func.sum(SalesData.total_amount).label("total_sales"),
            func.avg(SalesData.margin_percentage).label("avg_margin"),
        )
        .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
        .filter(
            SalesData.pharmacy_id == pharmacy_id,
            SalesEnrichment.ml_category == category,
            SalesData.sale_date >= date_from,
            SalesData.sale_date <= date_to,
            SalesData.quantity > 0,
        )
        .group_by(SalesData.codigo_nacional)
    )

    # Obtener stock para cada producto
    stock_query = (
        db.query(
            InventorySnapshot.product_code,
            func.sum(InventorySnapshot.stock_quantity).label("stock_qty"),
        )
        .filter(InventorySnapshot.pharmacy_id == pharmacy_id)
        .group_by(InventorySnapshot.product_code)
    )
    stock_map = {r.product_code: r.stock_qty for r in stock_query.all()}

    data_points = []
    total_sales = 0
    total_margin = 0
    count = 0

    for r in products_query.all():
        sales = float(r.total_sales or 0)
        margin = float(r.avg_margin or 0)
        stock = stock_map.get(r.codigo_nacional, 0)

        total_sales += sales
        total_margin += margin
        count += 1

        data_points.append({
            "codigo_nacional": r.codigo_nacional,
            "product_name": r.product_name or "Desconocido",
            "detected_brand": r.detected_brand,
            "x": round(sales, 2),
            "y": round(margin, 2),
            "size": float(stock) if stock else 10.0,
            "stock_qty": stock,
            "is_selected": r.codigo_nacional == product_code,
        })

    category_avg_sales = total_sales / count if count > 0 else 0
    category_avg_margin = total_margin / count if count > 0 else 0

    struct_logger.info(
        "competitive_analysis",
        pharmacy_id=str(pharmacy_id),
        product_code=product_code,
        category=category,
        products_in_category=count,
    )

    return CompetitiveAnalysisResponse(
        data_points=data_points,
        category=category,
        category_avg_sales=round(category_avg_sales, 2),
        category_avg_margin=round(category_avg_margin, 2),
        total_products_in_category=count,
    )


@router.get("/products/{product_code}/recommendation", response_model=RecommendationResponse)
async def get_product_recommendation(
    product_code: str,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Recomendación algorítmica para Decision Box (Issue #494).

    Genera recomendación basada en:
    - Días de cobertura (stock)
    - Tendencia MAT YoY
    - Margen vs categoría
    - GMROI
    - Existencia de mejores alternativas

    Semáforo:
    - ROJO: coverage_days > 90 Y trend_MAT < -10% (sobrestock en declive)
    - AMARILLO: margin_vs_category < 0 con volumen alto (generador de tráfico)
    - VERDE: coverage_days < 7 Y trend_MAT > 15% (oportunidad de compra)

    Args:
        product_code: Código Nacional del producto
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        RecommendationResponse con recomendación, factores y razones
    """
    pharmacy_id = current_user.pharmacy_id

    basket_service = BasketAnalysisService()
    result = basket_service.get_product_recommendation(
        db=db,
        pharmacy_id=pharmacy_id,
        product_code=product_code,
    )

    return RecommendationResponse(**result)


@router.get("/products/{product_code}/full-analysis", response_model=FullProductAnalysisResponse)
async def get_full_product_analysis(
    product_code: str,
    date_from: Optional[date] = Query(None, description="Fecha inicio"),
    date_to: Optional[date] = Query(None, description="Fecha fin"),
    alternatives_limit: int = Query(10, ge=1, le=50, description="Límite alternativas"),
    complementary_days: int = Query(90, ge=30, le=365, description="Días para MBA"),
    complementary_limit: int = Query(10, ge=1, le=50, description="Límite complementarios"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Análisis completo del producto en una sola llamada (Issue #494 optimization).

    Combina todos los datos del producto:
    - Ficha con KPIs (MAT, GMROI, cobertura, margen)
    - Alternativas (productos mismo NECESIDAD, diferente marca)
    - Complementarios (Market Basket Analysis)
    - Análisis competitivo (scatter plot de categoría)
    - Recomendación algorítmica (Decision Box)

    **Beneficio**: Reduce 5 API calls a 1, mejorando latencia de ~2-5s a ~0.5-1s.

    Args:
        product_code: Código Nacional del producto
        date_from: Fecha inicio para KPIs (default: hoy - 365 días)
        date_to: Fecha fin para KPIs (default: hoy)
        alternatives_limit: Máximo de alternativas a retornar
        complementary_days: Período para análisis MBA
        complementary_limit: Máximo de complementarios a retornar
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        FullProductAnalysisResponse con todos los datos combinados
    """
    from datetime import timedelta
    import logging

    logger = logging.getLogger(__name__)
    pharmacy_id = current_user.pharmacy_id

    # Defaults de fecha
    if not date_to:
        date_to = date.today()
    if not date_from:
        date_from = date_to - timedelta(days=365)

    basket_service = BasketAnalysisService()
    results = {
        "ficha": None,
        "alternatives": None,
        "complementary": None,
        "competitive": None,
        "recommendation": None,
    }

    # Ejecutar todas las consultas secuencialmente para mantener la sesión DB

    # 1. Ficha
    try:
        ficha_response = await get_product_ficha(
            product_code=product_code,
            date_from=date_from,
            date_to=date_to,
            db=db,
            current_user=current_user,
        )
        results["ficha"] = ficha_response
    except HTTPException as e:
        if e.status_code != 404:
            logger.warning(f"[full-analysis] ficha error for {product_code}: {e.detail}")
    except Exception as e:
        logger.warning(f"[full-analysis] ficha error: {e}")

    # 2. Alternativas
    try:
        alternatives_response = await get_product_alternatives(
            product_code=product_code,
            date_from=date_from,
            date_to=date_to,
            limit=alternatives_limit,
            db=db,
            current_user=current_user,
        )
        results["alternatives"] = alternatives_response
    except Exception as e:
        logger.warning(f"[full-analysis] alternatives error: {e}")

    # 3. Complementarios (MBA)
    try:
        complementary_response = await get_complementary_products(
            product_code=product_code,
            days=complementary_days,
            limit=complementary_limit,
            db=db,
            current_user=current_user,
        )
        results["complementary"] = complementary_response
    except Exception as e:
        logger.warning(f"[full-analysis] complementary error: {e}")

    # 4. Análisis competitivo
    try:
        competitive_response = await get_competitive_analysis(
            product_code=product_code,
            date_from=date_from,
            date_to=date_to,
            db=db,
            current_user=current_user,
        )
        results["competitive"] = competitive_response
    except Exception as e:
        logger.warning(f"[full-analysis] competitive error: {e}")

    # 5. Recomendación
    try:
        recommendation_response = await get_product_recommendation(
            product_code=product_code,
            db=db,
            current_user=current_user,
        )
        results["recommendation"] = recommendation_response
    except Exception as e:
        logger.warning(f"[full-analysis] recommendation error: {e}")

    return FullProductAnalysisResponse(**results)


# ============================================================================
# Issue #505: L2 Subcategories Endpoints
# ============================================================================


@router.get("/sales-by-l2/{l1_category}")
async def get_sales_by_l2(
    l1_category: L1WithL2Category,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    employee_ids: Optional[List[str]] = Query(None, description="IDs de empleados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Datos de ventas por subcategoría L2 para una categoría L1 (Issue #505).

    Drill-down L1→L2: Agrega ventas por ml_subcategory_l2 para la categoría L1 especificada.
    Solo disponible para: dermocosmetica, suplementos, higiene_bucal

    Args:
        l1_category: Categoría L1 (dermocosmetica, suplementos, higiene_bucal)
        date_from: Fecha inicio del rango de análisis
        date_to: Fecha fin del rango de análisis
        employee_ids: Lista de IDs de empleados para filtrar
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con nodes para treemap L2:
        {
            "nodes": [
                {"category": "solar_facial", "display_name": "Solar Facial",
                 "archetype": "🎯 Estratégico", "sales": 5000.0, "count": 45, "percentage": 33.3},
                ...
            ],
            "total_sales": 15000.0,
            "total_products": 135,
            "l1_category": "dermocosmetica"
        }

    Raises:
        422: Si l1_category no es un valor válido del Enum
    """
    return ventalibre_service.get_sales_by_l2(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        l1_category=l1_category.value,
        date_from=date_from,
        date_to=date_to,
        employee_ids=employee_ids
    )


@router.get("/subcategories/{l1_category}")
async def get_l2_subcategories(
    l1_category: L1WithL2Category,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista de subcategorías L2 para una categoría L1 (Issue #505).

    Devuelve la taxonomía L2 con nombres display y arquetipos de negocio.
    Útil para poblar selectores y leyendas.

    Args:
        l1_category: Categoría L1 (dermocosmetica, suplementos, higiene_bucal)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        Dict con lista de subcategorías L2:
        {
            "l1_category": "dermocosmetica",
            "l1_display_name": "Dermocosmética",
            "subcategories": [
                {"value": "solar_facial", "label": "Solar Facial",
                 "archetype": "🎯 Estratégico", "archetype_description": "Consejo farmacéutico"},
                ...
            ]
        }

    Raises:
        422: Si l1_category no es un valor válido del Enum
    """
    from app.schemas.symptom_taxonomy import (
        L1_TO_L2_CATEGORIES,
        L2_ARCHETYPES,
        L2_DISPLAY_NAMES,
        get_display_name,
    )

    l1_value = l1_category.value

    # Archetype descriptions
    archetype_descriptions = {
        "🪝 Gancho": "Productos de entrada, alta rotación",
        "💎 Joyas": "Alto margen, ticket elevado",
        "🔄 Mantenimiento": "Recompra frecuente",
        "🎯 Estratégico": "Consejo farmacéutico",
        "📅 Estacional": "Picos verano",
        "🔧 Específico": "Tratamiento especializado",
        "🔒 Fidelización": "Venta cruzada",
        "👴 Cronicidad": "Cliente fidelizado",
        "❄️ Coyuntural": "Picos invierno",
        "📈 Tendencia": "Crecimiento sostenido",
        "🏃 Zafarrancho": "Picos enero/mayo",
        "🛒 Commodity": "Precio competitivo",
        "🏥 Core Farmacia": "Consejo profesional",
        "✨ Deseo": "Compra impulsiva",
        "💊 Nicho": "Especialización",
        "➕ Complementarios": "Venta cruzada",
    }

    l2_categories = L1_TO_L2_CATEGORIES.get(l1_value, [])
    subcategories = []

    for l2_cat in l2_categories:
        archetype = L2_ARCHETYPES.get(l2_cat, "")
        subcategories.append({
            "value": l2_cat,
            "label": L2_DISPLAY_NAMES.get(l2_cat, l2_cat.replace("_", " ").title()),
            "archetype": archetype,
            "archetype_description": archetype_descriptions.get(archetype, ""),
        })

    return {
        "l1_category": l1_value,
        "l1_display_name": get_display_name(l1_value),
        "subcategories": subcategories,
    }


@router.get("/l2-coverage")
async def get_l2_coverage(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Estadísticas de cobertura L2 por categoría L1 (Issue #505).

    Muestra qué porcentaje de productos tienen clasificación L2
    en cada categoría L1 que soporta L2.

    Args:
        db: Sesión de base de datos
        current_user: Usuario autenticado (no usado, pero requerido para auth)

    Returns:
        Dict con cobertura L2:
        {
            "categories": [
                {"l1_category": "dermocosmetica", "total_products": 5000,
                 "with_l2": 4500, "coverage_percent": 90.0},
                ...
            ],
            "overall": {"total_products": 15000, "with_l2": 13500, "coverage_percent": 90.0}
        }
    """
    return ventalibre_service.get_l2_coverage(db=db)


# ============================================================================
# Issue #539: HHI Matrix - Scatter Plot HHI × Margen por Categoría
# ============================================================================


@router.get("/hhi-matrix")
async def get_hhi_matrix(
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Matriz HHI × Margen para scatter plot por categoría (Issue #539).

    Devuelve HHI y margen promedio para TODAS las categorías con ventas,
    permitiendo visualizar la estrategia de concentración de la farmacia.

    Diagnósticos por cuadrante:
    - 🎯 Especialista Exitoso: HHI alto + Margen alto (socio clave de proveedores)
    - ⚠️ Riesgo de Dependencia: HHI alto + Margen bajo (revisar condiciones)
    - ⭐ Generalista Premium: HHI bajo + Margen alto (surtido variado rentable)
    - 📊 Generalista de Volumen: HHI bajo + Margen bajo (negocio de rotación)
    - ⚖️ Surtido Balanceado: Zona intermedia (sin alarmas)

    Args:
        date_from: Fecha inicio del rango (default: últimos 12 meses)
        date_to: Fecha fin del rango (default: hoy)
        db: Sesión de base de datos
        current_user: Usuario autenticado

    Returns:
        {
            "categories": [
                {
                    "category": "proteccion_solar",
                    "hhi": 1850.5,
                    "avg_margin": 28.3,
                    "total_sales": 15000.0,
                    "brand_count": 8,
                    "top_brands": [
                        {"brand": "isdin", "share": 35.5},
                        {"brand": "lacer", "share": 22.1},
                        {"brand": "gum", "share": 15.0}
                    ],
                    "diagnosis": {
                        "emoji": "⭐",
                        "title": "Generalista Premium",
                        "color": "primary",
                        "quadrant": "generalist_premium"
                    }
                },
                ...
            ],
            "thresholds": {
                "hhi_low": 1500,
                "hhi_high": 2500,
                "margin_low": 15.0,
                "margin_high": 25.0
            },
            "summary": {
                "total_categories": 32,
                "avg_hhi": 2100.5,
                "avg_margin": 22.3,
                "total_sales": 125000.0
            },
            "targets": {
                "margin": [{"value": 18.0, "color": "#dc3545", ...}, ...],
                "hhi": [{"value": 2500, "color": "#ffc107", ...}]
            }
        }
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )
    result = ventalibre_service.get_hhi_matrix(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        date_from=date_from,
        date_to=date_to,
    )

    # Issue #510: Añadir targets de margen y HHI para líneas de referencia
    result["targets"] = {
        "margin": _get_margin_targets_for_chart(db, current_user.pharmacy_id),
        "hhi": _get_margin_targets_for_chart(db, current_user.pharmacy_id, ["hhi_max"]),
    }

    return result


@router.get("/value-quadrant-l2/{l1_category}")
async def get_l2_value_quadrant(
    l1_category: L1WithL2Category,
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Cuadrante de Valor por subcategoría L2 (Issue #505).

    Scatter plot Margen(%) vs Volumen(€) para subcategorías L2 dentro de una L1.
    Los thresholds son dinámicos (medianas), garantizando distribución en 4 cuadrantes.

    Cuadrantes:
    - star: Alto volumen, alto margen (Estrellas - proteger)
    - traffic: Alto volumen, bajo margen (Generadores de Tráfico)
    - opportunity: Bajo volumen, alto margen (Oportunidades - potenciar)
    - review: Bajo volumen, bajo margen (Revisar - evaluar)

    Args:
        l1_category: Categoría L1 (dermocosmetica, suplementos, higiene_bucal)
        date_from: Fecha inicio del rango
        date_to: Fecha fin del rango

    Returns:
        {
            "subcategories": [
                {"l2": "solar_facial", "display_name": "Solar Facial",
                 "archetype": "🎯 Estratégico", "sales": 5000, "margin_pct": 35.5,
                 "quadrant": "star", "quadrant_label": "Estrellas"},
                ...
            ],
            "thresholds": {"median_sales": 3000.0, "median_margin": 30.5},
            "l1_category": "dermocosmetica",
            "targets": [{"value": 18.0, "color": "#dc3545", "line_style": "dot", "label": "Mínimo 18%"}, ...]
        }

    Raises:
        422: Si l1_category no es un valor válido del Enum
    """
    result = ventalibre_service.get_l2_value_quadrant(
        db=db,
        pharmacy_id=current_user.pharmacy_id,
        l1_category=l1_category.value,
        date_from=date_from,
        date_to=date_to,
    )

    # Issue #510: Añadir targets de margen para líneas de referencia
    result["targets"] = _get_margin_targets_for_chart(db, current_user.pharmacy_id)

    return result


# =============================================================================
# Issue #512: Context Navigation - CSV Export Endpoint
# =============================================================================


@router.get("/products/export")
async def export_products_csv(
    category: str = Query(..., description="Categoría NECESIDAD (L1) a exportar"),
    date_from: Optional[date] = Query(None, description="Fecha inicio (YYYY-MM-DD)"),
    date_to: Optional[date] = Query(None, description="Fecha fin (YYYY-MM-DD)"),
    format: str = Query("csv", description="Formato de exportación (csv)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Exportar productos de una categoría NECESIDAD en formato CSV (Issue #512).

    Genera un CSV con los productos de la categoría seleccionada, incluyendo:
    - Nombre del producto
    - Marca detectada
    - Código Nacional
    - Ventas (€)
    - Unidades vendidas
    - Margen (%)
    - Subcategoría L2 (si aplica)

    Args:
        category: Código de categoría L1 (ej: 'dermocosmetica', 'respiratorio')
        date_from: Fecha inicio del rango de ventas
        date_to: Fecha fin del rango de ventas
        format: Formato de exportación (solo 'csv' soportado)

    Returns:
        Response: Archivo CSV para descarga

    Raises:
        400: Si la categoría no es válida
        403: Si el usuario no tiene farmacia asociada
    """
    import csv
    import io
    from fastapi.responses import StreamingResponse
    from app.models.sales_data import SalesData
    from app.models.sales_enrichment import SalesEnrichment

    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=403,
            detail="Usuario sin farmacia asociada. Contacte al administrador."
        )

    pharmacy_id = current_user.pharmacy_id

    # Validate category
    valid_categories = [
        "dolor_fiebre", "respiratorio", "digestivo", "piel", "bucal",
        "oftalmologia", "heridas", "muscular", "dermocosmetica", "suplementos",
        "higiene_bucal", "infantil", "sexual", "control_peso", "antitabaco",
        "pies", "cabello", "veterinaria", "hogar", "interno_no_venta", "otros"
    ]

    if category not in valid_categories:
        raise HTTPException(
            status_code=400,
            detail=f"Categoría '{category}' no válida. Categorías permitidas: {', '.join(valid_categories)}"
        )

    struct_logger.info(
        "ventalibre.export_products",
        pharmacy_id=str(pharmacy_id),
        category=category,
        date_from=str(date_from) if date_from else None,
        date_to=str(date_to) if date_to else None,
    )

    try:
        # Build query for products in category
        query = (
            db.query(
                SalesData.product_name,
                SalesData.national_code,
                SalesEnrichment.detected_brand,
                SalesEnrichment.ml_category,
                SalesEnrichment.ml_subcategory,
                SalesData.pvp,
                SalesData.pmc,
                SalesData.quantity,
            )
            .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesEnrichment.product_type == "venta_libre",
                SalesEnrichment.ml_category == category,
            )
        )

        # Apply date filters if provided
        if date_from:
            query = query.filter(SalesData.date >= date_from)
        if date_to:
            query = query.filter(SalesData.date <= date_to)

        # Order by sales descending
        query = query.order_by(SalesData.pvp.desc())

        # Limit to prevent huge exports
        products = query.limit(10000).all()

        struct_logger.info(
            "ventalibre.export_products.count",
            pharmacy_id=str(pharmacy_id),
            category=category,
            count=len(products),
        )

        # Generate CSV
        output = io.StringIO()
        writer = csv.writer(output, delimiter=";")

        # Header
        writer.writerow([
            "Producto",
            "Marca",
            "Código Nacional",
            "Categoría L1",
            "Subcategoría L2",
            "PVP (€)",
            "PMC (€)",
            "Unidades",
            "Margen (%)",
        ])

        # Rows
        for product in products:
            # Calculate margin
            pvp = product.pvp or 0
            pmc = product.pmc or 0
            margin_pct = ((pvp - pmc) / pvp * 100) if pvp > 0 else 0

            writer.writerow([
                product.product_name or "",
                product.detected_brand or "",
                product.national_code or "",
                product.ml_category or "",
                product.ml_subcategory or "",
                f"{pvp:.2f}".replace(".", ","),
                f"{pmc:.2f}".replace(".", ","),
                product.quantity or 0,
                f"{margin_pct:.1f}".replace(".", ","),
            ])

        # Prepare response
        output.seek(0)
        content = output.getvalue()

        # Use BOM for Excel compatibility with UTF-8
        bom = "\ufeff"
        content_with_bom = bom + content

        return StreamingResponse(
            iter([content_with_bom]),
            media_type="text/csv; charset=utf-8",
            headers={
                "Content-Disposition": f'attachment; filename="productos_{category}.csv"'
            },
        )

    except Exception as e:
        struct_logger.error(
            "ventalibre.export_products.error",
            pharmacy_id=str(pharmacy_id),
            category=category,
            error=str(e),
        )
        raise HTTPException(
            status_code=500,
            detail="Error al generar el archivo de exportación"
        )
