﻿# backend/app/api/upload.py
"""
API endpoints para upload de archivos de ventas.

Nota: La lógica de procesamiento de datos ha sido extraída a
services/upload_processing_service.py para mejorar modularidad.
"""
import asyncio
import io
import time
import uuid
from contextlib import asynccontextmanager
from datetime import datetime, time as time_type, timedelta
from pathlib import Path
from typing import Optional, Set

import pandas as pd
import structlog
from fastapi import APIRouter, Depends, File, HTTPException, UploadFile
from sqlalchemy import and_, cast, func, or_
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import Session, scoped_session, sessionmaker
from sqlalchemy.types import Interval

from app.api.error_handler import handle_exception
from app.core.security import get_current_active_user, require_permissions
from app.database import engine, get_db
from app.exceptions import InvalidERPFormatError, PharmacyNotFoundError
from app.models import FileType, FileUpload, Pharmacy, SalesData, UploadStatus
from app.models.sales_enrichment import SalesEnrichment
from app.models.user import User
from app.parsers.file_type_detector import FileContentType, detect_file_content_type
from app.parsers.parser_factory import ParserFactory
from app.services.enrichment_service import enrichment_service
from app.services.inventory_processing_service import get_inventory_processing_service
# Funciones de procesamiento extraídas a servicio separado (refactorización archivos largos)
from app.services.upload_processing_service import (
    build_sales_record,
    calculate_upload_percentage,
    get_enrichment_progress_summary,
    get_enrichment_status,
    load_existing_sales_for_duplicate_detection,
    process_sales_batch,
    save_batch_with_progress,
)
from app.utils.datetime_utils import make_aware, utc_now

logger = structlog.get_logger(__name__)

router = APIRouter()

# Configuración
UPLOAD_DIR = Path("data/uploads")
UPLOAD_DIR.mkdir(parents=True, exist_ok=True)
MAX_FILE_SIZE = 50 * 1024 * 1024  # 50MB

# Task Management - Thread-safe background task tracking
background_tasks: Set[asyncio.Task] = set()


def handle_task_exception(task: asyncio.Task):
    """
    Callback para manejar excepciones en background tasks
    """
    try:
        if task.exception():
            logger.error(
                "[TASK_ERROR] Background task falló",
                extra={
                    "task_name": task.get_name(),
                    "exception": str(task.exception()),
                    "exception_type": type(task.exception()).__name__,
                },
            )
    except asyncio.CancelledError:
        logger.info(f"[TASK_CANCELLED] Background task cancelado: {task.get_name()}")
    finally:
        # Cleanup task reference
        background_tasks.discard(task)


@asynccontextmanager
async def async_db_session():
    """
    Thread-safe async context manager para sesiones de base de datos.

    RESUELVE ISSUE CRÍTICO: Database Session Management - Thread Safety Violation

    Features:
    - Scoped session con scopefunc basado en asyncio task ID
    - Proper cleanup automático
    - Compatible con Render multi-worker environment
    """
    session_factory = scoped_session(sessionmaker(bind=engine), scopefunc=lambda: id(asyncio.current_task()))

    session = session_factory()
    try:
        yield session
        session.commit()
    except Exception:
        session.rollback()
        raise
    finally:
        session.close()
        session_factory.remove()



@router.post("/")
async def upload_file(
    file: UploadFile = File(...),
    pharmacy_id: Optional[str] = None,  # Auto-detect from available pharmacies
    erp_type: Optional[str] = None,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Endpoint para cargar archivos de ventas.

    ISSUE #332: Usa threading para procesamiento en lugar de background_tasks
    para sobrevivir al timeout de 60s en Render.
    """
    logger.info("upload.started", filename=file.filename)

    try:
        # Leer archivo TEMPRANO para obtener file_size
        logger.info("[UPLOAD] Leyendo contenido del archivo...")
        contents = await file.read()
        file_size = len(contents)
        logger.info(f"[UPLOAD] Tamaño del archivo: {file_size:,} bytes")

        # Validar tamaño del archivo
        if file_size > MAX_FILE_SIZE:
            logger.warning(f"[UPLOAD] Archivo demasiado grande: {file_size:,} bytes > {MAX_FILE_SIZE:,}")
            raise HTTPException(
                status_code=413,
                detail=f"El archivo es demasiado grande. Máximo permitido: {MAX_FILE_SIZE / 1024 / 1024:.1f}MB",
            )

        if file_size == 0:
            logger.warning("[UPLOAD] Archivo vacío")
            raise HTTPException(status_code=400, detail="El archivo está vacío")

        # Auto-detect pharmacy_id si no se proporciona
        logger.info("[UPLOAD] Detectando pharmacy_id...")
        if pharmacy_id is None:
            # Primero intentar usar la farmacia del usuario autenticado
            if current_user and current_user.pharmacy_id:
                pharmacy_id = str(current_user.pharmacy_id)
                logger.info(f"[UPLOAD] Usando farmacia del usuario: {pharmacy_id}")

        # NUEVO: Auto-detect erp_type de la farmacia si no se proporciona
        if erp_type is None and pharmacy_id:
            from app.models.pharmacy import Pharmacy

            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if pharmacy and pharmacy.erp_type:
                erp_type = pharmacy.erp_type
                logger.info(f"[UPLOAD] Usando ERP configurado en farmacia: {erp_type}")
            else:
                logger.warning(f"[UPLOAD] Farmacia {pharmacy_id} no tiene erp_type configurado, se usará detección automática")

        # Verificar que la farmacia existe ANTES de cualquier operación
        logger.info(f"[UPLOAD] Verificando farmacia: {pharmacy_id}")
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            logger.info("[UPLOAD] Farmacia no existe, creando nueva...")
            # Crear farmacia demo si no existe
            pharmacy = Pharmacy(
                id=pharmacy_id,
                name="Farmacia Demo",
                code="DEMO",
                email="demo@xfarma.es",
            )
            db.add(pharmacy)
            db.commit()
            logger.info(f"[UPLOAD] Farmacia creada: {pharmacy_id}")
        else:
            logger.info(f"[UPLOAD] Farmacia encontrada: {pharmacy.name}")

        # 🔍 PUNTO DE CONTROL CRÍTICO: Verificar nomenclator_local disponible
        logger.info("[UPLOAD] Verificando disponibilidad de nomenclator...")
        from app.models.nomenclator_local import NomenclatorLocal

        nomenclator_count = db.query(NomenclatorLocal).count()

        if nomenclator_count == 0:
            logger.warning(
                f"⚠️ [UPLOAD] Nomenclator vacío ({nomenclator_count} productos). FORZANDO CARGA AUTOMÁTICA..."
            )

            # 🎯 CREAR UPLOAD RECORD TEMPRANO para mostrar progreso al usuario
            upload_record = FileUpload(
                filename=file.filename,
                file_path="",  # Se actualizará después
                file_type=FileType.UNKNOWN,
                file_size=file_size,
                pharmacy_id=pharmacy_id,
                status=UploadStatus.PROCESSING,
                processing_notes="⚠️ Sistema preparándose: Descargando nomenclator oficial del Ministerio de Sanidad...",
                uploaded_at=utc_now(),
                processing_started_at=utc_now(),
            )
            db.add(upload_record)
            db.commit()
            db.refresh(upload_record)
            upload_id = upload_record.id

            # Forzar carga del nomenclator antes del upload
            try:
                from app.external_data.nomenclator_integration import nomenclator_integration

                logger.info("📥 [UPLOAD] Descargando y cargando nomenclator oficial...")

                # Actualizar estado para el usuario
                upload_record.processing_notes = (
                    "📥 Descargando datos oficiales de medicamentos... Esto puede tardar unos minutos la primera vez."
                )
                db.commit()

                result = nomenclator_integration.manual_update(db)

                # Verificar que la carga fue exitosa
                new_count = db.query(NomenclatorLocal).count()
                logger.info(f"✅ [UPLOAD] Nomenclator cargado: {new_count} productos disponibles")
                logger.info(f"📊 [UPLOAD] Resultado: {result}")

                # Actualizar estado: nomenclator listo, ahora procesar archivo
                upload_record.processing_notes = (
                    f"✅ Nomenclator cargado ({new_count:,} productos). Procesando archivo de ventas..."
                )
                db.commit()

            except Exception as e:
                logger.error(f"❌ [UPLOAD] Error cargando nomenclator: {e}")
                upload_record.status = UploadStatus.ERROR
                upload_record.error_message = f"No se pudo cargar nomenclator oficial: {str(e)}"
                upload_record.processing_completed_at = utc_now()
                db.commit()
                raise HTTPException(
                    status_code=503,
                    detail=f"Sistema no disponible: No se pudo cargar nomenclator oficial. Error: {str(e)}",
                )
        else:
            logger.info(f"✅ [UPLOAD] Nomenclator disponible: {nomenclator_count} productos")

        # Ya se leyó y validó el archivo anteriormente

        # Validar extensión
        logger.info("[UPLOAD] Validando extensión del archivo...")
        file_extension = Path(file.filename).suffix.lower()
        logger.info(f"[UPLOAD] Extensión detectada: {file_extension}")
        if file_extension not in [".csv", ".xlsx", ".xls"]:
            logger.error(f"[UPLOAD] Extensión no soportada: {file_extension}")
            raise HTTPException(
                status_code=400,
                detail=f"Formato de archivo no soportado: {file_extension}",
            )

        # ✅ ISSUE #476: Detección automática de tipo de archivo (ventas vs inventario)
        logger.info("[UPLOAD] Detectando tipo de archivo (ventas/inventario)...")

        # Crear archivo temporal en memoria para detectar tipo
        try:
            # Leer muestra del contenido
            if file_extension == ".csv":
                # Intentar diferentes encodings para CSV
                sample_df = None
                for encoding in ["utf-8", "latin1", "cp1252"]:
                    for sep in [";", ",", "\t"]:
                        try:
                            sample_df = pd.read_csv(
                                io.BytesIO(contents),
                                nrows=50,
                                sep=sep,
                                encoding=encoding,
                                on_bad_lines="skip",
                            )
                            if len(sample_df.columns) > 1:
                                break
                        except Exception:
                            continue
                    if sample_df is not None and len(sample_df.columns) > 1:
                        break
            else:
                # Excel
                sample_df = pd.read_excel(io.BytesIO(contents), nrows=50)

            if sample_df is not None and not sample_df.empty:
                content_type, detection_details = detect_file_content_type(sample_df)

                logger.info(
                    f"[UPLOAD] Tipo detectado: {content_type.value}",
                    extra=detection_details,
                )

                # Si es inventario, enrutar al procesador de inventario
                if content_type == FileContentType.INVENTORY:
                    logger.info("[UPLOAD] Detectado como INVENTARIO - enrutando a procesador de inventario")

                    # Determinar file_type para inventario
                    if erp_type and "farmatic" in erp_type.lower():
                        inv_file_type = FileType.INVENTORY_FARMATIC
                    else:
                        inv_file_type = FileType.INVENTORY_FARMANAGER

                    # Crear registro de upload para inventario
                    if "upload_record" not in locals():
                        upload_record = FileUpload(
                            pharmacy_id=pharmacy_id,
                            filename=file.filename,
                            file_path="",
                            file_size=file_size,
                            file_type=inv_file_type,
                            status=UploadStatus.QUEUED,
                            processing_notes="📦 Inventario detectado automáticamente. Procesando...",
                        )
                        db.add(upload_record)
                        db.commit()
                        db.refresh(upload_record)

                    # Preparar ruta del archivo
                    file_id = str(uuid.uuid4())
                    inv_file_path = UPLOAD_DIR / f"inventory_{file_id}{file_extension}"

                    # Lanzar procesamiento de inventario
                    inventory_service = get_inventory_processing_service()
                    processing_thread = Thread(
                        target=inventory_service.process_inventory_in_thread,
                        args=(
                            str(upload_record.id),
                            str(inv_file_path),
                            contents,
                            pharmacy_id,
                            erp_type,
                            None,  # snapshot_date: usar fecha actual
                        ),
                        name=f"inventory_upload_{upload_record.id}",
                        daemon=False,
                    )
                    processing_thread.start()

                    return {
                        "message": "📦 Inventario detectado automáticamente. Procesando...",
                        "upload_id": str(upload_record.id),
                        "filename": file.filename,
                        "size": file_size,
                        "status": "queued",
                        "detected_type": "inventory",
                        "detection_confidence": detection_details.get("confidence", "medium"),
                    }

                # Si es ventas, continuar con el flujo normal
                logger.info("[UPLOAD] Detectado como VENTAS - continuando flujo normal")

        except Exception as e:
            logger.warning(f"[UPLOAD] Error en detección automática: {e} - continuando como ventas")
            # En caso de error, asumir ventas (comportamiento anterior)

        # ✅ FIX ISSUE #274: Crear registro PRIMERO para retornar upload_id inmediatamente
        # Esto previene timeout del frontend (60s) con archivos grandes
        if "upload_record" not in locals():
            logger.info("[UPLOAD] Creando registro en base de datos (ANTES de guardar archivo)...")
            upload_record = FileUpload(
                pharmacy_id=pharmacy_id,
                filename=file.filename,
                file_path="",  # Se actualizará en background
                file_size=file_size,
                file_type=FileType.UNKNOWN,
                status=UploadStatus.PENDING,
                processing_notes="Guardando archivo en servidor...",
            )

            db.add(upload_record)
            db.commit()
            db.refresh(upload_record)
            logger.info(f"[UPLOAD] Registro creado: {upload_record.id}")
        else:
            # Upload record ya creado durante descarga de nomenclator
            logger.info(f"[UPLOAD] Usando registro existente de nomenclator: {upload_record.id}")
            upload_record.processing_notes = "✅ Nomenclator listo. Guardando archivo..."
            db.commit()

        # Guardar file_path para background processing
        file_id = str(uuid.uuid4())
        file_path = UPLOAD_DIR / f"{file_id}{file_extension}"

        # ✅ FIX ISSUE #332: Usar threading en lugar de background_tasks
        # Threading permite que el procesamiento sobreviva al ciclo de vida del HTTP request
        # Crítico para Render donde requests tienen timeout de 60s
        from threading import Thread
        from app.services.file_processing_service import file_processing_service

        # Actualizar status a QUEUED antes de lanzar thread
        upload_record.status = UploadStatus.QUEUED
        upload_record.processing_notes = "Archivo en cola para procesamiento..."
        db.commit()

        logger.info("[UPLOAD] Lanzando procesamiento en thread separado (Issue #332)...")
        processing_thread = Thread(
            target=file_processing_service.process_file_in_thread,
            args=(
                str(upload_record.id),
                str(file_path),
                contents,  # Pasar contenidos ya leídos
                pharmacy_id,
                erp_type,
            ),
            name=f"upload_processor_{upload_record.id}",
            daemon=False,  # ✅ CRITICAL: NO daemon para sobrevivir al request
        )
        processing_thread.start()

        # ✅ FIX ISSUE #332: Retornar INMEDIATAMENTE con status QUEUED
        logger.info("upload.queued", upload_id=str(upload_record.id), filename=file.filename)
        return {
            "message": "Archivo en cola para procesamiento. Use /status/{upload_id} para monitorear progreso.",
            "upload_id": str(upload_record.id),
            "filename": file.filename,
            "size": file_size,
            "status": "queued",  # Status inicial: queued (Issue #332)
        }

    except HTTPException:
        # Re-raise HTTP exceptions as-is
        raise
    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "upload_file", "pharmacy_id": pharmacy_id})
    except InvalidERPFormatError as e:
        raise handle_exception(e, {"endpoint": "upload_file", "filename": file.filename})
    except Exception as e:
        import traceback

        logger.error(
            "upload.unexpected_error",
            error=str(e),
            error_type=type(e).__name__,
            stack_trace=traceback.format_exc(),
        )
        raise handle_exception(e, {"endpoint": "upload_file", "pharmacy_id": pharmacy_id, "filename": file.filename})


async def process_file_with_save(
    upload_id: str,
    file_path: str,
    contents: bytes,
    pharmacy_id: str,
    erp_type: Optional[str],
):
    """
    Guarda archivo en disco y luego lo procesa en background.

    FIX ISSUE #274: Esta función wrapper previene timeout del frontend al separar:
    1. Guardado de archivo (I/O)
    2. Procesamiento (CPU-intensive)

    CRITICAL FIX: Usa async_db_session() para sesión independiente del HTTP request.
    Esto previene errores de sesión cerrada cuando el background task continúa
    después de que FastAPI cierra el request.

    Args:
        upload_id: ID del upload record
        file_path: Ruta donde guardar el archivo
        contents: Contenidos del archivo ya leídos
        pharmacy_id: ID de la farmacia
        erp_type: Tipo de ERP (opcional)
    """
    # ✅ CRITICAL: Crear sesión independiente para background task
    async with async_db_session() as db:
        upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
        if not upload:
            logger.error("upload.not_found", upload_id=upload_id)
            return

        try:
            # FASE 1: Guardar archivo en disco
            logger.info(f"[SAVE] Guardando archivo en disco: {file_path}")
            upload.processing_notes = "Guardando archivo en servidor..."
            db.commit()

            with open(file_path, "wb") as f:
                f.write(contents)

            logger.info(f"[SAVE] Archivo guardado exitosamente: {file_path}")

            # Actualizar file_path en DB
            upload.file_path = str(file_path)
            upload.processing_notes = "Archivo guardado. Iniciando procesamiento..."
            db.commit()

        except Exception as e:
            logger.error(f"[SAVE] Error guardando archivo: {str(e)}")
            upload.status = UploadStatus.ERROR
            upload.error_message = f"Error guardando archivo: {str(e)}"
            upload.processing_completed_at = utc_now()
            db.commit()
            return

    # FASE 2: Procesar archivo (delegado a función original con su propia sesión)
    await process_file(upload_id, file_path, pharmacy_id, erp_type)


async def process_file(
    upload_id: str,
    file_path: str,
    pharmacy_id: str,
    erp_type: Optional[str],
):
    """
    Procesa el archivo cargado en background con enriquecimiento automático asíncrono.

    FIX PR #284 Code Review (Session-Per-Batch Pattern):
    - PHASE 1: Parse CSV (no DB session needed)
    - PHASE 2: Load duplicates with dedicated short-lived session
    - PHASE 3: Process each batch with its own session
    - PHASE 4: Final stats update with dedicated session

    Benefits:
    - Reduces connection pool pressure in Render
    - Better error recovery (failed batch doesn't invalidate entire upload)
    - Scales to high-concurrency scenarios

    CRITICAL FIX Issue #274: Independent sessions prevent timeout errors.
    """
    start_time = time.time()
    logger.info(
        "[UPLOAD] Iniciando procesamiento asíncrono",
        extra={"upload_id": upload_id, "pharmacy_id": pharmacy_id, "erp_type": erp_type, "start_time": start_time},
    )

    # ========================================================================
    # PHASE 1: Parse CSV (NO DATABASE SESSION NEEDED)
    # ========================================================================
    try:
        # ✅ ISSUE #329: Actualizar estado a PARSING antes de parsear
        async with async_db_session() as db:
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if upload:
                upload.status = UploadStatus.PARSING
                upload.processing_started_at = utc_now()
                upload.processing_notes = "Parseando archivo ERP..."
                db.commit()

        # Parse file without holding DB connection
        df, stats = ParserFactory.parse_file(file_path, pharmacy_id, upload_id, erp_type)

        if df.empty:
            # Update upload with error using dedicated session
            async with async_db_session() as db:
                upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
                if upload:
                    upload.status = UploadStatus.ERROR
                    upload.error_message = "No se pudieron procesar datos del archivo"
                    upload.processing_completed_at = utc_now()
                    db.commit()
            return

        # ✅ ISSUE #329: Actualizar a VALIDATING después de parsear exitosamente
        async with async_db_session() as db:
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if not upload:
                logger.error("upload.not_found", upload_id=upload_id)
                return

            upload.status = UploadStatus.VALIDATING
            upload.processing_notes = f"Archivo parseado. Validando {len(df):,} registros..."
            upload.rows_total = len(df)
            upload.rows_processed = 0
            db.commit()

    except Exception as e:
        logger.error(f"[ERROR] Error parseando archivo: {str(e)}")
        async with async_db_session() as db:
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if upload:
                upload.status = UploadStatus.ERROR
                upload.error_message = f"Error parseando archivo: {str(e)}"
                upload.processing_completed_at = utc_now()
                db.commit()
        return

    # ========================================================================
    # PHASE 2: Load Existing Sales for Duplicate Detection (DEDICATED SESSION)
    # ========================================================================
    min_date = df["sale_date"].min()
    max_date = df["sale_date"].max()

    try:
        async with async_db_session() as db:
            # ✅ ISSUE #329: Actualizar estado a VALIDATING durante carga de duplicados
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if upload:
                upload.processing_notes = "Obteniendo ventana temporal de ventas existentes..."
                db.commit()

            # ✅ NUEVA LÓGICA (Issue #330): Obtener ventana temporal en lugar de set completo
            existing_sales_window = load_existing_sales_for_duplicate_detection(db, pharmacy_id, min_date, max_date)
    except Exception as e:
        logger.error(f"[ERROR] Error cargando duplicados: {str(e)}")
        async with async_db_session() as db:
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if upload:
                upload.status = UploadStatus.ERROR
                upload.error_message = f"Error cargando duplicados: {str(e)}"
                upload.processing_completed_at = utc_now()
                db.commit()
        return

    # ========================================================================
    # PHASE 3: Process Batches with Independent Sessions
    # ========================================================================
    records_saved = 0
    records_duplicated = 0
    batch_size = 500

    # ✅ ISSUE #329: Actualizar estado a SAVING antes de procesar batches
    async with async_db_session() as db:
        upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
        if upload:
            upload.status = UploadStatus.SAVING
            upload.processing_notes = f"Guardando {len(df):,} registros en base de datos..."
            db.commit()

    for idx in range(0, len(df), batch_size):
        batch_df = df.iloc[idx : idx + batch_size]

        # ✅ FIX #2 (Code Review): New session per batch
        async with async_db_session() as db:
            try:
                # REFACTORED (Issue #325): Process batch using helper function
                batch_records, batch_duplicates = process_sales_batch(
                    db, batch_df, existing_sales_window, pharmacy_id, upload_id
                )

                records_duplicated += batch_duplicates

                # REFACTORED (Issue #325): Save batch using helper function
                if batch_records:
                    saved_in_batch = save_batch_with_progress(
                        db=db,
                        sales_batch=batch_records,
                        upload_id=upload_id,
                        processed_count=idx + len(batch_df),
                        total_count=len(df),
                        records_saved_so_far=records_saved,
                        records_duplicated_so_far=records_duplicated
                    )
                    records_saved += saved_in_batch

            except Exception as e:
                logger.error(f"[BATCH] Error procesando batch: {e}")
                # Continue with next batch even if this one fails

    # ========================================================================
    # PHASE 4: Final Stats Update (DEDICATED SESSION)
    # ========================================================================
    logger.info(
        f"[INFO] Guardado completado: {records_saved} nuevos registros guardados, "
        f"{records_duplicated} duplicados omitidos (entre cargas)"
    )

    try:
        async with async_db_session() as db:
            upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
            if not upload:
                logger.error("upload.not_found", upload_id=upload_id)
                return

            # ✅ ISSUE #329: Transición a PROCESSING para enriquecimiento
            # (Si hay registros para enriquecer, si no → COMPLETED)
            if records_saved > 0:
                upload.status = UploadStatus.PROCESSING
                upload.processing_notes = (
                    f"Datos guardados ({records_saved:,} nuevos, {records_duplicated:,} duplicados). "
                    f"Iniciando enriquecimiento con CIMA/nomenclator..."
                )
            else:
                # Sin registros nuevos → marcar como completado
                upload.status = UploadStatus.COMPLETED
                upload.processing_notes = (
                    f"Sin datos nuevos. Todos los {records_duplicated} registros ya existían en la base de datos."
                )
                upload.processing_completed_at = utc_now()

            upload.rows_total = stats.get("total_rows", 0)
            upload.rows_processed = records_saved + records_duplicated
            upload.rows_with_errors = stats.get("error_rows", 0)
            upload.rows_duplicates = records_duplicated  # Issue #330: Guardar contador de duplicados

            # Detectar tipo de ERP si se pudo
            if "erp_type" in df.columns and not df["erp_type"].empty:
                detected_type = df["erp_type"].iloc[0]
                if detected_type == "farmatic":
                    upload.file_type = FileType.FARMATIC
                elif detected_type == "farmanager":
                    upload.file_type = FileType.FARMANAGER

            db.commit()
            logger.info(f"[OK] Archivo procesado: {records_saved} registros guardados")

    except Exception as e:
        logger.error(f"[ERROR] Error actualizando stats finales: {str(e)}")
        return

    # *** ENRIQUECIMIENTO AUTOMÁTICO ASÍNCRONO ***
    if records_saved > 0:
        enrichment_task = asyncio.create_task(
            _run_async_enrichment_pipeline(
                upload_id=upload_id, pharmacy_id=pharmacy_id, records_count=records_saved, start_time=start_time
            ),
            name=f"enrichment_pipeline_{upload_id[:8]}",
        )

        enrichment_task.add_done_callback(handle_task_exception)
        background_tasks.add(enrichment_task)

        logger.info(
            "[ENRICHMENT] Pipeline asíncrono iniciado en background",
            extra={
                "upload_id": upload_id,
                "records_saved": records_saved,
                "task_name": enrichment_task.get_name(),
                "processing_time_seconds": round(time.time() - start_time, 2),
            },
        )


@router.get("/status/{upload_id}")
async def get_upload_status(
    upload_id: str,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Obtiene el estado completo de un upload, incluyendo progreso de enriquecimiento.

    RESUELVE ISSUE #252: Enriquecimiento automático sin feedback de progreso
    RESUELVE ISSUE #274: Feedback de progreso en tiempo real (campo percentage)
    SECURITY (REGLA #7): Valida acceso cross-pharmacy según REGLA #10 (relación 1:1 User-Pharmacy)

    Returns:
        - Estado del upload (pending/processing/completed/error)
        - Progreso del enriquecimiento automático
        - Estadísticas detalladas de productos procesados
        - percentage: Porcentaje de progreso (0-100) basado en estado y registros
    """
    upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()

    if not upload:
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # ✅ SECURITY: Validar que el usuario tiene acceso a este upload (REGLA #7 + REGLA #10)
    if upload.pharmacy_id != current_user.pharmacy_id:
        # Usuario intentando acceder a upload de otra farmacia
        logger.warning(
            "[SECURITY] Intento de acceso no autorizado a upload",
            extra={
                "user_id": str(current_user.id),
                "user_pharmacy_id": str(current_user.pharmacy_id),
                "upload_id": upload_id,
                "upload_pharmacy_id": str(upload.pharmacy_id),
            },
        )
        # Devolver 404 en lugar de 403 para no revelar existencia del upload
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # Obtener estado y progreso del enriquecimiento
    enrichment_status = get_enrichment_status(db, upload_id)
    enrichment_progress = get_enrichment_progress_summary(db, upload_id)

    return {
        "upload_id": str(upload.id),
        "filename": upload.filename,
        "status": upload.status.value,
        "rows_total": upload.rows_total,
        "rows_processed": upload.rows_processed,
        "rows_with_errors": upload.rows_with_errors,
        "error_message": upload.error_message,
        "processing_notes": upload.processing_notes,  # ✅ NUEVO: Notas de progreso
        "percentage": calculate_upload_percentage(upload),  # ✅ NUEVO (Issue #274): Porcentaje calculado
        "uploaded_at": upload.uploaded_at,
        "processing_started_at": upload.processing_started_at,
        "processing_completed_at": upload.processing_completed_at,
        # ✅ NUEVO: Estado y progreso del enriquecimiento automático
        "enrichment_status": enrichment_status,
        "enrichment_progress": enrichment_progress,
    }


@router.get("/history")
async def get_upload_history(
    pharmacy_id: Optional[str] = None,  # Auto-detect from available pharmacies
    limit: int = 10,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Obtiene el historial de uploads de una farmacia con estadísticas detalladas.

    Incluye: fechas de ventas, totales de ventas, segregación prescripción/venta libre.
    Issue #330: Información completa para auditoría farmacéutica.

    OPTIMIZACIÓN: Consolidado de N+1 queries a 2 queries para mejor rendimiento.
    - Query 1: Obtiene uploads básicos
    - Query 2: Estadísticas agregadas para TODOS los uploads a la vez
    """
    from sqlalchemy import case

    from app.models.product_catalog import ProductCatalog

    # Auto-detect pharmacy_id si no se proporciona
    if pharmacy_id is None:
        pharmacy_id = await _get_default_pharmacy_id(db)

    # Query 1: Obtener uploads básicos (query simple, muy rápida)
    uploads = (
        db.query(FileUpload)
        .filter(FileUpload.pharmacy_id == pharmacy_id)
        .order_by(FileUpload.uploaded_at.desc())
        .limit(limit)
        .all()
    )

    if not uploads:
        return []

    # Obtener IDs de uploads para query de estadísticas
    upload_ids = [u.id for u in uploads]

    # Query 2: Estadísticas agregadas para TODOS los uploads en UNA SOLA query
    # Clasificación basada en xfarma_prescription_category:
    # - PRESCRIPCIÓN: xfarma_prescription_category IS NOT NULL
    # - LIBRE/OTC: xfarma_prescription_category IS NULL
    stats_query = db.query(
        SalesData.upload_id,
        func.min(SalesData.sale_date).label('fecha_desde'),
        func.max(SalesData.sale_date).label('fecha_hasta'),
        func.sum(SalesData.total_amount).label('total_ventas'),
        func.sum(
            case(
                (ProductCatalog.xfarma_prescription_category.isnot(None), SalesData.total_amount),
                else_=0
            )
        ).label('ventas_prescripcion'),
        func.sum(
            case(
                (ProductCatalog.xfarma_prescription_category.is_(None), SalesData.total_amount),
                else_=0
            )
        ).label('ventas_libre')
    ).select_from(SalesData)\
     .outerjoin(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)\
     .outerjoin(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)\
     .filter(SalesData.upload_id.in_(upload_ids))\
     .group_by(SalesData.upload_id)\
     .all()

    # Crear diccionario de estadísticas por upload_id
    stats_by_upload = {
        str(s.upload_id): {
            'fecha_desde': s.fecha_desde,
            'fecha_hasta': s.fecha_hasta,
            'total_ventas': s.total_ventas,
            'ventas_prescripcion': s.ventas_prescripcion,
            'ventas_libre': s.ventas_libre,
        }
        for s in stats_query
    }

    # Construir resultado combinando uploads con estadísticas
    result = []
    for u in uploads:
        upload_id_str = str(u.id)
        stats = stats_by_upload.get(upload_id_str, {})

        result.append({
            "upload_id": upload_id_str,
            "filename": u.filename,
            "status": u.status.value,
            "file_type": u.file_type.value if u.file_type else "unknown",
            "rows_total": u.rows_total,
            "rows_processed": u.rows_processed,
            "rows_duplicates": u.rows_duplicates or 0,
            "uploaded_at": u.uploaded_at,
            "error_message": u.error_message,
            # Estadísticas (Issue #330)
            "fecha_desde": stats.get('fecha_desde').isoformat() if stats.get('fecha_desde') else None,
            "fecha_hasta": stats.get('fecha_hasta').isoformat() if stats.get('fecha_hasta') else None,
            "total_ventas": float(stats.get('total_ventas') or 0),
            "ventas_prescripcion": float(stats.get('ventas_prescripcion') or 0),
            "ventas_libre": float(stats.get('ventas_libre') or 0),
        })

    return result


@router.post("/cleanup/zombies")
async def cleanup_zombie_uploads(
    max_age_hours: float = 1.0,
    current_user: User = Depends(get_current_active_user),  # SECURITY: JWT authentication required
    db: Session = Depends(get_db),
):
    """
    Limpia archivos que quedaron en estados zombies por más del tiempo especificado.
    Por defecto, archivos en estos estados por más de 1 hora se consideran zombies.

    Tipos de zombies detectados:
    - PENDING: Archivos subidos pero que nunca empezaron a procesarse (usa uploaded_at)
    - PROCESSING: Archivos atascados durante el procesamiento de datos (usa processing_started_at)
    - SAVING: Archivos atascados durante el guardado a base de datos (usa processing_started_at)

    SECURITY (Issue #302):
    - Requiere autenticación JWT válida
    - Admin: puede limpiar zombies de todas las farmacias
    - Non-admin: solo puede limpiar zombies de su propia farmacia
    - Incluye validación de pharmacy_id en el loop para prevenir modificaciones accidentales
    - Registra operación en processing_notes para auditoría
    """
    try:
        # SECURITY: Verificar rol admin
        is_admin = current_user.role == "admin" or getattr(current_user, "is_superuser", False)

        # Calcular timestamp límite
        cutoff_time = utc_now() - timedelta(hours=max_age_hours)

        # SECURITY: Query con scope de farmacia
        # Detectar zombies en PENDING, PROCESSING y SAVING
        query = db.query(FileUpload).filter(
            or_(
                FileUpload.status == UploadStatus.PENDING,
                FileUpload.status == UploadStatus.PROCESSING,
                FileUpload.status == UploadStatus.SAVING
            ),
            or_(
                # PENDING usa uploaded_at (nunca inició procesamiento)
                and_(
                    FileUpload.status == UploadStatus.PENDING,
                    FileUpload.uploaded_at < cutoff_time
                ),
                # PROCESSING y SAVING usan processing_started_at
                and_(
                    FileUpload.status.in_([UploadStatus.PROCESSING, UploadStatus.SAVING]),
                    FileUpload.processing_started_at < cutoff_time
                )
            )
        )

        # SECURITY: Si no es admin, filtrar por pharmacy_id
        if not is_admin:
            if not current_user.pharmacy_id:
                raise HTTPException(
                    status_code=403,
                    detail="Usuario sin farmacia asignada no puede ejecutar cleanup de zombies"
                )
            query = query.filter(FileUpload.pharmacy_id == current_user.pharmacy_id)
            logger.info(f"[CLEANUP] Usuario no-admin {current_user.email} - scope limitado a pharmacy {current_user.pharmacy_id}")
        else:
            logger.info(f"[CLEANUP] Usuario admin {current_user.email} - scope global")

        zombie_uploads = query.all()

        zombies_cleaned = 0
        zombie_details = []

        for upload in zombie_uploads:
            # SECURITY: Double-check pharmacy_id (defense in depth)
            if not is_admin and upload.pharmacy_id != current_user.pharmacy_id:
                logger.error(
                    f"[CLEANUP] SECURITY VIOLATION: Non-admin user {current_user.email} "
                    f"attempted to clean upload from different pharmacy. "
                    f"User pharmacy: {current_user.pharmacy_id}, Upload pharmacy: {upload.pharmacy_id}"
                )
                continue  # Skip este upload sin modificarlo

            # Calcular tiempo en estado zombie
            # FIX: Convertir naive datetime de SQLAlchemy a aware para evitar TypeError
            # PENDING usa uploaded_at, PROCESSING/SAVING usan processing_started_at
            if upload.status == UploadStatus.PENDING:
                time_reference = make_aware(upload.uploaded_at)
            else:
                time_reference = make_aware(upload.processing_started_at)

            time_in_state = utc_now() - time_reference
            hours_in_state = time_in_state.total_seconds() / 3600

            # Guardar status original para logging
            original_status = upload.status.value

            # Marcar como error con explicación
            upload.status = UploadStatus.ERROR
            upload.error_message = f"Zombie en status '{original_status}' interrumpido después de {hours_in_state:.1f} horas. Posible timeout o interrupción del servidor."
            upload.processing_completed_at = utc_now()
            # AUDIT: Registrar quién ejecutó la limpieza
            upload.processing_notes = (
                f"Limpiado automáticamente como zombie (status: {original_status}) el {utc_now().isoformat()} "
                f"por usuario {current_user.email} ({'admin' if is_admin else 'user'})"
            )

            zombie_details.append(
                {
                    "upload_id": str(upload.id),
                    "filename": upload.filename,
                    "pharmacy_id": str(upload.pharmacy_id),
                    "original_status": original_status,
                    "time_in_state_hours": round(hours_in_state, 2),
                    "time_reference": time_reference.isoformat(),
                    "time_reference_field": "uploaded_at" if original_status == "pending" else "processing_started_at",
                }
            )

            zombies_cleaned += 1
            logger.warning(
                f"[CLEANUP] Archivo zombie limpiado: {upload.id} - {upload.filename} "
                f"(status: {original_status}, tiempo: {hours_in_state:.1f} horas) "
                f"por {current_user.email} ({'admin' if is_admin else 'user'})"
            )

        db.commit()

        return {
            "success": True,
            "zombies_cleaned": zombies_cleaned,
            "cutoff_time": cutoff_time.isoformat(),
            "max_age_hours": max_age_hours,
            "zombie_details": zombie_details,
            "message": f"Se limpiaron {zombies_cleaned} archivos zombies",
            "scope": "global" if is_admin else f"pharmacy_{current_user.pharmacy_id}"  # AUDIT: Indicar scope
        }

    except HTTPException:
        # Re-raise HTTPException sin modificar
        raise
    except Exception as e:
        logger.error(f"[CLEANUP] Error limpiando zombies: {str(e)}")
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Error limpiando archivos zombies: {str(e)}")


@router.get("/status/summary")
async def get_upload_status_summary(
    pharmacy_id: Optional[str] = None,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Obtiene un resumen del estado de todos los uploads de una farmacia,
    incluyendo detección de posibles zombies.
    """

    # Auto-detect pharmacy_id si no se proporciona
    if pharmacy_id is None:
        pharmacy_id = await _get_default_pharmacy_id(db)

    try:
        # Contar uploads por estado
        status_counts = (
            db.query(FileUpload.status, func.count(FileUpload.id).label("count"))
            .filter(FileUpload.pharmacy_id == pharmacy_id)
            .group_by(FileUpload.status)
            .all()
        )

        # Detectar posibles zombies (processing > 1 hora)
        one_hour_ago = utc_now() - timedelta(hours=1)
        potential_zombies = (
            db.query(FileUpload)
            .filter(
                FileUpload.pharmacy_id == pharmacy_id,
                FileUpload.status == UploadStatus.PROCESSING,
                FileUpload.processing_started_at < one_hour_ago,
            )
            .all()
        )

        zombie_info = []
        for zombie in potential_zombies:
            time_in_processing = utc_now() - zombie.processing_started_at
            zombie_info.append(
                {
                    "upload_id": str(zombie.id),
                    "filename": zombie.filename,
                    "hours_in_processing": round(time_in_processing.total_seconds() / 3600, 2),
                    "processing_started_at": zombie.processing_started_at.isoformat(),
                }
            )

        # Formatear respuesta
        summary = {
            "pharmacy_id": pharmacy_id,
            "status_counts": {status.value: count for status, count in status_counts},
            "potential_zombies": {
                "count": len(potential_zombies),
                "details": zombie_info,
            },
            "timestamp": utc_now().isoformat(),
        }

        return summary

    except Exception as e:
        logger.error(f"[STATUS_SUMMARY] Error: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Error obteniendo resumen de estado: {str(e)}")


@router.get("/enrichment/progress")
async def get_enrichment_progress(
    pharmacy_id: Optional[str] = None,  # Auto-detect from available pharmacies
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Obtiene el progreso del enriquecimiento de datos para una farmacia
    """

    # Auto-detect pharmacy_id si no se proporciona
    if pharmacy_id is None:
        pharmacy_id = await _get_default_pharmacy_id(db)

    try:
        progress = enrichment_service.get_enrichment_progress(db, pharmacy_id)
        return {"success": True, "data": progress}
    except Exception as e:
        logger.error(f"Error obteniendo progreso de enriquecimiento: {str(e)}")
        raise HTTPException(
            status_code=500,
            detail=f"Error obteniendo progreso de enriquecimiento: {str(e)}",
        )


@router.get("/tasks/status")
async def get_background_tasks_status(
    current_user: User = Depends(get_current_active_user),
):
    """
    Monitorea el estado de los background tasks activos

    NUEVO ENDPOINT: Para monitorear tasks asíncronos y detectar problemas
    """
    try:
        active_tasks = []

        for task in background_tasks.copy():  # Copy para evitar modificaciones concurrentes
            if not task.done():
                task_info = {
                    "name": task.get_name(),
                    "created_at": getattr(task, "_created_at", None),
                    "state": "running",
                }
                active_tasks.append(task_info)

        # Cleanup de tasks completados
        background_tasks.difference_update({t for t in background_tasks if t.done()})

        return {
            "active_tasks_count": len(active_tasks),
            "active_tasks": active_tasks,
            "total_tracked_tasks": len(background_tasks),
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error("[TASKS_STATUS] Error obteniendo estado de tasks", extra={"error": str(e)})
        raise HTTPException(status_code=500, detail=f"Error obteniendo estado de background tasks: {str(e)}")


@router.post("/tasks/cancel")
async def cancel_background_tasks(
    task_name_pattern: Optional[str] = None,
    cancel_all: bool = False,
    current_user: User = Depends(require_permissions("admin")),
):
    """
    Cancela background tasks activos

    NUEVO ENDPOINT: Para cleanup de emergencia de tasks problemáticos

    Args:
        task_name_pattern: Patrón para filtrar tasks por nombre
        cancel_all: Si True, cancela TODOS los tasks activos
    """
    try:
        cancelled_tasks = []
        tasks_to_cancel = []

        # Seleccionar tasks a cancelar
        for task in background_tasks.copy():
            if not task.done():
                task_name = task.get_name()

                if cancel_all:
                    tasks_to_cancel.append(task)
                elif task_name_pattern and task_name_pattern in task_name:
                    tasks_to_cancel.append(task)

        # Cancelar tasks seleccionados
        for task in tasks_to_cancel:
            try:
                task.cancel()
                cancelled_tasks.append({"name": task.get_name(), "cancelled": True})
                logger.warning(
                    f"[TASK_CANCEL] Task cancelado: {task.get_name()}",
                    extra={"task_name": task.get_name(), "reason": "manual_cancellation"},
                )
            except Exception as e:
                cancelled_tasks.append({"name": task.get_name(), "cancelled": False, "error": str(e)})

        # Cleanup
        background_tasks.difference_update(tasks_to_cancel)

        return {
            "cancelled_count": len([t for t in cancelled_tasks if t.get("cancelled")]),
            "failed_cancellations": len([t for t in cancelled_tasks if not t.get("cancelled")]),
            "cancelled_tasks": cancelled_tasks,
            "remaining_active_tasks": len([t for t in background_tasks if not t.done()]),
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error("[TASK_CANCEL] Error cancelando tasks", extra={"error": str(e)})
        raise HTTPException(status_code=500, detail=f"Error cancelando background tasks: {str(e)}")


@router.post("/enrichment/trigger")
async def trigger_enrichment(
    pharmacy_id: Optional[str] = None,  # Auto-detect from available pharmacies
    upload_id: Optional[str] = None,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Dispara manualmente el proceso de enriquecimiento para una farmacia.

    ISSUE #332: Usa threading para sobrevivir a timeout de Render.
    """

    # Auto-detect pharmacy_id si no se proporciona
    if pharmacy_id is None:
        pharmacy_id = await _get_default_pharmacy_id(db)

    try:
        # Verificar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(status_code=404, detail="Farmacia no encontrada")

        # Ejecutar enriquecimiento en thread separado (Issue #332)
        from threading import Thread

        enrichment_thread = Thread(
            target=_trigger_enrichment_background,
            args=(pharmacy_id, upload_id, db),
            name=f"enrichment_trigger_{pharmacy_id}",
            daemon=False,
        )
        enrichment_thread.start()

        return {
            "message": "Enriquecimiento iniciado en segundo plano",
            "pharmacy_id": pharmacy_id,
            "upload_id": upload_id,
            "status": "processing",
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error disparando enriquecimiento: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Error disparando enriquecimiento: {str(e)}")


async def _get_default_pharmacy_id(db: Session) -> str:
    """
    Obtener pharmacy_id por defecto cuando no se proporciona.

    Estrategia:
    1. Buscar farmacia con más datos de ventas
    2. Si no hay datos, usar la primera farmacia disponible
    3. Fallback al ID hardcodeado como último recurso

    Returns:
        pharmacy_id de la farmacia por defecto
    """
    try:
        # Buscar farmacia con más datos de ventas
        pharmacy_with_data = (
            db.query(Pharmacy)
            .join(SalesData, Pharmacy.id == SalesData.pharmacy_id)
            .group_by(Pharmacy.id, Pharmacy.name)
            .order_by(db.func.count(SalesData.id).desc())
            .first()
        )

        if pharmacy_with_data:
            logger.info(f"Using pharmacy with data: {pharmacy_with_data.name} ({pharmacy_with_data.id})")
            return str(pharmacy_with_data.id)

        # Si no hay datos, usar primera farmacia disponible
        first_pharmacy = db.query(Pharmacy).first()
        if first_pharmacy:
            logger.info(f"Using first available pharmacy: {first_pharmacy.name} ({first_pharmacy.id})")
            return str(first_pharmacy.id)

        # Último recurso: ID hardcodeado
        fallback_id = "11111111-1111-1111-1111-111111111111"
        logger.warning(f"No pharmacies found, using fallback ID: {fallback_id}")
        return fallback_id

    except Exception as e:
        logger.error(f"Error getting default pharmacy_id: {e}")
        return "11111111-1111-1111-1111-111111111111"


async def _run_async_enrichment_pipeline(upload_id: str, pharmacy_id: str, records_count: int, start_time: float):
    """
    Ejecuta el pipeline completo de enriquecimiento automático de forma asíncrona.

    RESUELVE ISSUES CRÍTICOS:
    - Database Session Management: Usa async_db_session() thread-safe
    - Timeout Protection: 5 minutos máximo para prevenir resource exhaustion
    - Proper Error Handling: Structured logging y cleanup garantizado

    FLUJO ASÍNCRONO:
    1. Enriquecimiento de datos (CIMA + nomenclator)
    2. Inicialización de partners (paralelo)
    3. Logging estructurado detallado
    4. Manejo robusto de errores
    """
    pipeline_start = time.time()

    logger.info(
        "[ASYNC_ENRICHMENT] Pipeline iniciado",
        extra={
            "upload_id": upload_id,
            "pharmacy_id": pharmacy_id,
            "records_count": records_count,
            "upload_processing_time": round(time.time() - start_time, 2),
            "pipeline_start_time": pipeline_start,
        },
    )

    try:
        # FIX Issue #253: Aumentar timeout de 5 a 30 minutos para archivos grandes
        async with asyncio.timeout(1800):  # 30 minutos timeout
            # RESUELVE ISSUE CRÍTICO: Thread-safe DB session management
            async with async_db_session() as db:
                # TASK 1: Enriquecimiento de datos (asíncrono)
                enrichment_result = await _async_enrichment_wrapper(db, pharmacy_id, upload_id, records_count)

                # TASK 2: Inicialización de partners (asíncrono, independiente)
                partners_result = await _async_partners_wrapper(db, pharmacy_id, upload_id)

                # Actualizar upload con resultados del pipeline
                _update_upload_with_pipeline_results(db, upload_id, enrichment_result, partners_result, pipeline_start)

                # FIX Issue #253: Retry automático si tasa de enriquecimiento < 90%
                if enrichment_result.get("success") and enrichment_result.get("stats"):
                    stats = enrichment_result["stats"]
                    processed = stats.get("processed", 0)
                    enriched = stats.get("enriched", 0)

                    if processed > 0:
                        enrichment_rate = enriched / processed

                        if enrichment_rate < 0.90:
                            logger.warning(
                                f"⚠️ [AUTO_RETRY] Tasa de enriquecimiento baja ({enrichment_rate*100:.1f}%) - "
                                f"Iniciando retry automático...",
                                extra={
                                    "upload_id": upload_id,
                                    "pharmacy_id": pharmacy_id,
                                    "processed": processed,
                                    "enriched": enriched,
                                    "enrichment_rate": enrichment_rate,
                                },
                            )

                            # Ejecutar retry en mismo thread pool
                            loop = asyncio.get_event_loop()
                            retry_stats = await loop.run_in_executor(
                                None, enrichment_service.enrich_sales_batch, db, pharmacy_id, upload_id
                            )

                            logger.info(
                                f"✅ [AUTO_RETRY] Completado - Nuevos enriquecidos: {retry_stats.get('enriched', 0)}",
                                extra={"upload_id": upload_id, "retry_stats": retry_stats},
                            )

        total_pipeline_time = time.time() - pipeline_start

        # Issue #443: Invalidar cache de treemap para esta farmacia
        # (los datos de ventas cambiaron, el treemap debe recalcularse)
        try:
            from app.services.partner_analysis_service import partner_analysis_service
            invalidated = await partner_analysis_service.invalidate_treemap_cache(uuid.UUID(pharmacy_id))
            if invalidated > 0:
                logger.info(
                    f"[TREEMAP_CACHE] Invalidated {invalidated} cache entries after upload",
                    extra={"upload_id": upload_id, "pharmacy_id": pharmacy_id},
                )
        except Exception as cache_error:
            logger.debug(f"[TREEMAP_CACHE] Cache invalidation skipped: {cache_error}")

        logger.info(
            "[ASYNC_ENRICHMENT] Pipeline completado exitosamente",
            extra={
                "upload_id": upload_id,
                "total_pipeline_time_seconds": round(total_pipeline_time, 2),
                "enrichment_stats": enrichment_result.get("stats"),
                "partners_initialized": partners_result.get("count", 0),
                "success": True,
            },
        )

    except asyncio.TimeoutError:
        logger.error(
            "[ASYNC_ENRICHMENT] Pipeline timeout después de 30 minutos",
            extra={
                "upload_id": upload_id,
                "pipeline_time_seconds": round(time.time() - pipeline_start, 2),
                "timeout_seconds": 1800,
            },
        )
        # Actualizar upload con error de timeout
        async with async_db_session() as db:
            _update_upload_with_error(
                db, upload_id, "Pipeline timeout después de 30 minutos - operación demasiado lenta"
            )

    except Exception as e:
        logger.error(
            "[ASYNC_ENRICHMENT] Error en pipeline asíncrono",
            extra={
                "upload_id": upload_id,
                "error": str(e),
                "error_type": type(e).__name__,
                "pipeline_time_seconds": round(time.time() - pipeline_start, 2),
            },
        )
        # Actualizar upload con error
        try:
            async with async_db_session() as db:
                _update_upload_with_error(db, upload_id, str(e))
        except Exception as db_error:
            logger.error(
                "[ASYNC_ENRICHMENT] Error adicional actualizando DB",
                extra={"upload_id": upload_id, "original_error": str(e), "db_error": str(db_error)},
            )


async def _async_enrichment_wrapper(db: Session, pharmacy_id: str, upload_id: str, records_count: int) -> dict:
    """Wrapper asíncrono para el servicio de enriquecimiento"""
    start_time = time.time()

    logger.info(
        "[ENRICHMENT_TASK] Iniciando enriquecimiento", extra={"upload_id": upload_id, "records_count": records_count}
    )

    try:
        # Ejecutar enriquecimiento en thread pool para no bloquear event loop
        loop = asyncio.get_event_loop()
        stats = await loop.run_in_executor(None, enrichment_service.enrich_sales_batch, db, pharmacy_id, upload_id)

        processing_time = time.time() - start_time
        logger.info(
            "[ENRICHMENT_TASK] Completado",
            extra={"upload_id": upload_id, "stats": stats, "processing_time_seconds": round(processing_time, 2)},
        )

        return {"success": True, "stats": stats, "processing_time": processing_time}

    except AttributeError as e:
        # ERROR CRÍTICO: Método no encontrado en el servicio de enrichment
        processing_time = time.time() - start_time
        logger.error(
            "[ENRICHMENT_TASK] ERROR CRÍTICO: Método no encontrado en EnrichmentService",
            extra={
                "upload_id": upload_id,
                "pharmacy_id": pharmacy_id,
                "error_type": "AttributeError",
                "error": str(e),
                "service": "EnrichmentService",
                "processing_time_seconds": round(processing_time, 2),
                "severity": "critical",
            },
        )
        return {
            "success": False,
            "error": f"Método no encontrado: {str(e)}",
            "processing_time": processing_time,
            "error_type": "method_not_found",
        }

    except Exception as e:
        processing_time = time.time() - start_time
        logger.error(
            "[ENRICHMENT_TASK] Error inesperado en enriquecimiento",
            extra={
                "upload_id": upload_id,
                "pharmacy_id": pharmacy_id,
                "error_type": type(e).__name__,
                "error": str(e),
                "processing_time_seconds": round(processing_time, 2),
                "severity": "high",
            },
        )
        return {"success": False, "error": str(e), "processing_time": processing_time, "error_type": type(e).__name__}


async def _async_partners_wrapper(db: Session, pharmacy_id: str, upload_id: str) -> dict:
    """Wrapper asíncrono para la inicialización de partners"""
    start_time = time.time()

    logger.info(
        "[PARTNERS_TASK] Iniciando inicialización automática",
        extra={"upload_id": upload_id, "pharmacy_id": pharmacy_id},
    )

    try:
        # Import dinámico para evitar problemas de dependencias circulares
        from app.services.pharmacy_partners_service import PharmacyPartnersService

        # Ejecutar partners en thread pool
        loop = asyncio.get_event_loop()
        partners_service = PharmacyPartnersService()
        result = await loop.run_in_executor(None, partners_service.calculate_and_update_partners, db, pharmacy_id)

        processing_time = time.time() - start_time
        suggested_count = result.get("auto_suggested_count", 0)

        if result.get("success"):
            logger.info(
                "[PARTNERS_TASK] Completado exitosamente",
                extra={
                    "upload_id": upload_id,
                    "partners_initialized": suggested_count,
                    "processing_time_seconds": round(processing_time, 2),
                },
            )
        else:
            logger.warning(
                "[PARTNERS_TASK] No se pudieron inicializar",
                extra={
                    "upload_id": upload_id,
                    "message": result.get("message", "Unknown error"),
                    "processing_time_seconds": round(processing_time, 2),
                },
            )

        return {
            "success": result.get("success", False),
            "count": suggested_count,
            "message": result.get("message", ""),
            "processing_time": processing_time,
        }

    except AttributeError as e:
        # ERROR CRÍTICO: Método no encontrado en el servicio
        processing_time = time.time() - start_time
        logger.error(
            "[PARTNERS_TASK] ERROR CRÍTICO: Método no encontrado en PharmacyPartnersService",
            extra={
                "upload_id": upload_id,
                "pharmacy_id": pharmacy_id,
                "error_type": "AttributeError",
                "error": str(e),
                "service": "PharmacyPartnersService",
                "processing_time_seconds": round(processing_time, 2),
                "severity": "critical",
            },
        )
        # TODO: Enviar alerta a sistema de monitoring (Sentry, DataDog, etc.)
        return {
            "success": False,
            "error": f"Método no encontrado: {str(e)}",
            "processing_time": processing_time,
            "error_type": "method_not_found",
        }

    except Exception as e:
        processing_time = time.time() - start_time
        logger.error(
            "[PARTNERS_TASK] Error inesperado en cálculo de partners",
            extra={
                "upload_id": upload_id,
                "pharmacy_id": pharmacy_id,
                "error_type": type(e).__name__,
                "error": str(e),
                "processing_time_seconds": round(processing_time, 2),
                "severity": "high",
            },
        )
        return {"success": False, "error": str(e), "processing_time": processing_time, "error_type": type(e).__name__}


def _update_upload_with_pipeline_results(
    db: Session, upload_id: str, enrichment_result: dict, partners_result: dict, pipeline_start: float
):
    """
    Actualiza el upload con los resultados del pipeline asíncrono

    CORREGIDO: Función síncrona para usar con SQLAlchemy session manager
    """
    try:
        upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
        if not upload:
            logger.error(f"[PIPELINE_UPDATE] Upload {upload_id} no encontrado")
            return

        # Construir mensaje de estado del pipeline
        pipeline_time = round(time.time() - pipeline_start, 2)
        status_parts = []

        if enrichment_result.get("success"):
            stats = enrichment_result.get("stats", {})
            if isinstance(stats, dict):
                enriched_count = stats.get("enriched", "OK")
                status_parts.append(f"Enriquecimiento: {enriched_count} productos")
            else:
                status_parts.append(f"Enriquecimiento: {stats}")
        else:
            status_parts.append(f"Enriquecimiento: ERROR - {enrichment_result.get('error', 'Unknown')}")

        if partners_result.get("success"):
            status_parts.append(f"Partners: {partners_result.get('count', 0)} inicializados")
        else:
            status_parts.append(f"Partners: ERROR - {partners_result.get('error', 'Unknown')}")

        status_parts.append(f"Pipeline: {pipeline_time}s")

        # Actualizar upload
        current_message = upload.error_message or ""
        new_status = " | ".join(status_parts)
        upload.error_message = f"{current_message} | {new_status}" if current_message else new_status

        # ✅ CRÍTICO: Marcar upload como completado si pipeline tuvo éxito
        if enrichment_result.get("success") and partners_result.get("success"):
            upload.status = UploadStatus.COMPLETED
            upload.processing_completed_at = utc_now()
            upload.processing_notes = f"Pipeline completado exitosamente en {pipeline_time}s. " + (
                upload.processing_notes or ""
            )
        elif not enrichment_result.get("success"):
            # Pipeline falló por enriquecimiento
            upload.status = UploadStatus.ERROR
            upload.processing_completed_at = utc_now()
            upload.error_message = f"Error en enriquecimiento: {enrichment_result.get('error', 'Unknown')}"
        else:
            # Partners falló pero enriquecimiento OK → PARTIAL
            upload.status = UploadStatus.PARTIAL
            upload.processing_completed_at = utc_now()
            upload.error_message = f"Partners failed: {partners_result.get('error', 'Unknown')}"

        db.commit()  # ✅ CRÍTICO: Commitear cambios

        logger.info(
            "[PIPELINE_UPDATE] Upload actualizado con resultados del pipeline",
            extra={
                "upload_id": upload_id,
                "pipeline_time_seconds": pipeline_time,
                "enrichment_success": enrichment_result.get("success"),
                "partners_success": partners_result.get("success"),
                "final_status": upload.status.value,
            },
        )

    except Exception as e:
        logger.error("[PIPELINE_UPDATE] Error actualizando upload", extra={"upload_id": upload_id, "error": str(e)})


def _update_upload_with_error(db: Session, upload_id: str, error_msg: str):
    """
    Actualiza el upload con error del pipeline

    CORREGIDO: Función síncrona para usar con SQLAlchemy session manager
    """
    try:
        upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()
        if upload:
            current_message = upload.error_message or ""
            error_status = f"Pipeline ERROR: {error_msg}"
            upload.error_message = f"{current_message} | {error_status}" if current_message else error_status
            logger.warning(
                "[PIPELINE_ERROR_UPDATE] Upload marcado con error",
                extra={"upload_id": upload_id, "error_msg": error_msg},
            )
    except Exception as e:
        logger.error(
            "[PIPELINE_ERROR_UPDATE] Error actualizando con error", extra={"upload_id": upload_id, "error": str(e)}
        )


def _trigger_enrichment_background(pharmacy_id: str, upload_id: Optional[str], db: Session):
    """
    Ejecuta enriquecimiento en background (LEGACY - mantener para compatibilidad)
    """
    try:
        logger.info(f"[MANUAL_ENRICHMENT] Iniciando enriquecimiento manual para farmacia {pharmacy_id}")

        stats = enrichment_service.enrich_sales_batch(db=db, pharmacy_id=pharmacy_id, upload_id=upload_id)

        logger.info(f"[MANUAL_ENRICHMENT] Completado - {stats}")

    except Exception as e:
        logger.error(f"[MANUAL_ENRICHMENT] Error: {str(e)}")
        raise


# =============================================================================
# DELETE UPLOAD ENDPOINTS (Issue: Individual upload deletion)
# =============================================================================


@router.get("/{upload_id}/delete-preview")
async def preview_upload_deletion(
    upload_id: str,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Preview de lo que se eliminará si se borra este upload.

    Retorna estadísticas sin realizar ninguna modificación:
    - Nombre del archivo
    - Número de registros de ventas
    - Rango de fechas de los datos
    - Importe total de ventas

    SECURITY: Valida que el upload pertenece a la farmacia del usuario.
    """
    upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()

    if not upload:
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # SECURITY: Validar que el usuario tiene acceso (REGLA #7 + REGLA #10)
    if upload.pharmacy_id != current_user.pharmacy_id:
        logger.warning(
            "[SECURITY] Intento de preview no autorizado",
            extra={
                "user_id": str(current_user.id),
                "user_pharmacy_id": str(current_user.pharmacy_id),
                "upload_id": upload_id,
                "upload_pharmacy_id": str(upload.pharmacy_id),
            },
        )
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # Contar registros de ventas asociados
    sales_count = db.query(func.count(SalesData.id)).filter(SalesData.upload_id == upload_id).scalar() or 0

    # Contar registros de enriquecimiento asociados
    enrichment_count = (
        db.query(func.count(SalesEnrichment.id))
        .join(SalesData, SalesData.id == SalesEnrichment.sales_data_id)
        .filter(SalesData.upload_id == upload_id)
        .scalar()
        or 0
    )

    # Obtener rango de fechas y totales
    sales_stats = (
        db.query(
            func.min(SalesData.sale_date).label("fecha_desde"),
            func.max(SalesData.sale_date).label("fecha_hasta"),
            func.sum(SalesData.total_amount).label("total_amount"),
        )
        .filter(SalesData.upload_id == upload_id)
        .first()
    )

    return {
        "upload_id": str(upload.id),
        "filename": upload.filename,
        "uploaded_at": upload.uploaded_at.isoformat() if upload.uploaded_at else None,
        "sales_count": sales_count,
        "enrichment_count": enrichment_count,
        "date_range": {
            "from": sales_stats.fecha_desde.strftime("%d/%m/%Y") if sales_stats.fecha_desde else None,
            "to": sales_stats.fecha_hasta.strftime("%d/%m/%Y") if sales_stats.fecha_hasta else None,
        },
        "total_amount": float(sales_stats.total_amount) if sales_stats.total_amount else 0.0,
    }


@router.delete("/{upload_id}")
async def delete_upload(
    upload_id: str,
    batch_size: int = 5000,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Elimina un upload específico y todos los datos de ventas asociados.

    CHUNKED DELETE: Para datasets grandes (>5K registros), la eliminación
    se hace en batches para evitar timeouts. El frontend debe hacer polling
    mientras el status sea 'in_progress'.

    Args:
        upload_id: UUID del upload a eliminar
        batch_size: Número de registros a eliminar por batch (default: 5000)

    Returns:
        - status='in_progress': Hay más registros por eliminar, hacer otra llamada
        - status='completed': Todos los registros eliminados

    Esta operación es PERMANENTE y elimina:
    - El registro del upload (solo cuando status='completed')
    - Todos los datos de ventas de este upload
    - Todos los datos de enriquecimiento de las ventas afectadas

    SECURITY:
    - Requiere autenticación JWT
    - Usuario solo puede eliminar uploads de su propia farmacia

    AUDIT:
    - Registra la eliminación con conteos de registros afectados
    """
    upload = db.query(FileUpload).filter(FileUpload.id == upload_id).first()

    if not upload:
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # SECURITY: Validar que el usuario tiene acceso (REGLA #7 + REGLA #10)
    if upload.pharmacy_id != current_user.pharmacy_id:
        logger.warning(
            "[SECURITY] Intento de eliminación no autorizado",
            extra={
                "user_id": str(current_user.id),
                "user_pharmacy_id": str(current_user.pharmacy_id),
                "upload_id": upload_id,
                "upload_pharmacy_id": str(upload.pharmacy_id),
            },
        )
        raise HTTPException(status_code=404, detail="Upload no encontrado")

    # Guardar info para logs
    filename = upload.filename
    pharmacy_id = str(upload.pharmacy_id)

    try:
        # Contar registros restantes
        sales_remaining = (
            db.query(func.count(SalesData.id)).filter(SalesData.upload_id == upload_id).scalar() or 0
        )

        enrichment_remaining = (
            db.query(func.count(SalesEnrichment.id))
            .join(SalesData, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(SalesData.upload_id == upload_id)
            .scalar()
            or 0
        )

        logger.info(
            "[DELETE_UPLOAD] Estado actual",
            extra={
                "upload_id": upload_id,
                "sales_remaining": sales_remaining,
                "enrichment_remaining": enrichment_remaining,
                "batch_size": batch_size,
            },
        )

        deleted_enrichments = 0
        deleted_sales = 0

        # PASO 1: Eliminar enrichments en batch (si hay)
        if enrichment_remaining > 0:
            # Obtener IDs de sales_data de este upload
            sales_ids_subquery = (
                db.query(SalesData.id).filter(SalesData.upload_id == upload_id).subquery()
            )

            # Obtener batch de enrichments a eliminar
            enrichments_to_delete = (
                db.query(SalesEnrichment.id)
                .filter(SalesEnrichment.sales_data_id.in_(sales_ids_subquery))
                .limit(batch_size)
                .all()
            )

            if enrichments_to_delete:
                enrichment_ids = [e[0] for e in enrichments_to_delete]
                deleted_enrichments = (
                    db.query(SalesEnrichment)
                    .filter(SalesEnrichment.id.in_(enrichment_ids))
                    .delete(synchronize_session=False)
                )
                db.commit()

                logger.info(
                    "[DELETE_UPLOAD] Batch de enrichments eliminado",
                    extra={"deleted_enrichments": deleted_enrichments, "upload_id": upload_id},
                )

                # Recalcular restantes
                new_enrichment_remaining = enrichment_remaining - deleted_enrichments
                new_sales_remaining = sales_remaining

                return {
                    "success": True,
                    "status": "in_progress",
                    "message": f"Eliminando datos de '{filename}'...",
                    "progress": {
                        "phase": "enrichments",
                        "deleted_this_batch": deleted_enrichments,
                        "enrichments_remaining": max(0, new_enrichment_remaining),
                        "sales_remaining": new_sales_remaining,
                        "total_remaining": max(0, new_enrichment_remaining) + new_sales_remaining,
                    },
                }

        # PASO 2: Eliminar sales_data en batch (si no hay enrichments)
        if sales_remaining > 0:
            # Obtener batch de sales a eliminar
            sales_to_delete = (
                db.query(SalesData.id)
                .filter(SalesData.upload_id == upload_id)
                .limit(batch_size)
                .all()
            )

            if sales_to_delete:
                sales_ids = [s[0] for s in sales_to_delete]
                deleted_sales = (
                    db.query(SalesData)
                    .filter(SalesData.id.in_(sales_ids))
                    .delete(synchronize_session=False)
                )
                db.commit()

                logger.info(
                    "[DELETE_UPLOAD] Batch de sales eliminado",
                    extra={"deleted_sales": deleted_sales, "upload_id": upload_id},
                )

                # Recalcular restantes
                new_sales_remaining = sales_remaining - deleted_sales

                if new_sales_remaining > 0:
                    return {
                        "success": True,
                        "status": "in_progress",
                        "message": f"Eliminando ventas de '{filename}'...",
                        "progress": {
                            "phase": "sales",
                            "deleted_this_batch": deleted_sales,
                            "enrichments_remaining": 0,
                            "sales_remaining": new_sales_remaining,
                            "total_remaining": new_sales_remaining,
                        },
                    }

        # PASO 3: Todo eliminado, ahora eliminar el upload
        db.delete(upload)
        db.commit()

        logger.info(
            "[DELETE_UPLOAD] Upload eliminado exitosamente (completed)",
            extra={
                "upload_id": upload_id,
                "filename": filename,
                "pharmacy_id": pharmacy_id,
                "user_id": str(current_user.id),
                "user_email": current_user.email,
            },
        )

        # AUDIT: Registrar eliminación
        from app.services.audit_service import get_audit_service
        from app.models.audit_log import AuditAction

        try:
            audit_service = get_audit_service(db)
            audit_service.log_action(
                action=AuditAction.DELETE,
                user_id=str(current_user.id),
                resource_type="file_upload",
                resource_id=upload_id,
                details={
                    "filename": filename,
                    "pharmacy_id": pharmacy_id,
                },
            )
        except Exception as audit_error:
            logger.warning(f"[DELETE_UPLOAD] Error registrando audit: {audit_error}")

        return {
            "success": True,
            "status": "completed",
            "message": f"Upload '{filename}' eliminado correctamente",
            "deleted": {
                "upload_id": upload_id,
                "filename": filename,
            },
        }

    except Exception as e:
        db.rollback()
        logger.error(
            "[DELETE_UPLOAD] Error eliminando upload",
            extra={
                "upload_id": upload_id,
                "error": str(e),
                "error_type": type(e).__name__,
            },
        )
        raise HTTPException(status_code=500, detail=f"Error eliminando upload: {str(e)}")


# =============================================================================
# INVENTORY UPLOAD ENDPOINTS (Issue #476)
# =============================================================================


@router.post("/inventory")
async def upload_inventory_file(
    file: UploadFile = File(...),
    pharmacy_id: Optional[str] = None,
    erp_type: Optional[str] = None,  # farmanager (default), farmatic (futuro)
    snapshot_date: Optional[str] = None,  # YYYY-MM-DD format
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
):
    """
    Endpoint para cargar archivos de inventario desde ERPs.

    Issue #476: Carga de ficheros de inventario desde ERPs.

    Flujo similar a upload de ventas pero:
    - Sin enriquecimiento CIMA (no aplica a inventario)
    - Detección de duplicados por (pharmacy_id, snapshot_date)
    - Reemplazo de snapshot existente si hay conflicto

    Args:
        file: Archivo CSV de inventario (exportado desde ERP)
        pharmacy_id: ID de la farmacia (opcional, auto-detect si no se proporciona)
        erp_type: Tipo de ERP - "farmanager" (default). Farmatic se añadirá después.
        snapshot_date: Fecha del inventario en formato YYYY-MM-DD (default: hoy)

    Returns:
        dict con upload_id y status QUEUED para polling
    """
    from datetime import date as date_type
    from threading import Thread

    from app.services.inventory_processing_service import get_inventory_processing_service

    logger.info("[INVENTORY_UPLOAD] Iniciando upload de inventario", extra={"filename": file.filename})

    try:
        # 1. Leer contenido del archivo
        contents = await file.read()
        file_size = len(contents)
        logger.info(f"[INVENTORY_UPLOAD] Tamaño: {file_size:,} bytes")

        # 2. Validar tamaño
        if file_size > MAX_FILE_SIZE:
            raise HTTPException(
                status_code=413,
                detail=f"Archivo demasiado grande. Máximo: {MAX_FILE_SIZE / 1024 / 1024:.1f}MB",
            )

        if file_size == 0:
            raise HTTPException(status_code=400, detail="Archivo vacío")

        # 3. Determinar pharmacy_id
        if pharmacy_id is None:
            if current_user.pharmacy_id:
                pharmacy_id = str(current_user.pharmacy_id)
            else:
                raise HTTPException(
                    status_code=400,
                    detail="No se pudo determinar la farmacia. Proporciona pharmacy_id explícitamente.",
                )

        # 4. Validar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(status_code=404, detail="Farmacia no encontrada")

        # 5. Validar acceso del usuario a esta farmacia
        if str(current_user.pharmacy_id) != pharmacy_id:
            logger.warning(
                "[SECURITY] Intento de upload inventario no autorizado",
                extra={
                    "user_id": str(current_user.id),
                    "user_pharmacy_id": str(current_user.pharmacy_id),
                    "target_pharmacy_id": pharmacy_id,
                },
            )
            raise HTTPException(status_code=403, detail="No tienes permiso para esta farmacia")

        # 6. Parsear snapshot_date si se proporciona
        parsed_snapshot_date: Optional[date_type] = None
        if snapshot_date:
            try:
                from datetime import datetime as dt
                parsed_snapshot_date = dt.strptime(snapshot_date, "%Y-%m-%d").date()
            except ValueError:
                raise HTTPException(
                    status_code=400,
                    detail="Formato de fecha inválido. Usar YYYY-MM-DD",
                )

        # 7. Determinar tipo de archivo
        file_type = FileType.INVENTORY_FARMANAGER  # Default
        if erp_type:
            erp_lower = erp_type.lower()
            if "farmatic" in erp_lower:
                file_type = FileType.INVENTORY_FARMATIC
            elif "farmanager" in erp_lower:
                file_type = FileType.INVENTORY_FARMANAGER

        # 8. Crear registro FileUpload
        logger.info("[INVENTORY_UPLOAD] Creando registro en base de datos...")
        upload_record = FileUpload(
            pharmacy_id=pharmacy_id,
            filename=file.filename,
            file_path="",  # Se actualizará en background
            file_size=file_size,
            file_type=file_type,
            status=UploadStatus.PENDING,
            processing_notes="Preparando procesamiento de inventario...",
        )

        db.add(upload_record)
        db.commit()
        db.refresh(upload_record)

        logger.info(f"[INVENTORY_UPLOAD] Registro creado: {upload_record.id}")

        # 9. Preparar ruta del archivo
        file_id = uuid.uuid4()
        file_extension = Path(file.filename).suffix or ".csv"
        file_path = UPLOAD_DIR / f"inventory_{file_id}{file_extension}"

        # 10. Lanzar procesamiento en thread separado (Issue #332 pattern)
        upload_record.status = UploadStatus.QUEUED
        upload_record.processing_notes = "Inventario en cola para procesamiento..."
        db.commit()

        inventory_service = get_inventory_processing_service()

        logger.info("[INVENTORY_UPLOAD] Lanzando procesamiento en thread separado...")
        processing_thread = Thread(
            target=inventory_service.process_inventory_in_thread,
            args=(
                str(upload_record.id),
                str(file_path),
                contents,
                pharmacy_id,
                erp_type,
                parsed_snapshot_date,
            ),
            name=f"inventory_upload_{upload_record.id}",
            daemon=False,  # CRITICAL: NO daemon para sobrevivir al request
        )
        processing_thread.start()

        # 11. Retornar inmediatamente con status QUEUED
        logger.info(
            "[INVENTORY_UPLOAD] Archivo en cola",
            extra={"upload_id": str(upload_record.id), "filename": file.filename},
        )

        return {
            "message": "Inventario en cola para procesamiento. Use /status/{upload_id} para monitorear progreso.",
            "upload_id": str(upload_record.id),
            "status": "queued",
            "filename": file.filename,
            "file_size": file_size,
            "file_type": file_type.value,
            "snapshot_date": str(parsed_snapshot_date) if parsed_snapshot_date else str(date_type.today()),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.exception("[INVENTORY_UPLOAD] Error inesperado", extra={"error": str(e)})
        raise HTTPException(status_code=500, detail=f"Error procesando inventario: {str(e)}")
