﻿"""
Endpoints unificados para el estado del sistema.

Estos endpoints proporcionan una API consolidada para el estado del sistema
sin romper la compatibilidad con endpoints existentes.
"""

import logging
from datetime import timedelta
from typing import Any, Dict, List

from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.database import get_db
from app.models import FileUpload, ProductCatalog, SalesData
from app.services.catalog_maintenance_service import CatalogMaintenanceService
from app.utils.datetime_utils import utc_now

router = APIRouter(prefix="/api/v1/system-unified", tags=["system-unified"])
logger = logging.getLogger(__name__)


@router.get("/status-complete")
async def get_complete_system_status(
    include_metrics: bool = True,
    include_components: bool = True,
    include_recent_activity: bool = False,
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Obtiene el estado completo del sistema consolidado.

    Este endpoint combina:
    - Estado básico del sistema (compatible con /api/v1/system/status)
    - Métricas detalladas de rendimiento
    - Estado de componentes y servicios
    - Actividad reciente del sistema

    Args:
        include_metrics: Incluir métricas de rendimiento
        include_components: Incluir estado de componentes
        include_recent_activity: Incluir actividad reciente
    """
    try:
        result = {
            "status": "operational",
            "timestamp": utc_now().isoformat(),
            "version": "1.0.0",
            "environment": "development",
        }

        if include_components:
            result["components"] = await _get_components_status(db)

        if include_metrics:
            result["metrics"] = await _get_system_metrics(db)

        if include_recent_activity:
            result["recent_activity"] = await _get_recent_activity(db)

        # Determinar el estado general basado en componentes
        if include_components:
            overall_status = _determine_overall_status(result["components"])
            result["status"] = overall_status

        return {
            "success": True,
            "data": result,
            "message": "Estado del sistema obtenido exitosamente",
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error obteniendo estado del sistema: {e}")
        raise HTTPException(status_code=500, detail=f"Error interno del servidor: {str(e)}")


@router.get("/health-summary")
async def get_health_summary(
    level: str = "basic", db: Session = Depends(get_db)  # basic, detailed, diagnostic
) -> Dict[str, Any]:
    """
    Resumen de salud del sistema para diferentes audiencias.

    Args:
        level: Nivel de detalle
            - basic: Para farmacéuticos (estado simple)
            - detailed: Para administradores (métricas técnicas)
            - diagnostic: Para desarrolladores (información debug)
    """
    try:
        if level == "basic":
            data = await _get_basic_health_summary(db)
        elif level == "detailed":
            data = await _get_detailed_health_summary(db)
        elif level == "diagnostic":
            data = await _get_diagnostic_health_summary(db)
        else:
            raise HTTPException(status_code=400, detail="Nivel debe ser: basic, detailed, o diagnostic")

        return {
            "success": True,
            "data": data,
            "message": f"Resumen de salud ({level}) obtenido exitosamente",
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error obteniendo resumen de salud: {e}")
        raise HTTPException(status_code=500, detail=f"Error interno del servidor: {str(e)}")


async def _get_components_status(db: Session) -> Dict[str, Any]:
    """Obtiene el estado de todos los componentes del sistema"""
    components = {}

    # Estado de la base de datos
    try:
        catalog_count = db.query(ProductCatalog).count()
        sales_count = db.query(SalesData).count()

        components["database"] = {
            "status": "operational",
            "catalog_products": catalog_count,
            "sales_records": sales_count,
            "last_check": utc_now().isoformat(),
        }
    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        components["database"] = {
            "status": "error",
            "error": str(e),
            "last_check": utc_now().isoformat(),
        }

    # Estado del catálogo
    try:
        maintenance_service = CatalogMaintenanceService(db)
        catalog_status = await maintenance_service.get_sync_status()

        components["catalog"] = {
            "status": ("operational" if catalog_status.get("ready", False) else "maintenance"),
            "last_sync": catalog_status.get("last_update"),
            "total_products": catalog_status.get("total_products", 0),
            "cima_products": catalog_status.get("cima_products", 0),
            "nomenclator_products": catalog_status.get("nomenclator_products", 0),
        }
    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        components["catalog"] = {
            "status": "error",
            "error": str(e),
            "last_check": utc_now().isoformat(),
        }

    # Estado de CIMA (simulado por ahora)
    components["cima"] = {
        "status": "operational",
        "api_available": True,
        "last_sync": utc_now().isoformat(),
        "last_check": utc_now().isoformat(),
    }

    # Estado del nomenclator (simulado por ahora)
    components["nomenclator"] = {
        "status": "operational",
        "data_available": True,
        "last_sync": utc_now().isoformat(),
        "last_check": utc_now().isoformat(),
    }

    return components


async def _get_system_metrics(db: Session) -> Dict[str, Any]:
    """Obtiene métricas de rendimiento del sistema"""
    metrics = {}

    # Métricas de base de datos
    try:
        # Tiempo de respuesta de BD (simulado)
        metrics["database_response_time_ms"] = 45.2

        # Tamaño de base de datos
        db_size_query = text(
            """
        SELECT pg_size_pretty(pg_database_size(current_database())) as size;
        """
        )
        result = db.execute(db_size_query).fetchone()
        metrics["database_size"] = result[0] if result else "Unknown"

        # Conexiones activas
        connections_query = text(
            """
        SELECT count(*) FROM pg_stat_activity
        WHERE datname = current_database() AND state = 'active';
        """
        )
        result = db.execute(connections_query).fetchone()
        metrics["active_connections"] = result[0] if result else 0

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.warning(f"Error obteniendo métricas de BD: {e}")
        metrics["database_error"] = str(e)

    # Métricas del sistema (simuladas por ahora)
    metrics.update(
        {
            "memory_usage_mb": 245.7,
            "cpu_usage_percent": 15.3,
            "uptime_hours": 48.2,
            "last_updated": utc_now().isoformat(),
        }
    )

    return metrics


async def _get_recent_activity(db: Session) -> List[Dict[str, Any]]:
    """Obtiene actividad reciente del sistema"""
    try:
        # Últimas cargas de archivos
        recent_uploads = (
            db.query(FileUpload)
            .filter(FileUpload.created_at >= utc_now() - timedelta(hours=24))
            .order_by(FileUpload.created_at.desc())
            .limit(10)
            .all()
        )

        activity = []
        for upload in recent_uploads:
            activity.append(
                {
                    "type": "file_upload",
                    "timestamp": upload.created_at.isoformat(),
                    "description": f"Archivo {upload.filename} subido",
                    "status": upload.status,
                    "pharmacy_id": str(upload.pharmacy_id),
                }
            )

        return activity

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.warning(f"Error obteniendo actividad reciente: {e}")
        return []


def _determine_overall_status(components: Dict[str, Any]) -> str:
    """Determina el estado general basado en componentes"""
    if not components:
        return "unknown"

    statuses = [comp.get("status", "unknown") for comp in components.values()]

    if "error" in statuses:
        return "error"
    elif "maintenance" in statuses:
        return "maintenance"
    elif all(status == "operational" for status in statuses):
        return "operational"
    else:
        return "degraded"


async def _get_basic_health_summary(db: Session) -> Dict[str, Any]:
    """Resumen básico para farmacéuticos"""
    components = await _get_components_status(db)
    overall_status = _determine_overall_status(components)

    # Estado simplificado
    status_map = {
        "operational": {
            "text": "OPERATIVO",
            "color": "success",
            "icon": "check-circle",
        },
        "maintenance": {"text": "MANTENIMIENTO", "color": "warning", "icon": "tools"},
        "error": {"text": "ERROR", "color": "danger", "icon": "exclamation-triangle"},
        "degraded": {
            "text": "DEGRADADO",
            "color": "warning",
            "icon": "exclamation-circle",
        },
        "unknown": {
            "text": "DESCONOCIDO",
            "color": "secondary",
            "icon": "question-circle",
        },
    }

    return {
        "overall_status": overall_status,
        "status_display": status_map.get(overall_status, status_map["unknown"]),
        "catalog_ready": components.get("catalog", {}).get("status") == "operational",
        "total_products": components.get("catalog", {}).get("total_products", 0),
        "last_check": utc_now().isoformat(),
        "alerts": [],  # TODO: Implementar sistema de alertas
    }


async def _get_detailed_health_summary(db: Session) -> Dict[str, Any]:
    """Resumen detallado para administradores"""
    components = await _get_components_status(db)
    metrics = await _get_system_metrics(db)

    return {
        "components": components,
        "metrics": metrics,
        "overall_status": _determine_overall_status(components),
        "performance": {
            "database_response_time": metrics.get("database_response_time_ms", 0),
            "memory_usage": metrics.get("memory_usage_mb", 0),
            "cpu_usage": metrics.get("cpu_usage_percent", 0),
        },
        "last_check": utc_now().isoformat(),
    }


async def _get_diagnostic_health_summary(db: Session) -> Dict[str, Any]:
    """Resumen diagnóstico para desarrolladores"""
    detailed = await _get_detailed_health_summary(db)
    recent_activity = await _get_recent_activity(db)

    return {
        **detailed,
        "recent_activity": recent_activity,
        "debug_info": {
            "active_connections": detailed["metrics"].get("active_connections", 0),
            "database_size": detailed["metrics"].get("database_size", "Unknown"),
            "uptime_hours": detailed["metrics"].get("uptime_hours", 0),
        },
    }
