﻿"""
API endpoints para el estado del sistema y sincronización.
Endpoints consolidados para evitar duplicación.
"""

import os
import time
from datetime import timedelta
from typing import Any, Dict, Literal, Optional, Tuple

import structlog
from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Request
from pydantic import BaseModel
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.api.deps import get_current_admin_user, get_current_user
from app.core.rate_limiting import admin_sync_limit
from app.database import get_db
from app.models.file_upload import FileUpload
from app.models.homogeneous_group import HomogeneousGroup
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
from app.models.user import User
from app.services.catalog_maintenance_service import CatalogMaintenanceService
from app.services.cima_sync_monitor import get_cima_monitor_status, start_cima_monitoring, stop_cima_monitoring
from app.utils.datetime_utils import utc_now

logger = structlog.get_logger()
router = APIRouter()

# Cache simple para el status del sistema
_status_cache = {"data": None, "timestamp": 0, "ttl": 30}  # 30 segundos de cache


def get_or_create_component_status(db: Session, component: SystemComponent) -> SystemStatus:
    """Obtener o crear el estado de un componente"""
    status = db.query(SystemStatus).filter_by(component=component).first()
    if not status:
        status = SystemStatus(
            component=component,
            status=SystemStatusEnum.NOT_INITIALIZED,
            message=f"Componente {component.value} no inicializado",
        )
        db.add(status)
        db.commit()
    return status


@router.get("/system/sync/real-progress")
def get_real_sync_progress(
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Obtiene el progreso REAL de sincronización, sin teatro ni simulación.
    Datos directos desde el RealProgressTracker.
    """
    try:
        # Obtener estados de todos los componentes de sincronización
        components = [
            SystemComponent.CATALOG,
            SystemComponent.CIMA,
            SystemComponent.NOMENCLATOR,
        ]

        active_syncs = []
        component_details = {}

        for component in components:
            status = db.query(SystemStatus).filter_by(component=component).first()
            if status:
                # Parsear detalles JSON si existen
                details = None
                if status.details:
                    try:
                        import json

                        details = json.loads(status.details)
                    except (json.JSONDecodeError, TypeError, AttributeError, ValueError) as e:
                        logger.warning(f"Error parsing status details for {component.value}: {e}")
                        details = status.details

                component_data = {
                    "component": component.value,
                    "status": status.status.value if status.status else "unknown",
                    "progress": status.progress or 0,
                    "message": status.message,
                    "processed_items": status.processed_items,
                    "total_items": status.total_items,
                    "started_at": (status.started_at.isoformat() if status.started_at else None),
                    "updated_at": (status.updated_at.isoformat() if status.updated_at else None),
                }

                # Agregar detalles reales si existen
                if details and isinstance(details, dict):
                    component_data["real_metrics"] = {
                        "phase": details.get("phase", "unknown"),
                        "processed": details.get("processed", 0),
                        "total": details.get("total", 0),
                        "current_chunk": details.get("chunk", 0),
                        "total_chunks": details.get("total_chunks", 0),
                        "speed": details.get("speed", 0),
                        "eta_seconds": details.get("eta_seconds"),
                        "errors": details.get("errors", 0),
                    }

                    # Calcular tiempo transcurrido real
                    if status.started_at:
                        # Manejar timezone awareness
                        from datetime import timezone

                        now = utc_now()
                        started = status.started_at
                        # Si started_at tiene timezone, convertir now también
                        if started.tzinfo:
                            now = now.replace(tzinfo=timezone.utc)
                        # Si started_at no tiene timezone, quitar de now
                        elif now.tzinfo:
                            now = now.replace(tzinfo=None)
                        elapsed = (now - started).total_seconds()
                        component_data["real_metrics"]["elapsed_seconds"] = int(elapsed)

                component_details[component.value] = component_data

                # Si está activo, agregarlo a la lista
                if status.status in [
                    SystemStatusEnum.INITIALIZING,
                    SystemStatusEnum.UPDATING,
                ]:
                    active_syncs.append(component_data)

        # Determinar el componente principal activo
        primary_sync = None
        if active_syncs:
            # Prioridad: catalog > cima > nomenclator
            for comp in ["catalog", "cima", "nomenclator"]:
                sync = next((s for s in active_syncs if s["component"] == comp), None)
                if sync:
                    primary_sync = sync
                    break

        return {
            "is_syncing": len(active_syncs) > 0,
            "active_components": [s["component"] for s in active_syncs],
            "primary_sync": primary_sync,
            "components": component_details,
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("catalog.real_progress.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        return {
            "is_syncing": False,
            "error": str(e),
            "timestamp": utc_now().isoformat(),
        }


@router.get("/system/catalog/quick-status")
def get_catalog_quick_status(
    current_user: User = Depends(get_current_user),  # ✅ SECURITY FIX: Auth required (NOT admin-only, used by dashboard)
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Endpoint optimizado para el panel de control del catálogo.
    Solo obtiene información esencial del catálogo sin métricas pesadas.
    Diseñado para actualizaciones frecuentes (cada 30 segundos).
    """
    try:
        # Solo las consultas más básicas y rápidas
        catalog_count = db.query(ProductCatalog).count()

        # Obtener estado de componentes críticos del catálogo
        statuses = (
            db.query(SystemStatus)
            .filter(
                SystemStatus.component.in_(
                    [
                        SystemComponent.CATALOG,
                        SystemComponent.CIMA,
                        SystemComponent.NOMENCLATOR,
                    ]
                )
            )
            .all()
        )

        status_dict = {s.component.value: s.to_dict() for s in statuses}

        # Solo conteos básicos si hay productos (evitar consultas complejas cuando está vacío)
        basic_metrics = {
            "total_products": catalog_count,
            "with_cima": 0,
            "with_nomenclator": 0,
        }

        if catalog_count > 0:
            # Solo dos consultas simples para los conteos básicos
            basic_metrics["with_cima"] = (
                db.query(ProductCatalog).filter(ProductCatalog.data_sources.ilike("%cima%")).count()
            )
            basic_metrics["with_nomenclator"] = (
                db.query(ProductCatalog).filter(ProductCatalog.data_sources.ilike("%nomenclator%")).count()
            )

        # Determinar estado de sincronización
        is_syncing = any(s.status in [SystemStatusEnum.INITIALIZING, SystemStatusEnum.UPDATING] for s in statuses)

        # Estado general simplificado
        if catalog_count == 0:
            overall_status = "requires_initialization"
            overall_message = "El catálogo necesita inicialización"
        elif is_syncing:
            overall_status = "initializing"
            overall_message = "Sincronización en progreso..."
        elif any(s.status == SystemStatusEnum.ERROR for s in statuses):
            overall_status = "error"
            overall_message = "Error en componentes del catálogo"
        else:
            overall_status = "ready"
            overall_message = "Catálogo operativo"

        return {
            "status": overall_status,
            "message": overall_message,
            "timestamp": utc_now().isoformat(),
            "catalog": basic_metrics,
            "components": status_dict,
            "sync": {
                "in_progress": is_syncing,
                "available_sources": ["cima", "nomenclator"],
            },
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("catalog.quick_status.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/system/status")
def get_system_status(
    include_details: bool = True,
    current_user: User = Depends(get_current_user),  # ✅ SECURITY FIX: Auth required (NOT admin-only, used by all users)
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Endpoint unificado para obtener el estado completo del sistema.
    Combina información de inicialización, salud, catálogo y métricas.

    Args:
        include_details: Si incluir información detallada (para reducir payload cuando no es necesario)

    Returns:
        Estado completo del sistema con toda la información necesaria
    """
    # OPTIMIZATION: Extended cache TTL for production performance
    # Health checks shouldn't need real-time data
    current_time = time.time()
    cache_ttl = 60 if not include_details else _status_cache["ttl"]  # 60s for simple status

    if _status_cache["data"] and (current_time - _status_cache["timestamp"]) < cache_ttl:
        logger.debug("Retornando status desde cache")
        return _status_cache["data"]

    try:
        # OPTIMIZATION: Single consolidated query for ALL metrics (Issue #293)
        # Replaces 10+ individual queries with ONE efficient query
        start_time = time.time()

        # Build query using SQLAlchemy model table names to prevent hardcoding errors
        consolidated_query = text(
            f"""
            WITH catalog_stats AS (
                SELECT
                    COUNT(*) as total_products,
                    COUNT(*) FILTER (WHERE data_sources ILIKE '%cima%') as with_cima,
                    COUNT(*) FILTER (WHERE data_sources ILIKE '%nomenclator%') as with_nomen,
                    COUNT(*) FILTER (WHERE data_sources = 'cima') as only_cima,
                    COUNT(*) FILTER (WHERE data_sources IN ('nomenclator', 'nomenclator_oficial', '"nomenclator_oficial"')) as only_nomenclator,
                    COUNT(*) FILTER (WHERE data_sources ILIKE '%,%' AND (data_sources ILIKE '%cima%' OR data_sources ILIKE '%nomenclator%')) as both_sources,
                    COUNT(*) FILTER (WHERE data_sources IS NULL OR data_sources = '') as no_sources
                FROM {ProductCatalog.__tablename__}
            ),
            sales_stats AS (
                SELECT
                    COUNT(*) as total_sales,
                    MIN(sale_date) as min_date,
                    MAX(sale_date) as max_date
                FROM {SalesData.__tablename__}
            ),
            upload_stats AS (
                SELECT filename, uploaded_at
                FROM {FileUpload.__tablename__}
                ORDER BY uploaded_at DESC
                LIMIT 1
            ),
            groups_count AS (
                SELECT COUNT(*) as total_groups
                FROM {HomogeneousGroup.__tablename__}
            )
            SELECT
                c.total_products, c.with_cima, c.with_nomen, c.only_cima, c.only_nomenclator,
                c.both_sources, c.no_sources,
                s.total_sales, s.min_date, s.max_date,
                u.filename as last_upload_filename, u.uploaded_at as last_upload_date,
                g.total_groups
            FROM catalog_stats c
            CROSS JOIN sales_stats s
            CROSS JOIN groups_count g
            LEFT JOIN upload_stats u ON true
        """
        )

        result = db.execute(consolidated_query).first()

        # Extract all metrics from single query result
        catalog_count = result.total_products or 0
        catalog_with_cima = result.with_cima or 0
        catalog_with_nomen = result.with_nomen or 0
        only_cima = result.only_cima or 0
        only_nomenclator = result.only_nomenclator or 0
        both_sources = result.both_sources or 0
        no_sources = result.no_sources or 0
        sales_count = result.total_sales or 0
        groups_count = result.total_groups or 0

        # Get SystemStatus (lightweight query)
        statuses = db.query(SystemStatus).all()
        status_dict = {s.component.value: s.to_dict() for s in statuses}

        # Enrich homogeneous_groups component if exists
        if "homogeneous_groups" in status_dict:
            status_dict["homogeneous_groups"]["count"] = groups_count

        # Enrich CIMA/nomenclator components with product counts
        if "cima" in status_dict:
            status_dict["cima"]["products"] = catalog_with_cima
        if "nomenclator" in status_dict:
            status_dict["nomenclator"]["products"] = catalog_with_nomen

        # Build catalog metrics from consolidated query
        catalog_metrics = {
            "total_products": catalog_count,
            "with_cima": catalog_with_cima,
            "with_nomenclator": catalog_with_nomen,
            "distribution": {
                "only_cima": only_cima,
                "only_nomenclator": only_nomenclator,
                "both_sources": both_sources,
                "no_sources": no_sources,
            },
            "sync_status": {},  # Empty for performance - rarely used
            "coverage_percentage": (
                round((catalog_with_cima + catalog_with_nomen) / (catalog_count * 2) * 100, 1)
                if catalog_count > 0
                else 0
            ),
        }

        # Build sales metrics from consolidated query
        sales_metrics = {}
        if include_details:
            sales_metrics = {
                "total_records": sales_count,
                "date_range": {
                    "start": result.min_date.isoformat() if result.min_date else None,
                    "end": result.max_date.isoformat() if result.max_date else None,
                },
                "last_upload": {
                    "filename": result.last_upload_filename,
                    "date": result.last_upload_date.isoformat() if result.last_upload_date else None,
                },
            }

        # === ESTADO DE INICIALIZACIÓN ===
        critical_components = [
            SystemComponent.CATALOG,
            SystemComponent.NOMENCLATOR,
            SystemComponent.CIMA,
            SystemComponent.HOMOGENEOUS_GROUPS,
        ]

        initialization_progress = 0
        initialization_messages = []
        components_ready = []

        for component in critical_components:
            comp_status = status_dict.get(component.value, {})
            initialization_progress += comp_status.get("progress", 0)

            if comp_status.get("status") == SystemStatusEnum.INITIALIZING.value:
                initialization_messages.append(comp_status.get("message", f"Procesando {component.value}..."))
            elif comp_status.get("status") == SystemStatusEnum.READY.value:
                components_ready.append(component.value)

        overall_progress = int(initialization_progress / len(critical_components))

        # === ESTADO GENERAL DEL SISTEMA ===
        if catalog_count == 0:
            overall_status = "requires_initialization"
            overall_message = "El sistema necesita inicializar el catálogo de productos"
            initialization_required = True
        elif any(s.status == SystemStatusEnum.INITIALIZING for s in statuses):
            overall_status = "initializing"
            overall_message = (
                " ".join(initialization_messages) if initialization_messages else "Sistema inicializándose..."
            )
            initialization_required = False
        elif any(s.status == SystemStatusEnum.ERROR for s in statuses):
            overall_status = "error"
            overall_message = "Sistema con errores. Revisar componentes."
            initialization_required = False
        elif all(
            status_dict.get(c.value, {}).get("status") == SystemStatusEnum.READY.value for c in critical_components
        ):
            overall_status = "ready"
            overall_message = "Sistema completamente operativo"
            initialization_required = False
        else:
            overall_status = "partial"
            overall_message = "Sistema parcialmente operativo"
            initialization_required = False

        # === SALUD DE SERVICIOS ===
        health_checks = {
            "database": False,
            "catalog": False,
            "redis": False,  # TODO: Implementar cuando tengamos Redis
        }

        try:
            db.execute(text("SELECT 1"))
            health_checks["database"] = True
        except Exception as e:
            logger.error("health.database.check_failed", error=str(e), error_type=type(e).__name__, exc_info=True)
            health_checks["database"] = False

        health_checks["catalog"] = catalog_count > 0

        # === INFORMACIÓN DE SINCRONIZACIÓN ===
        # Obtener última sincronización de cada fuente
        last_sync_info = {}
        for component in [SystemComponent.CIMA, SystemComponent.NOMENCLATOR]:
            comp_status = status_dict.get(component.value, {})
            # Usar last_success_at del objeto status, no de details
            last_sync_date = comp_status.get("last_success_at")
            if last_sync_date or comp_status.get("status"):
                last_sync_info[component.value.lower()] = {
                    "last_sync": last_sync_date,
                    "status": comp_status.get("status"),
                    "message": comp_status.get("message"),
                }

        # === RESPUESTA CONSOLIDADA ===
        response = {
            # Estado general
            "status": overall_status,
            "message": overall_message,
            "progress": overall_progress,
            "timestamp": utc_now().isoformat(),
            # Información de inicialización (para compatibilidad con frontend actual)
            "initialization": {
                "required": initialization_required,
                "overall_status": overall_status,
                "overall_progress": overall_progress,
                "overall_message": overall_message,
                "catalog_empty": catalog_count == 0,
                "auto_initialize": catalog_count == 0,
                "components": status_dict if include_details else None,
            },
            # Métricas del catálogo (para compatibilidad con catalog/status)
            "catalog": catalog_metrics,
            # Información de sincronización
            "sync": {
                "last_updates": last_sync_info,
                "in_progress": any(s.status == SystemStatusEnum.INITIALIZING for s in statuses),
                "available_sources": ["cima", "nomenclator"],
            },
            # Salud del sistema
            "health": health_checks,
            # Acciones disponibles
            "actions_available": {
                "initialize_catalog": catalog_count == 0,
                "sync_nomenclator": True,
                "sync_cima": True,
                "calculate_groups": catalog_count > 0,
            },
        }

        # Agregar métricas de ventas si se solicitaron detalles
        if include_details and sales_metrics:
            response["sales"] = sales_metrics

        # Agregar componentes detallados si se solicitaron
        if include_details:
            response["components"] = status_dict
            # Actualizar cache
            _status_cache["data"] = response
            _status_cache["timestamp"] = time.time()

        # Log performance metrics (Issue #293)
        execution_time_ms = (time.time() - start_time) * 1000
        logger.info(
            "system.status.optimized.success",
            execution_time_ms=round(execution_time_ms, 2),
            queries_consolidated=2,  # consolidated_query + SystemStatus query
            speedup_factor="5-10x",
            total_products=catalog_count,
        )

        return response

    except Exception as e:
        # Log completo con traceback para debugging (Issue #293 - Enhanced error logging)
        execution_time_ms = round((time.time() - start_time) * 1000, 2) if "start_time" in locals() else None
        logger.error(
            "system.status.optimized.failure",
            error=str(e),
            error_type=type(e).__name__,
            execution_time_ms=execution_time_ms,
            include_details=include_details,
            exc_info=True,
        )
        raise HTTPException(status_code=500, detail=str(e))


class SyncRequest(BaseModel):
    target: Literal["all", "catalog", "cima", "nomenclator"] = "all"
    force: bool = False


@router.post("/system/sync")
@admin_sync_limit
async def sync_system(
    request: Request,  # Required by SlowAPI decorator, not used in function body
    sync_request: SyncRequest,  # Request body con los parámetros
    current_user: User = Depends(get_current_admin_user),  # Requiere admin
    background_tasks: BackgroundTasks = BackgroundTasks(),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Endpoint unificado para sincronización del sistema.
    Reemplaza los múltiples endpoints de sincronización.

    Args:
        sync_request: SyncRequest con target ("all"|"catalog"|"cima"|"nomenclator") y force
        current_user: Usuario autenticado con permisos de admin

    Returns:
        Estado de la sincronización iniciada con sync_id y componentes
    """
    try:
        # Log para debug
        logger.info(f"[SYNC] Request received - target: {sync_request.target}, force: {sync_request.force}")

        service = CatalogMaintenanceService(db)

        # Mapear target a componentes
        components_to_sync = []
        if sync_request.target == "all":
            components_to_sync = [
                SystemComponent.CATALOG,
                SystemComponent.NOMENCLATOR,
                SystemComponent.CIMA,
            ]
        elif sync_request.target == "catalog":
            components_to_sync = [SystemComponent.CATALOG]
        elif sync_request.target == "cima":
            components_to_sync = [SystemComponent.CIMA]
        elif sync_request.target == "nomenclator":
            components_to_sync = [SystemComponent.NOMENCLATOR]

        # Verificar si ya hay sincronización en progreso
        existing_sync = db.query(SystemStatus).filter(SystemStatus.status == SystemStatusEnum.INITIALIZING).first()

        if existing_sync and not sync_request.force:
            return {
                "status": "already_running",
                "message": f"Ya hay una sincronización en progreso: {existing_sync.component.value}",
                "component": existing_sync.component.value,
            }

        # Actualizar estados de componentes a sincronizando
        sync_id = f"sync_{utc_now().strftime('%Y%m%d_%H%M%S')}"
        for component in components_to_sync:
            status = get_or_create_component_status(db, component)
            status.status = SystemStatusEnum.INITIALIZING
            status.progress = 0
            status.message = f"Iniciando sincronización de {component.value}..."
            status.started_at = utc_now()
            import json

            status.details = json.dumps({"sync_id": sync_id})

        db.commit()

        # Lanzar sincronización real según el target (TODOS SÍNCRONOS desde 2025-11-02)
        if sync_request.target == "cima":
            # SYNC MANUAL: Ejecutar síncronamente para garantizar ejecución
            import time
            start_time = time.time()
            logger.info("[SYNC] Ejecutando CIMA sync síncronamente (manual trigger)")

            # Detectar si estamos en Render (variable de entorno o dominio)
            is_render = os.getenv("RENDER") == "true" or "onrender.com" in os.getenv("RENDER_EXTERNAL_URL", "")

            if is_render:
                # En Render: chunks de 300 productos (~2 min por chunk)
                chunk_size = 300
                logger.info(f"[SYNC] Render - chunks de {chunk_size} productos")
            else:
                # Local: chunks más grandes
                chunk_size = 1000
                logger.info(f"[SYNC] Local - chunks de {chunk_size} productos")

            try:
                result = await service.sync_cima_chunked(
                    db,
                    chunk_size=chunk_size,
                    force_update=sync_request.force,
                )
                duration = time.time() - start_time

                logger.info(
                    f"[SYNC] CIMA sync completado en {duration:.2f}s: {result}",
                    extra={"duration_seconds": duration, "result": result}
                )

                # ⚠️ Advertencia si se acerca al timeout de Gunicorn (180s en Render)
                if duration > 120:  # 2/3 del timeout
                    logger.warning(
                        f"[TIMING_WARNING] CIMA sync tardó {duration:.2f}s "
                        f"(>120s, cerca del límite de 180s de Gunicorn)"
                    )

                # Retornar resultado detallado con duración
                return {
                    "status": "completed",
                    "sync_id": sync_id,
                    "target": sync_request.target,
                    "components": [c.value for c in components_to_sync],
                    "message": "Sincronización de CIMA completada",
                    "result": result,
                    "duration_seconds": round(duration, 2),
                }
            except Exception as sync_error:
                duration = time.time() - start_time
                logger.error(
                    f"[SYNC] Error en CIMA sync después de {duration:.2f}s: {sync_error}",
                    exc_info=True,
                    extra={"duration_seconds": duration}
                )

                # Actualizar estado a error
                for component in components_to_sync:
                    status = get_or_create_component_status(db, component)
                    status.status = SystemStatusEnum.ERROR
                    status.message = f"Error: {str(sync_error)}"
                    status.last_error_at = utc_now()
                db.commit()

                raise HTTPException(
                    status_code=500,
                    detail=f"Error en sincronización de CIMA: {str(sync_error)}"
                )
        elif sync_request.target == "nomenclator":
            # SYNC MANUAL: Ejecutar síncronamente para garantizar ejecución
            # (Background tasks pueden perderse en Render con workers múltiples)
            import time
            start_time = time.time()
            logger.info("[SYNC] Ejecutando nomenclator sync síncronamente (manual trigger)")

            try:
                result = service.sync_nomenclator_only(db, force_update=sync_request.force)
                duration = time.time() - start_time

                logger.info(
                    f"[SYNC] Nomenclator sync completado en {duration:.2f}s: {result}",
                    extra={"duration_seconds": duration, "result": result}
                )

                # ⚠️ Advertencia si se acerca al timeout de Gunicorn (180s en Render)
                if duration > 120:  # 2/3 del timeout
                    logger.warning(
                        f"[TIMING_WARNING] Nomenclator sync tardó {duration:.2f}s "
                        f"(>120s, cerca del límite de 180s de Gunicorn)"
                    )

                # Retornar resultado detallado con duración
                return {
                    "status": "completed",
                    "sync_id": sync_id,
                    "target": sync_request.target,
                    "components": [c.value for c in components_to_sync],
                    "message": "Sincronización de nomenclator completada",
                    "result": result,
                    "duration_seconds": round(duration, 2),  # Agregar duración al response
                }
            except Exception as sync_error:
                duration = time.time() - start_time
                logger.error(
                    f"[SYNC] Error en nomenclator sync después de {duration:.2f}s: {sync_error}",
                    exc_info=True,
                    extra={"duration_seconds": duration}
                )

                # Actualizar estado a error
                for component in components_to_sync:
                    status = get_or_create_component_status(db, component)
                    status.status = SystemStatusEnum.ERROR
                    status.message = f"Error: {str(sync_error)}"
                    status.last_error_at = utc_now()
                db.commit()

                raise HTTPException(
                    status_code=500,
                    detail=f"Error en sincronización de nomenclator: {str(sync_error)}"
                )
        elif sync_request.target == "all":
            # SYNC MANUAL: Ejecutar síncronamente nomenclator + CIMA
            import time
            start_time = time.time()
            logger.info("[SYNC] Ejecutando sync 'all' síncronamente (nomenclator + CIMA)")

            # Detectar si estamos en Render
            is_render = os.getenv("RENDER") == "true" or "onrender.com" in os.getenv("RENDER_EXTERNAL_URL", "")

            if is_render:
                chunk_size = 300  # Render: chunks pequeños
                logger.info(f"[SYNC] Render - chunks de {chunk_size} productos")
            else:
                chunk_size = 1000  # Local: chunks más grandes
                logger.info(f"[SYNC] Local - chunks de {chunk_size} productos")

            try:
                # PASO 1: Nomenclator sync
                logger.info("[SYNC] Paso 1/2: Sincronizando nomenclator...")
                nomenclator_result = service.sync_nomenclator_only(db, force_update=sync_request.force)
                nomenclator_duration = time.time() - start_time
                logger.info(f"[SYNC] Nomenclator completado en {nomenclator_duration:.2f}s")

                # PASO 2: CIMA sync
                logger.info("[SYNC] Paso 2/2: Sincronizando CIMA...")
                cima_start = time.time()
                cima_result = await service.sync_cima_chunked(
                    db,
                    chunk_size=chunk_size,
                    force_update=sync_request.force,
                )
                cima_duration = time.time() - cima_start
                logger.info(f"[SYNC] CIMA completado en {cima_duration:.2f}s")

                # Duración total
                total_duration = time.time() - start_time

                logger.info(
                    f"[SYNC] Sync 'all' completado en {total_duration:.2f}s "
                    f"(nomenclator: {nomenclator_duration:.2f}s, CIMA: {cima_duration:.2f}s)",
                    extra={"duration_seconds": total_duration}
                )

                # ⚠️ Advertencia si se acerca al timeout de Gunicorn (180s en Render)
                if total_duration > 120:  # 2/3 del timeout
                    logger.warning(
                        f"[TIMING_WARNING] Sync 'all' tardó {total_duration:.2f}s "
                        f"(>120s, cerca del límite de 180s de Gunicorn)"
                    )

                # Retornar resultado detallado con duración
                return {
                    "status": "completed",
                    "sync_id": sync_id,
                    "target": sync_request.target,
                    "components": [c.value for c in components_to_sync],
                    "message": "Sincronización completa (nomenclator + CIMA) completada",
                    "result": {
                        "nomenclator": nomenclator_result,
                        "cima": cima_result
                    },
                    "duration_seconds": round(total_duration, 2),
                    "step_durations": {
                        "nomenclator": round(nomenclator_duration, 2),
                        "cima": round(cima_duration, 2)
                    }
                }
            except Exception as sync_error:
                duration = time.time() - start_time
                logger.error(
                    f"[SYNC] Error en sync 'all' después de {duration:.2f}s: {sync_error}",
                    exc_info=True,
                    extra={"duration_seconds": duration}
                )

                # Actualizar estado a error
                for component in components_to_sync:
                    status = get_or_create_component_status(db, component)
                    status.status = SystemStatusEnum.ERROR
                    status.message = f"Error: {str(sync_error)}"
                    status.last_error_at = utc_now()
                db.commit()

                raise HTTPException(
                    status_code=500,
                    detail=f"Error en sincronización completa: {str(sync_error)}"
                )

        # Este return ya no se alcanza porque todos los targets retornan directamente
        return {
            "status": "error",
            "sync_id": sync_id,
            "target": sync_request.target,
            "components": [c.value for c in components_to_sync],
            "message": f"Sincronización de {sync_request.target} iniciada",
            "estimated_duration": 180 if sync_request.target == "all" else 60,
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("sync.start.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/system/sync/cancel")
def cancel_active_syncs(
    component: Optional[str] = None,
    force: bool = False,
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Cancelar sincronizaciones activas.

    Args:
        component: Componente específico a cancelar (opcional)
        force: Forzar cancelación incluso si está progresando
        db: Sesión de base de datos

    Returns:
        Estado de las cancelaciones realizadas
    """
    try:
        # Filtro base: sincronizaciones activas
        query = db.query(SystemStatus).filter(SystemStatus.status == SystemStatusEnum.INITIALIZING)

        # Filtrar por componente si se especifica
        if component:
            try:
                component_enum = SystemComponent(component.lower())
                query = query.filter(SystemStatus.component == component_enum)
            except ValueError:
                raise HTTPException(
                    status_code=400,
                    detail=f"Componente inválido: {component}. Válidos: {[c.value for c in SystemComponent]}",
                )

        active_syncs = query.all()

        if not active_syncs:
            return {
                "status": "no_active_syncs",
                "message": "No hay sincronizaciones activas para cancelar",
                "component": component,
            }

        cancelled_components = []
        for sync in active_syncs:
            # Verificar si está realmente progresando (progreso > 0 y actualizado recientemente)
            recently_updated = sync.updated_at and sync.updated_at > utc_now() - timedelta(minutes=5)

            if sync.progress > 0 and recently_updated and not force:
                logger.info(f"Saltando {sync.component.value} - progresando activamente")
                continue

            logger.info(f"Cancelando sincronización: {sync.component.value}")

            # Cancelar sincronización
            sync.status = SystemStatusEnum.ERROR
            sync.message = "Sincronización cancelada manualmente"
            sync.progress = 0
            sync.last_error_at = utc_now()

            cancelled_components.append(
                {
                    "component": sync.component.value,
                    "was_progressing": sync.progress > 0 and recently_updated,
                    "runtime_minutes": (
                        int((utc_now() - sync.started_at).total_seconds() / 60) if sync.started_at else 0
                    ),
                }
            )

        db.commit()

        return {
            "status": "cancelled",
            "cancelled_count": len(cancelled_components),
            "cancelled_components": cancelled_components,
            "force": force,
            "timestamp": utc_now().isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("sync.cancel.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/system/sync/cleanup")
def auto_cleanup_stale_syncs(
    timeout_minutes: int = 60,
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Endpoint para limpieza automática de sincronizaciones enganchadas.

    Args:
        timeout_minutes: Timeout en minutos para considerar una sincronización enganchada
        db: Sesión de base de datos

    Returns:
        Resultado de la limpieza automática
    """
    try:
        # Validar timeout
        if timeout_minutes < 5:
            raise HTTPException(status_code=400, detail="Timeout mínimo: 5 minutos")
        if timeout_minutes > 480:  # 8 horas
            raise HTTPException(status_code=400, detail="Timeout máximo: 480 minutos")

        result = cleanup_stale_syncs(db, timeout_minutes)

        if result["cleaned_count"] > 0:
            logger.info(f"Limpieza automática completada: {result['cleaned_count']} componentes")
        else:
            logger.debug("Limpieza automática: no hay sincronizaciones enganchadas")

        return result

    except HTTPException:
        raise
    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("sync.cleanup.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/system/sync/{sync_id}")
def get_sync_status(
    sync_id: str,
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Obtener el estado de una sincronización específica.

    Args:
        sync_id: ID de la sincronización

    Returns:
        Estado actual de la sincronización
    """
    try:
        # Buscar componentes con este sync_id
        import json

        statuses = db.query(SystemStatus).filter(SystemStatus.details.contains(json.dumps({"sync_id": sync_id}))).all()

        if not statuses:
            raise HTTPException(status_code=404, detail="Sincronización no encontrada")

        components_status = []
        total_progress = 0

        for status in statuses:
            components_status.append(
                {
                    "component": status.component.value,
                    "status": status.status.value,
                    "progress": status.progress,
                    "message": status.message,
                }
            )
            total_progress += status.progress

        overall_progress = int(total_progress / len(statuses)) if statuses else 0
        all_complete = all(s.status == SystemStatusEnum.READY for s in statuses)
        any_error = any(s.status == SystemStatusEnum.ERROR for s in statuses)

        return {
            "sync_id": sync_id,
            "status": ("completed" if all_complete else "error" if any_error else "in_progress"),
            "overall_progress": overall_progress,
            "components": components_status,
            "timestamp": utc_now().isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("sync.status.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


# === GESTIÓN AUTOMÁTICA DE TIMEOUTS Y CANCELACIONES ===


def cleanup_stale_syncs(db: Session, timeout_minutes: int = 60) -> Dict[str, Any]:
    """
    Limpiar sincronizaciones enganchadas que excedan el timeout.

    Args:
        db: Sesión de base de datos
        timeout_minutes: Timeout en minutos (default: 60 minutos)

    Returns:
        Información sobre las sincronizaciones limpiadas
    """
    try:
        cutoff_time = utc_now() - timedelta(minutes=timeout_minutes)

        # Buscar sincronizaciones enganchadas
        stale_syncs = (
            db.query(SystemStatus)
            .filter(
                SystemStatus.status == SystemStatusEnum.INITIALIZING,
                SystemStatus.started_at < cutoff_time,
            )
            .all()
        )

        cleaned_components = []
        for sync in stale_syncs:
            logger.warning(
                f"Limpiando sincronización enganchada: {sync.component.value} "
                f"(iniciada: {sync.started_at}, timeout: {timeout_minutes}min)"
            )

            # Marcar como error con mensaje informativo
            sync.status = SystemStatusEnum.ERROR
            sync.message = f"Proceso interrumpido por timeout ({timeout_minutes}min)"
            sync.progress = 0
            sync.last_error_at = utc_now()

            cleaned_components.append(
                {
                    "component": sync.component.value,
                    "started_at": sync.started_at.isoformat(),
                    "runtime_minutes": int((utc_now() - sync.started_at).total_seconds() / 60),
                }
            )

        db.commit()

        return {
            "cleaned_count": len(cleaned_components),
            "cleaned_components": cleaned_components,
            "timeout_minutes": timeout_minutes,
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("sync.auto_cleanup.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        raise


# ==========================================
# ENDPOINTS DE MONITOREO AUTOMÁTICO CIMA
# ==========================================


@router.get("/system/cima/monitor/status")
def get_cima_sync_monitor_status(
    current_user: User = Depends(get_current_admin_user)  # ✅ SECURITY FIX: Admin required
) -> Dict[str, Any]:
    """
    Obtiene el estado completo del monitor automático de sincronización CIMA

    Returns:
        Estado del monitor: heartbeat, detección de estancamiento, recovery status, etc.
    """
    try:
        monitor_status = get_cima_monitor_status()

        logger.info(
            "monitor.status.requested",
            monitoring_active=monitor_status.get("monitoring_active", False),
            sync_active=monitor_status.get("sync_heartbeat") is not None,
            is_stalled=monitor_status.get("stall_detection", {}).get("is_stalled", False),
        )

        return {
            "status": "success",
            "data": monitor_status,
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("monitor.status.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        raise HTTPException(status_code=500, detail=f"Error obteniendo estado del monitor: {str(e)[:200]}")


@router.post("/system/cima/monitor/start")
def start_cima_sync_monitoring(
    current_user: User = Depends(get_current_admin_user)  # ✅ SECURITY FIX: Admin required
) -> Dict[str, Any]:
    """
    Inicia el monitoreo automático de sincronización CIMA

    Returns:
        Confirmación de inicio del monitoreo
    """
    try:
        success = start_cima_monitoring()

        if success:
            logger.info("monitor.started_via_api")
            return {
                "status": "success",
                "message": "Monitoreo automático CIMA iniciado correctamente",
                "monitoring_active": True,
                "timestamp": utc_now().isoformat(),
            }
        else:
            logger.error("monitor.start_failed")
            raise HTTPException(
                status_code=500, detail="No se pudo iniciar el monitoreo automático (Redis no disponible?)"
            )

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("monitor.start.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        raise HTTPException(status_code=500, detail=f"Error iniciando monitor: {str(e)[:200]}")


@router.post("/system/cima/monitor/stop")
def stop_cima_sync_monitoring(
    current_user: User = Depends(get_current_admin_user)  # ✅ SECURITY FIX: Admin required
) -> Dict[str, Any]:
    """
    Detiene el monitoreo automático de sincronización CIMA

    Returns:
        Confirmación de detención del monitoreo
    """
    try:
        stop_cima_monitoring()

        logger.info("monitor.stopped_via_api")

        return {
            "status": "success",
            "message": "Monitoreo automático CIMA detenido",
            "monitoring_active": False,
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("monitor.stop.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        raise HTTPException(status_code=500, detail=f"Error deteniendo monitor: {str(e)[:200]}")


@router.post("/system/cima/monitor/force-recovery")
def force_cima_recovery(
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Fuerza un intento de recovery de sincronización CIMA estancada
    (ignora límites de cooldown y attempts, para uso manual)

    Returns:
        Resultado del intento de recovery forzado
    """
    try:
        from app.services.cima_sync_monitor import cima_sync_monitor

        # Verificar si hay estancamiento
        is_stalled, diagnosis = cima_sync_monitor.detector.is_sync_stalled()

        if not is_stalled:
            return {
                "status": "not_needed",
                "message": "No se detecta estancamiento en sincronización CIMA",
                "diagnosis": diagnosis,
                "timestamp": utc_now().isoformat(),
            }

        logger.warning("monitor.forced_recovery.requested", diagnosis=diagnosis)

        # Forzar recovery ignorando límites
        # Limpiar estado de recovery temporal para permitir intento
        redis_client = cima_sync_monitor.redis
        if redis_client:
            recovery_key = cima_sync_monitor.config.REDIS_RECOVERY_KEY
            # Temporarily clear recovery limits for manual override
            redis_client.delete(recovery_key)

        # Intentar recovery
        recovery_success, recovery_info = cima_sync_monitor.recovery.attempt_recovery()

        result = {
            "status": "recovery_attempted",
            "recovery_successful": recovery_success,
            "forced_recovery": True,
            "stall_diagnosis": diagnosis,
            "recovery_info": recovery_info,
            "timestamp": utc_now().isoformat(),
        }

        if recovery_success:
            logger.info("monitor.forced_recovery.successful", result=result)
        else:
            logger.error("monitor.forced_recovery.failed", result=result)

        return result

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("monitor.forced_recovery.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        raise HTTPException(status_code=500, detail=f"Error en recovery forzado: {str(e)[:200]}")


@router.get("/system/cima/monitor/health")
def get_cima_sync_health(
    current_user: User = Depends(get_current_user)  # ✅ SECURITY FIX: Auth required (used by monitoring, not only admin)
) -> Dict[str, Any]:
    """
    Health check específico para sincronización CIMA
    Endpoint ligero para monitoreo externo

    Returns:
        Estado de salud simplificado de la sincronización CIMA
    """
    try:
        from app.services.cima_sync_monitor import cima_sync_monitor

        monitor_status = cima_sync_monitor.get_monitor_status()
        heartbeat_data = monitor_status.get("sync_heartbeat")
        stall_info = monitor_status.get("stall_detection", {})

        # Determinar estado general
        if heartbeat_data:
            if stall_info.get("is_stalled", False):
                health_status = "stalled"
                health_message = f"Sync estancada: {stall_info.get('diagnosis', {}).get('reason', 'unknown')}"
            else:
                health_status = "syncing"
                progress = heartbeat_data.get("progress_percentage", 0)
                phase = heartbeat_data.get("current_phase", "unknown")
                health_message = f"Sincronizando: {progress}% - {phase}"
        else:
            health_status = "idle"
            health_message = "No hay sincronización activa"

        # Información esencial para health checks
        health_data = {
            "status": health_status,
            "message": health_message,
            "monitoring_active": monitor_status.get("monitoring_active", False),
            "timestamp": utc_now().isoformat(),
        }

        # Agregar datos de progreso si hay sync activa
        if heartbeat_data:
            health_data.update(
                {
                    "sync_progress": {
                        "percentage": heartbeat_data.get("progress_percentage", 0),
                        "processed": heartbeat_data.get("processed_items", 0),
                        "total": heartbeat_data.get("total_items", 0),
                        "current_phase": heartbeat_data.get("current_phase", "unknown"),
                        "last_update": heartbeat_data.get("last_update"),
                    }
                }
            )

        return health_data

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("monitor.health_check.error", error=str(e), error_type=type(e).__name__, exc_info=True)

        # Health check debe ser resiliente - retornar datos básicos en caso de error
        # API response con error truncado para evitar leakage
        return {
            "status": "error",
            "message": f"Error en health check: {str(e)[:100]}",
            "monitoring_active": False,
            "timestamp": utc_now().isoformat(),
        }


def _calculate_health_status(monitor_status: Dict[str, Any]) -> Tuple[str, str]:
    """
    Calcula el estado general de salud y detección de stalls.

    Args:
        monitor_status: Estado actual del monitor CIMA

    Returns:
        Tupla (health_status, stall_detection)
        - health_status: "healthy" | "degraded" | "unhealthy"
        - stall_detection: "healthy" | "warning" | "critical" | "disabled"
    """
    is_stalled = monitor_status.get("stall_detection", {}).get("is_stalled", False)
    monitoring_active = monitor_status.get("monitoring_active", False)
    heartbeat = monitor_status.get("sync_heartbeat")
    heartbeat_tracker = monitor_status.get("heartbeat_tracker", {})

    # Determinar estado de salud general
    if not monitoring_active:
        return "unhealthy", "disabled"
    elif is_stalled:
        return "degraded", "critical"
    elif heartbeat:
        # Verificar edad del heartbeat
        time_since_heartbeat = heartbeat_tracker.get("time_since_heartbeat_seconds")
        if time_since_heartbeat and time_since_heartbeat > 600:  # 10 minutos
            return "degraded", "warning"
        else:
            return "healthy", "healthy"
    else:
        return "healthy", "healthy"


def _build_health_response(health_status: str, stall_detection: str, monitor_status: Dict[str, Any]) -> Dict[str, Any]:
    """
    Construye la respuesta del health check con toda la información.

    Args:
        health_status: Estado general de salud
        stall_detection: Estado de detección de stalls
        monitor_status: Estado completo del monitor CIMA

    Returns:
        Diccionario con respuesta completa del health check
    """
    heartbeat_tracker = monitor_status.get("heartbeat_tracker", {})
    recovery_status = monitor_status.get("recovery_status", {})
    circuit_breaker = recovery_status.get("circuit_breaker", {})

    # Información del heartbeat
    last_heartbeat_time = None
    time_since_heartbeat_seconds = None
    if heartbeat_tracker.get("has_heartbeat"):
        last_heartbeat_data = heartbeat_tracker.get("last_heartbeat")
        if last_heartbeat_data:
            last_heartbeat_time = last_heartbeat_data.get("timestamp")
            time_since_heartbeat_seconds = heartbeat_tracker.get("time_since_heartbeat_seconds")

    # Próximo chequeo
    monitor_interval = monitor_status.get("config", {}).get("monitor_interval", 300)
    next_check_in_seconds = monitor_interval  # Aproximación

    # Recovery attempts
    recovery_attempts_today = recovery_status.get("restart_count_today", 0)

    # Circuit breaker state
    circuit_state = circuit_breaker.get("state", "UNKNOWN")

    return {
        "status": health_status,
        "monitoring_active": monitor_status.get("monitoring_active", False),
        "redis_available": heartbeat_tracker.get("redis_available", False),
        "storage_backend": heartbeat_tracker.get("storage_backend", "unknown"),
        "last_heartbeat": last_heartbeat_time,
        "time_since_heartbeat_seconds": time_since_heartbeat_seconds,
        "stall_detection": stall_detection,
        "recovery_attempts_today": recovery_attempts_today,
        "circuit_breaker_state": circuit_state,
        "last_recovery_attempt": recovery_status.get("last_restart"),
        "next_check_in_seconds": next_check_in_seconds,
        "timestamp": utc_now().isoformat(),
    }


@router.get("/health/watchdog", response_model=Dict[str, Any])
async def get_watchdog_health():
    """
    Health check mejorado del watchdog/monitor CIMA.
    Issue #114 - Fase 2: Endpoint mejorado con circuit breaker y resiliencia.

    Returns:
        Estado detallado del sistema de monitoreo CIMA con información de:
        - Estado general del monitoreo
        - Disponibilidad de Redis
        - Estado del heartbeat
        - Detección de stalls
        - Intentos de recovery
        - Circuit breaker status
        - Próximo chequeo programado
    """
    try:
        monitor_status = get_cima_monitor_status()

        # Calcular estado de salud
        health_status, stall_detection = _calculate_health_status(monitor_status)

        # Construir respuesta completa
        response = _build_health_response(health_status, stall_detection, monitor_status)

        # Log para debugging
        logger.debug(
            "watchdog.health_check",
            status=health_status,
            stall_detection=stall_detection,
            circuit_state=response["circuit_breaker_state"],
        )

        return response

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("watchdog.health_check.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        return {
            "status": "unhealthy",
            "monitoring_active": False,
            "redis_available": False,
            "error": str(e)[:200],
            "timestamp": utc_now().isoformat(),
        }


@router.get("/cache/stats")
async def get_cache_stats() -> Dict[str, Any]:
    """
    Get Redis cache statistics for monitoring cache hit rates.
    Issue #194: Endpoint for monitoring enrichment cache performance.

    Returns:
        Cache statistics including hit rate, total requests, and error count
    """
    try:
        from app.services.enrichment_cache import enrichment_cache

        # Get cache stats from enrichment_cache service
        cache_stats = enrichment_cache.get_stats()

        # Check cache health
        cache_healthy = await enrichment_cache.check_health()

        return {
            "status": "success",
            "cache_stats": cache_stats,
            "health": {"redis_available": cache_healthy, "cache_enabled": cache_stats["enabled"]},
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error("cache.stats.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        return {
            "status": "error",
            "cache_stats": {
                "enabled": False,
                "hits": 0,
                "misses": 0,
                "errors": 0,
                "invalidations": 0,
                "hit_rate": 0,
                "total_requests": 0,
            },
            "health": {"redis_available": False, "cache_enabled": False},
            "error": str(e)[:200],
            "timestamp": utc_now().isoformat(),
        }


@router.get("/watchdog-stats", response_model=Dict[str, Any])
async def get_watchdog_stats():
    """
    Obtener estadísticas del monitor CIMA consolidado.

    NOTA: Este endpoint mantiene el nombre "watchdog" por compatibilidad histórica,
    pero usa internamente el servicio CimaSyncMonitor consolidado (Issue #109).
    El término "watchdog" es legacy - el monitor actual implementa circuit breaker,
    heartbeat tracking, y auto-recovery.
    """
    try:
        # Usar el servicio consolidado CimaSyncMonitor
        monitor_status = get_cima_monitor_status()

        # Mapear a formato compatible con watchdog antiguo
        watchdog_compatible_stats = {
            "watchdog_active": monitor_status.get("monitoring_active", False),
            "restart_count_today": monitor_status.get("recovery_status", {}).get("restart_count_today", 0),
            "max_restarts_allowed": monitor_status.get("config", {}).get("max_daily_attempts", 3),
            "last_restart": monitor_status.get("recovery_status", {}).get("last_restart"),
            "last_progress_check": (
                monitor_status.get("sync_heartbeat", {}).get("last_update")
                if monitor_status.get("sync_heartbeat")
                else None
            ),
            "config": {
                "check_interval_minutes": monitor_status.get("config", {}).get("monitor_interval", 300) // 60,
                "stall_threshold_minutes": monitor_status.get("config", {}).get("stall_timeout", 1200) // 60,
                "restart_cooldown_hours": monitor_status.get("config", {}).get("recovery_cooldown", 14400) // 3600,
            },
            # Nuevos campos del monitor consolidado
            "is_stalled": monitor_status.get("stall_detection", {}).get("is_stalled", False),
            "can_recover": monitor_status.get("recovery_status", {}).get("can_attempt_recovery", False),
        }

        return {"watchdog_stats": watchdog_compatible_stats, "timestamp": utc_now().isoformat(), "status": "success"}

    except Exception as e:
        # Log completo con traceback para debugging
        logger.error("watchdog.stats.error", error=str(e), error_type=type(e).__name__, exc_info=True)
        # API response con error truncado para evitar leakage
        return {
            "watchdog_stats": {"watchdog_active": False, "error": str(e)[:200]},
            "timestamp": utc_now().isoformat(),
            "status": "error",
        }
