# backend/app/api/subscriptions.py
"""
API endpoints for subscription management.

Issue #444: Implementar caducidad automatica de planes PRO y MAX.

Includes:
- User subscription status endpoint
- Admin subscription management endpoints
- Subscription expiration check endpoint
"""
import logging
from datetime import datetime
from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query, Request, status
from pydantic import BaseModel, ConfigDict, Field
from slowapi import Limiter
from slowapi.util import get_remote_address
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.core.security import require_permissions
from app.core.subscription_limits import Permission, SubscriptionPlan
from app.models.user import User
from app.services.audit_service import AuditAction, AuditService
from app.services.subscription_expiration_service import get_subscription_expiration_service

logger = logging.getLogger(__name__)
router = APIRouter()

# Rate limiter for admin operations
limiter = Limiter(key_func=get_remote_address)


# ============================================================================
# Pydantic Schemas
# ============================================================================


class SubscriptionStatusResponse(BaseModel):
    """Response schema for subscription status"""

    model_config = ConfigDict(from_attributes=True)

    plan: str = Field(..., description="Current subscription plan (free, pro, max)")
    is_active: bool = Field(..., description="Whether the user account is active")
    subscription_start: Optional[datetime] = Field(
        None, description="When the subscription started"
    )
    subscription_end: Optional[datetime] = Field(
        None, description="When the subscription expires (null for FREE)"
    )
    days_remaining: Optional[int] = Field(
        None, description="Days until expiration (null for FREE)"
    )
    is_expiring_soon: bool = Field(
        False, description="True if <= 7 days remaining"
    )
    is_expired: bool = Field(False, description="True if subscription has expired")


class SubscriptionUpdateRequest(BaseModel):
    """Request schema for updating subscription"""

    plan: str = Field(
        ...,
        description="New subscription plan (free, pro, max)",
        pattern="^(free|pro|max)$",
    )
    subscription_end: Optional[datetime] = Field(
        None, description="New expiration date (required for pro/max)"
    )
    subscription_start: Optional[datetime] = Field(
        None, description="New start date (defaults to now)"
    )


class SubscriptionUpdateResponse(BaseModel):
    """Response schema for subscription update"""

    success: bool
    message: str
    user_id: str
    plan: str
    subscription_end: Optional[datetime]


class ExpirationCheckResponse(BaseModel):
    """Response schema for expiration check"""

    total_checked: int = Field(..., description="Total subscriptions checked")
    expired_count: int = Field(..., description="Number of expired subscriptions")
    downgraded_user_ids: list[str] = Field(
        ..., description="List of downgraded user IDs (GDPR-compliant, no emails)"
    )
    errors: list[str] = Field(..., description="Any errors encountered")


class SubscriptionStatsResponse(BaseModel):
    """Response schema for subscription statistics"""

    by_plan: dict[str, int] = Field(..., description="Count of users by plan")
    expiring_soon_7_days: int = Field(
        ..., description="Users expiring in next 7 days"
    )
    expired_not_downgraded: int = Field(
        ..., description="Expired but not yet downgraded"
    )
    total_users: int = Field(..., description="Total active users")


class AdminExpiringUserResponse(BaseModel):
    """Response schema for admin expiring-soon endpoint with user identification"""

    model_config = ConfigDict(from_attributes=True)

    user_id: str = Field(..., description="User UUID")
    email: str = Field(..., description="User email for admin contact")
    full_name: Optional[str] = Field(None, description="User full name")
    plan: str = Field(..., description="Current subscription plan")
    subscription_end: Optional[datetime] = Field(None, description="Expiration date")
    days_remaining: Optional[int] = Field(None, description="Days until expiration")
    is_expiring_soon: bool = Field(False, description="True if <= 7 days remaining")


class PaginatedExpiringUsersResponse(BaseModel):
    """Paginated response for expiring users"""

    items: list[AdminExpiringUserResponse] = Field(..., description="Users list")
    total: int = Field(..., description="Total matching users")
    skip: int = Field(..., description="Records skipped")
    limit: int = Field(..., description="Max records returned")


# ============================================================================
# User Endpoints (authenticated users can check their own subscription)
# ============================================================================


@router.get(
    "/subscription/status",
    response_model=SubscriptionStatusResponse,
    summary="Get current subscription status",
    description="Returns the current user's subscription status including days remaining",
)
async def get_subscription_status(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Get the current user's subscription status.

    Returns subscription plan, expiration date, days remaining,
    and warning flags for expiring subscriptions.

    **Authentication required:** Yes (any authenticated user)

    Returns:
        SubscriptionStatusResponse with current subscription details
    """
    service = get_subscription_expiration_service(db)
    status = service.get_subscription_status(current_user)

    return SubscriptionStatusResponse(
        plan=status.plan,
        is_active=status.is_active,
        subscription_start=status.subscription_start,
        subscription_end=status.subscription_end,
        days_remaining=status.days_remaining,
        is_expiring_soon=status.is_expiring_soon,
        is_expired=status.is_expired,
    )


# ============================================================================
# Admin Endpoints (require MANAGE_USERS permission)
# ============================================================================


@router.put(
    "/admin/users/{user_id}/subscription",
    response_model=SubscriptionUpdateResponse,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))],
    summary="Update user subscription",
    description="Update a user's subscription plan and expiration date",
)
@limiter.limit("10/minute")
async def update_user_subscription(
    request: Request,
    user_id: UUID,
    subscription_update: SubscriptionUpdateRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Update a user's subscription plan and dates.

    **Permissions required:** MANAGE_USERS

    **Rate limit:** 10 requests per minute

    Args:
        user_id: ID of the user to update
        subscription_update: New subscription details

    Returns:
        SubscriptionUpdateResponse with result

    Raises:
        400: Invalid plan or missing expiration date for PRO/MAX
        404: User not found
        403: Insufficient permissions
        429: Rate limit exceeded
    """
    # Find target user
    target_user = db.query(User).filter(User.id == user_id).first()

    if not target_user:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"User with ID {user_id} not found",
        )

    # Validate: PRO/MAX plans should have expiration date
    if subscription_update.plan in [SubscriptionPlan.PRO, SubscriptionPlan.MAX]:
        if not subscription_update.subscription_end:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"subscription_end is required for {subscription_update.plan} plan",
            )

    service = get_subscription_expiration_service(db)
    success, message = service.update_subscription(
        user=target_user,
        plan=subscription_update.plan,
        subscription_end=subscription_update.subscription_end,
        subscription_start=subscription_update.subscription_start,
        admin_user=current_user,
    )

    if not success:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=message,
        )

    return SubscriptionUpdateResponse(
        success=True,
        message=message,
        user_id=str(user_id),
        plan=subscription_update.plan,
        subscription_end=target_user.subscription_end,
    )


@router.post(
    "/admin/subscriptions/check-expirations",
    response_model=ExpirationCheckResponse,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))],
    summary="Check and expire subscriptions",
    description="Manually trigger subscription expiration check",
)
@limiter.limit("5/minute")
async def check_expirations(
    request: Request,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Manually trigger the subscription expiration check.

    This endpoint allows admins to manually run the expiration check
    that normally runs daily. It will:
    1. Find all PRO/MAX subscriptions that have expired
    2. Downgrade them to FREE
    3. Create audit logs for each downgrade

    **Permissions required:** MANAGE_USERS

    **Rate limit:** 5 requests per minute

    Returns:
        ExpirationCheckResponse with results

    Raises:
        403: Insufficient permissions
        429: Rate limit exceeded
    """
    service = get_subscription_expiration_service(db)
    result = service.check_and_expire_subscriptions()

    # Audit log for manual check
    audit_service = AuditService(db)
    audit_service.log_action(
        action=AuditAction.ADMIN_ACTION,
        method="POST",
        endpoint="/api/v1/admin/subscriptions/check-expirations",
        user=current_user,
        request=request,
        resource_type="subscription",
        resource_id="batch",
        description=f"Manual expiration check: {result.expired_count} downgraded",
        details={
            "total_checked": result.total_checked,
            "expired_count": result.expired_count,
            "downgraded_users": result.downgraded_users,
            "errors": result.errors,
        },
        success=len(result.errors) == 0,
    )

    return ExpirationCheckResponse(
        total_checked=result.total_checked,
        expired_count=result.expired_count,
        downgraded_user_ids=result.downgraded_user_ids,
        errors=result.errors,
    )


@router.get(
    "/admin/subscriptions/stats",
    response_model=SubscriptionStatsResponse,
    dependencies=[Depends(require_permissions(Permission.VIEW_SYSTEM_STATS.value))],
    summary="Get subscription statistics",
    description="Get statistics about subscription plans and expirations",
)
async def get_subscription_stats(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Get subscription statistics for admin dashboard.

    Returns counts by plan, expiring soon, and expired but not downgraded.

    **Permissions required:** VIEW_SYSTEM_STATS

    Returns:
        SubscriptionStatsResponse with statistics
    """
    service = get_subscription_expiration_service(db)
    stats = service.get_subscription_stats()

    return SubscriptionStatsResponse(**stats)


@router.get(
    "/admin/subscriptions/expiring-soon",
    response_model=PaginatedExpiringUsersResponse,
    dependencies=[Depends(require_permissions(Permission.VIEW_USERS.value))],
    summary="Get users with expiring subscriptions",
    description="Get paginated list of users whose subscriptions expire soon",
)
async def get_expiring_soon(
    days: int = Query(7, ge=1, le=30, description="Days to look ahead"),
    skip: int = Query(0, ge=0, description="Number of records to skip"),
    limit: int = Query(50, ge=1, le=100, description="Max records to return"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Get users whose subscriptions expire within N days (paginated).

    **Permissions required:** VIEW_USERS

    Args:
        days: Number of days to look ahead (1-30, default 7)
        skip: Number of records to skip for pagination (default 0)
        limit: Maximum records to return (1-100, default 50)

    Returns:
        PaginatedExpiringUsersResponse with users and pagination info
    """
    service = get_subscription_expiration_service(db)
    expiring_users = service.get_expiring_soon(days=days)

    # Apply pagination
    total = len(expiring_users)
    paginated_users = expiring_users[skip:skip + limit]

    items = []
    for user in paginated_users:
        status = service.get_subscription_status(user)
        items.append(
            AdminExpiringUserResponse(
                user_id=str(user.id),
                email=user.email,
                full_name=user.full_name,
                plan=status.plan,
                subscription_end=status.subscription_end,
                days_remaining=status.days_remaining,
                is_expiring_soon=status.is_expiring_soon,
            )
        )

    return PaginatedExpiringUsersResponse(
        items=items,
        total=total,
        skip=skip,
        limit=limit,
    )
