﻿# backend/app/api/sales.py
"""
API endpoints para análisis de datos de ventas.
Proporciona datos para el dashboard frontend.
Incluye endpoints de análisis enriquecidos con datos CIMA/nomenclator.
"""

from datetime import date, datetime, timedelta
from typing import Any, Dict, List, Optional
from uuid import UUID

import structlog
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import and_, desc, extract, func
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now
from app.utils.subscription_helpers import get_data_date_limit_for_user

from ..database import get_db
from ..models.enums import PrescriptionCategory
from ..models.file_upload import FileUpload
from ..models.pharmacy import Pharmacy
from ..models.product_catalog import ProductCatalog

# Imports lazy de modelos para evitar conflictos SQLAlchemy MetaData
from ..models.sales_data import SalesData
from ..models.sales_enrichment import SalesEnrichment
from ..models.user import User
from .deps import get_current_active_user
from .error_handler import handle_exception

logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/sales", tags=["sales"])


# =============================================================================
# HELPER FUNCTIONS (Issue #16 Gap 2)
# =============================================================================


def _apply_prescription_category_filter(
    query,
    prescription_categories: Optional[List[str]],
    db: Session,
    already_joined_enrichment: bool = False,
    already_joined_catalog: bool = False,
):
    """
    Aplicar filtro de categorias de prescripcion a una query de SalesData.

    Args:
        query: Query base de SQLAlchemy sobre SalesData
        prescription_categories: Lista de categorias a filtrar (valores del enum PrescriptionCategory)
        db: Session de base de datos (no usado directamente, pero consistente con patron)
        already_joined_enrichment: Si la query ya tiene join con SalesEnrichment
        already_joined_catalog: Si la query ya tiene join con ProductCatalog

    Returns:
        Query modificada con filtro de categorias

    Raises:
        HTTPException 400: Si alguna categoria no es valida
    """
    if not prescription_categories:
        return query

    # Validar y convertir categorias a enum
    try:
        category_enums = [PrescriptionCategory(cat) for cat in prescription_categories]
    except ValueError as e:
        raise HTTPException(
            status_code=400,
            detail=f"Categoría de prescripción inválida: {str(e)}. "
            f"Valores válidos: {[c.value for c in PrescriptionCategory]}"
        )

    # Aplicar joins necesarios si no estan ya aplicados
    if not already_joined_enrichment:
        query = query.join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)

    if not already_joined_catalog:
        query = query.join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)

    # Aplicar filtro por categorias
    return query.filter(ProductCatalog.xfarma_prescription_category.in_(category_enums))


@router.get("/pharmacies")
async def get_available_pharmacies(
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Obtener lista de farmacias con datos de ventas disponibles.

    Returns:
        Lista de farmacias con información básica
    """
    try:
        # Obtener farmacias con estadísticas agregadas en una sola query optimizada
        pharmacy_stats = (
            db.query(
                Pharmacy.id,
                Pharmacy.name,
                Pharmacy.code,
                Pharmacy.city,
                func.count(SalesData.id).label("sales_count"),
                func.count(SalesEnrichment.id).label("enrichment_count"),
            )
            .join(SalesData, Pharmacy.id == SalesData.pharmacy_id)
            .outerjoin(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .group_by(Pharmacy.id, Pharmacy.name, Pharmacy.code, Pharmacy.city)
            .all()
        )

        result = []
        for stats in pharmacy_stats:
            sales_count = stats.sales_count or 0
            enrichment_count = stats.enrichment_count or 0

            result.append(
                {
                    "id": str(stats.id),
                    "name": stats.name,
                    "code": stats.code,
                    "city": stats.city,
                    "sales_records": sales_count,
                    "enriched_records": enrichment_count,
                    "enrichment_rate": (round((enrichment_count / sales_count * 100), 1) if sales_count > 0 else 0),
                }
            )

        return {
            "pharmacies": result,
            "total_pharmacies": len(result),
            "default_pharmacy_id": result[0]["id"] if result else None,
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("pharmacies.fetch_error", extra={"error": str(e), "endpoint": "get_available_pharmacies"})
        raise handle_exception(e, {"endpoint": "get_available_pharmacies"})


@router.get("/summary")
async def get_sales_summary(
    days: int = Query(30, ge=1, le=365, description="Número de días hacia atrás"),
    pharmacy_id: Optional[str] = Query(None, description="ID de farmacia específica"),
    prescription_categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (ej: medicamentos, conjunto_homogeneo, veterinaria)"
    ),
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Obtener resumen de ventas para KPIs del dashboard.

    Args:
        days: Número de días hacia atrás a incluir
        pharmacy_id: ID de farmacia (opcional)
        prescription_categories: Lista de categorías de prescripción a filtrar (opcional)
        db: Sesión de base de datos

    Returns:
        Diccionario con métricas de resumen
    """

    try:
        # Calcular fechas
        end_date = utc_now()
        start_date = end_date - timedelta(days=days)

        # REGLA #18 (Issue #142): Restricción de 3 meses para usuarios FREE
        free_tier_date_limit = get_data_date_limit_for_user(current_user, db)
        if free_tier_date_limit:
            # Usuario FREE: aplicar límite de 3 meses desde última venta
            original_start = start_date
            start_date = max(start_date, free_tier_date_limit)

            if start_date != original_start:
                logger.info(
                    "sales_summary.free_tier_restriction",
                    extra={
                        "user_id": str(current_user.id),
                        "pharmacy_id": pharmacy_id or "all",
                        "original_start": original_start.date().isoformat(),
                        "adjusted_start": start_date.date().isoformat(),
                        "restriction": "3_months_from_last_sale"
                    }
                )

        # Período anterior para comparación
        previous_start = start_date - timedelta(days=days)
        previous_end = start_date

        # Query base
        base_query = db.query(SalesData)
        if pharmacy_id:
            base_query = base_query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Aplicar filtro de categorías de prescripción (Issue #16 Gap 2)
        base_query = _apply_prescription_category_filter(base_query, prescription_categories, db)

        # Datos del período actual
        current_query = base_query.filter(
            and_(
                SalesData.sale_date >= start_date.date(),
                SalesData.sale_date <= end_date.date(),
            )
        )

        # Datos del período anterior
        previous_query = base_query.filter(
            and_(
                SalesData.sale_date >= previous_start.date(),
                SalesData.sale_date < previous_end.date(),
            )
        )

        # Métricas actuales
        current_summary = current_query.with_entities(
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.quantity).label("products_sold"),
            func.count(SalesData.id).label("total_transactions"),
            func.avg(SalesData.total_amount).label("avg_transaction"),
        ).first()

        # Métricas anteriores
        previous_summary = previous_query.with_entities(
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.quantity).label("products_sold"),
            func.count(SalesData.id).label("total_transactions"),
            func.avg(SalesData.total_amount).label("avg_transaction"),
        ).first()

        # Formatear respuesta
        return {
            "total_sales": float(current_summary.total_sales or 0),
            "products_sold": int(current_summary.products_sold or 0),
            "total_transactions": int(current_summary.total_transactions or 0),
            "avg_ticket": float(current_summary.avg_transaction or 0),
            "gross_margin": 0.0,  # Sin datos de margen disponibles
            # Valores anteriores para comparación
            "previous_total_sales": (
                float(previous_summary.total_sales or 0) if previous_summary.total_sales else None
            ),
            "previous_products_sold": (
                int(previous_summary.products_sold or 0) if previous_summary.products_sold else None
            ),
            "previous_avg_ticket": (
                float(previous_summary.avg_transaction or 0) if previous_summary.avg_transaction else None
            ),
            "previous_gross_margin": None,  # Sin datos de margen disponibles
            # Metadatos
            "period_days": days,
            "start_date": start_date.date().isoformat(),
            "end_date": end_date.date().isoformat(),
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.summary_error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener resumen de ventas")


@router.get("/trends")
async def get_sales_trends(
    period: str = Query(
        "daily",
        enum=["daily", "weekly", "monthly"],
        description="Período de agregación",
    ),
    days: int = Query(30, ge=1, le=365, description="Número de días hacia atrás"),
    pharmacy_id: Optional[str] = Query(None, description="ID de farmacia específica"),
    prescription_categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (ej: medicamentos, conjunto_homogeneo, veterinaria)"
    ),
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> List[Dict[str, Any]]:
    """
    Obtener tendencias de ventas para gráficos.

    Args:
        period: Período de agregación (daily, weekly, monthly)
        days: Número de días hacia atrás
        pharmacy_id: ID de farmacia (opcional)
        prescription_categories: Lista de categorías de prescripción a filtrar (opcional)
        db: Sesión de base de datos

    Returns:
        Lista de puntos de datos con fecha y ventas
    """

    try:
        # Calcular fechas
        end_date = utc_now()
        start_date = end_date - timedelta(days=days)

        # REGLA #18 (Issue #142): Restricción de 3 meses para usuarios FREE
        free_tier_date_limit = get_data_date_limit_for_user(current_user, db)
        if free_tier_date_limit:
            # Usuario FREE: aplicar límite de 3 meses desde última venta
            original_start = start_date
            start_date = max(start_date, free_tier_date_limit)

            if start_date != original_start:
                logger.info(
                    "sales_trends.free_tier_restriction",
                    extra={
                        "user_id": str(current_user.id),
                        "pharmacy_id": pharmacy_id or "all",
                        "original_start": original_start.date().isoformat(),
                        "adjusted_start": start_date.date().isoformat(),
                        "restriction": "3_months_from_last_sale"
                    }
                )

        # Query base
        query = db.query(SalesData)
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Aplicar filtro de categorías de prescripción (Issue #16 Gap 2)
        query = _apply_prescription_category_filter(query, prescription_categories, db)

        query = query.filter(
            and_(
                SalesData.sale_date >= start_date.date(),
                SalesData.sale_date <= end_date.date(),
            )
        )

        # Agregación según el período
        if period == "daily":
            # Agrupar por día
            trends = (
                query.group_by(SalesData.sale_date)
                .with_entities(
                    SalesData.sale_date.label("date"),
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.sum(SalesData.quantity).label("total_quantity"),
                    func.count(SalesData.id).label("total_transactions"),
                )
                .order_by(SalesData.sale_date)
                .all()
            )

            # Convertir a formato esperado
            result = []
            for trend in trends:
                result.append(
                    {
                        "date": trend.date.isoformat(),
                        "total_sales": float(trend.total_sales or 0),
                        "total_quantity": int(trend.total_quantity or 0),
                        "total_transactions": int(trend.total_transactions or 0),
                    }
                )

        elif period == "weekly":
            # Agrupar por semana
            trends = (
                query.with_entities(
                    func.date_trunc("week", SalesData.sale_date).label("week_start"),
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.sum(SalesData.quantity).label("total_quantity"),
                    func.count(SalesData.id).label("total_transactions"),
                )
                .group_by(func.date_trunc("week", SalesData.sale_date))
                .order_by("week_start")
                .all()
            )

            result = []
            for trend in trends:
                result.append(
                    {
                        "date": trend.week_start.isoformat(),
                        "total_sales": float(trend.total_sales or 0),
                        "total_quantity": int(trend.total_quantity or 0),
                        "total_transactions": int(trend.total_transactions or 0),
                    }
                )

        elif period == "monthly":
            # Agrupar por mes
            trends = (
                query.with_entities(
                    extract("year", SalesData.sale_date).label("year"),
                    extract("month", SalesData.sale_date).label("month"),
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.sum(SalesData.quantity).label("total_quantity"),
                    func.count(SalesData.id).label("total_transactions"),
                )
                .group_by(
                    extract("year", SalesData.sale_date),
                    extract("month", SalesData.sale_date),
                )
                .order_by("year", "month")
                .all()
            )

            result = []
            for trend in trends:
                # Crear fecha del primer día del mes
                month_date = datetime(int(trend.year), int(trend.month), 1)
                result.append(
                    {
                        "date": month_date.date().isoformat(),
                        "month": f"{int(trend.month):02d}/{int(trend.year)}",
                        "total_sales": float(trend.total_sales or 0),
                        "total_quantity": int(trend.total_quantity or 0),
                        "total_transactions": int(trend.total_transactions or 0),
                    }
                )

        return result

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.trends.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener tendencias de ventas")


@router.get("/top-products")
async def get_top_products(
    limit: int = Query(10, ge=1, le=50, description="Número de productos a devolver"),
    days: int = Query(30, ge=1, le=365, description="Número de días hacia atrás"),
    pharmacy_id: Optional[str] = Query(None, description="ID de farmacia específica"),
    sort_by: str = Query("quantity", enum=["quantity", "amount"], description="Criterio de ordenación"),
    prescription_categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (ej: medicamentos, conjunto_homogeneo, veterinaria)"
    ),
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> List[Dict[str, Any]]:
    """
    Obtener productos más vendidos.

    Args:
        limit: Número máximo de productos a devolver
        days: Número de días hacia atrás
        pharmacy_id: ID de farmacia (opcional)
        sort_by: Criterio de ordenación (quantity o amount)
        prescription_categories: Lista de categorías de prescripción a filtrar (opcional)
        db: Sesión de base de datos

    Returns:
        Lista de productos más vendidos
    """

    try:
        # Calcular fechas
        end_date = utc_now()
        start_date = end_date - timedelta(days=days)

        # Query base
        query = db.query(SalesData)
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Aplicar filtro de categorías de prescripción (Issue #16 Gap 2)
        query = _apply_prescription_category_filter(query, prescription_categories, db)

        query = query.filter(
            and_(
                SalesData.sale_date >= start_date.date(),
                SalesData.sale_date <= end_date.date(),
            )
        )

        # Agrupar por producto
        if sort_by == "quantity":
            order_field = desc(func.sum(SalesData.quantity))
        else:
            order_field = desc(func.sum(SalesData.total_amount))

        top_products = (
            query.group_by(SalesData.codigo_nacional, SalesData.product_name)
            .with_entities(
                SalesData.codigo_nacional,
                SalesData.product_name,
                func.sum(SalesData.quantity).label("quantity_sold"),
                func.sum(SalesData.total_amount).label("amount_sold"),
                func.avg(SalesData.unit_price).label("avg_price"),
                func.count(SalesData.id).label("transaction_count"),
            )
            .order_by(order_field)
            .limit(limit)
            .all()
        )

        # Formatear respuesta
        result = []
        for i, product in enumerate(top_products):
            result.append(
                {
                    "rank": i + 1,
                    "codigo_nacional": product.codigo_nacional,  # Campo correcto según DATA_CATALOG.md
                    "product_name": product.product_name,
                    "quantity_sold": int(product.quantity_sold or 0),
                    "amount_sold": float(product.amount_sold or 0),
                    "avg_price": float(product.avg_price or 0),
                    "transaction_count": int(product.transaction_count or 0),
                    "avg_per_transaction": (
                        float(product.quantity_sold / product.transaction_count) if product.transaction_count > 0 else 0
                    ),
                }
            )

        return result

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.top_products.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener productos más vendidos")


@router.get("/categories")
async def get_sales_by_category(
    days: int = Query(30, ge=1, le=365, description="Número de días hacia atrás"),
    pharmacy_id: Optional[str] = Query(None, description="ID de farmacia específica"),
    prescription_categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (ej: medicamentos, conjunto_homogeneo, veterinaria)"
    ),
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> List[Dict[str, Any]]:
    """
    Obtener ventas por categorías de producto usando datos enriquecidos.
    Usa product_type y therapeutic_category de SalesEnrichment.

    Args:
        days: Número de días hacia atrás
        pharmacy_id: ID de farmacia (opcional)
        prescription_categories: Lista de categorías de prescripción a filtrar (opcional)
        db: Sesión de base de datos

    Returns:
        Lista de ventas por categoría
    """

    try:
        # Obtener datos reales usando enriquecimiento
        end_date = utc_now()
        start_date = end_date - timedelta(days=days)

        query = db.query(SalesData)
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Aplicar filtro de categorías de prescripción (Issue #16 Gap 2)
        # Ya tendremos join con SalesEnrichment y ProductCatalog despues
        query = _apply_prescription_category_filter(query, prescription_categories, db)
        has_prescription_filter = prescription_categories is not None and len(prescription_categories) > 0

        query = query.filter(
            and_(
                SalesData.sale_date >= start_date.date(),
                SalesData.sale_date <= end_date.date(),
            )
        )

        # Usar datos enriquecidos para categorización real
        # Si ya aplicamos filtro de prescripcion, ya tenemos join con SalesEnrichment
        if has_prescription_filter:
            categories_query = (
                query.with_entities(
                    SalesEnrichment.product_type.label("category"),
                    func.sum(SalesData.total_amount).label("sales"),
                    func.sum(SalesData.quantity).label("quantity"),
                    func.count(SalesData.id).label("transactions"),
                )
                .filter(SalesEnrichment.product_type.isnot(None))
                .group_by(SalesEnrichment.product_type)
                .order_by(desc("sales"))
                .all()
            )
        else:
            categories_query = (
                query.join(SalesEnrichment)
                .with_entities(
                    SalesEnrichment.product_type.label("category"),
                    func.sum(SalesData.total_amount).label("sales"),
                    func.sum(SalesData.quantity).label("quantity"),
                    func.count(SalesData.id).label("transactions"),
                )
                .filter(SalesEnrichment.product_type.isnot(None))
                .group_by(SalesEnrichment.product_type)
                .order_by(desc("sales"))
                .all()
            )

        # Calcular total para porcentajes (convertir a float)
        total_sales = sum(float(cat.sales) for cat in categories_query)

        if total_sales == 0:
            # Fallback a datos simulados si no hay enriquecimiento
            fallback_total = query.with_entities(func.sum(SalesData.total_amount)).scalar() or 0
            if fallback_total == 0:
                return []
            return [
                {
                    "category": "Sin Categorizar",
                    "sales": fallback_total,
                    "percentage": 1.0,
                    "quantity": 0,
                    "transactions": 0,
                }
            ]

        # Formatear respuesta con datos reales
        categories = []
        for cat in categories_query:
            categories.append(
                {
                    "category": (cat.category.title() if cat.category else "Sin Categorizar"),
                    "sales": float(cat.sales),
                    "quantity": int(cat.quantity),
                    "transactions": int(cat.transactions),
                    "percentage": round(float(cat.sales) / total_sales, 3),
                }
            )

        return categories

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.by_category.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener ventas por categoría")


@router.get("/laboratories")
async def get_top_laboratories(
    limit: int = Query(10, ge=1, le=20, description="Número de laboratorios a devolver"),
    days: int = Query(30, ge=1, le=365, description="Número de días hacia atrás"),
    pharmacy_id: Optional[str] = Query(None, description="ID de farmacia específica"),
    prescription_categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (ej: medicamentos, conjunto_homogeneo, veterinaria)"
    ),
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> List[Dict[str, Any]]:
    """
    Obtener laboratorios con más ventas.

    Args:
        limit: Número máximo de laboratorios
        days: Número de días hacia atrás
        pharmacy_id: ID de farmacia (opcional)
        prescription_categories: Lista de categorías de prescripción a filtrar (opcional)
        db: Sesión de base de datos

    Returns:
        Lista de laboratorios más vendidos
    """

    try:
        # Calcular fechas
        end_date = utc_now()
        start_date = end_date - timedelta(days=days)

        # Query base
        query = db.query(SalesData)
        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        query = query.filter(
            and_(
                SalesData.sale_date >= start_date.date(),
                SalesData.sale_date <= end_date.date(),
            )
        )

        # JOIN con datos enriquecidos para obtener laboratorio
        query = (
            query.join(SalesEnrichment)
            .join(ProductCatalog)
            .filter(
                ProductCatalog.nomen_laboratorio.isnot(None),
                ProductCatalog.nomen_laboratorio != "",
            )
        )

        # Aplicar filtro de categorías de prescripción (Issue #16 Gap 2)
        # Ya tenemos join con SalesEnrichment y ProductCatalog
        query = _apply_prescription_category_filter(
            query, prescription_categories, db,
            already_joined_enrichment=True,
            already_joined_catalog=True
        )

        # Agrupar por laboratorio (ahora desde ProductCatalog)
        top_labs = (
            query.group_by(ProductCatalog.nomen_laboratorio)
            .with_entities(
                ProductCatalog.nomen_laboratorio,
                func.sum(SalesData.total_amount).label("total_sales"),
                func.sum(SalesData.quantity).label("total_quantity"),
                func.count(SalesData.id).label("product_count"),
                func.avg(SalesData.margin_percentage).label("avg_margin"),
            )
            .order_by(desc(func.sum(SalesData.total_amount)))
            .limit(limit)
            .all()
        )

        # Formatear respuesta
        result = []
        for i, lab in enumerate(top_labs):
            result.append(
                {
                    "rank": i + 1,
                    "laboratory": lab.nomen_laboratorio,
                    "total_sales": float(lab.total_sales or 0),
                    "total_quantity": int(lab.total_quantity or 0),
                    "product_count": int(lab.product_count or 0),
                    "avg_margin": (float(lab.avg_margin or 0) / 100 if lab.avg_margin else 0),
                }
            )

        return result

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.top_laboratories.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener laboratorios más vendidos")


# ENDPOINTS DE ANÁLISIS ENRIQUECIDO CON DATOS CIMA/NOMENCLATOR


@router.get("/enriched/summary/{pharmacy_id}")
async def get_enriched_sales_summary(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
) -> Dict[str, Any]:
    """
    Resumen de ventas enriquecidas con datos CIMA/nomenclator
    Métricas clave para dashboard principal
    """
    try:
        # Base query
        query = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id)

        # Filtros de fecha
        if start_date:
            query = query.filter(SalesData.sale_date >= start_date)
        if end_date:
            query = query.filter(SalesData.sale_date <= end_date)

        # Estadísticas básicas
        total_sales = query.count()
        total_amount = query.with_entities(func.sum(SalesData.total_amount)).scalar() or 0

        # Ventas enriquecidas
        enriched_query = query.join(SalesEnrichment).join(ProductCatalog)
        enriched_sales = enriched_query.count()
        enrichment_rate = (enriched_sales / total_sales * 100) if total_sales > 0 else 0

        # Análisis de genéricos (basado en tipo de fármaco del nomenclator)
        generics_data = (
            enriched_query.filter(ProductCatalog.nomen_tipo_farmaco.ilike("%genérico%"))
            .with_entities(
                func.count().label("count"),
                func.sum(SalesData.total_amount).label("amount"),
            )
            .first()
        )

        generic_count = generics_data.count if generics_data else 0
        generic_amount = generics_data.amount if generics_data else 0

        # Categorías terapéuticas top
        therapeutic_categories = (
            enriched_query.with_entities(
                SalesEnrichment.therapeutic_category,
                func.count().label("count"),
                func.sum(SalesData.total_amount).label("amount"),
            )
            .filter(SalesEnrichment.therapeutic_category.isnot(None))
            .group_by(SalesEnrichment.therapeutic_category)
            .order_by(desc("amount"))
            .limit(5)
            .all()
        )

        return {
            "period": {
                "start_date": start_date.isoformat() if start_date else None,
                "end_date": end_date.isoformat() if end_date else None,
            },
            "metrics": {
                "total_sales": total_sales,
                "total_amount": float(total_amount),
                "enriched_sales": enriched_sales,
                "enrichment_rate": round(enrichment_rate, 2),
                "generic_sales": {
                    "count": generic_count,
                    "amount": float(generic_amount or 0),
                    "percentage": round((generic_count / total_sales * 100) if total_sales > 0 else 0, 2),
                },
            },
            "therapeutic_categories": [
                {
                    "category": cat.therapeutic_category,
                    "sales_count": cat.count,
                    "amount": float(cat.amount),
                }
                for cat in therapeutic_categories
            ],
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.enriched_summary.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener resumen de ventas enriquecidas")


@router.get("/enriched/generic-opportunities/{pharmacy_id}")
async def get_generic_opportunities(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
    min_savings: float = Query(0.0, description="Mínimo ahorro potencial en €"),
) -> Dict[str, Any]:
    """
    Análisis detallado de oportunidades de genéricos
    Comparación marca vs genérico con datos oficiales
    """
    try:
        # Consulta base con enriquecimiento
        query = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id)

        if start_date:
            query = query.filter(SalesData.sale_date >= start_date)
        if end_date:
            query = query.filter(SalesData.sale_date <= end_date)

        # Ventas de marca con genérico disponible
        branded_sales = (
            query.join(SalesEnrichment)
            .join(ProductCatalog, SalesEnrichment.product_catalog_id == ProductCatalog.id)
            .filter(
                ProductCatalog.nomen_tipo_farmaco == "MARCA",
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.nomen_pvp.isnot(None),
            )
            .all()
        )

        opportunities = []
        total_potential_savings = 0

        # Porcentaje de descuento por defecto para laboratorios partners
        default_discount_percentage = 25.0  # 25% descuento sobre PVP

        for sale in branded_sales:
            # Verificar que tiene conjunto homogéneo (indica que hay alternativas disponibles)
            if sale.enrichment.product_catalog.nomen_codigo_homogeneo:
                product_catalog = sale.enrichment.product_catalog
                quantity = sale.quantity or 0
                pvp = float(product_catalog.nomen_pvp or 0)  # Precio Venta al Público

                # CÁLCULO CORRECTO DEL AHORRO POTENCIAL:
                # En el mismo conjunto homogéneo, genérico y marca tienen el MISMO PVL
                # El ahorro viene del descuento que pueden ofrecer los laboratorios partners
                potential_savings = 0
                savings_calculation = {}

                if pvp > 0 and quantity > 0:
                    # Ahorro = PVP × descuento_% × cantidad
                    potential_savings = (pvp * default_discount_percentage / 100) * quantity
                    total_potential_savings += potential_savings

                    savings_calculation = {
                        "pvp_unit": pvp,
                        "quantity": quantity,
                        "discount_percentage": default_discount_percentage,
                        "total_current_cost": pvp * quantity,
                        "cost_with_discount": pvp * (1 - default_discount_percentage / 100) * quantity,
                        "potential_savings": potential_savings,
                    }

                # Solo incluir en oportunidades si supera el mínimo de ahorro
                if potential_savings >= min_savings:
                    opportunities.append(
                        {
                            "sale_info": {
                                "sale_id": str(sale.id),
                                "sale_date": sale.sale_date.isoformat(),
                                "quantity": quantity,
                                "total_paid": float(sale.total_amount),
                            },
                            "branded_product": {
                                "name": product_catalog.cima_nombre_comercial,
                                "laboratory": product_catalog.nomen_laboratorio,
                                "official_price": pvp,
                                "homogeneous_group": product_catalog.nomen_codigo_homogeneo,
                            },
                            "calculations": savings_calculation,
                            "has_generic_alternatives": True,
                            "note": f"Ahorro calculado con {default_discount_percentage}% descuento sobre PVP",
                        }
                    )

        # Estadísticas agregadas
        opportunities_count = len(opportunities)

        # Calcular promedio basado en las oportunidades que realmente aparecen en la respuesta
        displayed_opportunities_savings = sum(
            opp["calculations"].get("potential_savings", 0) for opp in opportunities if "calculations" in opp
        )
        avg_savings_per_opportunity = (
            displayed_opportunities_savings / opportunities_count if opportunities_count > 0 else 0
        )

        return {
            "summary": {
                "total_opportunities": opportunities_count,
                "total_potential_savings": round(displayed_opportunities_savings, 2),
                "average_savings_per_opportunity": round(avg_savings_per_opportunity, 2),
                "discount_percentage_applied": default_discount_percentage,
                "total_branded_sales_analyzed": len(branded_sales),
                "min_savings_filter": min_savings,
            },
            "opportunities": opportunities,
            "period": {
                "start_date": start_date.isoformat() if start_date else None,
                "end_date": end_date.isoformat() if end_date else None,
            },
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.generic_opportunities.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener oportunidades de genéricos")


@router.get("/enriched/therapeutic-analysis/{pharmacy_id}")
async def get_therapeutic_analysis(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
) -> Dict[str, Any]:
    """
    Análisis por categorías terapéuticas ATC
    Distribución de ventas por grupo farmacológico
    """
    try:
        query = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id)

        if start_date:
            query = query.filter(SalesData.sale_date >= start_date)
        if end_date:
            query = query.filter(SalesData.sale_date <= end_date)

        # Análisis por grupo ATC (primeros 3 caracteres)
        atc_analysis = (
            query.join(SalesEnrichment)
            .join(ProductCatalog)
            .with_entities(
                func.substring(ProductCatalog.atc_code, 1, 3).label("atc_group"),
                ProductCatalog.therapeutic_group,
                func.count().label("sales_count"),
                func.sum(SalesData.total_amount).label("total_amount"),
                func.sum(SalesData.quantity).label("total_quantity"),
                func.avg(SalesData.total_amount).label("avg_sale_amount"),
            )
            .filter(ProductCatalog.atc_code.isnot(None))
            .group_by(
                func.substring(ProductCatalog.atc_code, 1, 3),
                ProductCatalog.therapeutic_group,
            )
            .order_by(desc("total_amount"))
            .all()
        )

        # Análisis de prescripción vs OTC
        prescription_analysis = (
            query.join(SalesEnrichment)
            .join(ProductCatalog)
            .with_entities(
                ProductCatalog.requires_prescription,
                func.count().label("count"),
                func.sum(SalesData.total_amount).label("amount"),
            )
            .filter(ProductCatalog.requires_prescription.isnot(None))
            .group_by(ProductCatalog.requires_prescription)
            .all()
        )

        return {
            "atc_groups": [
                {
                    "atc_code": result.atc_group,
                    "therapeutic_group": result.therapeutic_group,
                    "sales_count": result.sales_count,
                    "total_amount": float(result.total_amount),
                    "total_quantity": result.total_quantity,
                    "average_sale": round(float(result.avg_sale_amount), 2),
                }
                for result in atc_analysis
            ],
            "prescription_analysis": [
                {
                    "requires_prescription": result.requires_prescription,
                    "sales_count": result.count,
                    "total_amount": float(result.amount),
                    "percentage": round(
                        (result.count / sum(r.count for r in prescription_analysis) * 100),
                        2,
                    ),
                }
                for result in prescription_analysis
            ],
            "period": {
                "start_date": start_date.isoformat() if start_date else None,
                "end_date": end_date.isoformat() if end_date else None,
            },
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.therapeutic_analysis.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener análisis terapéutico")


@router.get("/enrichment/status/{pharmacy_id}")
async def get_enrichment_status(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Estado del proceso de enriquecimiento
    Métricas de calidad y cobertura de datos
    """
    try:
        # Estadísticas de archivos subidos
        uploads_stats = (
            db.query(FileUpload)
            .filter(FileUpload.pharmacy_id == pharmacy_id)
            .with_entities(
                func.count().label("total_files"),
                func.sum(FileUpload.rows_processed).label("total_records"),
            )
            .first()
        )

        # Estadísticas de enriquecimiento
        enrichment_stats = (
            db.query(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .outerjoin(SalesEnrichment)
            .with_entities(
                func.count(SalesData.id).label("total_sales"),
                func.count(SalesEnrichment.id).label("enriched_sales"),
                func.avg(SalesEnrichment.match_confidence).label("avg_confidence"),
            )
            .first()
        )

        # Distribución por método de matching
        match_methods = (
            db.query(SalesEnrichment)
            .join(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .with_entities(SalesEnrichment.match_method, func.count().label("count"))
            .group_by(SalesEnrichment.match_method)
            .all()
        )

        # Fuentes de enriquecimiento
        enrichment_sources = (
            db.query(SalesEnrichment)
            .join(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .with_entities(SalesEnrichment.enrichment_source, func.count().label("count"))
            .group_by(SalesEnrichment.enrichment_source)
            .all()
        )

        enrichment_rate = 0
        if enrichment_stats.total_sales > 0:
            enrichment_rate = enrichment_stats.enriched_sales / enrichment_stats.total_sales * 100

        return {
            "data_processing": {
                "total_files_uploaded": uploads_stats.total_files or 0,
                "total_records_processed": uploads_stats.total_records or 0,
                "processing_success_rate": 100.0,  # Simplificado por falta de campos de error
            },
            "enrichment_coverage": {
                "total_sales": enrichment_stats.total_sales or 0,
                "enriched_sales": enrichment_stats.enriched_sales or 0,
                "enrichment_rate": round(enrichment_rate, 2),
                "average_confidence": round(float(enrichment_stats.avg_confidence or 0), 2),
            },
            "match_methods": [
                {
                    "method": method.match_method,
                    "count": method.count,
                    "percentage": round(
                        (
                            (method.count / enrichment_stats.enriched_sales * 100)
                            if enrichment_stats.enriched_sales > 0
                            else 0
                        ),
                        2,
                    ),
                }
                for method in match_methods
            ],
            "enrichment_sources": [
                {"source": source.enrichment_source, "count": source.count} for source in enrichment_sources
            ],
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.enrichment_status.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener estado de enriquecimiento")


@router.get("/date-range/{pharmacy_id}")
async def get_sales_date_range(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, str]:
    """
    Obtener rango de fechas de ventas disponibles para una farmacia
    """
    try:
        # Obtener fechas mínima y máxima
        date_range = (
            db.query(
                func.min(SalesData.sale_date).label("min_date"),
                func.max(SalesData.sale_date).label("max_date"),
            )
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .first()
        )

        if not date_range or not date_range.min_date or not date_range.max_date:
            # Si no hay datos, usar un rango predeterminado
            from datetime import timedelta

            today = utc_now().date()
            min_date = today - timedelta(days=365)  # Un año atrás
            max_date = today
        else:
            min_date = date_range.min_date
            max_date = date_range.max_date

        return {"min_date": min_date.isoformat(), "max_date": max_date.isoformat()}

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error("sales.date_range.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener rango de fechas")


@router.get("/yoy-kpis/{pharmacy_id}")
async def get_yoy_kpis(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    KPIs de variación Year-over-Year (YoY) para el dashboard.

    Calcula la variación de ventas comparando:
    - QTD YoY: Lo que va de trimestre vs mismo período año anterior
    - MTD YoY: Lo que va de mes vs mismo período año anterior
    - WTD YoY: Lo que va de semana vs mismo período año anterior

    Returns:
        Diccionario con los 3 KPIs de variación YoY
    """
    from dateutil.relativedelta import relativedelta

    try:
        today = utc_now().date()

        # ========================================
        # QTD YoY: Quarter-to-Date Year-over-Year
        # ========================================
        quarter = ((today.month - 1) // 3) + 1
        quarter_start_month = (quarter - 1) * 3 + 1
        qtd_start = date(today.year, quarter_start_month, 1)
        qtd_end = today

        # Mismo período del año anterior
        qtd_prev_start = qtd_start - relativedelta(years=1)
        qtd_prev_end = qtd_end - relativedelta(years=1)

        # Consultas QTD
        qtd_current = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= qtd_start,
                SalesData.sale_date <= qtd_end,
            )
            .scalar() or 0
        )

        qtd_previous = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= qtd_prev_start,
                SalesData.sale_date <= qtd_prev_end,
            )
            .scalar() or 0
        )

        qtd_yoy_change = 0.0
        if qtd_previous > 0:
            qtd_yoy_change = ((float(qtd_current) - float(qtd_previous)) / float(qtd_previous)) * 100
        elif qtd_current > 0:
            qtd_yoy_change = 100.0  # Nuevo período con ventas

        # ========================================
        # MTD YoY: Month-to-Date Year-over-Year
        # ========================================
        mtd_start = date(today.year, today.month, 1)
        mtd_end = today

        mtd_prev_start = mtd_start - relativedelta(years=1)
        mtd_prev_end = mtd_end - relativedelta(years=1)

        mtd_current = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= mtd_start,
                SalesData.sale_date <= mtd_end,
            )
            .scalar() or 0
        )

        mtd_previous = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= mtd_prev_start,
                SalesData.sale_date <= mtd_prev_end,
            )
            .scalar() or 0
        )

        mtd_yoy_change = 0.0
        if mtd_previous > 0:
            mtd_yoy_change = ((float(mtd_current) - float(mtd_previous)) / float(mtd_previous)) * 100
        elif mtd_current > 0:
            mtd_yoy_change = 100.0

        # ========================================
        # WTD YoY: Week-to-Date Year-over-Year
        # ========================================
        # Lunes de esta semana (ISO week starts on Monday)
        wtd_start = today - timedelta(days=today.weekday())
        wtd_end = today

        wtd_prev_start = wtd_start - relativedelta(years=1)
        wtd_prev_end = wtd_end - relativedelta(years=1)

        wtd_current = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= wtd_start,
                SalesData.sale_date <= wtd_end,
            )
            .scalar() or 0
        )

        wtd_previous = (
            db.query(func.sum(SalesData.total_amount))
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.sale_date >= wtd_prev_start,
                SalesData.sale_date <= wtd_prev_end,
            )
            .scalar() or 0
        )

        wtd_yoy_change = 0.0
        if wtd_previous > 0:
            wtd_yoy_change = ((float(wtd_current) - float(wtd_previous)) / float(wtd_previous)) * 100
        elif wtd_current > 0:
            wtd_yoy_change = 100.0

        # Construir respuesta
        return {
            "qtd_yoy": {
                "label": f"Q{quarter} YoY",
                "change_percent": round(qtd_yoy_change, 1),
                "current_value": float(qtd_current),
                "previous_value": float(qtd_previous),
                "period": {
                    "current": {"from": qtd_start.isoformat(), "to": qtd_end.isoformat()},
                    "previous": {"from": qtd_prev_start.isoformat(), "to": qtd_prev_end.isoformat()},
                },
            },
            "mtd_yoy": {
                "label": "Mes YoY",
                "change_percent": round(mtd_yoy_change, 1),
                "current_value": float(mtd_current),
                "previous_value": float(mtd_previous),
                "period": {
                    "current": {"from": mtd_start.isoformat(), "to": mtd_end.isoformat()},
                    "previous": {"from": mtd_prev_start.isoformat(), "to": mtd_prev_end.isoformat()},
                },
            },
            "wtd_yoy": {
                "label": "Semana YoY",
                "change_percent": round(wtd_yoy_change, 1),
                "current_value": float(wtd_current),
                "previous_value": float(wtd_previous),
                "period": {
                    "current": {"from": wtd_start.isoformat(), "to": wtd_end.isoformat()},
                    "previous": {"from": wtd_prev_start.isoformat(), "to": wtd_prev_end.isoformat()},
                },
            },
            "reference_date": today.isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error("sales.yoy_kpis.error", error=str(e))
        raise HTTPException(status_code=500, detail="Error al obtener KPIs YoY")
