# backend/app/api/roi_tracker.py
"""
API endpoints para ROI Tracker (Feedback Loop de Acciones).

Issue #514: Cerrar el círculo diagnóstico → acción → resultado.
Demostrar ROI de xFarma con datos reales.

Endpoints:
- Actions CRUD: Create, list, get, execute, discard, postpone
- ROI Summary: Current month, specific month, history, recalculate
- Dashboard Widget: Quick stats for frontend display
"""

import logging
from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query, status as http_status
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.action_tracking import ActionStatus
from app.models.user import User
from app.schemas.roi_tracker import (
    ActionCreate,
    ActionDiscardRequest,
    ActionExecuteRequest,
    ActionListResponse,
    ActionPostponeRequest,
    ActionResponse,
    ROIDashboardWidget,
    ROIHistoryResponse,
    ROISummaryResponse,
)
from app.services.action_impact_calculator import ActionImpactCalculator
from app.services.action_tracking_service import ActionTrackingService
from app.services.roi_summary_service import ROISummaryService

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/roi-tracker", tags=["roi-tracker"])


# =============================================================================
# HELPER FUNCTIONS
# =============================================================================


def _action_to_response(action) -> ActionResponse:
    """Convert ActionTracking model to ActionResponse schema."""
    return ActionResponse(
        id=action.id,
        pharmacy_id=action.pharmacy_id,
        action_type=action.action_type,
        action_description=action.action_description,
        affected_products=action.affected_products or [],
        expected_impact_eur=action.expected_impact_eur,
        expected_impact_description=action.expected_impact_description,
        status=action.status,
        executed_at=action.executed_at,
        discarded_at=action.discarded_at,
        discard_reason=action.discard_reason,
        postponed_until=action.postponed_until,
        actual_impact_eur=action.actual_impact_eur,
        actual_impact_description=action.actual_impact_description,
        impact_calculated_at=action.impact_calculated_at,
        insight_hash=action.insight_hash,
        created_at=action.created_at,
        updated_at=action.updated_at,
        is_pending=action.is_pending,
        is_executed=action.is_executed,
        has_impact_calculated=action.has_impact_calculated,
        impact_delta=action.impact_delta,
        impact_accuracy_pct=action.impact_accuracy_pct,
    )


def _summary_to_response(summary) -> ROISummaryResponse:
    """Convert MonthlyROISummary model to ROISummaryResponse schema."""
    return ROISummaryResponse(
        id=summary.id,
        pharmacy_id=summary.pharmacy_id,
        month=summary.month_str,
        actions_suggested=summary.actions_suggested,
        actions_executed=summary.actions_executed,
        actions_discarded=summary.actions_discarded,
        actions_postponed=summary.actions_postponed,
        execution_rate=round(summary.execution_rate, 1),
        total_expected_impact=round(summary.total_expected_impact, 2),
        total_actual_impact=round(summary.total_actual_impact, 2),
        impact_by_type=summary.impact_by_type or {},
        subscription_cost=summary.subscription_cost,
        roi_percentage=round(summary.roi_percentage, 1) if summary.roi_percentage else None,
        net_roi=round(summary.net_roi, 2),
        calculated_at=summary.calculated_at,
    )


# =============================================================================
# ACTIONS ENDPOINTS
# =============================================================================


@router.get("/actions", response_model=ActionListResponse)
async def list_actions(
    status_filter: Optional[str] = Query(
        None,
        alias="status",
        description="Filter by status: pending, executed, discarded, postponed",
    ),
    action_type: Optional[str] = Query(
        None,
        description="Filter by type: liquidation, restock, pricing, diversify",
    ),
    limit: int = Query(50, ge=1, le=200, description="Max results to return"),
    offset: int = Query(0, ge=0, description="Pagination offset"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    List actions for the current pharmacy.

    Supports filtering by status and action type.
    Returns counts by status for UI display.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)

    # Get actions with filters
    actions = service.get_actions(
        pharmacy_id=current_user.pharmacy_id,
        status=status_filter,
        action_type=action_type,
        limit=limit,
        offset=offset,
    )

    # Get counts by status
    counts = service.count_by_status(current_user.pharmacy_id)

    return ActionListResponse(
        actions=[_action_to_response(a) for a in actions],
        total=sum(counts.values()),
        pending_count=counts.get(ActionStatus.PENDING.value, 0),
        executed_count=counts.get(ActionStatus.EXECUTED.value, 0),
        discarded_count=counts.get(ActionStatus.DISCARDED.value, 0),
        postponed_count=counts.get(ActionStatus.POSTPONED.value, 0),
    )


@router.get("/actions/{action_id}", response_model=ActionResponse)
async def get_action(
    action_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """Get a specific action by ID."""
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)
    action = service.get_action_for_pharmacy(action_id, current_user.pharmacy_id)

    if not action:
        raise HTTPException(
            status_code=http_status.HTTP_404_NOT_FOUND,
            detail=f"Acción {action_id} no encontrada",
        )

    return _action_to_response(action)


@router.post("/actions", response_model=ActionResponse, status_code=http_status.HTTP_201_CREATED)
async def create_action(
    action_data: ActionCreate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Create a new suggested action.

    Called by insight engine when generating recommendations.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)

    try:
        # Convert AffectedProduct list to dict list for storage
        affected_products = [p.model_dump() for p in action_data.affected_products]

        action = service.create_action(
            pharmacy_id=current_user.pharmacy_id,
            action_type=action_data.action_type,
            action_description=action_data.action_description,
            expected_impact_eur=action_data.expected_impact_eur,
            affected_products=affected_products,
            expected_impact_description=action_data.expected_impact_description,
            insight_hash=action_data.insight_hash,
        )

        logger.info(
            f"Created action {action.id} for pharmacy {current_user.pharmacy_id}"
        )

        return _action_to_response(action)

    except ValueError as e:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )


@router.post("/actions/{action_id}/execute", response_model=ActionResponse)
async def execute_action(
    action_id: UUID,
    request: ActionExecuteRequest = None,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Mark an action as executed.

    User confirms they performed the suggested action.
    Optionally provide execution timestamp and notes.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)

    executed_at = request.executed_at if request else None
    notes = request.notes if request else None

    action = service.mark_executed(
        action_id=action_id,
        pharmacy_id=current_user.pharmacy_id,
        executed_at=executed_at,
        notes=notes,
    )

    if not action:
        raise HTTPException(
            status_code=http_status.HTTP_404_NOT_FOUND,
            detail=f"Acción {action_id} no encontrada",
        )

    logger.info(f"Action {action_id} marked as executed by user {current_user.id}")

    return _action_to_response(action)


@router.post("/actions/{action_id}/discard", response_model=ActionResponse)
async def discard_action(
    action_id: UUID,
    request: ActionDiscardRequest = None,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Discard an action.

    User explicitly rejects the suggested action.
    Optionally provide a reason for discarding.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)

    reason = request.reason if request else None

    action = service.mark_discarded(
        action_id=action_id,
        pharmacy_id=current_user.pharmacy_id,
        reason=reason,
    )

    if not action:
        raise HTTPException(
            status_code=http_status.HTTP_404_NOT_FOUND,
            detail=f"Acción {action_id} no encontrada",
        )

    logger.info(f"Action {action_id} discarded by user {current_user.id}: {reason}")

    return _action_to_response(action)


@router.post("/actions/{action_id}/postpone", response_model=ActionResponse)
async def postpone_action(
    action_id: UUID,
    request: ActionPostponeRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Postpone an action until a future date.

    User wants to consider the action later.
    Required: postpone_until date (must be in the future).
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ActionTrackingService(db)

    action = service.mark_postponed(
        action_id=action_id,
        pharmacy_id=current_user.pharmacy_id,
        postpone_until=request.postpone_until,
        notes=request.notes,
    )

    if not action:
        raise HTTPException(
            status_code=http_status.HTTP_404_NOT_FOUND,
            detail=f"Acción {action_id} no encontrada",
        )

    logger.info(
        f"Action {action_id} postponed until {request.postpone_until} "
        f"by user {current_user.id}"
    )

    return _action_to_response(action)


# =============================================================================
# ROI SUMMARY ENDPOINTS
# =============================================================================


@router.get("/summary", response_model=ROISummaryResponse)
async def get_current_summary(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Get ROI summary for the current month.

    Calculates if not exists or returns cached version.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ROISummaryService(db)
    summary = service.get_current_month_summary(current_user.pharmacy_id)

    if not summary:
        # Calculate if not exists
        from app.utils.datetime_utils import utc_now

        now = utc_now()
        summary = service.calculate_monthly_summary(
            pharmacy_id=current_user.pharmacy_id,
            year=now.year,
            month=now.month,
        )

    return _summary_to_response(summary)


@router.get("/summary/history", response_model=ROIHistoryResponse)
async def get_summary_history(
    months: int = Query(6, ge=1, le=24, description="Number of months to retrieve"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Get ROI summary history for the last N months.

    Returns list of monthly summaries ordered by date DESC.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ROISummaryService(db)

    summaries = service.get_last_n_months(current_user.pharmacy_id, months)
    stats = service.get_historical_stats(current_user.pharmacy_id, months)

    return ROIHistoryResponse(
        summaries=[_summary_to_response(s) for s in summaries],
        total_months=stats["total_months"],
        avg_roi_percentage=stats["avg_roi_percentage"],
        total_accumulated_impact=stats["total_accumulated_impact"],
        total_subscription_cost=stats["total_subscription_cost"],
    )


@router.get("/summary/{month}", response_model=ROISummaryResponse)
async def get_summary_by_month(
    month: str,  # Format: YYYY-MM
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Get ROI summary for a specific month.

    Args:
        month: Month in YYYY-MM format (e.g., "2026-01")
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    # Parse month string
    try:
        year, month_num = month.split("-")
        year = int(year)
        month_num = int(month_num)
        if not (1 <= month_num <= 12):
            raise ValueError("Mes debe estar entre 1 y 12")
    except (ValueError, AttributeError) as e:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail=f"Formato de mes inválido. Use YYYY-MM (ej: 2026-01): {e}",
        )

    service = ROISummaryService(db)
    summary = service.get_or_calculate(
        pharmacy_id=current_user.pharmacy_id,
        year=year,
        month=month_num,
    )

    return _summary_to_response(summary)


@router.post("/summary/recalculate", response_model=ROISummaryResponse)
async def recalculate_summary(
    month: Optional[str] = Query(
        None,
        description="Month in YYYY-MM format. If not provided, recalculates current month.",
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Force recalculation of ROI summary for a month.

    Useful after updating action impacts or fixing data.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    # Determine month to recalculate
    if month:
        try:
            year, month_num = month.split("-")
            year = int(year)
            month_num = int(month_num)
            if not (1 <= month_num <= 12):
                raise ValueError("Mes debe estar entre 1 y 12")
        except (ValueError, AttributeError) as e:
            raise HTTPException(
                status_code=http_status.HTTP_400_BAD_REQUEST,
                detail=f"Formato de mes inválido. Use YYYY-MM: {e}",
            )
    else:
        from app.utils.datetime_utils import utc_now

        now = utc_now()
        year = now.year
        month_num = now.month

    service = ROISummaryService(db)
    summary = service.calculate_monthly_summary(
        pharmacy_id=current_user.pharmacy_id,
        year=year,
        month=month_num,
    )

    logger.info(
        f"Recalculated ROI summary for pharmacy {current_user.pharmacy_id}, "
        f"month {year}-{month_num:02d}"
    )

    return _summary_to_response(summary)


# =============================================================================
# DASHBOARD WIDGET ENDPOINT
# =============================================================================


@router.get("/widget", response_model=ROIDashboardWidget)
async def get_dashboard_widget(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Get data for the ROI dashboard widget.

    Returns current month stats, pending actions, and trend vs previous month.
    Optimized for quick loading in the main dashboard.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    service = ROISummaryService(db)
    widget_data = service.get_dashboard_data(current_user.pharmacy_id)

    return ROIDashboardWidget(**widget_data)


# =============================================================================
# IMPACT CALCULATION ENDPOINTS
# =============================================================================


@router.post("/actions/{action_id}/calculate-impact", response_model=ActionResponse)
async def calculate_action_impact(
    action_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Calculate actual impact for an executed action.

    Can be called manually or scheduled automatically after execution.
    Requires action to be in 'executed' status.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    tracking_service = ActionTrackingService(db)
    action = tracking_service.get_action_for_pharmacy(action_id, current_user.pharmacy_id)

    if not action:
        raise HTTPException(
            status_code=http_status.HTTP_404_NOT_FOUND,
            detail=f"Acción {action_id} no encontrada",
        )

    if not action.is_executed:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Solo se puede calcular impacto de acciones ejecutadas",
        )

    calculator = ActionImpactCalculator(db)
    updated_action = calculator.calculate_and_save(action)

    logger.info(
        f"Calculated impact for action {action_id}: "
        f"{updated_action.actual_impact_eur}EUR"
    )

    return _action_to_response(updated_action)


@router.post("/calculate-pending-impacts")
async def calculate_pending_impacts(
    min_days: int = Query(
        7,
        ge=1,
        le=90,
        description="Minimum days since execution before calculating impact",
    ),
    batch_size: int = Query(100, ge=1, le=500, description="Batch size for processing"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Calculate impacts for all pending executed actions.

    Admin endpoint to trigger batch impact calculation.
    Processes actions executed at least N days ago without impact calculated.
    """
    if not current_user.pharmacy_id:
        raise HTTPException(
            status_code=http_status.HTTP_400_BAD_REQUEST,
            detail="Usuario no tiene farmacia asignada",
        )

    # Verify admin role for batch operations
    if current_user.role != "admin":
        raise HTTPException(
            status_code=http_status.HTTP_403_FORBIDDEN,
            detail="Solo administradores pueden ejecutar cálculos batch",
        )

    calculator = ActionImpactCalculator(db)
    processed = calculator.calculate_pending_impacts(
        pharmacy_id=current_user.pharmacy_id,
        min_days_since_execution=min_days,
        batch_size=batch_size,
    )

    return {
        "status": "completed",
        "processed_count": processed,
        "min_days_since_execution": min_days,
    }
