# backend/app/api/reports.py
"""
API endpoints para Sistema de Reportes (Issue #511).

Endpoints:
- GET /reports/{pharmacy_id}/direccion: PDF One-Pager ejecutivo
- GET /reports/{pharmacy_id}/suelo: Excel Pick-Lists operativos

Requiere autenticación JWT. Solo usuarios PRO tienen acceso.
"""

import re
from datetime import date, datetime, timezone
from uuid import UUID

import structlog
from fastapi import APIRouter, Depends, HTTPException, Query, status
from fastapi.responses import StreamingResponse
from sqlalchemy.orm import Session
import io

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.services.report_service import report_service
from app.services.pdf_generator import pdf_generator
from app.services.excel_generator import excel_generator
from app.services.word_action_plan_generator import word_action_plan_generator
from app.services.weekly_pdf_generator import weekly_pdf_generator
from app.schemas.action_plans import ActionPlanExportRequest, ActionPlanExportResponse

logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/reports", tags=["reports"])


def _check_pro_tier(user: User, db: Session) -> None:
    """
    Verificar que el usuario tiene tier PRO.

    Raises:
        HTTPException 403 si no es PRO
    """
    from app.models.user import Subscription

    try:
        subscription = db.query(Subscription).filter(
            Subscription.user_id == user.id,
            Subscription.is_active == True,
        ).first()

        if subscription and subscription.tier in ("pro", "enterprise", "premium"):
            return  # Acceso permitido

    except Exception as e:
        # Si el modelo Subscription no existe aún, log y continuar
        logger.debug("reports.pro_check.subscription_query_failed", error=str(e))

    # Feature flag: Permitir bypass SOLO en desarrollo/testing
    import os
    environment = os.getenv("ENVIRONMENT", "production")
    if environment in ("development", "testing") and os.getenv("REPORTS_BYPASS_PRO_CHECK", "false").lower() == "true":
        logger.warning("reports.pro_check.bypassed_by_env", environment=environment)
        return
    # En producción, bypass silenciosamente ignorado

    raise HTTPException(
        status_code=status.HTTP_403_FORBIDDEN,
        detail="Esta funcionalidad requiere suscripción PRO"
    )


def _validate_date_range(start_date: date, end_date: date) -> None:
    """
    Validar rango de fechas.

    Raises:
        HTTPException 400 si el rango es inválido
    """
    if start_date > end_date:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="start_date debe ser anterior a end_date"
        )

    # Límite FREE tier: 3 meses (REGLA #18)
    max_days = 93  # ~3 meses
    if (end_date - start_date).days > max_days:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Rango máximo permitido: {max_days} días"
        )


def _sanitize_filename(name: str) -> str:
    """
    Sanitizar nombre para uso seguro en filenames.

    Elimina caracteres especiales que podrían causar problemas.
    """
    # Reemplazar caracteres no alfanuméricos (excepto guiones) por underscore
    sanitized = re.sub(r'[^\w\-]', '_', name, flags=re.ASCII)
    # Eliminar underscores múltiples consecutivos
    sanitized = re.sub(r'_+', '_', sanitized)
    # Eliminar underscores al inicio/final
    return sanitized.strip('_')


@router.get("/{pharmacy_id}/direccion")
async def download_direccion_pdf(
    pharmacy_id: UUID,
    start_date: date = Query(..., description="Fecha inicio del período"),
    end_date: date = Query(..., description="Fecha fin del período"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> StreamingResponse:
    """
    Descargar PDF One-Pager para Dirección.

    Genera un resumen ejecutivo de una página con:
    - KPIs: Ventas, Margen, Stock, SKUs + cambios YoY
    - Top 3 categorías por crecimiento
    - Top 5 alertas por impacto económico
    - Recomendación principal

    Args:
        pharmacy_id: UUID de la farmacia
        start_date: Fecha inicio del período a analizar
        end_date: Fecha fin del período

    Returns:
        StreamingResponse con PDF

    Raises:
        403: Usuario no autorizado o sin tier PRO
        404: Farmacia no encontrada
    """
    logger.info(
        "reports.direccion.request",
        pharmacy_id=str(pharmacy_id),
        start_date=str(start_date),
        end_date=str(end_date),
        user_id=str(current_user.id),
    )

    # Verificar acceso PRO
    _check_pro_tier(current_user, db)

    # Validar fechas
    _validate_date_range(start_date, end_date)

    # Verificar que el usuario tiene acceso a esta farmacia
    if current_user.pharmacy_id != pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia"
        )

    try:
        # Obtener datos
        report_data = await report_service.get_direccion_data(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=start_date,
            end_date=end_date,
        )

        # Generar PDF
        pdf_bytes = pdf_generator.generate(report_data)

        # Nombre de archivo (sanitizado para seguridad)
        pharmacy_name = _sanitize_filename(report_data.pharmacy_name)
        period_str = start_date.strftime('%Y%m')
        filename = f"Informe_Direccion_{pharmacy_name}_{period_str}.pdf"

        logger.info(
            "reports.direccion.success",
            pharmacy_id=str(pharmacy_id),
            pdf_size=len(pdf_bytes),
        )

        return StreamingResponse(
            io.BytesIO(pdf_bytes),
            media_type="application/pdf",
            headers={
                "Content-Disposition": f'attachment; filename="{filename}"',
                "Content-Length": str(len(pdf_bytes)),
            },
        )

    except Exception as e:
        logger.error(
            "reports.direccion.error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,  # Full traceback in logs for debugging
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error generando informe PDF. Contacte a soporte si el problema persiste."
        )


@router.get("/{pharmacy_id}/suelo")
async def download_suelo_excel(
    pharmacy_id: UUID,
    include_liquidation: bool = Query(True, description="Incluir hoja Liquidar"),
    include_restock: bool = Query(True, description="Incluir hoja Reponer"),
    include_promotion: bool = Query(True, description="Incluir hoja Promocionar"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> StreamingResponse:
    """
    Descargar Excel Pick-Lists para Suelo.

    Genera un workbook con 3 hojas de productos accionables:
    - Liquidar: Stock muerto (>180 días sin venta)
    - Reponer: Stock crítico (<7 días cobertura)
    - Promocionar: Alto margen, baja rotación

    Args:
        pharmacy_id: UUID de la farmacia
        include_liquidation: Incluir hoja de liquidación
        include_restock: Incluir hoja de reposición
        include_promotion: Incluir hoja de promoción

    Returns:
        StreamingResponse con Excel

    Raises:
        403: Usuario no autorizado o sin tier PRO
        404: Farmacia no encontrada
    """
    logger.info(
        "reports.suelo.request",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
        include_liquidation=include_liquidation,
        include_restock=include_restock,
        include_promotion=include_promotion,
    )

    # Verificar acceso PRO
    _check_pro_tier(current_user, db)

    # Verificar que el usuario tiene acceso a esta farmacia
    if current_user.pharmacy_id != pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia"
        )

    try:
        # Obtener datos
        report_data = await report_service.get_suelo_data(
            db=db,
            pharmacy_id=pharmacy_id,
            include_liquidation=include_liquidation,
            include_restock=include_restock,
            include_promotion=include_promotion,
        )

        # Generar Excel
        excel_bytes = excel_generator.generate(report_data)

        # Nombre de archivo (sanitizado para seguridad, UTC para consistencia)
        pharmacy_name = _sanitize_filename(report_data.pharmacy_name)
        date_str = datetime.now(timezone.utc).strftime("%Y%m%d")
        filename = f"PickLists_{pharmacy_name}_{date_str}.xlsx"

        logger.info(
            "reports.suelo.success",
            pharmacy_id=str(pharmacy_id),
            excel_size=len(excel_bytes),
        )

        return StreamingResponse(
            io.BytesIO(excel_bytes),
            media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
            headers={
                "Content-Disposition": f'attachment; filename="{filename}"',
                "Content-Length": str(len(excel_bytes)),
            },
        )

    except Exception as e:
        logger.error(
            "reports.suelo.error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,  # Full traceback in logs for debugging
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error generando informe Excel. Contacte a soporte si el problema persiste."
        )


@router.get("/{pharmacy_id}/action-plan")
async def download_action_plan(
    pharmacy_id: UUID,
    start_date: date = Query(..., description="Fecha inicio del período"),
    end_date: date = Query(..., description="Fecha fin del período"),
    format: str = Query("docx", description="Formato: docx o pdf"),
    max_actions: int = Query(20, ge=1, le=50, description="Máximo de acciones"),
    min_economic_value: float = Query(0.0, ge=0, description="Valor mínimo"),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> StreamingResponse:
    """
    Descargar Plan de Acción en Word/PDF (Issue #513).

    Genera un documento estructurado con:
    - Portada: Farmacia, período, oportunidad total
    - Resumen Ejecutivo: KPIs y distribución de acciones
    - Acciones por Prioridad: Tablas agrupadas con pasos concretos
    - Checklist de Seguimiento: Tabla para marcar progreso

    Args:
        pharmacy_id: UUID de la farmacia
        start_date: Fecha inicio del período a analizar
        end_date: Fecha fin del período
        format: Formato de salida (docx o pdf)
        max_actions: Número máximo de acciones a incluir
        min_economic_value: Valor económico mínimo para incluir acción

    Returns:
        StreamingResponse con documento Word

    Raises:
        403: Usuario no autorizado o sin tier PRO
        404: Farmacia no encontrada
    """
    logger.info(
        "reports.action_plan.request",
        pharmacy_id=str(pharmacy_id),
        start_date=str(start_date),
        end_date=str(end_date),
        format=format,
        user_id=str(current_user.id),
    )

    # Verificar acceso PRO
    _check_pro_tier(current_user, db)

    # Validar fechas
    _validate_date_range(start_date, end_date)

    # Verificar que el usuario tiene acceso a esta farmacia
    if current_user.pharmacy_id != pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia"
        )

    # Validar formato (PDF no implementado aún)
    if format != "docx":
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Solo formato 'docx' soportado actualmente. PDF disponible próximamente."
        )

    try:
        # Obtener datos del plan de acción
        action_plan_data = await report_service.get_action_plan_data(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=start_date,
            end_date=end_date,
            max_actions=max_actions,
            min_economic_value=min_economic_value,
        )

        # Generar documento Word
        doc_bytes = word_action_plan_generator.generate(action_plan_data)

        # Nombre de archivo (sanitizado para seguridad)
        pharmacy_name = _sanitize_filename(action_plan_data.pharmacy_name)
        period_str = start_date.strftime('%Y%m')
        filename = f"Plan_Accion_{pharmacy_name}_{period_str}.docx"

        # Media type para Word
        media_type = "application/vnd.openxmlformats-officedocument.wordprocessingml.document"

        logger.info(
            "reports.action_plan.success",
            pharmacy_id=str(pharmacy_id),
            doc_size=len(doc_bytes),
            actions_count=len(action_plan_data.actions),
        )

        return StreamingResponse(
            io.BytesIO(doc_bytes),
            media_type=media_type,
            headers={
                "Content-Disposition": f'attachment; filename="{filename}"',
                "Content-Length": str(len(doc_bytes)),
            },
        )

    except Exception as e:
        logger.error(
            "reports.action_plan.error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error generando plan de acción. Contacte a soporte si el problema persiste."
        )


@router.get("/{pharmacy_id}/weekly")
async def download_weekly_pdf(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> StreamingResponse:
    """
    Descargar PDF Resumen Semanal (Issue #552).

    Genera el "latido" del sistema: un PDF resumen de la semana
    que el farmacéutico puede encontrar cada lunes sin abrir la app.

    Secciones:
    - Resumen KPIs: Ventas semana vs año pasado
    - Top 3 Categorías con crecimiento
    - Alertas prioritarias (top 3)
    - Recomendación principal

    El PDF se genera para la semana anterior (lunes a domingo).

    Args:
        pharmacy_id: UUID de la farmacia

    Returns:
        StreamingResponse con PDF

    Raises:
        403: Usuario no autorizado
        404: Farmacia no encontrada
    """
    from datetime import timedelta

    logger.info(
        "reports.weekly.request",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
    )

    # Verificar que el usuario tiene acceso a esta farmacia
    if current_user.pharmacy_id != pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia"
        )

    try:
        # Calcular semana anterior (lunes a domingo)
        today = date.today()
        # Encontrar el lunes de la semana pasada
        days_since_monday = today.weekday()
        last_monday = today - timedelta(days=days_since_monday + 7)
        last_sunday = last_monday + timedelta(days=6)

        # Obtener datos usando ReportService existente
        report_data = await report_service.get_direccion_data(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=last_monday,
            end_date=last_sunday,
        )

        # Generar PDF con el generador semanal
        pdf_bytes = weekly_pdf_generator.generate(report_data)

        # Nombre de archivo
        pharmacy_name = _sanitize_filename(report_data.pharmacy_name)
        week_str = last_monday.strftime('%Y_W%W')
        filename = f"Resumen_Semanal_{pharmacy_name}_{week_str}.pdf"

        logger.info(
            "reports.weekly.success",
            pharmacy_id=str(pharmacy_id),
            week=week_str,
            pdf_size=len(pdf_bytes),
        )

        return StreamingResponse(
            io.BytesIO(pdf_bytes),
            media_type="application/pdf",
            headers={
                "Content-Disposition": f'attachment; filename="{filename}"',
                "Content-Length": str(len(pdf_bytes)),
            },
        )

    except Exception as e:
        logger.error(
            "reports.weekly.error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error generando resumen semanal. Contacte a soporte si el problema persiste."
        )


@router.post("/{pharmacy_id}/weekly/generate")
async def generate_weekly_pdf_to_file(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> dict:
    """
    Generar PDF Semanal y guardarlo en carpeta exports.

    Este endpoint es llamado por el ETL nocturno cada lunes a las 4:00 AM
    para generar automáticamente el PDF semanal.

    Args:
        pharmacy_id: UUID de la farmacia

    Returns:
        dict con path del archivo generado

    Raises:
        403: Usuario no autorizado
    """
    from datetime import timedelta
    from pathlib import Path
    import os

    logger.info(
        "reports.weekly.generate_to_file",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
    )

    # Verificar acceso
    if current_user.pharmacy_id != pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia"
        )

    try:
        # Calcular semana anterior
        today = date.today()
        days_since_monday = today.weekday()
        last_monday = today - timedelta(days=days_since_monday + 7)
        last_sunday = last_monday + timedelta(days=6)

        # Obtener datos
        report_data = await report_service.get_direccion_data(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=last_monday,
            end_date=last_sunday,
        )

        # Preparar path de salida
        exports_dir = Path(os.getenv("EXPORTS_DIR", "/app/exports"))
        exports_dir.mkdir(parents=True, exist_ok=True)

        pharmacy_name = _sanitize_filename(report_data.pharmacy_name)
        week_str = last_monday.strftime('%Y_W%W')
        filename = f"Resumen_Semanal_{pharmacy_name}_{week_str}.pdf"
        output_path = exports_dir / filename

        # Generar y guardar PDF
        pdf_bytes = weekly_pdf_generator.generate(
            report_data,
            output_path=str(output_path),
        )

        logger.info(
            "reports.weekly.generate_to_file.success",
            pharmacy_id=str(pharmacy_id),
            output_path=str(output_path),
            size_kb=len(pdf_bytes) // 1024,
        )

        return {
            "status": "success",
            "filename": filename,
            "path": str(output_path),
            "size_kb": len(pdf_bytes) // 1024,
            "period": {
                "start": last_monday.isoformat(),
                "end": last_sunday.isoformat(),
            },
        }

    except Exception as e:
        logger.error(
            "reports.weekly.generate_to_file.error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error generando PDF: {str(e)}"
        )
