﻿# backend/app/api/reenrichment.py
from typing import Optional

from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Query
from sqlalchemy import and_
from sqlalchemy.orm import Session

from app.api.deps import get_current_admin_user
from app.database import get_db, SessionLocal
from app.models import Pharmacy, SalesData, SalesEnrichment
from app.models.user import User
from app.services.reenrichment_service import ReEnrichmentService
from app.utils.datetime_utils import utc_now

router = APIRouter(prefix="/reenrichment", tags=["reenrichment"])


@router.get("/status/{pharmacy_id}")
async def check_reenrichment_status(pharmacy_id: str, db: Session = Depends(get_db)):
    """
    Verifica qué datos necesitan re-enriquecimiento para una farmacia
    """

    service = ReEnrichmentService(db)
    status = service.identify_stale_enrichments(pharmacy_id)

    return {
        "pharmacy_id": pharmacy_id,
        "total_sales": status["total_sales"],
        "needs_update": status["needs_update"],
        "products_affected": len(status["products_to_update"]),
        "reasons": status["reasons"],
        "recommendation": ("Re-enriquecimiento recomendado" if status["needs_update"] > 0 else "Datos actualizados"),
    }


@router.post("/execute/{pharmacy_id}")
async def execute_reenrichment(pharmacy_id: str, background_tasks: BackgroundTasks, db: Session = Depends(get_db)):
    """
    Ejecuta re-enriquecimiento para una farmacia específica
    """

    service = ReEnrichmentService(db)

    # Primero verificar si es necesario
    status = service.identify_stale_enrichments(pharmacy_id)

    if status["needs_update"] == 0:
        return {
            "status": "no_action_needed",
            "message": "No hay datos que necesiten actualización",
        }

    # Ejecutar en background para no bloquear
    background_tasks.add_task(service.schedule_re_enrichment, pharmacy_id)

    return {
        "status": "initiated",
        "message": f"Re-enriquecimiento iniciado para {status['needs_update']} registros",
        "products_affected": len(status["products_to_update"]),
        "estimated_time_seconds": (status["needs_update"] / 1000) * 2,  # Estimación
    }


@router.post("/execute-all")
async def execute_global_reenrichment(
    background_tasks: BackgroundTasks,
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required (Issue #452)
    db: Session = Depends(get_db),
):
    """
    Ejecuta re-enriquecimiento para TODAS las farmacias.

    Requires: Admin authentication
    """

    service = ReEnrichmentService(db)

    # Ejecutar en background
    background_tasks.add_task(service.schedule_re_enrichment)

    return {
        "status": "initiated",
        "message": "Re-enriquecimiento global iniciado",
        "check_progress_at": "/api/v1/reenrichment/progress",
    }


@router.get("/catalog-changes")
async def get_recent_catalog_changes(since_hours: int = 24, db: Session = Depends(get_db)):
    """
    Muestra productos del catálogo actualizados recientemente
    """
    from datetime import timedelta

    from sqlalchemy import func

    from app.models import ProductCatalog

    since = utc_now() - timedelta(hours=since_hours)

    # Contar productos actualizados
    updated_count = db.query(func.count(ProductCatalog.id)).filter(ProductCatalog.updated_at > since).scalar()

    # Obtener muestra de productos actualizados
    sample = (
        db.query(
            ProductCatalog.national_code,
            ProductCatalog.nomen_nombre,
            ProductCatalog.updated_at,
            ProductCatalog.data_sources,
        )
        .filter(ProductCatalog.updated_at > since)
        .order_by(ProductCatalog.updated_at.desc())
        .limit(20)
        .all()
    )

    return {
        "since": since.isoformat(),
        "total_updated": updated_count,
        "sample_products": [
            {
                "codigo_nacional": p.national_code,
                "nombre": p.nomen_nombre,
                "actualizado": p.updated_at.isoformat(),
                "fuentes": p.data_sources,
            }
            for p in sample
        ],
    }


@router.post("/failed-records/{pharmacy_id}")
async def reenrich_failed_records(
    pharmacy_id: str,
    background_tasks: BackgroundTasks,
    db: Session = Depends(get_db),
    limit: Optional[int] = 100,
):
    """
    Reenriquece registros que fallaron previamente o están en revisión manual.
    Útil para aprovechar mejoras en el algoritmo de matching o nuevos datos del catálogo.
    """

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(status_code=404, detail="Farmacia no encontrada")

    # Ejecutar reenriquecimiento en background para no bloquear
    background_tasks.add_task(_execute_failed_reenrichment, pharmacy_id, limit, db)

    return {
        "status": "initiated",
        "message": f"Reenriquecimiento de registros fallidos iniciado para farmacia {pharmacy_id}",
        "limit": limit,
        "estimated_time_seconds": (limit / 50) * 2,  # Estimación basada en batch size
    }


@router.post("/failed-records-sync/{pharmacy_id}")
async def reenrich_failed_records_sync(pharmacy_id: str, limit: Optional[int] = 10, db: Session = Depends(get_db)):
    """
    Reenriquece registros fallidos de forma síncrona (para testing o pequeños lotes).
    """
    from app.services.enrichment_service import enrichment_service

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(status_code=404, detail="Farmacia no encontrada")

    # Ejecutar directamente
    stats = enrichment_service.reenrich_failed_records(db, pharmacy_id, limit)

    return {
        "status": "completed",
        "pharmacy_id": pharmacy_id,
        "stats": stats,
        "success_rate": (
            f"{(stats['reenriched'] / stats['processed'] * 100):.1f}%" if stats["processed"] > 0 else "0%"
        ),
    }


def _execute_failed_reenrichment(pharmacy_id: str, limit: int, db: Session):
    """
    Función helper para ejecutar reenriquecimiento en background
    """
    import logging

    from app.services.enrichment_service import enrichment_service

    logger = logging.getLogger(__name__)

    try:
        logger.info(f"[BACKGROUND] Iniciando reenriquecimiento de registros fallidos para farmacia {pharmacy_id}")
        stats = enrichment_service.reenrich_failed_records(db, pharmacy_id, limit)
        logger.info(f"[BACKGROUND] Reenriquecimiento completado: {stats}")
    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"[BACKGROUND] Error en reenriquecimiento: {str(e)}")
        raise


@router.post("/scale/{pharmacy_id}")
async def enrich_large_scale(
    pharmacy_id: str,
    background_tasks: BackgroundTasks,
    db: Session = Depends(get_db),
    batch_size: Optional[int] = 1000,
):
    """
    Enriquecimiento optimizado para gran escala (200K-400K registros).

    Características:
    - Procesamiento en chunks para evitar OOM
    - Bulk operations para máximo rendimiento
    - Cache inteligente con productos frecuentes
    - Progress tracking en tiempo real
    """

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(status_code=404, detail="Farmacia no encontrada")

    # Estimar volumen para procesar
    pending_count = (
        db.query(SalesData)
        .outerjoin(SalesEnrichment)
        .filter(and_(SalesData.pharmacy_id == pharmacy_id, SalesEnrichment.id.is_(None)))
        .count()
    )

    if pending_count == 0:
        return {
            "status": "no_action_needed",
            "message": "No hay registros pendientes de enriquecimiento",
        }

    # Ejecutar en background con progress tracking
    background_tasks.add_task(_execute_scale_enrichment, pharmacy_id, batch_size, db)

    return {
        "status": "initiated",
        "message": f"Enriquecimiento iniciado para {pending_count} registros",
        "batch_size": batch_size,
        "estimated_chunks": (pending_count // batch_size) + 1,
        "estimated_time_minutes": (pending_count / 1000) * 2,  # 2 min por 1K registros
        "monitor_progress_at": f"/api/v1/enrichment/progress/{pharmacy_id}",
    }


@router.post("/scale-sync/{pharmacy_id}")
async def enrich_large_scale_sync(pharmacy_id: str, batch_size: Optional[int] = 500, db: Session = Depends(get_db)):
    """
    Enriquecimiento optimizado síncrono (para testing con lotes pequeños).
    """
    from app.services.enrichment_service import enrichment_service

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(status_code=404, detail="Farmacia no encontrada")

    # Progress tracking simple para modo síncrono
    progress_data = {"current_progress": None}

    def progress_callback(progress):
        progress_data["current_progress"] = progress
        # En producción, esto se guardaría en Redis/cache para consulta
        print(f"Progress: {progress['percentage']:.1f}% - {progress['processed']}/{progress['total_estimated']}")

    # Ejecutar directamente
    stats = enrichment_service.enrich_sales_batch(
        db, pharmacy_id, batch_size=batch_size, progress_callback=progress_callback
    )

    return {
        "status": "completed",
        "pharmacy_id": pharmacy_id,
        "stats": stats,
        "performance": {
            "success_rate": (
                f"{(stats['enriched'] / stats['processed'] * 100):.1f}%" if stats["processed"] > 0 else "0%"
            ),
            "cache_efficiency": (
                f"{(stats['cache_hits'] / (stats['cache_hits'] + stats['cache_misses']) * 100):.1f}%"
                if (stats["cache_hits"] + stats["cache_misses"]) > 0
                else "0%"
            ),
            "chunks_processed": stats["total_chunks"],
            "avg_records_per_chunk": (stats["processed"] / stats["total_chunks"] if stats["total_chunks"] > 0 else 0),
        },
    }


def _execute_scale_enrichment(pharmacy_id: str, batch_size: int, db: Session):
    """
    Función helper para ejecutar enriquecimiento optimizado en background
    """
    import logging

    from app.services.enrichment_service import enrichment_service

    logger = logging.getLogger(__name__)

    try:
        logger.info(f"[BACKGROUND] Iniciando enriquecimiento optimizado para farmacia {pharmacy_id}")

        def progress_callback(progress):
            # En producción, esto se guardaría en Redis para consulta
            logger.info(
                f"[PROGRESS] {progress['percentage']:.1f}% - Chunk {progress['current_chunk']} - "
                f"Cache efficiency: {progress['cache_efficiency']:.1f}%"
            )

        stats = enrichment_service.enrich_sales_batch(
            db, pharmacy_id, batch_size=batch_size, progress_callback=progress_callback
        )

        logger.info(f"[BACKGROUND] Completado: {stats}")
    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"[BACKGROUND] Error: {str(e)}")
        raise


# =============================================================================
# BACKFILL NULL PRODUCT_TYPE (Issue #446)
# =============================================================================


@router.get("/backfill-product-type/status")
async def check_null_product_type_status(
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
    db: Session = Depends(get_db),
):
    """
    Verifica cuántos registros tienen product_type NULL con catálogo.

    Este es el bug encontrado donde reenrichment_service creaba enrichments
    sin establecer product_type. El bug fue corregido, este endpoint permite
    verificar y corregir registros históricos.

    Requires: Admin authentication
    """
    from sqlalchemy import func

    from app.models import ProductCatalog

    # Contar total NULL con catálogo
    null_with_catalog = (
        db.query(func.count(SalesEnrichment.id))
        .filter(SalesEnrichment.product_type.is_(None), SalesEnrichment.product_catalog_id.isnot(None))
        .scalar()
    )

    # Breakdown por categoría
    breakdown = (
        db.query(ProductCatalog.xfarma_prescription_category, func.count(SalesEnrichment.id).label("count"))
        .join(SalesEnrichment, SalesEnrichment.product_catalog_id == ProductCatalog.id)
        .filter(SalesEnrichment.product_type.is_(None))
        .group_by(ProductCatalog.xfarma_prescription_category)
        .all()
    )

    breakdown_dict = {}
    expected_prescription = 0
    expected_venta_libre = 0

    for category, count in breakdown:
        key = category if category else "NULL_category"
        breakdown_dict[key] = count
        if category is None:
            expected_venta_libre += count
        else:
            expected_prescription += count

    # Distribución actual de product_type
    distribution = (
        db.query(SalesEnrichment.product_type, func.count(SalesEnrichment.id))
        .group_by(SalesEnrichment.product_type)
        .all()
    )

    distribution_dict = {ptype if ptype else "NULL": count for ptype, count in distribution}

    return {
        "needs_backfill": null_with_catalog > 0,
        "null_with_catalog": null_with_catalog,
        "breakdown_by_category": breakdown_dict,
        "expected_after_backfill": {
            "to_prescription": expected_prescription,
            "to_venta_libre": expected_venta_libre,
        },
        "current_distribution": distribution_dict,
        "action": (
            f"POST /api/v1/reenrichment/backfill-product-type para corregir {null_with_catalog} registros"
            if null_with_catalog > 0
            else "No se requiere acción"
        ),
    }


@router.post("/backfill-product-type")
async def backfill_null_product_type(
    background_tasks: BackgroundTasks,
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
    db: Session = Depends(get_db),
    dry_run: bool = False,
):
    """
    Corrige registros con product_type NULL que tienen catálogo.

    Issue #446: Bug donde reenrichment_service.py creaba enrichments sin
    establecer product_type. El bug fue corregido en el código, este endpoint
    permite corregir registros históricos afectados.

    Requires: Admin authentication

    Args:
        dry_run: Si es True, solo reporta cuántos registros se corregirían
    """
    from sqlalchemy import func

    # Contar registros a corregir
    null_count = (
        db.query(func.count(SalesEnrichment.id))
        .filter(SalesEnrichment.product_type.is_(None), SalesEnrichment.product_catalog_id.isnot(None))
        .scalar()
    )

    if null_count == 0:
        return {
            "status": "no_action_needed",
            "message": "No hay registros con product_type NULL que tengan catálogo",
            "null_with_catalog": 0,
        }

    if dry_run:
        return {
            "status": "dry_run",
            "message": f"Se corregirían {null_count} registros (ejecutar sin dry_run=true para aplicar)",
            "null_with_catalog": null_count,
        }

    # Ejecutar en background (sin pasar db session - el background task crea la suya)
    background_tasks.add_task(_execute_product_type_backfill)

    return {
        "status": "initiated",
        "message": f"Backfill iniciado para {null_count} registros con product_type NULL",
        "records_to_fix": null_count,
        "estimated_time_seconds": (null_count / 500) * 2,
        "check_status_at": "/api/v1/reenrichment/backfill-product-type/status",
    }


def _execute_product_type_backfill():
    """
    Función helper para ejecutar backfill de product_type en background.

    IMPORTANTE: Crea su propia sesión de DB para evitar DetachedInstanceError
    cuando la sesión del request se cierra antes de que termine el background task.
    """
    import logging

    from app.models import ProductCatalog
    from app.services.enrichment_service import EnrichmentService

    logger = logging.getLogger(__name__)

    # Crear sesión propia para el background task
    db = SessionLocal()

    try:
        logger.info("[BACKFILL] Iniciando corrección de product_type NULL")

        enrichment_service = EnrichmentService()
        batch_size = 500
        updated_count = 0
        to_prescription = 0
        to_venta_libre = 0

        while True:
            # Obtener batch de registros NULL con catálogo
            batch = (
                db.query(SalesEnrichment)
                .filter(SalesEnrichment.product_type.is_(None), SalesEnrichment.product_catalog_id.isnot(None))
                .limit(batch_size)
                .all()
            )

            if not batch:
                break

            # OPTIMIZACIÓN: Batch load de catálogos (evita N+1 queries)
            catalog_ids = [e.product_catalog_id for e in batch if e.product_catalog_id]
            catalogs = db.query(ProductCatalog).filter(ProductCatalog.id.in_(catalog_ids)).all()
            catalog_map = {c.id: c for c in catalogs}

            for enrichment in batch:
                product_catalog = catalog_map.get(enrichment.product_catalog_id)

                if product_catalog:
                    new_product_type = enrichment_service._derive_product_type(product_catalog)
                    enrichment.product_type = new_product_type

                    if new_product_type == "prescription":
                        to_prescription += 1
                    else:
                        to_venta_libre += 1
                else:
                    # Sin catálogo, asumir venta_libre
                    enrichment.product_type = "venta_libre"
                    to_venta_libre += 1

                updated_count += 1

            db.commit()
            logger.info(f"[BACKFILL] Progreso: {updated_count} registros corregidos")

        logger.info(
            f"[BACKFILL] Completado: {updated_count} registros "
            f"(prescription: {to_prescription}, venta_libre: {to_venta_libre})"
        )

    except Exception as e:
        logger.error(f"[BACKFILL] Error: {str(e)}")
        db.rollback()
        raise
    finally:
        db.close()  # IMPORTANTE: Cerrar sesión al terminar


# =============================================================================
# REDETECT BRANDS (Issue #446)
# =============================================================================


@router.get("/redetect-brands/status")
async def check_redetect_brands_status(
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
    db: Session = Depends(get_db),
):
    """
    Verifica cuántos productos venta_libre necesitan detección de marca.

    Issue #446: Cuando se añaden nuevas marcas al BrandDetectionService,
    los productos existentes no las tienen detectadas. Este endpoint
    permite verificar el estado y ejecutar la redetección.

    Requires: Admin authentication
    """
    from sqlalchemy import func

    from app.services.brand_detection_service import brand_detection_service

    # Total venta_libre
    total_venta_libre = (
        db.query(func.count(SalesEnrichment.id))
        .filter(SalesEnrichment.product_type == "venta_libre")
        .scalar()
    )

    # Sin marca detectada
    without_brand = (
        db.query(func.count(SalesEnrichment.id))
        .filter(
            SalesEnrichment.product_type == "venta_libre",
            SalesEnrichment.detected_brand.is_(None),
        )
        .scalar()
    )

    # Con marca detectada
    with_brand = total_venta_libre - without_brand

    # Distribución por marca
    brand_distribution = (
        db.query(SalesEnrichment.detected_brand, func.count(SalesEnrichment.id).label("count"))
        .filter(
            SalesEnrichment.product_type == "venta_libre",
            SalesEnrichment.detected_brand.isnot(None),
        )
        .group_by(SalesEnrichment.detected_brand)
        .order_by(func.count(SalesEnrichment.id).desc())
        .limit(20)
        .all()
    )

    return {
        "brands_configured": len(brand_detection_service.brand_priority),
        "total_venta_libre": total_venta_libre,
        "with_brand_detected": with_brand,
        "without_brand_detected": without_brand,
        "coverage_percentage": round(with_brand / total_venta_libre * 100, 1) if total_venta_libre > 0 else 0,
        "top_brands": [{"brand": brand, "count": count} for brand, count in brand_distribution],
        "action": (
            f"POST /api/v1/reenrichment/redetect-brands para detectar marcas en {without_brand} productos"
            if without_brand > 0
            else "Todos los productos tienen marca detectada o no es detectable"
        ),
    }


@router.post("/redetect-brands")
async def redetect_brands(
    background_tasks: BackgroundTasks,
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required
    db: Session = Depends(get_db),
    pharmacy_id: Optional[str] = None,
    limit: Optional[int] = None,
    sync: bool = False,
):
    """
    Re-detecta marcas en productos venta_libre usando BrandDetectionService.

    Issue #446: Permite actualizar detected_brand cuando se añaden nuevas marcas
    al servicio de detección sin necesidad de re-enriquecer todo el catálogo.

    Requires: Admin authentication

    Args:
        pharmacy_id: Opcional - solo procesar una farmacia específica
        limit: Opcional - limitar número de registros (para testing)
        sync: Si True, ejecuta síncronamente (para pequeños lotes)
    """
    from sqlalchemy import func

    # Contar registros a procesar
    query = db.query(func.count(SalesEnrichment.id)).filter(
        SalesEnrichment.product_type == "venta_libre",
        SalesEnrichment.detected_brand.is_(None),
    )

    if pharmacy_id:
        query = query.join(SalesData, SalesData.id == SalesEnrichment.sales_data_id).filter(
            SalesData.pharmacy_id == pharmacy_id
        )

    pending_count = query.scalar()

    if pending_count == 0:
        return {
            "status": "no_action_needed",
            "message": "No hay productos venta_libre sin marca detectada",
        }

    effective_limit = min(limit, pending_count) if limit else pending_count

    if sync and effective_limit <= 1000:
        # Ejecutar síncronamente para lotes pequeños
        service = ReEnrichmentService(db)
        results = service.redetect_brands(pharmacy_id=pharmacy_id, limit=limit)

        return {
            "status": "completed",
            "processed": results["processed"],
            "updated": results["updated"],
            "no_brand_found": results["no_brand_found"],
            "coverage_improvement": f"+{results['updated']} productos con marca",
            "brands_found": results["brands_found"],
        }

    # Ejecutar en background
    background_tasks.add_task(_execute_redetect_brands, pharmacy_id, limit)

    return {
        "status": "initiated",
        "message": f"Redetección de marcas iniciada para {effective_limit} productos",
        "records_to_process": effective_limit,
        "estimated_time_seconds": (effective_limit / 1000) * 2,
        "check_status_at": "/api/v1/reenrichment/redetect-brands/status",
    }


def _execute_redetect_brands(pharmacy_id: str = None, limit: int = None):
    """
    Función helper para ejecutar redetección de marcas en background.
    """
    import logging

    logger = logging.getLogger(__name__)

    # Crear sesión propia para el background task
    db = SessionLocal()

    try:
        logger.info(f"[REDETECT_BRANDS] Iniciando redetección de marcas")

        service = ReEnrichmentService(db)
        results = service.redetect_brands(pharmacy_id=pharmacy_id, limit=limit)

        logger.info(
            f"[REDETECT_BRANDS] Completado: {results['updated']}/{results['processed']} "
            f"productos actualizados con marca"
        )

    except Exception as e:
        logger.error(f"[REDETECT_BRANDS] Error: {str(e)}")
        db.rollback()
        raise
    finally:
        db.close()


# =============================================================================
# RECLASSIFY INVENTORY (Issue #487)
# =============================================================================


@router.get("/reclassify-inventory/status")
async def check_reclassify_inventory_status(
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
):
    """
    Verifica cuántos productos de inventario necesitan re-clasificación.

    Issue #487: Productos OTC de CIMA (FRENADOL, VOLTADOL, etc.) estaban
    siendo clasificados como "prescription" cuando deberían ser "venta_libre".

    Este endpoint verifica el estado de clasificación del inventario.

    Requires: Admin authentication
    """
    from sqlalchemy import func

    from app.models import InventorySnapshot, ProductCatalog

    # Contar por product_type actual
    type_counts = (
        db.query(
            InventorySnapshot.product_type,
            func.count(InventorySnapshot.id).label("count"),
        )
        .group_by(InventorySnapshot.product_type)
        .all()
    )

    type_stats = {row.product_type or "null": row.count for row in type_counts}

    # Contar productos con product_catalog_id que podrían estar mal clasificados
    # (marcados como prescription pero sin xfarma_prescription_category)
    potential_misclassified = (
        db.query(func.count(InventorySnapshot.id))
        .join(ProductCatalog, InventorySnapshot.product_catalog_id == ProductCatalog.id)
        .filter(
            InventorySnapshot.product_type == "prescription",
            ProductCatalog.xfarma_prescription_category.is_(None),
        )
        .scalar()
    )

    # Contar OTC de CIMA (cima_requiere_receta=False) clasificados como prescription
    otc_as_prescription = (
        db.query(func.count(InventorySnapshot.id))
        .join(ProductCatalog, InventorySnapshot.product_catalog_id == ProductCatalog.id)
        .filter(
            InventorySnapshot.product_type == "prescription",
            ProductCatalog.cima_requiere_receta == False,  # noqa: E712
            ProductCatalog.xfarma_prescription_category.is_(None),
        )
        .scalar()
    )

    return {
        "current_distribution": type_stats,
        "potential_misclassified": potential_misclassified,
        "otc_as_prescription": otc_as_prescription,
        "needs_reclassification": otc_as_prescription > 0,
        "action": (
            f"POST /api/v1/reenrichment/reclassify-inventory para corregir {otc_as_prescription} productos OTC"
            if otc_as_prescription > 0
            else "No action needed"
        ),
    }


@router.post("/reclassify-inventory")
async def reclassify_inventory(
    background_tasks: BackgroundTasks,
    sync: bool = Query(False, description="Ejecutar síncronamente (solo para lotes pequeños)"),
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
):
    """
    Re-clasifica productos de inventario usando la lógica correcta.

    Issue #487: Fix para productos OTC de CIMA que estaban siendo
    clasificados como "prescription" cuando deberían ser "venta_libre".

    La lógica correcta (replica EnrichmentService._derive_product_type):
    - xfarma_prescription_category != NULL (excepto VETERINARIA) → "prescription"
    - VETERINARIA → depende de cima_requiere_receta
    - xfarma_prescription_category == NULL → "venta_libre"

    Requires: Admin authentication
    """
    from sqlalchemy import func

    from app.models import InventorySnapshot, ProductCatalog

    # Verificar cuántos productos necesitan corrección
    to_update = (
        db.query(func.count(InventorySnapshot.id))
        .filter(
            InventorySnapshot.product_catalog_id.isnot(None),
        )
        .scalar()
    )

    if to_update == 0:
        return {
            "status": "no_action_needed",
            "message": "No hay productos de inventario con catálogo para re-clasificar",
        }

    if sync and to_update <= 1000:
        # Ejecutar síncronamente para lotes pequeños
        updated, changes = _execute_inventory_reclassify_sync(db)
        return {
            "status": "completed",
            "total_reviewed": to_update,
            "updated": updated,
            "changes": changes,
        }

    # Ejecutar en background para lotes grandes
    background_tasks.add_task(_execute_inventory_reclassify)

    return {
        "status": "initiated",
        "message": f"Re-clasificación de inventario iniciada para {to_update} productos",
        "records_to_process": to_update,
        "estimated_time_seconds": (to_update / 1000) * 3,
        "check_status_at": "/api/v1/reenrichment/reclassify-inventory/status",
    }


def _execute_inventory_reclassify_sync(db: Session) -> tuple[int, dict]:
    """
    Ejecuta re-clasificación de inventario de forma síncrona.

    Returns:
        Tuple of (updated_count, changes_dict)
    """
    from app.models import InventorySnapshot, ProductCatalog

    changes = {"prescription_to_venta_libre": 0, "venta_libre_to_prescription": 0, "unchanged": 0}

    # Obtener todos los snapshots con catálogo
    snapshots = (
        db.query(InventorySnapshot)
        .filter(InventorySnapshot.product_catalog_id.isnot(None))
        .all()
    )

    # Cargar catálogos en batch
    catalog_ids = [s.product_catalog_id for s in snapshots]
    catalogs = db.query(ProductCatalog).filter(ProductCatalog.id.in_(catalog_ids)).all()
    catalog_map = {c.id: c for c in catalogs}

    updated = 0
    for snapshot in snapshots:
        catalog = catalog_map.get(snapshot.product_catalog_id)
        if not catalog:
            continue

        new_type = _derive_product_type_for_inventory(
            catalog.xfarma_prescription_category,
            catalog.cima_requiere_receta,
        )

        if snapshot.product_type != new_type:
            old_type = snapshot.product_type
            snapshot.product_type = new_type
            updated += 1

            if old_type == "prescription" and new_type == "venta_libre":
                changes["prescription_to_venta_libre"] += 1
            elif old_type == "venta_libre" and new_type == "prescription":
                changes["venta_libre_to_prescription"] += 1
        else:
            changes["unchanged"] += 1

    db.commit()
    return updated, changes


def _execute_inventory_reclassify():
    """
    Función helper para ejecutar re-clasificación de inventario en background.
    """
    import logging

    from app.models import InventorySnapshot, ProductCatalog

    logger = logging.getLogger(__name__)
    db = SessionLocal()

    try:
        logger.info("[RECLASSIFY_INVENTORY] Iniciando re-clasificación de inventario")

        batch_size = 500
        total_updated = 0
        to_venta_libre = 0
        to_prescription = 0
        offset = 0

        while True:
            # Obtener batch de snapshots con catálogo
            batch = (
                db.query(InventorySnapshot)
                .filter(InventorySnapshot.product_catalog_id.isnot(None))
                .offset(offset)
                .limit(batch_size)
                .all()
            )

            if not batch:
                break

            # Cargar catálogos en batch
            catalog_ids = [s.product_catalog_id for s in batch]
            catalogs = db.query(ProductCatalog).filter(ProductCatalog.id.in_(catalog_ids)).all()
            catalog_map = {c.id: c for c in catalogs}

            for snapshot in batch:
                catalog = catalog_map.get(snapshot.product_catalog_id)
                if not catalog:
                    continue

                new_type = _derive_product_type_for_inventory(
                    catalog.xfarma_prescription_category,
                    catalog.cima_requiere_receta,
                )

                if snapshot.product_type != new_type:
                    old_type = snapshot.product_type
                    snapshot.product_type = new_type
                    total_updated += 1

                    if new_type == "venta_libre":
                        to_venta_libre += 1
                    else:
                        to_prescription += 1

            db.commit()
            offset += batch_size
            logger.info(f"[RECLASSIFY_INVENTORY] Progreso: {offset} registros revisados, {total_updated} actualizados")

        logger.info(
            f"[RECLASSIFY_INVENTORY] Completado: {total_updated} registros actualizados "
            f"(→prescription: {to_prescription}, →venta_libre: {to_venta_libre})"
        )

    except Exception as e:
        logger.error(f"[RECLASSIFY_INVENTORY] Error: {str(e)}")
        db.rollback()
        raise
    finally:
        db.close()


def _derive_product_type_for_inventory(
    xfarma_prescription_category: str | None,
    cima_requiere_receta: bool | None,
) -> str:
    """
    Determina el tipo de venta basado en clasificación de prescripción.

    Replica la lógica de EnrichmentService._derive_product_type.
    Issue #487: Fix para clasificar correctamente productos OTC de CIMA.
    """
    # EXCEPCIÓN: Categoría VETERINARIA
    if xfarma_prescription_category == "VETERINARIA":
        if cima_requiere_receta is True:
            return "prescription"
        elif cima_requiere_receta is False:
            return "venta_libre"
        else:
            return "prescription"

    # REGLA GENERAL: Tener categoría → prescription
    elif xfarma_prescription_category is not None:
        return "prescription"

    # Sin categoría (NULL) → Venta Libre
    else:
        return "venta_libre"
