"""
API endpoints para Análisis de Tendencias y Estacionalidad.

Issue #489: Tab 3 "Tendencias y Estacionalidad" en /prescription.
Issue #498: STL Decomposition + Forecast con Holt-Winters.
Issue #500: Stock-out Risk Matrix.
Issue #501: Anomaly Detection + Alerts.
Issue #502: Export CSV + Badges.

Endpoints:
- GET /prescription/{pharmacy_id}/seasonality/hourly-heatmap - Heatmap día×hora
- GET /prescription/{pharmacy_id}/seasonality/monthly-index - Índice estacionalidad mensual
- GET /prescription/{pharmacy_id}/seasonality/kpis - KPIs de estacionalidad
- GET /prescription/{pharmacy_id}/seasonality/trend-decomposition - Descomposición STL
- GET /prescription/{pharmacy_id}/seasonality/forecast - Forecast Holt-Winters
- GET /prescription/{pharmacy_id}/seasonality/category-patterns - Patrones por categoría
- GET /prescription/{pharmacy_id}/seasonality/stockout-risk - Matriz de riesgo de stock
- GET /prescription/{pharmacy_id}/seasonality/anomalies - Detección de anomalías
- GET /prescription/{pharmacy_id}/seasonality/export - Exportar forecast a CSV
- GET /prescription/{pharmacy_id}/seasonality/badges - Badges para tab header
"""

import logging
from datetime import date
from typing import Any, Dict, List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Path, Query, status
from sqlalchemy import func
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.pharmacy import Pharmacy
from app.models.user import User
from app.schemas.seasonality import (
    AnomaliesResponse,
    CategoryPatternsResponse,
    ExportForecastResponse,
    ForecastResponse,
    HourlyHeatmapResponse,
    MonthlyIndexResponse,
    SeasonalityBadges,
    SeasonalityKPIs,
    StockoutRiskResponse,
    TrendDecompositionResponse,
)
from app.services.prescription_seasonality_service import prescription_seasonality_service

logger = logging.getLogger(__name__)

router = APIRouter(
    prefix="/prescription",
    tags=["prescription", "seasonality"]
)


def _verify_pharmacy_access(
    pharmacy_id: UUID,
    current_user: User,
    db: Session,
) -> Pharmacy:
    """
    Verificar que el usuario tiene acceso a la farmacia.

    REGLA #10: Relación 1:1 User-Pharmacy.
    """
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()

    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar acceso (admin puede ver todo, user solo su farmacia)
    if current_user.role != "admin" and str(current_user.pharmacy_id) != str(pharmacy_id):
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    return pharmacy


@router.get(
    "/{pharmacy_id}/seasonality/hourly-heatmap",
    response_model=Dict[str, Any],
    summary="Heatmap de ventas por día y hora",
    description=(
        "Genera un heatmap de ventas de prescripción agregadas por día de la semana "
        "y hora del día. Útil para gestión de turnos y personal. "
        "Requiere autenticación JWT."
    ),
)
async def get_hourly_heatmap(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (YYYY-MM-DD)"),
    categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías de prescripción (opcional)"
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Genera heatmap de ventas por día de semana y hora.

    **Estructura de respuesta:**
    - `heatmap_data`: Lista de celdas con weekday, hour, sales, units
    - `summary`: Pico de ventas, totales, promedios
    - `period`: Rango de fechas analizado

    **Dimensiones:**
    - Eje Y: Días de la semana (1=Lunes a 7=Domingo)
    - Eje X: Horas del día (0-23)
    - Valor: Ventas totales o unidades
    """
    logger.info(
        f"[API] GET hourly-heatmap: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_hourly_heatmap(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            categories=categories,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in hourly-heatmap: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in hourly-heatmap: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al generar heatmap",
        )


@router.get(
    "/{pharmacy_id}/seasonality/monthly-index",
    response_model=Dict[str, Any],
    summary="Índice de estacionalidad mensual",
    description=(
        "Calcula el índice de estacionalidad para cada mes del año. "
        "El índice 1.0 representa el promedio anual; valores superiores indican "
        "meses con más ventas que el promedio. Útil para planificación de compras. "
        "Requiere autenticación JWT."
    ),
)
async def get_monthly_index(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (YYYY-MM-DD)"),
    category: Optional[str] = Query(
        None,
        description="Filtrar por categoría específica (opcional)"
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Calcula el índice de estacionalidad mensual.

    **Estructura de respuesta:**
    - `monthly_index`: Lista de 12 puntos (uno por mes) con índice y promedios
    - `summary`: Mes pico, mes valle, total anual
    - `period`: Rango de fechas y años incluidos
    - `data_quality`: Indicadores de suficiencia de datos

    **Interpretación del índice:**
    - 1.0 = Promedio anual
    - 1.3 = 30% más ventas que el promedio (ej: Enero para antigripales)
    - 0.7 = 30% menos ventas que el promedio (ej: Agosto)
    """
    logger.info(
        f"[API] GET monthly-index: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}, category={category}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_monthly_index(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            category=category,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in monthly-index: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in monthly-index: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al calcular índice mensual",
        )


@router.get(
    "/{pharmacy_id}/seasonality/kpis",
    response_model=Dict[str, Any],
    summary="KPIs de estacionalidad",
    description=(
        "Obtiene los KPIs principales de estacionalidad para el header del Tab 3. "
        "Incluye tendencia YoY, próximo pico estacional y alertas de anomalías. "
        "Requiere autenticación JWT."
    ),
)
async def get_seasonality_kpis(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (YYYY-MM-DD)"),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Obtiene KPIs principales de estacionalidad.

    **KPIs retornados:**
    - `trend_yoy`: Tendencia año sobre año (% cambio)
    - `trend_direction`: Dirección ("up", "down", "stable")
    - `next_peak_category`: Próxima categoría con pico estacional
    - `next_peak_months`: Meses del próximo pico (ej: "Nov-Feb")
    - `days_until_peak`: Días hasta el próximo pico
    - `anomalies_count`: Número de anomalías detectadas (Issue #501)
    """
    logger.info(
        f"[API] GET seasonality-kpis: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_seasonality_kpis(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in seasonality-kpis: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in seasonality-kpis: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al calcular KPIs de estacionalidad",
        )


@router.get(
    "/{pharmacy_id}/seasonality/trend-decomposition",
    response_model=Dict[str, Any],
    summary="Descomposición STL de tendencia",
    description=(
        "Descompone la serie temporal de ventas usando STL "
        "(Seasonal-Trend decomposition using LOESS). Separa la serie en "
        "tendencia, componente estacional y residuo. Requiere mínimo 24 meses de datos. "
        "Requiere autenticación JWT."
    ),
)
async def get_trend_decomposition(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: Optional[date] = Query(
        None,
        description="Fecha inicio del período (opcional - si no se especifica, usa todo el historial)"
    ),
    date_to: date = Query(..., description="Fecha fin del período (YYYY-MM-DD)"),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Descompone la serie temporal usando STL.

    **Componentes:**
    - `observed`: Valores observados (ventas reales)
    - `trend`: Tendencia de largo plazo (serie suavizada)
    - `seasonal`: Componente estacional (patrón repetitivo cada 12 meses)
    - `residual`: Residuo (ruido, anomalías)

    **Requisitos:**
    - Mínimo 12 meses de datos para descomposición robusta
    - Si date_from no se especifica, usa todo el historial disponible
    - Si no hay datos suficientes, retorna data_quality.sufficient_data = false
    """
    # Si no se especifica date_from, obtener fecha mínima de datos
    if date_from is None:
        from app.models import SalesData
        min_date_result = db.query(func.min(SalesData.sale_date)).filter(
            SalesData.pharmacy_id == pharmacy_id
        ).scalar()
        date_from = min_date_result.date() if min_date_result else date_to

    logger.info(
        f"[API] GET trend-decomposition: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_trend_decomposition(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in trend-decomposition: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in trend-decomposition: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al calcular descomposición STL",
        )


@router.get(
    "/{pharmacy_id}/seasonality/forecast",
    response_model=Dict[str, Any],
    summary="Forecast de ventas con Holt-Winters",
    description=(
        "Genera predicción de ventas para los próximos N meses usando "
        "Holt-Winters Exponential Smoothing. Incluye intervalo de confianza del 95%. "
        "Requiere mínimo 24 meses de datos históricos. "
        "Requiere autenticación JWT."
    ),
)
async def get_forecast(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: Optional[date] = Query(
        None,
        description="Fecha inicio del período (opcional - si no se especifica, usa todo el historial)"
    ),
    date_to: date = Query(..., description="Fecha fin del período histórico (YYYY-MM-DD)"),
    periods_ahead: int = Query(
        default=3,
        ge=1,
        le=12,
        description="Número de meses a predecir (1-12)",
    ),
    confidence: float = Query(
        default=0.95,
        ge=0.80,
        le=0.99,
        description="Nivel de confianza para el intervalo (0.80-0.99)",
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Genera forecast de ventas de prescripción.

    **Modelo:** Holt-Winters Exponential Smoothing con estacionalidad aditiva.

    **Respuesta:**
    - `forecast`: Lista de predicciones con fecha, valor e intervalo de confianza
    - `historical`: Últimos 6 meses de datos reales para contexto
    - `model_info`: Parámetros del modelo utilizado
    - `summary`: Resumen con total predicho, promedio mensual, tendencia

    **Interpretación:**
    - "Lo normal es vender entre €45K y €51K" (límites del intervalo)
    - Cuanto más ancho el intervalo, mayor incertidumbre
    - Si date_from no se especifica, usa todo el historial disponible
    """
    # Si no se especifica date_from, obtener fecha mínima de datos
    if date_from is None:
        from app.models import SalesData
        min_date_result = db.query(func.min(SalesData.sale_date)).filter(
            SalesData.pharmacy_id == pharmacy_id
        ).scalar()
        date_from = min_date_result.date() if min_date_result else date_to

    logger.info(
        f"[API] GET forecast: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}, periods={periods_ahead}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_forecast(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            periods_ahead=periods_ahead,
            confidence=confidence,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in forecast: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in forecast: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al generar forecast",
        )


@router.get(
    "/{pharmacy_id}/seasonality/category-patterns",
    response_model=CategoryPatternsResponse,
    summary="Patrones de estacionalidad por categoría",
    description=(
        "Obtiene los patrones de estacionalidad para las principales categorías "
        "de prescripción. Muestra qué categorías tienen picos estacionales "
        "(ej: antigripales en Nov-Feb, antihistamínicos en Mar-Jun). "
        "Requiere autenticación JWT."
    ),
)
async def get_category_patterns(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (YYYY-MM-DD)"),
    top_n: int = Query(
        default=10,
        ge=1,
        le=20,
        description="Número máximo de categorías a analizar (1-20)",
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> CategoryPatternsResponse:
    """
    Obtiene patrones de estacionalidad por categoría de prescripción.

    **Issue #499**: Fase 3 - Estacionalidad por Categoría ATC.

    **Estructura de respuesta:**
    - `category_patterns`: Lista de categorías con sus índices mensuales
    - `summary`: Categoría más/menos estacional
    - `period`: Rango de fechas analizado

    **Casos de uso:**
    - Antigripales: Pico Nov-Feb (índice ~1.5)
    - Antihistamínicos: Pico Mar-Jun (índice ~1.3)
    - Dietoterápicos: Sin estacionalidad marcada (índice ~1.0)

    **Interpretación del índice:**
    - 1.0 = Promedio anual
    - >1.1 = Mes por encima del promedio (pico)
    - <0.9 = Mes por debajo del promedio (valle)
    """
    logger.info(
        f"[API] GET category-patterns: pharmacy={pharmacy_id}, "
        f"date_from={date_from}, date_to={date_to}, top_n={top_n}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_category_patterns(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            top_n=top_n,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in category-patterns: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in category-patterns: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al calcular patrones por categoría",
        )


# =============================================================================
# STOCK-OUT RISK MATRIX (Issue #500)
# =============================================================================


@router.get(
    "/{pharmacy_id}/seasonality/stockout-risk",
    response_model=StockoutRiskResponse,
    summary="Matriz de riesgo de rotura de stock",
    description="""
    Calcula el riesgo de rotura de stock cruzando estacionalidad con inventario.

    Issue #500: Fase 3.5 - Stock-out Risk Matrix.

    Combina:
    - Índice de estacionalidad del mes actual
    - Stock actual del inventario
    - Ventas promedio diarias (últimos 30 días)

    Niveles de riesgo (por ratio de cobertura vs 7 días):
    - CRÍTICO (<0.5): Stock insuficiente para 3-4 días
    - ALTO (0.5-0.8): Stock para menos de una semana
    - OK (0.8-1.2): Stock adecuado
    - MEDIO (1.2-2.0): Exceso leve
    - EXCESO (>2.0): Sobrestock significativo
    """,
)
async def get_stockout_risk(
    pharmacy_id: UUID,
    days_until_restock: int = Query(
        7,
        ge=1,
        le=30,
        description="Días hasta la próxima reposición",
    ),
    top_n: int = Query(
        50,
        ge=10,
        le=200,
        description="Número máximo de productos a retornar",
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> StockoutRiskResponse:
    """
    Endpoint para obtener matriz de riesgo de rotura de stock.

    Requiere:
    - Snapshot de inventario reciente (product_type='prescription')
    - Datos de ventas de los últimos 30 días
    - Datos históricos para calcular índice de estacionalidad
    """
    logger.info(
        f"[API] stockout-risk request for pharmacy {pharmacy_id}, "
        f"restock_days={days_until_restock}, top_n={top_n}"
    )

    # Validar acceso (usar helper centralizado)
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_stockout_risk(
            db=db,
            pharmacy_id=pharmacy_id,
            days_until_restock=days_until_restock,
            top_n=top_n,
            atc_code=atc_code,
        )
        return result

    except Exception as e:
        logger.error(f"[API] Error in stockout-risk: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al calcular matriz de riesgo de stock",
        )


# =============================================================================
# ANOMALY DETECTION (Issue #501)
# =============================================================================


@router.get(
    "/{pharmacy_id}/seasonality/anomalies",
    response_model=AnomaliesResponse,
    summary="Detección de anomalías en ventas",
    description="""
    Detecta anomalías estadísticas en las ventas de prescripción.

    Issue #501: Fase 4 - Detector de Anomalías + Alertas.

    Algoritmo:
    - Calcula Z-score sobre ventas diarias
    - Identifica días con |Z-score| > sensibilidad (default: 2.0)

    Filtros anti-falsos positivos:
    - Excluye festivos nacionales españoles (Semana Santa, Navidad, etc.)
    - Excluye días con ventas en lote (>30% del diario = institucional)
    - Excluye días con inventario cero (roturas de stock) [próximamente]

    Niveles de severidad:
    - ALTA: |Z-score| > 3 (muy inusual)
    - MEDIA: |Z-score| > 2.5
    - BAJA: |Z-score| > 2
    """,
)
async def get_anomalies(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: Optional[date] = Query(
        None,
        description="Fecha inicio del período (default: 90 días atrás)",
    ),
    date_to: Optional[date] = Query(
        None,
        description="Fecha fin del período (default: hoy)",
    ),
    sensitivity: float = Query(
        2.0,
        ge=1.5,
        le=4.0,
        description="Umbral Z-score para detectar anomalías (1.5-4.0)",
    ),
    exclude_holidays: bool = Query(
        True,
        description="Excluir festivos nacionales españoles",
    ),
    exclude_bulk_sales: bool = Query(
        True,
        description="Excluir días con ventas en lote (institucionales)",
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> AnomaliesResponse:
    """
    Endpoint para detectar anomalías en ventas.

    Detecta días con comportamiento estadísticamente inusual,
    excluyendo causas conocidas (festivos, ventas institucionales).

    Útil para:
    - Identificar problemas operativos (cierres, errores)
    - Detectar oportunidades (demanda excepcional)
    - Auditoría de patrones de venta
    """
    logger.info(
        f"[API] anomalies request for pharmacy {pharmacy_id}, "
        f"sensitivity={sensitivity}, exclude_holidays={exclude_holidays}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.detect_anomalies(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            sensitivity=sensitivity,
            exclude_holidays=exclude_holidays,
            exclude_zero_stock=False,  # TODO: Implementar cuando tengamos stock diario
            exclude_bulk_sales_pct=0.3 if exclude_bulk_sales else 0,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in anomalies: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in anomalies: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al detectar anomalías",
        )


# =============================================================================
# EXPORT CSV + BADGES (Issue #502)
# =============================================================================


@router.get(
    "/{pharmacy_id}/seasonality/export",
    response_model=ExportForecastResponse,
    summary="Exportar forecast a CSV",
    description="""
    Genera datos de forecast listos para exportar a CSV/Excel.

    Issue #502: Fase 5 - Exportación CSV + Badges.

    Combina:
    - Forecast general de Holt-Winters
    - Patrones de estacionalidad por categoría
    - Índices de estacionalidad mensual

    Formato de salida:
    - Categoría, Mes, Forecast, Límite Inferior, Límite Superior, Índice Estacionalidad

    Útil para:
    - Importar en Excel para análisis adicional
    - Planificación de compras por categoría
    - Informes mensuales de previsión
    """,
)
async def export_forecast(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del histórico (YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del histórico (YYYY-MM-DD)"),
    periods_ahead: int = Query(
        default=3,
        ge=1,
        le=12,
        description="Meses a predecir (1-12)",
    ),
    top_categories: int = Query(
        default=10,
        ge=1,
        le=20,
        description="Número de categorías a incluir (1-20)",
    ),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> ExportForecastResponse:
    """
    Endpoint para exportar forecast de estacionalidad.

    Genera una tabla con previsiones por categoría y mes,
    lista para copiar a Excel o descargar como CSV.
    """
    logger.info(
        f"[API] export request for pharmacy {pharmacy_id}, "
        f"periods={periods_ahead}, categories={top_categories}"
    )

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.export_forecast(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            periods_ahead=periods_ahead,
            top_categories=top_categories,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in export: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in export: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al generar exportación de forecast",
        )


@router.get(
    "/{pharmacy_id}/seasonality/badges",
    response_model=SeasonalityBadges,
    summary="Badges para tab de estacionalidad",
    description="""
    Obtiene badges visuales para el header del tab de estacionalidad.

    Issue #502: Fase 5 - Badges visuales en tab header.

    Badges incluidos:
    - Número de anomalías en últimos 30 días (con color según severidad)
    - Próximo pico estacional (ej: "Antigripales en 45d")
    - Alertas de stockout (productos con riesgo crítico/alto)

    Colores Bootstrap:
    - danger: Urgente (anomalías severas, stockout crítico)
    - warning: Atención (anomalías, stockout alto, pico cercano)
    - info: Informativo (pico moderado)
    - secondary: Normal (sin alertas)
    """,
)
async def get_badges(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    atc_code: Optional[str] = Query(
        None,
        description="Filtrar por código ATC nivel 1 (A-V). Issue #532",
        min_length=1,
        max_length=1,
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
) -> SeasonalityBadges:
    """
    Endpoint para obtener badges del tab de estacionalidad.

    Proporciona información rápida sobre:
    - Estado de anomalías
    - Próximo pico estacional
    - Alertas de stockout
    """
    logger.info(f"[API] badges request for pharmacy {pharmacy_id}")

    # Verificar acceso a farmacia
    _verify_pharmacy_access(pharmacy_id, current_user, db)

    try:
        result = prescription_seasonality_service.get_badges(
            db=db,
            pharmacy_id=pharmacy_id,
            atc_code=atc_code,
        )
        return result

    except ValueError as e:
        logger.warning(f"[API] Validation error in badges: {e}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(f"[API] Error in badges: {e}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener badges de estacionalidad",
        )