"""
API endpoints para Dashboard de Análisis de Ventas de Prescripción.

Issue #400 Sprint 1: Endpoints protegidos por JWT para análisis terapéutico.

Endpoints:
- GET /prescription/{pharmacy_id}/overview - KPIs + time series + resumen categoría
- GET /prescription/{pharmacy_id}/distribution-by-atc - Distribución jerárquica ATC
- GET /prescription/{pharmacy_id}/waterfall-analysis - Análisis waterfall YoY/MoM/QoQ
"""

import logging
from datetime import date
from typing import Any, Dict, List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Path, Query, status
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.pharmacy import Pharmacy
from app.models.user import User
from app.schemas.prescription_analytics import (
    ATCDistributionResponse,
    PrescriptionOverviewResponse,
    TopContributorsResponse,
    WaterfallAnalysisResponse,
)
from app.services.prescription_analytics_service import PrescriptionAnalyticsService

logger = logging.getLogger(__name__)

router = APIRouter(
    prefix="/prescription",
    tags=["prescription", "analytics"]
)


@router.get(
    "/{pharmacy_id}/overview",
    response_model=PrescriptionOverviewResponse,
    summary="Vista general de prescripción",
    description=(
        "KPIs globales + evolución temporal + resumen por categoría. "
        "Requiere autenticación JWT. El usuario debe tener acceso a la farmacia solicitada."
    ),
)
async def get_prescription_overview(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (formato YYYY-MM-DD)"),
    categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías específicas de prescripción (opcional)"
    ),
    atc_codes: Optional[List[str]] = Query(
        None,
        description="Filtrar por códigos ATC nivel 1 (A-V). Ej: ['A', 'C', 'N']"
    ),
    laboratory_codes: Optional[List[str]] = Query(
        None,
        description="Filtrar por códigos de laboratorio CIMA/nomenclator (opcional). Issue #444"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene KPIs globales, evolución temporal y resumen por categoría de prescripción.

    **KPIs retornados:**
    - Total de ventas de prescripción en el período
    - Total de unidades vendidas
    - Porcentaje del total de ventas (prescripción vs total farmacia)
    - Ticket promedio de productos de prescripción
    - Coverage de códigos ATC (% de ventas con código ATC asignado)

    **Time Series:**
    - Formato long/flat (cada punto = mes + categoría)
    - Optimizado para gráficos Plotly
    - Agregación mensual

    **Category Summary:**
    - Resumen agregado por las 14 categorías de prescripción
    - Ordenado por ventas descendente

    **Autenticación:**
    - Requiere JWT válido
    - El usuario debe tener acceso a la farmacia solicitada

    **Validaciones:**
    - date_from < date_to
    - Rango máximo: 24 meses
    - Farmacia existe y usuario tiene acceso

    **Ejemplo request:**
    ```bash
    GET /api/v1/prescription/123e4567-e89b-12d3-a456-426614174000/overview?date_from=2025-01-01&date_to=2025-03-31
    Authorization: Bearer {JWT_TOKEN}
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando overview "
        f"para farmacia {pharmacy_id} ({date_from} - {date_to})"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorización: el usuario debe pertenecer a esta farmacia
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intentó acceder a farmacia {pharmacy_id} "
            f"pero pertenece a farmacia {current_user.pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Validar rango máximo (24 meses)
    months_diff = (date_to.year - date_from.year) * 12 + (date_to.month - date_from.month)
    if months_diff > 24:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"El rango máximo es de 24 meses, pero se solicitaron {months_diff} meses",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.calculate_overview_kpis(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            categories=categories,
            atc_codes=atc_codes,  # Issue #441
            laboratory_codes=laboratory_codes,  # Issue #444
        )

        logger.info(
            f"[PRESCRIPTION_API] Overview calculado exitosamente: {result.kpis.total_sales} ventas, "
            f"{len(result.time_series)} puntos time series"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validación: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error calculando overview: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error calculando análisis de prescripción: {str(e)}",
        )


@router.get(
    "/{pharmacy_id}/distribution-by-atc",
    response_model=ATCDistributionResponse,
    summary="Distribución por códigos ATC",
    description=(
        "Distribución jerárquica por códigos ATC con drill-down (5 niveles). "
        "Requiere autenticación JWT."
    ),
)
async def get_atc_distribution(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (formato YYYY-MM-DD)"),
    atc_level: int = Query(
        1,
        ge=1,
        le=5,
        description=(
            "Nivel de detalle ATC (1-5): "
            "1=Anatómico, 2=Terapéutico, 3=Farmacológico, 4=Químico, 5=Principio activo"
        )
    ),
    categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías específicas de prescripción (opcional)"
    ),
    atc_codes: Optional[List[str]] = Query(
        None,
        description="Filtrar por códigos ATC nivel 1 (A-V). Issue #444"
    ),
    laboratory_codes: Optional[List[str]] = Query(
        None,
        description="Filtrar por códigos de laboratorio CIMA/nomenclator (opcional). Issue #444"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene distribución jerárquica por códigos ATC.

    **Niveles ATC:**
    - Nivel 1: Primer carácter (A-V) - Grupo anatómico principal
    - Nivel 2: Primeros 3 caracteres (A02) - Grupo terapéutico
    - Nivel 3: Primeros 4 caracteres (A02B) - Subgrupo farmacológico
    - Nivel 4: Primeros 5 caracteres (A02BC) - Subgrupo químico
    - Nivel 5: Completo 7 caracteres (A02BC01) - Principio activo

    **Response:**
    - atc_distribution: Lista de nodos ATC con ventas, unidades y porcentaje
    - uncategorized: Productos sin código ATC (sales, units, percentage)

    **Autenticación:**
    - Requiere JWT válido
    - El usuario debe tener acceso a la farmacia solicitada

    **Ejemplo request:**
    ```bash
    GET /api/v1/prescription/123e4567-e89b-12d3-a456-426614174000/distribution-by-atc?date_from=2025-01-01&date_to=2025-03-31&atc_level=2
    Authorization: Bearer {JWT_TOKEN}
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando distribución ATC nivel {atc_level} "
        f"para farmacia {pharmacy_id}"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorización
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intentó acceder a farmacia {pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar atc_level (FastAPI ya lo valida con ge/le, pero double-check)
    if not 1 <= atc_level <= 5:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"atc_level debe estar entre 1 y 5, recibido: {atc_level}",
        )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.get_atc_distribution(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            atc_level=atc_level,
            categories=categories,
            atc_codes=atc_codes,  # Issue #444
            laboratory_codes=laboratory_codes,  # Issue #444
        )

        logger.info(
            f"[PRESCRIPTION_API] Distribución ATC calculada: {len(result.atc_distribution)} nodos"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validación: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error calculando distribución ATC: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error calculando distribución ATC: {str(e)}",
        )


@router.get(
    "/{pharmacy_id}/waterfall-analysis",
    response_model=WaterfallAnalysisResponse,
    summary="Análisis waterfall de crecimiento",
    description=(
        "Análisis waterfall de crecimiento/decrecimiento por categoría. "
        "Comparaciones: YoY (año sobre año), MoM (mes sobre mes), QoQ (quarter sobre quarter). "
        "Requiere autenticación JWT."
    ),
)
async def get_waterfall_analysis(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período actual (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período actual (formato YYYY-MM-DD)"),
    comparison_period: str = Query(
        "yoy",
        pattern="^(yoy|mom|qoq|custom)$",
        description="Tipo de comparación: yoy, mom, qoq, o 'custom' para períodos explícitos"
    ),
    # Issue #441: Soporte para períodos explícitos
    comparison_date_from: Optional[date] = Query(
        None,
        description="Fecha inicio período comparación (solo si comparison_period='custom')"
    ),
    comparison_date_to: Optional[date] = Query(
        None,
        description="Fecha fin período comparación (solo si comparison_period='custom')"
    ),
    categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías específicas de prescripción (opcional)"
    ),
    laboratory_codes: Optional[List[str]] = Query(
        None,
        description="Filtrar por códigos de laboratorio CIMA/nomenclator (opcional). Issue #444"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene análisis waterfall de crecimiento por categoría.

    **Comparaciones:**
    - **YoY** (Year over Year): Compara con el mismo período del año anterior
    - **MoM** (Month over Month): Compara con el mes anterior
    - **QoQ** (Quarter over Quarter): Compara con el trimestre anterior

    **Response:**
    - comparison: Metadata de comparación (type, current_period, previous_period)
    - waterfall_data: Cambios por categoría (current_sales, previous_sales, absolute_change, percentage_change, contribution_to_growth)
    - summary: Resumen agregado (total_current, total_previous, total_growth)

    **Waterfall Data:**
    - Ordenado por contribución al crecimiento (descendente por valor absoluto)
    - Categorías con contribución positiva = crecimiento
    - Categorías con contribución negativa = decrecimiento

    **Autenticación:**
    - Requiere JWT válido
    - El usuario debe tener acceso a la farmacia solicitada

    **Ejemplo request:**
    ```bash
    GET /api/v1/prescription/123e4567-e89b-12d3-a456-426614174000/waterfall-analysis?date_from=2025-01-01&date_to=2025-03-31&comparison_period=yoy
    Authorization: Bearer {JWT_TOKEN}
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando waterfall {comparison_period.upper()} "
        f"para farmacia {pharmacy_id}"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorización
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intentó acceder a farmacia {pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar comparison_period (FastAPI ya lo valida con regex, pero double-check)
    if comparison_period not in ["yoy", "mom", "qoq", "custom"]:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"comparison_period debe ser 'yoy', 'mom', 'qoq' o 'custom', recibido: {comparison_period}",
        )

    # Issue #441: Validar períodos explícitos si comparison_period='custom'
    if comparison_period == "custom":
        if not comparison_date_from or not comparison_date_to:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="comparison_date_from y comparison_date_to son requeridos cuando comparison_period='custom'",
            )
        if comparison_date_from >= comparison_date_to:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=f"comparison_date_from ({comparison_date_from}) debe ser menor que comparison_date_to ({comparison_date_to})",
            )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.calculate_waterfall(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            comparison_period=comparison_period,
            comparison_date_from=comparison_date_from,  # Issue #441
            comparison_date_to=comparison_date_to,  # Issue #441
            categories=categories,
            laboratory_codes=laboratory_codes,  # Issue #444
        )

        logger.info(
            f"[PRESCRIPTION_API] Waterfall calculado: {len(result.waterfall_data)} categorías, "
            f"crecimiento total={float(result.summary['total_growth']):.2f}"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validación: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error calculando waterfall: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error calculando análisis waterfall: {str(e)}",
        )


@router.get(
    "/{pharmacy_id}/top-contributors",
    response_model=TopContributorsResponse,
    summary="Top contributors por principio activo",
    description=(
        "Obtiene los principios activos que mas contribuyen al cambio en ventas de prescripcion. "
        "Comparaciones: YoY (anio sobre anio), MoM (mes sobre mes), QoQ (quarter sobre quarter). "
        "Filtros: limit (10/20/50), direction (all/up/down). "
        "Requiere autenticacion JWT."
    ),
)
async def get_top_contributors(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del periodo actual (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del periodo actual (formato YYYY-MM-DD)"),
    comparison_period: str = Query(
        "yoy",
        pattern="^(yoy|mom|qoq|custom)$",
        description="Tipo de comparacion: yoy, mom, qoq, o 'custom' para periodos explicitos"
    ),
    comparison_date_from: Optional[date] = Query(
        None,
        description="Fecha inicio periodo comparacion (solo si comparison_period='custom')"
    ),
    comparison_date_to: Optional[date] = Query(
        None,
        description="Fecha fin periodo comparacion (solo si comparison_period='custom')"
    ),
    limit: int = Query(
        10,
        description="Numero de resultados: 10, 20 o 50"
    ),
    direction: str = Query(
        "all",
        pattern="^(all|up|down)$",
        description="Filtrar por direccion: all (todos), up (solo subidas), down (solo bajadas)"
    ),
    categories: Optional[List[str]] = Query(
        None,
        description="Filtrar por categorías específicas de prescripción (opcional). Issue #457"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene los principios activos que mas contribuyen al cambio en ventas de prescripcion.

    Issue #436 Fase 2: Top Contributors por Principio Activo.

    **Tabla resultante:**
    | Principio Activo | Categoria | Var Euro | Var % | Impacto Total |
    |------------------|-----------|----------|-------|---------------|
    | Omeprazol 20mg   | Conj. Homogeneo | +850 | +45% | +1.8% |
    | Metformina 850mg | Medicamentos | +620 | +22% | +1.4% |

    **Comparaciones:**
    - **YoY** (Year over Year): Compara con el mismo periodo del anio anterior
    - **MoM** (Month over Month): Compara con el mes anterior
    - **QoQ** (Quarter over Quarter): Compara con el trimestre anterior

    **Filtros:**
    - **limit**: Numero de resultados (10, 20 o 50)
    - **direction**: all (todos), up (solo subidas), down (solo bajadas)

    **Response:**
    - contributors: Lista de top contributors ordenados por impacto absoluto
    - total_change: Cambio total en ventas de prescripcion
    - comparison_period: Tipo de comparacion usada
    - filters_applied: Filtros aplicados (limit, direction, fechas)

    **Autenticacion:**
    - Requiere JWT valido
    - El usuario debe tener acceso a la farmacia solicitada

    **Ejemplo request:**
    ```bash
    GET /api/v1/prescription/123e4567-e89b-12d3-a456-426614174000/top-contributors?date_from=2025-01-01&date_to=2025-03-31&comparison_period=yoy&limit=10&direction=all
    Authorization: Bearer {JWT_TOKEN}
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando top contributors "
        f"{comparison_period.upper()} para farmacia {pharmacy_id} (limit={limit}, direction={direction})"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorizacion
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intento acceder a farmacia {pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar comparison_period (FastAPI ya lo valida con regex, pero double-check)
    if comparison_period not in ["yoy", "mom", "qoq", "custom"]:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"comparison_period debe ser 'yoy', 'mom', 'qoq' o 'custom', recibido: {comparison_period}",
        )

    # Issue #441: Validar períodos custom
    if comparison_period == "custom":
        if not comparison_date_from or not comparison_date_to:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="comparison_date_from y comparison_date_to son requeridos cuando comparison_period='custom'",
            )
        if comparison_date_from >= comparison_date_to:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="comparison_date_from debe ser anterior a comparison_date_to",
            )

    # Validar limit
    if limit not in [10, 20, 50]:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"limit debe ser 10, 20 o 50, recibido: {limit}",
        )

    # Validar direction (FastAPI ya lo valida con regex, pero double-check)
    if direction not in ["all", "up", "down"]:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"direction debe ser 'all', 'up' o 'down', recibido: {direction}",
        )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.get_top_contributors(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            comparison_period=comparison_period,
            comparison_date_from=comparison_date_from,  # Issue #441
            comparison_date_to=comparison_date_to,  # Issue #441
            limit=limit,
            direction=direction,
            categories=categories,  # Issue #457: Filtro por categorías
        )

        logger.info(
            f"[PRESCRIPTION_API] Top contributors calculados: {len(result.contributors)} items, "
            f"cambio total={float(result.total_change):.2f}"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validacion: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error calculando top contributors: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error calculando top contributors: {str(e)}",
        )


# Issue #441: Endpoint para productos por categoría (acordeón drill-down)
@router.get(
    "/{pharmacy_id}/products-by-category",
    response_model=Dict,
    status_code=status.HTTP_200_OK,
    summary="Obtener productos por categoría de prescripción",
    description=(
        "Retorna los productos agrupados por categoría de prescripción para drill-down en acordeón. "
        "Incluye top 10 productos por ventas en cada categoría con métricas. "
        "Requiere autenticación JWT."
    ),
)
async def get_products_by_category(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (formato YYYY-MM-DD)"),
    category: Optional[str] = Query(
        None,
        description="Filtrar por una categoría específica (si None, retorna todas)"
    ),
    limit: int = Query(
        10,
        ge=1,
        le=50,
        description="Número máximo de productos por categoría (default 10)"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene productos agrupados por categoría de prescripción.

    Issue #441 Fase 4: Acordeón de categorías con drill-down a productos.

    **Response:**
    ```json
    {
        "categories": [
            {
                "category_key": "medicamentos",
                "category_name": "Medicamentos",
                "total_sales": 15000.50,
                "total_units": 450,
                "products": [
                    {
                        "national_code": "123456",
                        "name": "Omeprazol 20mg",
                        "sales": 850.25,
                        "units": 35,
                        "percentage": 5.67
                    },
                    ...
                ]
            },
            ...
        ],
        "summary": {
            "total_categories": 5,
            "total_products": 50,
            "total_sales": 45000.00
        }
    }
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando productos por categoría "
        f"para farmacia {pharmacy_id}, category={category}"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorización
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intento acceder a farmacia {pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.get_products_by_category(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            category=category,
            limit=limit,
        )

        logger.info(
            f"[PRESCRIPTION_API] Productos por categoría: {len(result['categories'])} categorías, "
            f"{result['summary']['total_products']} productos totales"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validación: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error obteniendo productos por categoría: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error obteniendo productos por categoría: {str(e)}",
        )


# Issue #441: Endpoint para productos por principio activo (acordeón drill-down)
# Issue #451: Extendido para soportar drill-down por grupo homogéneo
@router.get(
    "/{pharmacy_id}/products-by-active-ingredient",
    response_model=Dict,
    status_code=status.HTTP_200_OK,
    summary="Obtener productos por grupo de análisis",
    description=(
        "Retorna los productos de un grupo de análisis específico para drill-down en acordeón. "
        "Soporta drill-down por grupo homogéneo, principio activo o producto. "
        "Incluye top 10 productos por ventas con métricas. "
        "Requiere autenticación JWT."
    ),
)
async def get_products_by_active_ingredient(
    pharmacy_id: UUID = Path(..., description="ID de la farmacia"),
    date_from: date = Query(..., description="Fecha inicio del período (formato YYYY-MM-DD)"),
    date_to: date = Query(..., description="Fecha fin del período (formato YYYY-MM-DD)"),
    active_ingredient: str = Query(..., description="Nombre del grupo de análisis"),
    limit: int = Query(
        10,
        ge=1,
        le=50,
        description="Número máximo de productos (default 10)"
    ),
    comparison_date_from: Optional[date] = Query(
        None,
        description="Fecha inicio período comparación YoY (opcional). Issue #444"
    ),
    comparison_date_to: Optional[date] = Query(
        None,
        description="Fecha fin período comparación YoY (opcional). Issue #444"
    ),
    grouping_type: Optional[str] = Query(
        None,
        description="Tipo de agrupación: homogeneous_group, active_ingredient, product. Issue #451"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene productos de un principio activo específico.

    Issue #441 Fase 4: Acordeón de principios activos con drill-down a productos.
    Issue #444: Añade comparación YoY con % contribución al cambio por producto.

    **Response:**
    ```json
    {
        "active_ingredient": "OMEPRAZOL",
        "products": [
            {
                "national_code": "123456",
                "product_name": "Omeprazol Cinfa 20mg",
                "total_sales": 850.25,
                "total_units": 35,
                "percentage": 45.2,
                "variation_euros": 120.50,
                "contribution_percent": 15.5
            },
            ...
        ],
        "summary": {
            "total_products": 5,
            "total_sales": 1880.50,
            "total_units": 85,
            "total_variation": 350.25
        }
    }
    ```
    """
    logger.info(
        f"[PRESCRIPTION_API] Usuario {current_user.email} solicitando productos para principio activo "
        f"'{active_ingredient}' en farmacia {pharmacy_id}"
    )

    # Verificar que la farmacia existe
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada",
        )

    # Verificar autorización
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        logger.warning(
            f"[PRESCRIPTION_API] Usuario {current_user.email} intento acceder a farmacia {pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes acceso a esta farmacia",
        )

    # Validar fechas
    if date_from >= date_to:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"date_from ({date_from}) debe ser menor que date_to ({date_to})",
        )

    # Llamar al servicio
    try:
        service = PrescriptionAnalyticsService()
        result = service.get_products_by_active_ingredient(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
            active_ingredient=active_ingredient,
            limit=limit,
            comparison_date_from=comparison_date_from,  # Issue #444
            comparison_date_to=comparison_date_to,
            grouping_type=grouping_type,  # Issue #451: Drill-down adaptativo
        )

        logger.info(
            f"[PRESCRIPTION_API] Productos por {grouping_type or 'active_ingredient'} '{active_ingredient}': "
            f"{len(result['products'])} productos encontrados"
        )

        return result

    except ValueError as e:
        logger.error(f"[PRESCRIPTION_API] Error de validación: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e),
        )
    except Exception as e:
        logger.error(
            f"[PRESCRIPTION_API] Error obteniendo productos por principio activo: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error obteniendo productos por principio activo: {str(e)}",
        )
