# backend/app/api/pharmacy_targets.py
"""
API endpoints para gestión de targets/benchmarks por farmacia.

Issue #510: Benchmarks y Líneas de Referencia en Gráficos
Permite configurar líneas de referencia (mínimo, objetivo, excelente)
para contextualizar rendimiento en gráficos del dashboard VentaLibre.
"""

import logging
from typing import Any, Dict, List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from ..api.deps import get_current_user, verify_pharmacy_access
from ..database import get_db
from ..models.pharmacy_targets import DEFAULT_TARGETS, PharmacyTarget
from ..models.user import User
from ..schemas.pharmacy_targets import (
    BulkTargetCreate,
    DeleteResponse,
    InitDefaultsResponse,
    TargetChartFormat,
    TargetCreate,
    TargetListResponse,
    TargetResponse,
    TargetType,
    TargetUpdate,
)
from ..utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/pharmacy-targets", tags=["pharmacy-targets"])


@router.get("/{pharmacy_id}", response_model=TargetListResponse)
async def get_pharmacy_targets(
    pharmacy_id: UUID,
    target_type: Optional[TargetType] = Query(None, description="Filtrar por tipo"),
    category: Optional[str] = Query(None, description="Filtrar por categoría"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Obtiene todos los targets configurados para una farmacia.

    Args:
        pharmacy_id: ID de la farmacia
        target_type: Filtro opcional por tipo de target
        category: Filtro opcional por categoría NECESIDAD

    Returns:
        Lista de targets configurados
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        query = db.query(PharmacyTarget).filter(PharmacyTarget.pharmacy_id == pharmacy_id)

        if target_type:
            query = query.filter(PharmacyTarget.target_type == target_type.value)

        if category is not None:
            if category == "":
                # Filtrar solo targets globales (sin categoría)
                query = query.filter(PharmacyTarget.category.is_(None))
            else:
                query = query.filter(PharmacyTarget.category == category)

        targets = query.order_by(PharmacyTarget.target_type, PharmacyTarget.category).all()

        logger.info(
            "pharmacy_targets.get",
            extra={
                "pharmacy_id": str(pharmacy_id),
                "count": len(targets),
                "filter_type": target_type.value if target_type else None,
                "filter_category": category,
            },
        )

        return {
            "success": True,
            "pharmacy_id": pharmacy_id,
            "targets": [TargetResponse.model_validate(t) for t in targets],
            "count": len(targets),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting targets for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al obtener targets. Contacte al administrador.",
        )


@router.get("/{pharmacy_id}/chart-format", response_model=List[TargetChartFormat])
async def get_targets_for_chart(
    pharmacy_id: UUID,
    target_types: List[TargetType] = Query(..., description="Tipos de target a obtener"),
    category: Optional[str] = Query(None, description="Categoría específica o null para global"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> List[Dict[str, Any]]:
    """
    Obtiene targets en formato optimizado para Plotly charts.

    Args:
        pharmacy_id: ID de la farmacia
        target_types: Lista de tipos de target a obtener
        category: Categoría específica o null para targets globales

    Returns:
        Lista de targets en formato {value, color, line_style, label}
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        query = db.query(PharmacyTarget).filter(
            PharmacyTarget.pharmacy_id == pharmacy_id,
            PharmacyTarget.target_type.in_([t.value for t in target_types]),
        )

        if category:
            # Buscar targets específicos de categoría, con fallback a globales
            specific = query.filter(PharmacyTarget.category == category).all()
            if specific:
                targets = specific
            else:
                targets = query.filter(PharmacyTarget.category.is_(None)).all()
        else:
            targets = query.filter(PharmacyTarget.category.is_(None)).all()

        return [t.to_chart_dict() for t in targets]

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting chart targets for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al obtener targets. Contacte al administrador.",
        )


@router.post("/{pharmacy_id}", response_model=TargetResponse)
async def create_or_update_target(
    pharmacy_id: UUID,
    target: TargetCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Crea o actualiza un target para una farmacia.

    Si ya existe un target con el mismo (pharmacy_id, target_type, category),
    se actualiza. Si no existe, se crea nuevo.

    Args:
        pharmacy_id: ID de la farmacia
        target: Datos del target a crear/actualizar

    Returns:
        Target creado o actualizado
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        # Buscar target existente
        existing = (
            db.query(PharmacyTarget)
            .filter(
                PharmacyTarget.pharmacy_id == pharmacy_id,
                PharmacyTarget.target_type == target.target_type.value,
                PharmacyTarget.category == target.category,
            )
            .first()
        )

        if existing:
            # Actualizar existente
            existing.value = target.value
            existing.color = target.color
            existing.line_style = target.line_style.value
            existing.label = target.label
            existing.user_configured = True
            existing.updated_at = utc_now()
            db.commit()
            db.refresh(existing)

            logger.info(
                "pharmacy_targets.updated",
                extra={
                    "pharmacy_id": str(pharmacy_id),
                    "target_type": target.target_type.value,
                    "category": target.category,
                    "value": float(target.value),
                },
            )

            return TargetResponse.model_validate(existing)
        else:
            # Crear nuevo
            new_target = PharmacyTarget(
                pharmacy_id=pharmacy_id,
                target_type=target.target_type.value,
                category=target.category,
                value=target.value,
                color=target.color,
                line_style=target.line_style.value,
                label=target.label,
                user_configured=True,
            )
            db.add(new_target)
            db.commit()
            db.refresh(new_target)

            logger.info(
                "pharmacy_targets.created",
                extra={
                    "pharmacy_id": str(pharmacy_id),
                    "target_type": target.target_type.value,
                    "category": target.category,
                    "value": float(target.value),
                },
            )

            return TargetResponse.model_validate(new_target)

    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        logger.error(f"Error creating/updating target for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al crear/actualizar target. Contacte al administrador.",
        )


@router.post("/{pharmacy_id}/bulk", response_model=TargetListResponse)
async def bulk_create_targets(
    pharmacy_id: UUID,
    request: BulkTargetCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Crea o actualiza múltiples targets a la vez (operación atómica).

    Args:
        pharmacy_id: ID de la farmacia
        request: Lista de targets a crear/actualizar

    Returns:
        Lista de targets creados/actualizados
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        results = []
        for target in request.targets:
            # Upsert logic inline para operación atómica (no commit por item)
            existing = (
                db.query(PharmacyTarget)
                .filter(
                    PharmacyTarget.pharmacy_id == pharmacy_id,
                    PharmacyTarget.target_type == target.target_type.value,
                    PharmacyTarget.category == target.category,
                )
                .first()
            )

            if existing:
                existing.value = target.value
                existing.color = target.color
                existing.line_style = target.line_style.value
                existing.label = target.label
                existing.user_configured = True
                existing.updated_at = utc_now()
                results.append(existing)
            else:
                new_target = PharmacyTarget(
                    pharmacy_id=pharmacy_id,
                    target_type=target.target_type.value,
                    category=target.category,
                    value=target.value,
                    color=target.color,
                    line_style=target.line_style.value,
                    label=target.label,
                    user_configured=True,
                )
                db.add(new_target)
                results.append(new_target)

        # Single atomic commit
        db.commit()
        for t in results:
            db.refresh(t)

        logger.info(
            f"pharmacy_targets.bulk_created: pharmacy_id={pharmacy_id}, count={len(results)}"
        )

        return {
            "success": True,
            "pharmacy_id": pharmacy_id,
            "targets": [TargetResponse.model_validate(t) for t in results],
            "count": len(results),
        }

    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        logger.error(f"Error bulk creating targets for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al crear targets. Contacte al administrador.",
        )


@router.post("/{pharmacy_id}/defaults", response_model=InitDefaultsResponse)
async def initialize_default_targets(
    pharmacy_id: UUID,
    force: bool = Query(False, description="Sobrescribir targets existentes"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Inicializa targets con valores por defecto.

    Args:
        pharmacy_id: ID de la farmacia
        force: Si True, sobrescribe targets existentes no configurados por usuario

    Returns:
        Targets inicializados
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        created_count = 0
        created_targets = []

        for default in DEFAULT_TARGETS:
            # Verificar si ya existe
            existing = (
                db.query(PharmacyTarget)
                .filter(
                    PharmacyTarget.pharmacy_id == pharmacy_id,
                    PharmacyTarget.target_type == default["target_type"],
                    PharmacyTarget.category.is_(None),
                )
                .first()
            )

            if existing:
                if force and not existing.user_configured:
                    # Actualizar solo si force=True y no fue configurado por usuario
                    existing.value = default["value"]
                    existing.color = default["color"]
                    existing.line_style = default["line_style"]
                    existing.label = default["label"]
                    existing.updated_at = utc_now()
                    created_targets.append(existing)
                    created_count += 1
                else:
                    created_targets.append(existing)
            else:
                # Crear nuevo
                new_target = PharmacyTarget(
                    pharmacy_id=pharmacy_id,
                    target_type=default["target_type"],
                    category=None,
                    value=default["value"],
                    color=default["color"],
                    line_style=default["line_style"],
                    label=default["label"],
                    user_configured=False,
                )
                db.add(new_target)
                created_targets.append(new_target)
                created_count += 1

        db.commit()

        # Refrescar para obtener IDs
        for t in created_targets:
            db.refresh(t)

        logger.info(
            "pharmacy_targets.defaults_initialized",
            extra={
                "pharmacy_id": str(pharmacy_id),
                "created_count": created_count,
                "force": force,
            },
        )

        return {
            "success": True,
            "message": f"Initialized {created_count} default targets",
            "created_count": created_count,
            "targets": [TargetResponse.model_validate(t) for t in created_targets],
        }

    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        logger.error(f"Error initializing defaults for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al inicializar targets. Contacte al administrador.",
        )


@router.delete("/{pharmacy_id}/{target_type}", response_model=DeleteResponse)
async def delete_target(
    pharmacy_id: UUID,
    target_type: TargetType,
    category: Optional[str] = Query(None, description="Categoría específica o null para global"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Elimina un target específico.

    Args:
        pharmacy_id: ID de la farmacia
        target_type: Tipo de target a eliminar
        category: Categoría específica o null para target global

    Returns:
        Confirmación de eliminación
    """
    try:
        # Verificar acceso a la farmacia (REGLA #7 - Security)
        verify_pharmacy_access(pharmacy_id, current_user)

        query = db.query(PharmacyTarget).filter(
            PharmacyTarget.pharmacy_id == pharmacy_id,
            PharmacyTarget.target_type == target_type.value,
        )

        if category:
            query = query.filter(PharmacyTarget.category == category)
        else:
            query = query.filter(PharmacyTarget.category.is_(None))

        target = query.first()

        if not target:
            raise HTTPException(
                status_code=404,
                detail=f"Target {target_type.value} not found for pharmacy",
            )

        db.delete(target)
        db.commit()

        logger.info(
            "pharmacy_targets.deleted",
            extra={
                "pharmacy_id": str(pharmacy_id),
                "target_type": target_type.value,
                "category": category,
            },
        )

        return {
            "success": True,
            "message": f"Deleted target {target_type.value}",
            "deleted_type": target_type.value,
            "deleted_category": category,
        }

    except HTTPException:
        raise
    except Exception as e:
        db.rollback()
        logger.error(f"Error deleting target for pharmacy {pharmacy_id}: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error interno al eliminar target. Contacte al administrador.",
        )
