﻿# backend/app/api/pharmacy_partners.py
"""
API endpoints para gestión de laboratorios partners por farmacia
Permite inicialización automática y selección personalizada
"""

import logging
from typing import Any, Dict, List, Optional, Tuple
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from pydantic import BaseModel
import sqlalchemy as sa
from sqlalchemy import and_, case, desc, func
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

from ..database import get_db
from ..models.pharmacy_partners import PharmacyPartner
from ..models.user import User
from .deps import get_current_user, get_current_with_partner_access  # Issue #541: Partner access control
from ..models.product_catalog import ProductCatalog
from ..models.sales_data import SalesData
from ..models.sales_enrichment import SalesEnrichment

logger = logging.getLogger(__name__)

# Constantes para auto-inicialización de partners
AUTO_SUGGESTED_PARTNERS_COUNT = 8  # Top N laboratorios por volumen de ventas

router = APIRouter(prefix="/pharmacy-partners", tags=["pharmacy-partners"])


class PartnerSelectionRequest(BaseModel):
    """Request para actualizar selección de partners"""

    laboratory_name: str
    is_selected: bool


class BulkPartnerSelectionRequest(BaseModel):
    """Request para actualizar múltiples partners"""

    selections: List[PartnerSelectionRequest]


@router.post("/initialize/{pharmacy_id}")
async def initialize_pharmacy_partners(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    force_refresh: bool = Query(False, description="Forzar re-inicialización"),
    user_initiated: bool = Query(False, description="Refresh iniciado manualmente por usuario (ignora user_configured)"),
    current_user: User = Depends(get_current_user),  # Issue #541: Auth required
    _: User = Depends(get_current_with_partner_access),  # Issue #541: Titular access required
) -> Dict[str, Any]:
    """
    Inicializa automáticamente los laboratorios partners para una farmacia
    Identifica laboratorios de genéricos y selecciona top N por ventas como sugeridos

    Args:
        pharmacy_id: ID de farmacia
        force_refresh: Forzar re-inicialización aunque ya existan partners

    Returns:
        Partners inicializados con sugerencias automáticas
    """

    try:
        # Verificar si ya tiene partners inicializados
        existing_partners = db.query(PharmacyPartner).filter(PharmacyPartner.pharmacy_id == pharmacy_id).count()

        if existing_partners > 0 and not force_refresh:
            logger.info(f"Pharmacy {pharmacy_id} already has {existing_partners} partners initialized")
            return await get_pharmacy_partners(pharmacy_id, db)

        # 🛡️ PROTECCIÓN: NO borrar si usuario configuró manualmente (SOLO si refresh automático)
        if force_refresh:
            user_configured_count = db.query(PharmacyPartner).filter(
                PharmacyPartner.pharmacy_id == pharmacy_id,
                PharmacyPartner.user_configured == True
            ).count()

            # Si refresh AUTOMÁTICO y hay user_configured → RECHAZAR
            if user_configured_count > 0 and not user_initiated:
                logger.warning(
                    f"[INIT] Pharmacy {pharmacy_id} tiene {user_configured_count} partners "
                    f"con user_configured=TRUE. RECHAZANDO refresh automático para RESPETAR "
                    f"configuración del usuario. Retornando partners existentes."
                )
                return await get_pharmacy_partners(pharmacy_id, db)

            # Si refresh MANUAL (user_initiated=True) → Permitir borrar y recalcular
            if user_initiated:
                logger.info(
                    f"[INIT] Refresh MANUAL (user_initiated=True) - "
                    f"Recalculando partners ignorando {user_configured_count} user_configured. "
                    f"Borrando {existing_partners} partners."
                )
            else:
                logger.info(f"[INIT] force_refresh=True y sin user_configured - borrando {existing_partners} partners")

            db.query(PharmacyPartner).filter(PharmacyPartner.pharmacy_id == pharmacy_id).delete()
            db.commit()

        # 1. Identificar laboratorios de genéricos basándose en su CATÁLOGO GENERAL
        # NO en las ventas de esta farmacia específica
        generic_labs_query = (
            db.query(
                ProductCatalog.nomen_laboratorio,
                func.count(ProductCatalog.id).label("total_products"),
                func.sum(case((ProductCatalog.nomen_tipo_farmaco == "GENERICO", 1), else_=0)).label("generic_products"),
                func.round(
                    func.avg(
                        case(
                            (ProductCatalog.nomen_tipo_farmaco == "GENERICO", 100.0),
                            else_=0.0,
                        )
                    ),
                    2,
                ).label("generic_percentage"),
            )
            .filter(
                ProductCatalog.nomen_laboratorio.isnot(None),
                ProductCatalog.nomen_laboratorio != "",
                ProductCatalog.nomen_estado == "ALTA",  # Solo productos en ALTA
            )
            .group_by(ProductCatalog.nomen_laboratorio)
            .having(func.count(ProductCatalog.id) >= 5)  # Mínimo 5 productos para ser representativo
            .all()
        )

        # Filtrar laboratorios con >50% genéricos
        generic_laboratories = []
        for lab in generic_labs_query:
            percentage = float(lab.generic_percentage or 0)
            if percentage >= 50.0:
                generic_laboratories.append(
                    {
                        "laboratory": lab.nomen_laboratorio,
                        "total_products": lab.total_products,
                        "generic_products": lab.generic_products,
                        "generic_percentage": round(percentage, 2),
                    }
                )

        if not generic_laboratories:
            logger.warning(f"No generic laboratories found for pharmacy {pharmacy_id}")
            return {
                "pharmacy_id": str(pharmacy_id),
                "initialized": False,
                "message": "No se encontraron laboratorios de genéricos para inicializar",
                "partners": [],
            }

        # 2. De los laboratorios genéricos, calcular ventas en esta farmacia (últimos 13 meses)
        # ✅ OPTIMIZATION: Usar materialized view pre-calculada (100x speedup)
        # View refresh: diario via background job (suficiente para partners suggestion)
        # Query time: 180s → < 2s (elimina 3-way JOIN sobre 650k+ sales rows)
        generic_lab_names = [lab["laboratory"] for lab in generic_laboratories]

        # Query optimizado usando materialized view
        # La view ya tiene agregaciones pre-calculadas por (pharmacy_id, laboratory)
        sales_by_lab = (
            db.execute(
                sa.text("""
                    SELECT
                        laboratory_name AS nomen_laboratorio,
                        total_sales,
                        total_quantity,
                        total_transactions
                    FROM pharmacy_lab_sales_summary
                    WHERE pharmacy_id = :pharmacy_id
                        AND laboratory_name = ANY(:generic_labs)
                    ORDER BY total_sales DESC
                    LIMIT :limit
                """),
                {
                    "pharmacy_id": pharmacy_id,
                    "generic_labs": generic_lab_names,
                    "limit": AUTO_SUGGESTED_PARTNERS_COUNT
                }
            ).fetchall()
        )

        # 🔧 FALLBACK: Si la materialized view está vacía para esta farmacia,
        # usar query directa (más lenta pero garantiza datos actuales)
        # Esto ocurre cuando la view no se ha refrescado después del upload
        if not sales_by_lab:
            logger.warning(
                f"[INIT_PARTNERS] Materialized view vacía para farmacia {pharmacy_id}. "
                f"Usando fallback con query directa (puede ser lento)."
            )
            sales_by_lab = (
                db.execute(
                    sa.text("""
                        SELECT
                            pc.nomen_laboratorio,
                            COALESCE(SUM(sd.total_amount), 0) AS total_sales,
                            COALESCE(SUM(sd.quantity), 0) AS total_quantity,
                            COUNT(sd.id) AS total_transactions
                        FROM sales_data sd
                        JOIN sales_enrichment se ON se.sales_data_id = sd.id
                        JOIN product_catalog pc ON pc.id = se.product_catalog_id
                        WHERE sd.pharmacy_id = :pharmacy_id
                            AND sd.sale_date >= NOW() - INTERVAL '13 months'
                            AND pc.nomen_estado = 'ALTA'
                            AND pc.nomen_laboratorio IS NOT NULL
                            AND pc.nomen_laboratorio = ANY(:generic_labs)
                        GROUP BY pc.nomen_laboratorio
                        HAVING SUM(sd.total_amount) > 0
                        ORDER BY total_sales DESC
                        LIMIT :limit
                    """),
                    {
                        "pharmacy_id": pharmacy_id,
                        "generic_labs": generic_lab_names,
                        "limit": AUTO_SUGGESTED_PARTNERS_COUNT
                    }
                ).fetchall()
            )
            if sales_by_lab:
                logger.info(
                    f"[INIT_PARTNERS] Fallback exitoso: encontrados {len(sales_by_lab)} labs "
                    f"con ventas para farmacia {pharmacy_id}"
                )

        # 3. Crear PharmacyPartner records
        partners_created = []

        # Top N como sugeridos y seleccionados
        for i, lab_sales in enumerate(sales_by_lab):
            partner = PharmacyPartner(
                pharmacy_id=pharmacy_id,
                laboratory_name=lab_sales.nomen_laboratorio,
                is_selected=True,
                is_auto_suggested=True,
                user_configured=False,  # Explicitly mark as auto-initialized (not user-configured)
                suggestion_rank=str(i + 1),
                suggestion_sales_amount=f"{float(lab_sales.total_sales or 0):,.2f}€",
                suggestion_reason=f"Top {i+1} por ventas ({float(lab_sales.total_sales or 0):,.0f}€ en 13 meses)",
            )
            db.add(partner)
            partners_created.append(partner.to_dict())

        # Resto de laboratorios genéricos como no seleccionados
        suggested_labs = [lab_sales.nomen_laboratorio for lab_sales in sales_by_lab]
        for lab_info in generic_laboratories:
            if lab_info["laboratory"] not in suggested_labs:
                partner = PharmacyPartner(
                    pharmacy_id=pharmacy_id,
                    laboratory_name=lab_info["laboratory"],
                    is_selected=False,
                    is_auto_suggested=False,
                    user_configured=False,  # Explicitly mark as auto-initialized (not user-configured)
                    suggestion_reason=f"Laboratorio de genéricos ({lab_info['generic_percentage']}% genéricos)",
                )
                db.add(partner)
                partners_created.append(partner.to_dict())

        db.commit()

        logger.info(f"Initialized {len(partners_created)} partners for pharmacy {pharmacy_id}")

        return {
            "pharmacy_id": str(pharmacy_id),
            "initialized": True,
            "auto_suggested_count": len(sales_by_lab),
            "total_generic_labs": len(generic_laboratories),
            "partners": partners_created,
            "initialization_summary": {
                "criteria": f"Top {AUTO_SUGGESTED_PARTNERS_COUNT} laboratorios de genéricos por ventas (13 meses)",
                "min_generic_percentage": 50.0,
                "data_source": "Materialized view (refreshed daily)",
            },
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error initializing partners for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al inicializar laboratorios partners")


@router.get("/{pharmacy_id}")
async def get_pharmacy_partners(pharmacy_id: UUID, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Obtiene laboratorios partners de una farmacia

    NO AUTO-INICIALIZA: Retorna partners existentes (vacío si no hay).
    La inicialización ocurre automáticamente en el pipeline de upload de ventas.

    Args:
        pharmacy_id: ID de farmacia

    Returns:
        Lista de partners con estado de selección (puede estar vacía)
    """

    try:
        # GET simple: Solo leer partners existentes (sin auto-inicialización)
        # Esto previene el timeout de 180s que ocurría con ensure_partners_initialized()
        partners = (
            db.query(PharmacyPartner)
            .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
            .order_by(
                PharmacyPartner.is_selected.desc(),  # Seleccionados primero
                PharmacyPartner.is_auto_suggested.desc(),  # Sugeridos después
                PharmacyPartner.laboratory_name  # Alfabético al final
            )
            .all()
        )

        if not partners:
            # FALLBACK: Si hay ventas pero no partners → Inicializar en background
            # Esto cubre farmacias existentes que subieron ventas antes de este fix
            has_sales = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id).limit(1).first() is not None

            if has_sales:
                logger.info(f"Pharmacy {pharmacy_id} has sales but no partners - triggering background init")
                # Disparar inicialización en background (NO bloquear request)
                import asyncio
                from app.services.pharmacy_partners_service import PharmacyPartnersService

                async def init_partners_background():
                    try:
                        partners_service = PharmacyPartnersService()
                        loop = asyncio.get_event_loop()
                        await loop.run_in_executor(
                            None,
                            partners_service.calculate_and_update_partners,
                            db,
                            pharmacy_id
                        )
                        logger.info(f"Background partners init completed for pharmacy {pharmacy_id}")
                    except Exception as e:
                        logger.error(f"Background partners init failed for pharmacy {pharmacy_id}: {e}")

                # Fire and forget (no await)
                asyncio.create_task(init_partners_background())
            else:
                logger.info(f"No partners found for pharmacy {pharmacy_id} - will be initialized on first upload")

            return {
                "pharmacy_id": str(pharmacy_id),
                "partners": [],
                "summary": {
                    "total_partners": 0,
                    "selected_count": 0,
                    "auto_suggested_count": 0,
                },
            }

        # Issue #540: Enriquecer TODOS los partners con ventas actuales
        # El dropdown debe mostrar ventas para todos, no solo los 8 auto-sugeridos
        lab_names = [p.laboratory_name for p in partners]

        # Query ventas actuales de todos los partners (últimos 13 meses)
        sales_by_lab = {}
        if lab_names:
            sales_query = db.execute(
                sa.text("""
                    SELECT laboratory_name, total_sales, total_quantity
                    FROM pharmacy_lab_sales_summary
                    WHERE pharmacy_id = :pharmacy_id
                      AND laboratory_name = ANY(:lab_names)
                """),
                {"pharmacy_id": pharmacy_id, "lab_names": lab_names}
            ).fetchall()
            sales_by_lab = {row.laboratory_name: row for row in sales_query}

        # Enriquecer cada partner con sus ventas actuales
        partners_data = []
        for partner in partners:
            p_dict = partner.to_dict()

            # Añadir ventas calculadas (para partners sin suggestion_sales_amount)
            lab_sales = sales_by_lab.get(partner.laboratory_name)
            if lab_sales:
                p_dict["total_sales"] = float(lab_sales.total_sales or 0)
                p_dict["total_quantity"] = int(lab_sales.total_quantity or 0)
                # Si no tiene suggestion_sales_amount, usar el calculado
                if not p_dict.get("suggestion_sales_amount"):
                    p_dict["suggestion_sales_amount"] = f"{float(lab_sales.total_sales or 0):,.2f}€"
            else:
                p_dict["total_sales"] = 0
                p_dict["total_quantity"] = 0

            partners_data.append(p_dict)

        selected_partners = [p for p in partners_data if p["is_selected"]]
        suggested_partners = [p for p in partners_data if p["is_auto_suggested"]]

        return {
            "pharmacy_id": str(pharmacy_id),
            "partners": partners_data,
            "summary": {
                "total_partners": len(partners_data),
                "selected_count": len(selected_partners),
                "auto_suggested_count": len(suggested_partners),
            },
            "selected_partners": [p["laboratory_name"] for p in selected_partners],
            "needs_initialization": False,
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error getting partners for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener laboratorios partners")


@router.put("/{pharmacy_id}/selection")
async def update_partner_selection(
    pharmacy_id: UUID,
    request: BulkPartnerSelectionRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),  # Issue #541: Auth required
    _: User = Depends(get_current_with_partner_access),  # Issue #541: Titular access required
) -> Dict[str, Any]:
    """
    Actualiza selección de múltiples partners

    Args:
        pharmacy_id: ID de farmacia
        request: Nuevas selecciones de partners

    Returns:
        Estado actualizado de partners
    """

    try:
        updated_count = 0
        now = utc_now()

        # 🔍 DEBUG: Log para detectar problemas de encoding UTF-8
        if any('TECNIMEDE' in sel.laboratory_name or 'ESPAÑA' in sel.laboratory_name for sel in request.selections):
            logger.debug(
                f"[ENCODING_DEBUG] Recibido en backend: "
                f"{[sel.laboratory_name for sel in request.selections if 'TECNIMEDE' in sel.laboratory_name or 'ESPAÑA' in sel.laboratory_name]}"
            )

        for selection in request.selections:
            partner = (
                db.query(PharmacyPartner)
                .filter(
                    and_(
                        PharmacyPartner.pharmacy_id == pharmacy_id,
                        PharmacyPartner.laboratory_name == selection.laboratory_name,
                    )
                )
                .with_for_update()  # Lock row to prevent concurrent modifications (Render multi-worker)
                .first()
            )

            if partner:
                partner.is_selected = selection.is_selected
                partner.user_modified_at = now  # Marcar como modificación manual
                partner.user_configured = True  # PERSISTE después de restarts (Issue #PARTNERS_PERSISTENCE)
                updated_count += 1

                # 🔍 DEBUG PERSISTENCIA: Log cada partner actualizado
                if selection.is_selected:
                    logger.info(
                        f"[PERSISTENCIA_DEBUG] Setting user_configured=True for {selection.laboratory_name}: "
                        f"is_selected={selection.is_selected}"
                    )

        db.commit()

        logger.info(f"Updated {updated_count} partner selections for pharmacy {pharmacy_id}")

        # Issue #443: Invalidar cache de treemap (partners cambiaron → afecta análisis)
        try:
            from app.services.partner_analysis_service import partner_analysis_service
            import asyncio
            invalidated = await partner_analysis_service.invalidate_treemap_cache(pharmacy_id)
            if invalidated > 0:
                logger.info(
                    f"[TREEMAP_CACHE] Invalidated {invalidated} cache entries after partner selection update"
                )
        except Exception as cache_error:
            logger.debug(f"[TREEMAP_CACHE] Cache invalidation skipped: {cache_error}")

        # 🔍 DEBUG: Verificar que user_configured se guardó correctamente
        verify_query = db.query(PharmacyPartner).filter(
            PharmacyPartner.pharmacy_id == pharmacy_id,
            PharmacyPartner.is_selected == True
        ).all()
        user_configured_count = sum(1 for p in verify_query if p.user_configured)
        logger.info(
            f"[PERSISTENCIA_DEBUG] After commit: {len(verify_query)} selected, "
            f"{user_configured_count} have user_configured=True"
        )

        # Devolver estado actualizado
        return await get_pharmacy_partners(pharmacy_id, db)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error updating partner selection for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al actualizar selección de partners")


@router.get("/{pharmacy_id}/selected")
async def get_selected_partners(pharmacy_id: UUID, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Obtiene solo los laboratorios partners seleccionados para una farmacia

    AUTO-INICIALIZACIÓN: Si la farmacia no tiene partners seleccionados,
    los inicializa automáticamente antes de retornar.

    Args:
        pharmacy_id: ID de farmacia

    Returns:
        Lista de partners seleccionados para uso en análisis
    """

    try:
        # AUTO-INICIALIZAR partners si es necesario
        await ensure_partners_initialized(pharmacy_id, db)

        selected_partners = (
            db.query(PharmacyPartner)
            .filter(
                and_(
                    PharmacyPartner.pharmacy_id == pharmacy_id,
                    PharmacyPartner.is_selected.is_(True),
                )
            )
            .order_by(
                PharmacyPartner.suggestion_rank.nulls_last(),
                PharmacyPartner.laboratory_name,
            )
            .all()
        )

        partner_names = [p.laboratory_name for p in selected_partners]

        return {
            "pharmacy_id": str(pharmacy_id),
            "selected_partner_names": partner_names,
            "count": len(partner_names),
            "partners_detail": [p.to_dict() for p in selected_partners],
        }

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error getting selected partners for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener partners seleccionados")


@router.post("/{pharmacy_id}/reset-to-defaults")
async def reset_partners_to_defaults(pharmacy_id: UUID, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Resetea la selección de partners a los valores por defecto.

    Selecciona automáticamente los partners auto-sugeridos (top 8 por ventas)
    y deselecciona el resto. Útil para usuarios que accidentalmente deseleccionaron
    todos sus partners.

    Args:
        pharmacy_id: ID de farmacia

    Returns:
        Estado actualizado de partners con auto-sugeridos seleccionados
    """
    try:
        now = utc_now()

        # Obtener partners auto-sugeridos
        auto_suggested = (
            db.query(PharmacyPartner)
            .filter(
                PharmacyPartner.pharmacy_id == pharmacy_id,
                PharmacyPartner.is_auto_suggested == True,
            )
            .all()
        )

        if not auto_suggested:
            raise HTTPException(
                status_code=404,
                detail="No hay partners auto-sugeridos para resetear. Prueba a re-inicializar partners."
            )

        # Reset all partners: auto-suggested → selected, rest → deselected
        updated_count = 0

        # Seleccionar auto-sugeridos
        for partner in auto_suggested:
            if not partner.is_selected:
                partner.is_selected = True
                partner.user_modified_at = now
                partner.user_configured = False  # Reset user_configured flag
                updated_count += 1

        # Deseleccionar no auto-sugeridos
        non_suggested = (
            db.query(PharmacyPartner)
            .filter(
                PharmacyPartner.pharmacy_id == pharmacy_id,
                PharmacyPartner.is_auto_suggested == False,
                PharmacyPartner.is_selected == True,
            )
            .all()
        )

        for partner in non_suggested:
            partner.is_selected = False
            partner.user_modified_at = now
            partner.user_configured = False
            updated_count += 1

        db.commit()

        logger.info(
            f"[RESET_DEFAULTS] Reset {updated_count} partners to defaults for pharmacy {pharmacy_id}. "
            f"Auto-suggested selected: {len(auto_suggested)}"
        )

        return await get_pharmacy_partners(pharmacy_id, db)

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error resetting partners for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al resetear partners")


async def ensure_partners_initialized(pharmacy_id: UUID, db: Session) -> Tuple[bool, Optional[List[PharmacyPartner]]]:
    """
    Helper function que asegura que una farmacia tenga partners inicializados.
    Si no tiene partners o ninguno está seleccionado, los inicializa automáticamente.

    Este helper previene el problema de partners vacíos que ocurría al:
    1. Crear una nueva farmacia
    2. Subir ventas sin haber inicializado partners
    3. Intentar usar /generics sin partners seleccionados

    OPTIMIZACIÓN: Retorna los partners para evitar N+1 query pattern.

    Args:
        pharmacy_id: ID de la farmacia
        db: Sesión de base de datos

    Returns:
        Tuple[bool, Optional[List[PharmacyPartner]]]:
            - bool: True si tiene partners (o fueron inicializados exitosamente)
            - List[PharmacyPartner]: Lista de partners ordenada (None si falló)
    """

    # Verificar si ya tiene partners (query única optimizada)
    result = (
        db.query(
            func.count(PharmacyPartner.id).label("total"),
            func.count(case((PharmacyPartner.is_selected == True, 1))).label("selected"),
            func.count(case((PharmacyPartner.user_modified_at.isnot(None), 1))).label("modified"),
            func.count(case((PharmacyPartner.user_configured == True, 1))).label("user_configured"),
        )
        .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
        .first()
    )

    existing_partners_count = result.total if result else 0
    selected_count = result.selected if result else 0
    modified_count = result.modified if result else 0
    user_configured_count = result.user_configured if result else 0

    # 🔍 DEBUG PERSISTENCIA: Log valores actuales de BD
    logger.info(
        f"[ENSURE_PARTNERS] Pharmacy {pharmacy_id} estado actual BD: "
        f"total={existing_partners_count}, selected={selected_count}, "
        f"user_configured={user_configured_count}, modified={modified_count}"
    )

    # CASO 1: Ya tiene partners seleccionados → retornarlos (NO re-inicializar)
    if existing_partners_count > 0 and selected_count > 0:
        logger.info(f"[AUTO_INIT] Farmacia {pharmacy_id} ya tiene {selected_count} partners seleccionados")
        partners = (
            db.query(PharmacyPartner)
            .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
            .order_by(
                PharmacyPartner.suggestion_rank.asc().nullslast(),
                PharmacyPartner.is_selected.desc(),
            )
            .all()
        )
        return True, partners

    # CASO 2: Tiene partners pero ninguno seleccionado
    if existing_partners_count > 0 and selected_count == 0:
        # Si user_configured=TRUE → Usuario configuró manualmente (PERSISTE después de restarts)
        # NUNCA re-inicializar cuando user_configured=TRUE
        if user_configured_count > 0:
            logger.info(
                f"[AUTO_INIT] Farmacia {pharmacy_id} tiene {existing_partners_count} partners "
                f"pero ninguno seleccionado. user_configured=TRUE ({user_configured_count} partners). "
                f"RESPETANDO decisión del usuario (NO re-inicializar). "
                f"PERSISTE después de restarts."
            )
            partners = (
                db.query(PharmacyPartner)
                .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
                .order_by(
                    PharmacyPartner.suggestion_rank.asc().nullslast(),
                    PharmacyPartner.is_selected.desc(),
                )
                .all()
            )
            return True, partners

        # FALLBACK: Verificar user_modified_at (legacy logic, pre-user_configured)
        # Mantener por compatibilidad con datos existentes sin migración
        if modified_count > 0:
            logger.info(
                f"[AUTO_INIT] Farmacia {pharmacy_id} tiene {existing_partners_count} partners "
                f"pero ninguno seleccionado. Usuario hizo {modified_count} modificaciones manuales "
                f"(user_modified_at NOT NULL - legacy). RESPETANDO decisión del usuario."
            )
            partners = (
                db.query(PharmacyPartner)
                .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
                .order_by(
                    PharmacyPartner.suggestion_rank.asc().nullslast(),
                    PharmacyPartner.is_selected.desc(),
                )
                .all()
            )
            return True, partners

        # Si NO hay modificaciones manuales NI user_configured → probablemente datos corruptos → re-inicializar
        logger.warning(
            f"[AUTO_INIT] Farmacia {pharmacy_id} tiene {existing_partners_count} partners "
            f"pero ninguno seleccionado, user_configured=FALSE y sin modificaciones del usuario. "
            f"Re-inicializando."
        )

    # No tiene partners o no tiene ninguno seleccionado - inicializar
    try:
        logger.info(f"[AUTO_INIT] Inicializando partners automáticamente para farmacia {pharmacy_id}")
        init_result = await initialize_pharmacy_partners(
            pharmacy_id=pharmacy_id,
            db=db,
            force_refresh=(existing_partners_count > 0),  # Forzar refresh si ya existen
        )

        # Verificar que la inicialización fue exitosa
        if init_result.get("initialized"):
            logger.info(
                f"[AUTO_INIT] Partners inicializados exitosamente: "
                f"{init_result.get('auto_suggested_count', 0)} auto-sugeridos"
            )

            # Retornar partners recién inicializados (evita query adicional)
            partners = (
                db.query(PharmacyPartner)
                .filter(PharmacyPartner.pharmacy_id == pharmacy_id)
                .order_by(
                    PharmacyPartner.suggestion_rank.asc().nullslast(),
                    PharmacyPartner.is_selected.desc(),
                )
                .all()
            )
            return True, partners
        else:
            logger.error(f"[AUTO_INIT] Inicialización falló para farmacia {pharmacy_id}")
            return False, None

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"[AUTO_INIT] Error auto-inicializando partners para farmacia {pharmacy_id}: {str(e)}")
        return False, None
