﻿# backend/app/api/pharmacy.py
"""
API endpoints para gestión de datos de farmacia
"""

import logging
from datetime import date, timedelta
from typing import Any, Dict, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException
from pydantic import BaseModel, field_validator
from sqlalchemy import func
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now
from app.core.subscription_limits import SubscriptionPlan

from ..api.error_handler import handle_exception
from ..database import get_db
from ..exceptions import PharmacyNotFoundError
from ..models.file_upload import FileUpload
from ..models.pharmacy import Pharmacy
from ..models.product_catalog import ProductCatalog
from ..models.sales_data import SalesData
from ..models.sales_enrichment import SalesEnrichment
from ..models.system_status import SystemComponent, SystemStatus, SystemStatusEnum
from ..models.homogeneous_group import HomogeneousGroup
from ..models.user import User
from .deps import get_current_user

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/pharmacy", tags=["pharmacy"])


class PharmacyUpdateRequest(BaseModel):
    """Request para actualizar datos de farmacia"""

    name: Optional[str] = None
    code: Optional[str] = None
    address: Optional[str] = None
    city: Optional[str] = None
    province: Optional[str] = None
    postal_code: Optional[str] = None
    phone: Optional[str] = None
    mobile: Optional[str] = None
    email: Optional[str] = None
    website: Optional[str] = None
    erp_type: Optional[str] = None
    erp_version: Optional[str] = None
    subscription_plan: Optional[str] = None  # Sync with User.subscription_plan (Issue #348)

    @field_validator('subscription_plan')
    @classmethod
    def validate_subscription_plan(cls, v):
        """Valida que subscription_plan sea un valor permitido"""
        if v is not None:
            allowed_plans = {plan.value for plan in SubscriptionPlan}
            if v not in allowed_plans:
                raise ValueError(
                    f"subscription_plan debe ser uno de {allowed_plans}, recibido: '{v}'"
                )
        return v


@router.get("/{pharmacy_id}")
async def get_pharmacy(pharmacy_id: UUID, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Obtiene datos de una farmacia

    Args:
        pharmacy_id: ID de farmacia

    Returns:
        Datos completos de la farmacia
    """

    try:
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()

        if not pharmacy:
            raise PharmacyNotFoundError(pharmacy_id=str(pharmacy_id))

        # Convertir a dict con todos los campos necesarios para el formulario
        data = {
            "id": str(pharmacy.id),
            "name": pharmacy.name or "",
            "code": pharmacy.code or "",
            "address": pharmacy.address or "",
            "city": pharmacy.city or "",
            "province": pharmacy.province or "",
            "postal_code": pharmacy.postal_code or "",
            "phone": pharmacy.phone or "",
            "mobile": pharmacy.mobile or "",
            "email": pharmacy.email or "",
            "website": pharmacy.website or "",
            "erp_type": pharmacy.erp_type or "",
            "erp_version": pharmacy.erp_version or "",
            "subscription_plan": pharmacy.subscription_plan,
            "is_active": pharmacy.is_active,
            "created_at": (pharmacy.created_at.isoformat() if pharmacy.created_at else None),
            "updated_at": (pharmacy.updated_at.isoformat() if pharmacy.updated_at else None),
        }

        return {"success": True, "data": data}

    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "pharmacy_operation", "pharmacy_id": str(pharmacy_id)})
    except Exception as e:
        logger.error(f"Error getting pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener datos de farmacia")


@router.put("/{pharmacy_id}")
async def update_pharmacy(
    pharmacy_id: UUID, request: PharmacyUpdateRequest, db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Actualiza datos de una farmacia

    Args:
        pharmacy_id: ID de farmacia
        request: Datos a actualizar

    Returns:
        Datos actualizados de la farmacia
    """

    try:
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()

        if not pharmacy:
            raise PharmacyNotFoundError(pharmacy_id=str(pharmacy_id))

        # Actualizar solo los campos proporcionados
        # Validar que campos NOT NULL no se actualicen a null
        not_null_fields = ['name']  # Campos que no pueden ser null en la BD
        request_data = request.dict(exclude_unset=True)

        for field in not_null_fields:
            if field in request_data and request_data[field] is None:
                raise HTTPException(
                    status_code=400,
                    detail=f"El campo '{field}' no puede ser null"
                )

        updated_fields = []
        for field, value in request_data.items():
            if hasattr(pharmacy, field):
                setattr(pharmacy, field, value)
                updated_fields.append(field)

        # Actualizar timestamp

        pharmacy.updated_at = utc_now()

        db.commit()
        db.refresh(pharmacy)

        logger.info(f"Updated pharmacy {pharmacy_id}: fields {updated_fields}")

        return {
            "success": True,
            "message": "Datos de farmacia actualizados correctamente",
            "updated_fields": updated_fields,
            "data": {
                "id": str(pharmacy.id),
                "name": pharmacy.name,
                "code": pharmacy.code,
                "address": pharmacy.address,
                "phone": pharmacy.phone,
                "erp_type": pharmacy.erp_type,
                "erp_version": pharmacy.erp_version,
                "updated_at": pharmacy.updated_at.isoformat(),
            },
        }

    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "pharmacy_operation", "pharmacy_id": str(pharmacy_id)})
    except Exception as e:
        logger.error(f"Error updating pharmacy {pharmacy_id}: {str(e)}")
        db.rollback()
        raise HTTPException(status_code=500, detail="Error al actualizar datos de farmacia")


@router.get("/by-code/{pharmacy_code}")
async def get_pharmacy_by_code(pharmacy_code: str, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Obtiene datos de una farmacia por código

    Args:
        pharmacy_code: Código de farmacia

    Returns:
        Datos completos de la farmacia
    """

    try:
        pharmacy = db.query(Pharmacy).filter(Pharmacy.code == pharmacy_code).first()

        if not pharmacy:
            raise PharmacyNotFoundError(cif=str(pharmacy_code))

        return await get_pharmacy(pharmacy.id, db)

    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "pharmacy_operation", "pharmacy_code": str(pharmacy_code)})
    except Exception as e:
        logger.error(f"Error getting pharmacy by code {pharmacy_code}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener datos de farmacia")


@router.get("/{pharmacy_id}/dashboard-summary")
async def get_dashboard_summary(pharmacy_id: UUID, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Obtiene un resumen optimizado de datos para el dashboard de la farmacia.
    Endpoint ligero específico para la homepage.

    Args:
        pharmacy_id: ID de la farmacia

    Returns:
        Resumen de métricas clave para el dashboard
    """
    try:
        # Verificar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise PharmacyNotFoundError(pharmacy_id=str(pharmacy_id))

        # === MÉTRICAS DE VENTAS ===
        # Total de registros de ventas
        sales_count = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy_id).count()

        # Registros enriquecidos (optimizado con JOIN)
        enriched_count = (
            db.query(func.count(func.distinct(SalesEnrichment.sales_data_id)))
            .join(SalesData)
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .scalar()
            or 0
        )

        # Rango de fechas de ventas
        sales_date_range = (
            db.query(func.min(SalesData.sale_date), func.max(SalesData.sale_date))
            .filter(SalesData.pharmacy_id == pharmacy_id)
            .first()
            if sales_count > 0
            else (None, None)
        )

        # Último archivo subido
        last_upload = (
            db.query(FileUpload)
            .filter(FileUpload.pharmacy_id == pharmacy_id)
            .order_by(FileUpload.uploaded_at.desc())
            .first()
        )

        # === MÉTRICAS RÁPIDAS DEL CATÁLOGO ===
        # Solo contar productos totales (el resto viene del system/status)
        catalog_count = db.query(ProductCatalog).count()

        # Calcular porcentaje de enriquecimiento
        enrichment_percentage = 0
        if sales_count > 0:
            enrichment_percentage = round((enriched_count / sales_count * 100), 1)

        return {
            "success": True,
            "pharmacy": {
                "id": str(pharmacy.id),
                "name": pharmacy.name,
                "code": pharmacy.code,
            },
            "sales": {
                "total_records": sales_count,
                "enriched_records": enriched_count,
                "enrichment_percentage": enrichment_percentage,
                "date_range": {
                    "start": (sales_date_range[0].isoformat() if sales_date_range[0] else None),
                    "end": (sales_date_range[1].isoformat() if sales_date_range[1] else None),
                },
                "last_upload": (
                    {
                        "filename": last_upload.filename if last_upload else None,
                        "date": (last_upload.uploaded_at.isoformat() if last_upload else None),
                        "rows": last_upload.rows_processed if last_upload else 0,
                    }
                    if last_upload
                    else None
                ),
            },
            "quick_stats": {
                "products_in_catalog": catalog_count,
                "has_sales_data": sales_count > 0,
                "has_enrichment": enriched_count > 0,
            },
            "timestamp": utc_now().isoformat(),
        }

    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "pharmacy_operation", "pharmacy_id": str(pharmacy_id)})
    except Exception as e:
        logger.error(f"Error getting dashboard summary for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener resumen del dashboard")


@router.get("/{pharmacy_id}/initial-data")
async def get_initial_dashboard_data(
    pharmacy_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
) -> Dict[str, Any]:
    """
    Endpoint unificado para carga inicial del dashboard.
    Combina system_status + pharmacy_summary + sales_trend en una sola llamada.

    Optimización: Reduce 5-6 API calls a 1 sola llamada HTTP.
    Issue: Performance - Homepage initial load optimization.

    Args:
        pharmacy_id: ID de la farmacia del usuario
        current_user: Usuario autenticado
        db: Sesión de base de datos

    Returns:
        Datos combinados para renderizar el dashboard completo
    """
    import time
    from sqlalchemy import text

    start_time = time.time()

    try:
        # Verificar que la farmacia existe y pertenece al usuario
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise PharmacyNotFoundError(pharmacy_id=str(pharmacy_id))

        # =========================================================
        # PARTE 1: DATOS DE LA FARMACIA (antes: dashboard-summary)
        # =========================================================

        # Query consolidada para métricas de farmacia
        pharmacy_stats_query = text("""
            WITH sales_stats AS (
                SELECT
                    COUNT(*) as total_records,
                    COUNT(DISTINCT codigo_nacional) as unique_products,
                    SUM(COALESCE(total_amount, quantity * unit_price, 0)) as total_sales_amount,
                    MIN(sale_date) as min_date,
                    MAX(sale_date) as max_date
                FROM sales_data
                WHERE pharmacy_id = :pharmacy_id
            ),
            enrichment_stats AS (
                SELECT COUNT(DISTINCT se.sales_data_id) as enriched_count
                FROM sales_enrichment se
                JOIN sales_data sd ON se.sales_data_id = sd.id
                WHERE sd.pharmacy_id = :pharmacy_id
            ),
            upload_stats AS (
                SELECT filename, uploaded_at, rows_processed
                FROM file_uploads
                WHERE pharmacy_id = :pharmacy_id
                ORDER BY uploaded_at DESC
                LIMIT 1
            )
            SELECT
                s.total_records, s.unique_products, s.total_sales_amount,
                s.min_date, s.max_date,
                e.enriched_count,
                u.filename as last_upload_filename,
                u.uploaded_at as last_upload_date,
                u.rows_processed as last_upload_rows
            FROM sales_stats s
            CROSS JOIN enrichment_stats e
            LEFT JOIN upload_stats u ON true
        """)

        pharmacy_result = db.execute(pharmacy_stats_query, {"pharmacy_id": str(pharmacy_id)}).first()

        sales_count = pharmacy_result.total_records or 0
        unique_products = pharmacy_result.unique_products or 0
        total_sales_amount = float(pharmacy_result.total_sales_amount or 0)
        enriched_count = pharmacy_result.enriched_count or 0
        enrichment_rate = round((enriched_count / sales_count * 100), 1) if sales_count > 0 else 0

        # =========================================================
        # PARTE 2: ESTADO DEL SISTEMA (antes: system/status)
        # =========================================================

        # Query consolidada para catálogo, sistema, CIMA y nomenclátor
        system_stats_query = text("""
            WITH catalog_stats AS (
                SELECT
                    COUNT(*) as total_products,
                    COUNT(*) FILTER (WHERE data_sources ILIKE '%cima%') as with_cima,
                    COUNT(*) FILTER (WHERE data_sources ILIKE '%nomenclator%') as with_nomen,
                    MAX(updated_at) FILTER (WHERE data_sources ILIKE '%cima%') as cima_catalog_updated
                FROM product_catalog
            ),
            groups_stats AS (
                SELECT COUNT(*) as total_groups
                FROM homogeneous_groups_master
            ),
            last_cima_sync AS (
                SELECT sync_date, records_updated
                FROM catalog_sync_history
                WHERE sync_type = 'cima'
                  AND status IN ('success', 'completed', 'COMPLETED')
                ORDER BY sync_date DESC
                LIMIT 1
            ),
            last_nomenclator_sync AS (
                SELECT sync_date, records_updated
                FROM catalog_sync_history
                WHERE sync_type IN ('NOMENCLATOR', 'nomenclator')
                  AND status IN ('success', 'completed', 'COMPLETED')
                ORDER BY sync_date DESC
                LIMIT 1
            ),
            new_groups AS (
                -- Grupos "nuevos" = creados después de la importación inicial
                -- Detectamos la fecha de importación masiva (>100 grupos mismo día)
                SELECT COUNT(*) as new_groups_count
                FROM homogeneous_groups_master hg
                WHERE hg.created_at > (
                    SELECT MIN(created_at) + INTERVAL '1 day'
                    FROM homogeneous_groups_master
                )
            )
            SELECT
                c.total_products, c.with_cima, c.with_nomen,
                c.cima_catalog_updated,
                g.total_groups,
                COALESCE(lcs.sync_date, c.cima_catalog_updated) as cima_last_sync,
                lcs.records_updated as cima_records,
                lns.sync_date as nomenclator_last_sync,
                lns.records_updated as nomenclator_records,
                ng.new_groups_count
            FROM catalog_stats c
            CROSS JOIN groups_stats g
            LEFT JOIN last_cima_sync lcs ON true
            LEFT JOIN last_nomenclator_sync lns ON true
            LEFT JOIN new_groups ng ON true
        """)

        system_result = db.execute(system_stats_query).first()

        catalog_count = system_result.total_products or 0
        catalog_with_cima = system_result.with_cima or 0
        catalog_with_nomen = system_result.with_nomen or 0
        groups_count = system_result.total_groups or 0
        cima_last_sync = system_result.cima_last_sync
        cima_records = system_result.cima_records or 0
        nomenclator_last_sync = system_result.nomenclator_last_sync
        nomenclator_records = system_result.nomenclator_records or 0
        new_groups_count = system_result.new_groups_count or 0

        # Obtener estados de componentes del sistema
        statuses = db.query(SystemStatus).filter(
            SystemStatus.component.in_([
                SystemComponent.CATALOG,
                SystemComponent.CIMA,
                SystemComponent.NOMENCLATOR,
                SystemComponent.HOMOGENEOUS_GROUPS,
            ])
        ).all()

        status_dict = {s.component.value: s.to_dict() for s in statuses}

        # Agregar conteo de productos CIMA al componente (esperado por frontend KPIs)
        if "cima" in status_dict:
            status_dict["cima"]["products"] = catalog_with_cima
        else:
            # Si no hay registro CIMA, crear estructura mínima
            status_dict["cima"] = {"products": catalog_with_cima, "status": "unknown"}

        # Determinar estado general del sistema
        is_syncing = any(s.status == SystemStatusEnum.INITIALIZING for s in statuses)
        has_errors = any(s.status == SystemStatusEnum.ERROR for s in statuses)

        if catalog_count == 0:
            overall_status = "requires_initialization"
            overall_message = "El catálogo necesita inicialización"
        elif is_syncing:
            overall_status = "initializing"
            overall_message = "Sincronización en progreso..."
        elif has_errors:
            overall_status = "error"
            overall_message = "Error en componentes del sistema"
        else:
            overall_status = "ready"
            overall_message = "Sistema operativo"

        # =========================================================
        # PARTE 3: KPIs YEAR-OVER-YEAR CONSOLIDADOS (Issue #XXX)
        # =========================================================
        # Optimización: 12 queries → 1 query con CASE WHEN
        # Calcula QTD, MTD, WTD vs mismo período año anterior

        today = utc_now().date()

        # Calcular rangos de fechas para YoY
        # QTD: Desde inicio del trimestre hasta hoy
        trimestre = ((today.month - 1) // 3) + 1
        if trimestre == 1:
            qtd_start = date(today.year, 1, 1)
        elif trimestre == 2:
            qtd_start = date(today.year, 4, 1)
        elif trimestre == 3:
            qtd_start = date(today.year, 7, 1)
        else:
            qtd_start = date(today.year, 10, 1)

        # MTD: Desde inicio del mes hasta hoy
        mtd_start = date(today.year, today.month, 1)

        # WTD: Desde lunes de esta semana hasta hoy
        wtd_start = today - timedelta(days=today.weekday())

        # Mismos períodos del año anterior
        def safe_date_prev_year(d: date) -> date:
            """Maneja años bisiestos (29 feb → 28 feb)"""
            try:
                return date(d.year - 1, d.month, d.day)
            except ValueError:
                return date(d.year - 1, d.month, 28)

        qtd_start_prev = safe_date_prev_year(qtd_start)
        qtd_end_prev = safe_date_prev_year(today)
        mtd_start_prev = date(today.year - 1, today.month, 1)
        mtd_end_prev = safe_date_prev_year(today)

        # WTD año anterior: misma semana ISO
        iso_year, iso_week, _ = today.isocalendar()
        first_week_prev = date(iso_year - 1, 1, 4)
        first_monday_prev = first_week_prev - timedelta(days=first_week_prev.weekday())
        wtd_start_prev = first_monday_prev + timedelta(weeks=iso_week - 1)
        wtd_end_prev = wtd_start_prev + timedelta(days=today.weekday())

        # Query consolidada: 1 query en lugar de 12
        yoy_query = text("""
            SELECT
                -- QTD actual
                SUM(CASE WHEN sale_date >= :qtd_start AND sale_date <= :today
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as qtd_ventas,
                SUM(CASE WHEN sale_date >= :qtd_start AND sale_date <= :today
                    THEN quantity ELSE 0 END) as qtd_unidades,
                -- QTD año anterior
                SUM(CASE WHEN sale_date >= :qtd_start_prev AND sale_date <= :qtd_end_prev
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as qtd_ventas_prev,
                SUM(CASE WHEN sale_date >= :qtd_start_prev AND sale_date <= :qtd_end_prev
                    THEN quantity ELSE 0 END) as qtd_unidades_prev,
                -- MTD actual
                SUM(CASE WHEN sale_date >= :mtd_start AND sale_date <= :today
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as mtd_ventas,
                SUM(CASE WHEN sale_date >= :mtd_start AND sale_date <= :today
                    THEN quantity ELSE 0 END) as mtd_unidades,
                -- MTD año anterior
                SUM(CASE WHEN sale_date >= :mtd_start_prev AND sale_date <= :mtd_end_prev
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as mtd_ventas_prev,
                SUM(CASE WHEN sale_date >= :mtd_start_prev AND sale_date <= :mtd_end_prev
                    THEN quantity ELSE 0 END) as mtd_unidades_prev,
                -- WTD actual
                SUM(CASE WHEN sale_date >= :wtd_start AND sale_date <= :today
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as wtd_ventas,
                SUM(CASE WHEN sale_date >= :wtd_start AND sale_date <= :today
                    THEN quantity ELSE 0 END) as wtd_unidades,
                -- WTD año anterior
                SUM(CASE WHEN sale_date >= :wtd_start_prev AND sale_date <= :wtd_end_prev
                    THEN COALESCE(total_amount, quantity * unit_price, 0) ELSE 0 END) as wtd_ventas_prev,
                SUM(CASE WHEN sale_date >= :wtd_start_prev AND sale_date <= :wtd_end_prev
                    THEN quantity ELSE 0 END) as wtd_unidades_prev
            FROM sales_data
            WHERE pharmacy_id = :pharmacy_id
              AND sale_date >= :min_date
        """)

        # Fecha mínima para la query (optimización: solo últimos 2 años)
        min_yoy_date = date(today.year - 1, 1, 1)

        yoy_result = db.execute(yoy_query, {
            "pharmacy_id": str(pharmacy_id),
            "today": today,
            "qtd_start": qtd_start,
            "qtd_start_prev": qtd_start_prev,
            "qtd_end_prev": qtd_end_prev,
            "mtd_start": mtd_start,
            "mtd_start_prev": mtd_start_prev,
            "mtd_end_prev": mtd_end_prev,
            "wtd_start": wtd_start,
            "wtd_start_prev": wtd_start_prev,
            "wtd_end_prev": wtd_end_prev,
            "min_date": min_yoy_date,
        }).first()

        # Calcular variaciones YoY
        def calc_yoy_pct(current: float, previous: float) -> float:
            if previous == 0:
                return 0.0 if current == 0 else 100.0
            return round(((current - previous) / previous) * 100, 2)

        # Nombres de meses en español
        meses_es = ["Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio",
                    "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"]

        yoy_kpis = {
            "qtd_yoy": {
                "tipo": "QTD",
                "periodo_actual": {
                    "trimestre": f"Q{trimestre} {today.year}",
                    "inicio": qtd_start.isoformat(),
                    "fin": today.isoformat(),
                    "ventas": float(yoy_result.qtd_ventas or 0),
                    "unidades": int(yoy_result.qtd_unidades or 0),
                },
                "periodo_anterior": {
                    "trimestre": f"Q{trimestre} {today.year - 1}",
                    "inicio": qtd_start_prev.isoformat(),
                    "fin": qtd_end_prev.isoformat(),
                    "ventas": float(yoy_result.qtd_ventas_prev or 0),
                    "unidades": int(yoy_result.qtd_unidades_prev or 0),
                },
                "variacion": {
                    "ventas_porcentual": calc_yoy_pct(
                        float(yoy_result.qtd_ventas or 0),
                        float(yoy_result.qtd_ventas_prev or 0)
                    ),
                },
            },
            "mtd_yoy": {
                "tipo": "MTD",
                "periodo_actual": {
                    "mes": f"{meses_es[today.month - 1]} {today.year}",
                    "inicio": mtd_start.isoformat(),
                    "fin": today.isoformat(),
                    "ventas": float(yoy_result.mtd_ventas or 0),
                    "unidades": int(yoy_result.mtd_unidades or 0),
                },
                "periodo_anterior": {
                    "mes": f"{meses_es[today.month - 1]} {today.year - 1}",
                    "inicio": mtd_start_prev.isoformat(),
                    "fin": mtd_end_prev.isoformat(),
                    "ventas": float(yoy_result.mtd_ventas_prev or 0),
                    "unidades": int(yoy_result.mtd_unidades_prev or 0),
                },
                "variacion": {
                    "ventas_porcentual": calc_yoy_pct(
                        float(yoy_result.mtd_ventas or 0),
                        float(yoy_result.mtd_ventas_prev or 0)
                    ),
                },
            },
            "wtd_yoy": {
                "tipo": "WTD",
                "periodo_actual": {
                    "semana": f"Semana {iso_week} {today.year}",
                    "inicio": wtd_start.isoformat(),
                    "fin": today.isoformat(),
                    "ventas": float(yoy_result.wtd_ventas or 0),
                    "unidades": int(yoy_result.wtd_unidades or 0),
                },
                "periodo_anterior": {
                    "semana": f"Semana {iso_week} {today.year - 1}",
                    "inicio": wtd_start_prev.isoformat(),
                    "fin": wtd_end_prev.isoformat(),
                    "ventas": float(yoy_result.wtd_ventas_prev or 0),
                    "unidades": int(yoy_result.wtd_unidades_prev or 0),
                },
                "variacion": {
                    "ventas_porcentual": calc_yoy_pct(
                        float(yoy_result.wtd_ventas or 0),
                        float(yoy_result.wtd_ventas_prev or 0)
                    ),
                },
            },
        }

        # =========================================================
        # PARTE 4: TENDENCIA DE VENTAS (antes: sales/summary)
        # =========================================================

        # Ventas agregadas por día (últimos 30 días)
        end_date = utc_now()
        start_date = end_date - timedelta(days=30)

        sales_trend_query = text("""
            SELECT
                DATE(sale_date) as date,
                SUM(COALESCE(total_amount, quantity * unit_price, 0)) as total_sales,
                COUNT(*) as transaction_count
            FROM sales_data
            WHERE pharmacy_id = :pharmacy_id
              AND sale_date >= :start_date
              AND sale_date <= :end_date
            GROUP BY DATE(sale_date)
            ORDER BY date ASC
        """)

        sales_trend_result = db.execute(
            sales_trend_query,
            {
                "pharmacy_id": str(pharmacy_id),
                "start_date": start_date.date(),
                "end_date": end_date.date()
            }
        ).fetchall()

        sales_by_day = [
            {
                "date": row.date.isoformat() if row.date else None,
                "total_sales": float(row.total_sales) if row.total_sales else 0,
                "transaction_count": row.transaction_count or 0
            }
            for row in sales_trend_result
        ]

        # Calcular totales del período
        total_period_sales = sum(day["total_sales"] for day in sales_by_day)
        total_period_transactions = sum(day["transaction_count"] for day in sales_by_day)

        # =========================================================
        # RESPUESTA UNIFICADA
        # =========================================================

        execution_time_ms = (time.time() - start_time) * 1000

        logger.info(
            f"[INITIAL_DATA] Loaded in {execution_time_ms:.2f}ms for pharmacy {pharmacy_id}",
            extra={
                "pharmacy_id": str(pharmacy_id),
                "execution_time_ms": round(execution_time_ms, 2),
                "sales_count": sales_count,
                "catalog_count": catalog_count
            }
        )

        return {
            "success": True,
            # Datos de la farmacia
            "pharmacy": {
                "id": str(pharmacy.id),
                "name": pharmacy.name,
                "code": pharmacy.code,
            },
            # Métricas de ventas (campos esperados por KPIs del frontend)
            "sales": {
                "total_records": sales_count,
                "total_sales": total_sales_amount,  # KPI: Total ventas (€)
                "unique_products": unique_products,  # KPI: Productos únicos
                "enriched_records": enriched_count,
                "enrichment_rate": enrichment_rate,  # KPI: % Enriquecimiento
                "date_range": {
                    "start": pharmacy_result.min_date.isoformat() if pharmacy_result.min_date else None,
                    "end": pharmacy_result.max_date.isoformat() if pharmacy_result.max_date else None,
                },
                "last_upload": {
                    "filename": pharmacy_result.last_upload_filename,
                    "date": pharmacy_result.last_upload_date.isoformat() if pharmacy_result.last_upload_date else None,
                    "rows": pharmacy_result.last_upload_rows or 0,
                } if pharmacy_result.last_upload_filename else None,
            },
            # Estado del catálogo/sistema
            "catalog": {
                "total_products": catalog_count,
                "with_cima": catalog_with_cima,
                "with_nomenclator": catalog_with_nomen,
                "homogeneous_groups": groups_count,
                "coverage_percentage": (
                    round((catalog_with_cima + catalog_with_nomen) / (catalog_count * 2) * 100, 1)
                    if catalog_count > 0 else 0
                ),
            },
            # CIMA (última sincronización)
            "cima": {
                "last_sync": cima_last_sync.isoformat() if cima_last_sync else None,
                "records_updated": cima_records,
                "total_products": catalog_with_cima,
            },
            # Nomenclátor y grupos homogéneos
            "nomenclator": {
                "last_sync": nomenclator_last_sync.isoformat() if nomenclator_last_sync else None,
                "records_updated": nomenclator_records,
                "homogeneous_groups": groups_count,
                "new_groups": new_groups_count,  # Grupos creados desde última sync
            },
            # Estado del sistema
            "system": {
                "status": overall_status,
                "message": overall_message,
                "is_syncing": is_syncing,
                "components": status_dict,
            },
            # Tendencia de ventas (para el gráfico)
            "sales_trend": {
                "period_days": 30,
                "start_date": start_date.date().isoformat(),
                "end_date": end_date.date().isoformat(),
                "by_day": sales_by_day,
                "totals": {
                    "sales": total_period_sales,
                    "transactions": total_period_transactions,
                }
            },
            # KPIs Year-over-Year (optimización: incluidos en initial-data)
            "yoy_kpis": yoy_kpis,
            # Metadatos
            "timestamp": utc_now().isoformat(),
            "execution_time_ms": round(execution_time_ms, 2),
        }

    except PharmacyNotFoundError as e:
        raise handle_exception(e, {"endpoint": "initial_dashboard_data", "pharmacy_id": str(pharmacy_id)})
    except Exception as e:
        logger.error(f"Error getting initial dashboard data for pharmacy {pharmacy_id}: {str(e)}")
        raise HTTPException(status_code=500, detail="Error al obtener datos iniciales del dashboard")
