# backend/app/api/partner_analysis.py
"""
API endpoints para Partner Analysis - Panel Partners Redesign

Implementa los endpoints especializados para el nuevo diseño del panel de partners:
- Contexto fijo del universo sustituible
- Análisis dinámico con partners seleccionados
- Drill-down temporal (trimestre → mes → quincena)
- Detalle de conjuntos homogéneos específicos

Arquitectura basada en filtros composables tipo Power BI con separación clara
entre métricas fijas y análisis dinámico según partners seleccionados.
"""

import logging
from typing import List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import func
from sqlalchemy.orm import Session

from ..api.deps import get_current_user, verify_pharmacy_access
from ..core.constants import NOMENCLATOR_MIN_THRESHOLD
from ..database import get_db
from ..models.product_catalog import ProductCatalog
from ..models.sales_data import SalesData
from ..models.user import User
from ..schemas.partner_analysis import (
    HomogeneousGroupDetailResponse,
    PartnerDynamicAnalysisRequest,
    PartnerDynamicAnalysisResponse,
    PartnerProductReferencesResponse,
    SubstitutableUniverseResponse,
    TemporalBreakdownResponse,
)
from ..services.partner_analysis_service import partner_analysis_service

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/analysis", tags=["partner-analysis"])


@router.get("/substitutable-universe/{pharmacy_id}")
async def get_substitutable_universe_context(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[str] = Query(
        None,
        description="Fecha inicio análisis (YYYY-MM-DD). Tiene prioridad sobre period_months",
    ),
    end_date: Optional[str] = Query(
        None,
        description="Fecha fin análisis (YYYY-MM-DD). Tiene prioridad sobre period_months",
    ),
    period_months: int = Query(
        12,
        ge=1,
        le=24,
        description="DEPRECADO: Usar start_date/end_date. Fallback: período en meses desde hoy",
    ),
    limit: int = Query(
        100,
        ge=1,
        le=1000,
        description="Límite de grupos homogéneos a retornar (max 1000 para Render)",
    ),
    offset: int = Query(
        0,
        ge=0,
        description="Offset para paginación de grupos homogéneos",
    ),
) -> SubstitutableUniverseResponse:
    """
    Obtiene el contexto fijo del universo sustituible para la farmacia.

    **UNIVERSO SUSTITUIBLE DEFINIDO COMO:**
    - Solo conjuntos homogéneos que tienen AL MENOS un laboratorio genérico disponible
    - Métricas que NO cambian con la selección de partners
    - Base para todos los análisis posteriores de oportunidades

    **Casos de Uso:**
    - Carga inicial del panel de partners
    - Contexto para análisis comparativos
    - Métricas de referencia del potencial total

    **Performance:**
    - Cacheable: Resultado no cambia frecuentemente
    - Optimizado para datasets grandes (>100k registros)
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia (Issue #254, PR #289)
        verify_pharmacy_access(pharmacy_id, current_user)

        logger.info(f"[PARTNER_ANALYSIS_API] Obteniendo universo sustituible para farmacia {pharmacy_id}")

        # ✅ VALIDACIÓN: Detectar si nomenclator no está sincronizado
        # El servicio calcula grupos en tiempo real desde product_catalog.nomen_codigo_homogeneo
        # Si ese campo está vacío, devuelve 0 grupos aunque haya ventas
        sales_count = db.query(func.count(SalesData.id)).filter(
            SalesData.pharmacy_id == pharmacy_id
        ).scalar() or 0

        # Optimized: Use LIMIT to avoid full table scan on 67k+ products (Render 512MB constraint)
        nomenclator_ready = (
            db.query(ProductCatalog.id)
            .filter(ProductCatalog.nomen_codigo_homogeneo.isnot(None))
            .limit(NOMENCLATOR_MIN_THRESHOLD)
            .count()
        )

        if sales_count > 0 and nomenclator_ready < NOMENCLATOR_MIN_THRESHOLD:
            # Nomenclator no sincronizado - mensaje claro en lugar de "sin datos"
            logger.warning(
                f"[PARTNER_ANALYSIS_API] Farmacia {pharmacy_id}: nomenclator sync required. "
                f"Sales: {sales_count}, products with homogeneous code: {nomenclator_ready}/{NOMENCLATOR_MIN_THRESHOLD}"
            )
            raise HTTPException(
                status_code=503,
                detail="El catálogo de productos necesita sincronización. Contacta al administrador del sistema.",
            )

        context = partner_analysis_service.get_substitutable_universe_context(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=start_date,
            end_date=end_date,
            period_months=period_months,
            limit=limit,
            offset=offset,
        )

        logger.info(
            f"[PARTNER_ANALYSIS_API] Universo sustituible obtenido: {context['universe_summary']['total_substitutable_groups']} conjuntos"
        )

        return SubstitutableUniverseResponse(**context)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error obteniendo universo sustituible: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al obtener contexto del universo sustituible",
        )


@router.post("/partner-dynamic/{pharmacy_id}")
async def calculate_partner_dynamic_analysis(
    pharmacy_id: UUID,
    request: PartnerDynamicAnalysisRequest,
    limit: Optional[int] = None,  # P1.3: Paginación opcional
    offset: int = 0,  # P1.3: Offset para paginación
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> PartnerDynamicAnalysisResponse:
    """
    Realiza análisis dinámico basado en partners seleccionados.

    **LÓGICA DE FILTRADO EMPRESARIAL:**
    - Solo conjuntos donde existe AL MENOS UNA alternativa de partners seleccionados
    - Ahorro potencial ÚNICAMENTE sobre ventas NO-partners en conjuntos con partners
    - Exclusión automática de conjuntos sin partners disponibles

    **Ejemplo Filtrado:**
    ```
    Conjunto "OMEPRAZOL 20MG" labs disponibles: [KERN, NORMON, RATIOPHARM]
    Partners seleccionados: [CINFA, TEVA]
    Resultado: EXCLUIDO (no hay intersección)

    Conjunto "PARACETAMOL 1G" labs disponibles: [CINFA, KERN, NORMON]
    Partners seleccionados: [CINFA, TEVA]
    Resultado: INCLUIDO (CINFA disponible)
    ```

    **Métricas Calculadas:**
    - Ventas analizables: Solo en conjuntos con partners disponibles
    - Penetración partners: % actual de ventas que son de partners
    - Ahorro potencial: Ventas NO-partners × PVL base × % descuento

    **Paginación (P1.3):**
    - limit: Número máximo de grupos a retornar (None = sin límite)
    - offset: Número de grupos a saltar (default: 0)
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia (Issue #254, PR #289)
        verify_pharmacy_access(pharmacy_id, current_user)

        logger.info(
            f"[PARTNER_ANALYSIS_API] Análisis dinámico farmacia {pharmacy_id} con partner codes: {request.selected_partner_codes}, limit={limit}, offset={offset}"
        )

        analysis = partner_analysis_service.calculate_partner_dynamic_analysis(
            db=db,
            pharmacy_id=pharmacy_id,
            selected_partner_codes=request.selected_partner_codes,
            start_date=request.start_date,  # Issue #xxx: Fechas directas
            end_date=request.end_date,  # Issue #xxx: Fechas directas
            period_months=request.period_months,  # Fallback si no hay fechas
            discount_percentage=request.discount_percentage,
            limit=limit,  # P1.3: Pasar parámetros de paginación
            offset=offset,  # P1.3: Pasar parámetros de paginación
            selected_employee_names=request.selected_employee_names,  # Issue #402
        )

        logger.info(
            f"[PARTNER_ANALYSIS_API] Análisis dinámico completado: {analysis['analyzable_universe']['groups_count']} conjuntos analizables"
        )

        return PartnerDynamicAnalysisResponse(**analysis)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación análisis dinámico: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error análisis dinámico: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al calcular análisis dinámico de partners",
        )


@router.get("/temporal-breakdown/{pharmacy_id}")
async def get_temporal_breakdown(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    partner_codes: List[str] = Query(
        ...,
        description="Códigos de partners seleccionados (ej: ['111', '426'])",
    ),
    level: str = Query(
        "quarter",
        description="Nivel temporal: 'quarter', 'month', 'fortnight'",
    ),
    period_months: int = Query(
        12,
        ge=1,
        le=24,
        description="Período de análisis en meses",
    ),
    homogeneous_code: Optional[str] = Query(
        None,
        regex=r'^\d+(\.\d+)?$',  # Solo números y punto decimal
        description="Código de conjunto homogéneo para filtrar (opcional, ej: '1160.0'). Solo se permiten números y punto decimal.",
    ),
    employee_names: Optional[List[str]] = Query(
        None,
        description="Filtro de empleados (Issue #428)",
    ),
) -> TemporalBreakdownResponse:
    """
    Obtiene breakdown temporal para drill-down por períodos.

    **Niveles de Drill-Down:**
    - `quarter`: Análisis trimestral (Q1, Q2, Q3, Q4)
    - `month`: Análisis mensual (Ene, Feb, Mar...)
    - `fortnight`: Análisis quincenal (S1, S2, S3...)

    **Casos de Uso:**
    - Identificar tendencias estacionales en penetración partners
    - Planificar campañas por períodos específicos
    - Análisis de performance temporal de partners

    **Métricas por Período:**
    - Penetración partners (%)
    - Unidades de oportunidad (NO-partners)
    - Base de ahorro potencial (PVL)
    - Evolución temporal de métricas clave

    **NUEVO (Issue #346):**
    - Si se proporciona `homogeneous_code`, filtra solo productos de ese conjunto homogéneo
    - Útil para drill-down desde gráfico de conjuntos homogéneos hacia evolución temporal

    **NUEVO (Issue #428):**
    - Si se proporciona `employee_names`, filtra solo ventas de esos empleados
    - Útil para análisis específico por empleado o grupo de empleados
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia (Issue #254, PR #289)
        verify_pharmacy_access(pharmacy_id, current_user)

        # ✅ FASTAPI STANDARD: Lista directa sin parsing manual
        selected_partner_codes = [p.strip() for p in partner_codes if p.strip()]

        if not selected_partner_codes:
            raise ValueError("Debe especificar al menos un partner")

        # ✅ Issue #346: Logging con filtro opcional de conjunto homogéneo
        filter_msg = f" [filtro: conjunto {homogeneous_code}]" if homogeneous_code else ""
        employee_msg = f", empleados: {employee_names or 'todos'}"
        logger.info(f"[PARTNER_ANALYSIS_API] Breakdown temporal farmacia {pharmacy_id}, nivel: {level}{filter_msg}{employee_msg}")

        breakdown = partner_analysis_service.get_temporal_breakdown(
            db=db,
            pharmacy_id=pharmacy_id,
            selected_partner_codes=selected_partner_codes,
            level=level,
            period_months=period_months,
            homogeneous_code=homogeneous_code,  # ✅ Issue #346: Filtro opcional
            selected_employee_names=employee_names,  # ✅ Issue #428: Filtro de empleados
        )

        logger.info(
            f"[PARTNER_ANALYSIS_API] Breakdown temporal completado: {breakdown['summary']['periods_count']} períodos{filter_msg}"
        )

        return TemporalBreakdownResponse(**breakdown)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación temporal: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error breakdown temporal: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al obtener breakdown temporal",
        )


@router.get("/partner-products/{pharmacy_id}")
async def get_partner_product_references(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    partner_codes: List[str] = Query(
        ...,
        description="Códigos de partners seleccionados",
    ),
    homogeneous_code: Optional[str] = Query(
        None,
        description="Código conjunto homogéneo opcional para filtrar",
    ),
    period_months: int = Query(
        12,
        ge=1,
        le=24,
        description="Período de análisis en meses",
    ),
) -> PartnerProductReferencesResponse:
    """
    Obtiene tabla de referencias de productos vendidos de partners seleccionados.

    **Uso Principal:**
    - Mostrar tabla detallada en panel de partners
    - Ver productos específicos que se venden de cada partner
    - Análisis de portfolio de productos por laboratorio

    **Datos mostrados por producto:**
    - Código nacional y descripción
    - Laboratorio partner
    - Unidades vendidas y ventas totales
    - Precio medio por unidad

    **Filtros opcionales:**
    - Por conjunto homogéneo específico
    - Por período temporal

    **Ordenamiento:**
    - Descendente por unidades vendidas (productos más vendidos primero)
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia (Issue #254, PR #289)
        verify_pharmacy_access(pharmacy_id, current_user)

        # ✅ FASTAPI STANDARD: Lista directa sin parsing manual
        selected_partner_codes = [p.strip() for p in partner_codes if p.strip()]

        if not selected_partner_codes:
            raise ValueError("Debe especificar al menos un partner")

        logger.info(
            f"[PARTNER_ANALYSIS_API] Referencias productos partners farmacia {pharmacy_id}, partner_codes: {selected_partner_codes}"
        )

        references = partner_analysis_service.get_partner_product_references(
            db=db,
            pharmacy_id=pharmacy_id,
            selected_partner_codes=selected_partner_codes,
            homogeneous_code=homogeneous_code,
            period_months=period_months,
        )

        logger.info(f"[PARTNER_ANALYSIS_API] Referencias obtenidas: {references['total_products']} productos")

        return PartnerProductReferencesResponse(**references)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación referencias: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error obteniendo referencias productos: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al obtener referencias de productos partners",
        )


@router.get("/homogeneous-detail/{pharmacy_id}/{homogeneous_code}")
async def get_homogeneous_group_detail(
    pharmacy_id: UUID,
    homogeneous_code: str,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    partner_codes: List[str] = Query(
        ...,
        description="Códigos de partners seleccionados",
    ),
    period_months: int = Query(
        12,
        ge=1,
        le=24,
        description="Período de análisis en meses",
    ),
) -> HomogeneousGroupDetailResponse:
    """
    Obtiene detalle completo de un conjunto homogéneo específico.

    **Información Detallada:**
    - Todos los productos en el conjunto con sus laboratorios
    - Unidades vendidas por producto y laboratorio
    - Identificación clara partners vs no-partners
    - Cálculo de ahorro individual por producto NO-partner

    **Casos de Uso:**
    - Drill-down desde vista agregada de conjuntos
    - Análisis detallado para negociación con laboratorios
    - Identificación de productos específicos con mayor oportunidad

    **Métricas por Producto:**
    - Unidades vendidas y facturación
    - PVP vs Precio de Referencia
    - Base de ahorro individual (cantidad × PVL)
    - Clasificación partner/no-partner
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia (Issue #254, PR #289)
        verify_pharmacy_access(pharmacy_id, current_user)

        # ✅ FASTAPI STANDARD: Lista directa sin parsing manual
        selected_partner_codes = [p.strip() for p in partner_codes if p.strip()]

        if not selected_partner_codes:
            raise ValueError("Debe especificar al menos un partner")

        logger.info(f"[PARTNER_ANALYSIS_API] Detalle conjunto {homogeneous_code} para farmacia {pharmacy_id}")

        detail = partner_analysis_service.get_homogeneous_group_detail(
            db=db,
            pharmacy_id=pharmacy_id,
            homogeneous_code=homogeneous_code,
            selected_partner_codes=selected_partner_codes,
            period_months=period_months,
        )

        logger.info(
            f"[PARTNER_ANALYSIS_API] Detalle conjunto completado: {detail['group_summary']['total_products']} productos"
        )

        return HomogeneousGroupDetailResponse(**detail)

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación detalle: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error detalle conjunto: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al obtener detalle del conjunto homogéneo",
        )


@router.get("/context-treemap/{pharmacy_id}")
async def get_context_treemap(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[str] = Query(None, description="Fecha inicio análisis (YYYY-MM-DD)"),
    end_date: Optional[str] = Query(None, description="Fecha fin análisis (YYYY-MM-DD)"),
    employee_names: Optional[List[str]] = Query(None, description="Filtro de empleados (Issue #402)"),
):
    """
    Obtiene datos para treemap jerárquico del contexto de ventas (Issue #415).

    **Jerarquía del Treemap:**
    ```
    Total Ventas (100%)
    ├─ Sustituible (conjuntos con AL MENOS un genérico disponible)
    │  ├─ Analizable (cubierta por partners seleccionados)
    │  │  └─ Penetración (ya sustituido por genéricos)
    │  └─ No Analizable (sin partners que cubran)
    └─ No Sustituible (sin genéricos disponibles)
    ```

    **Filtros Composables:**
    - `start_date` / `end_date`: Rango temporal (default: últimos 2 años)
    - `employee_names`: Filtrar por empleados específicos (PRO feature, Issue #402)

    **Integración:**
    - Issue #415: Mejora de página /generics con layout optimizado
    - Issue #402: Filtro de empleados PRO
    - REGLA #18: Aplica restricción FREE tier automáticamente

    **Response:**
    Estructura jerárquica con valores absolutos (€) y porcentajes relativos
    """
    try:
        # ✅ SECURITY: Verificar que usuario tiene acceso a esta farmacia
        verify_pharmacy_access(pharmacy_id, current_user)

        # Parse dates if provided (con timezone UTC para evitar comparación naive vs aware)
        from datetime import datetime, timezone

        parsed_start_date = None
        parsed_end_date = None

        if start_date:
            try:
                parsed_start_date = datetime.fromisoformat(start_date)
                # Asegurar timezone-aware (UTC) si es naive
                if parsed_start_date.tzinfo is None:
                    parsed_start_date = parsed_start_date.replace(tzinfo=timezone.utc)
            except ValueError:
                raise HTTPException(status_code=400, detail="Formato start_date inválido (use YYYY-MM-DD)")

        if end_date:
            try:
                parsed_end_date = datetime.fromisoformat(end_date)
                # Asegurar timezone-aware (UTC) si es naive
                if parsed_end_date.tzinfo is None:
                    parsed_end_date = parsed_end_date.replace(tzinfo=timezone.utc)
            except ValueError:
                raise HTTPException(status_code=400, detail="Formato end_date inválido (use YYYY-MM-DD)")

        logger.info(
            f"[PARTNER_ANALYSIS_API] Contexto treemap para farmacia {pharmacy_id}, "
            f"período: {start_date or 'default'} a {end_date or 'default'}, "
            f"empleados: {employee_names or 'todos'}"
        )

        treemap_data = partner_analysis_service.get_context_treemap_data(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=parsed_start_date,
            end_date=parsed_end_date,
            employee_names=employee_names,
        )

        logger.info(
            f"[PARTNER_ANALYSIS_API] Contexto treemap completado: "
            f"Total={treemap_data['hierarchy']['value']:.2f}€"
        )

        return treemap_data

    except HTTPException:
        raise
    except ValueError as e:
        logger.warning(f"[PARTNER_ANALYSIS_API] Error validación contexto treemap: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logger.error(f"[PARTNER_ANALYSIS_API] Error contexto treemap: {e}")
        raise HTTPException(
            status_code=500,
            detail="Error al calcular contexto treemap",
        )
