"""
Optimal Partners API - Endpoint para recomendación de partners óptimos
Parte del módulo de análisis de genéricos (Issue #415)
"""

from fastapi import APIRouter, Depends, Query, HTTPException, status
from typing import Optional, List
from datetime import date
from uuid import UUID
from sqlalchemy.orm import Session
import logging

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.models.pharmacy import Pharmacy
from app.services.optimal_partners_service import (
    calculate_optimal_partners,
    OptimalPartnersResponse
)

logger = logging.getLogger(__name__)

router = APIRouter()


@router.get("/{pharmacy_id}", response_model=OptimalPartnersResponse)
async def get_optimal_partners(
    pharmacy_id: UUID,
    start_date: Optional[date] = Query(
        None,
        description="Fecha inicio del análisis (por defecto: hace 2 años)"
    ),
    end_date: Optional[date] = Query(
        None,
        description="Fecha fin del análisis (por defecto: hoy)"
    ),
    employee_names: Optional[List[str]] = Query(
        None,
        description="Filtrar por empleados específicos (opcional, feature PRO Issue #402)"
    ),
    max_partners: int = Query(
        8,
        ge=1,
        le=20,
        description="Número máximo de partners a retornar"
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Calcula partners óptimos para maximizar cobertura de ventas sustituibles.

    **Algoritmo de Selección Greedy:**
    1. Calcula cobertura individual de cada partner (% ventas sustituibles que puede cubrir)
    2. Selecciona partner con mayor cobertura como primario (position 1)
    3. Para cada posición adicional:
       - Calcula cobertura incremental (productos NO cubiertos por partners previos)
       - Ordena por mayor incremento de cobertura
       - Añade a lista, acumula cobertura total
    4. Retorna lista ordenada con métricas de cobertura

    **Cálculo de Cobertura:**
    - Ventas sustituibles totales en el período
    - Por cada partner: valor en € de productos que puede sustituir
    - Incremental: ventas de NUEVOS productos que añade este partner
    - Acumulada: suma running de cobertura %

    **Estimación de Ahorros:**
    - Asume 15-20% descuento promedio vs. marca
    - Calcula basado en ventas sustituibles cubiertas por el partner

    **Seguridad:**
    - Verifica que usuario tiene acceso a la farmacia (JWT obligatorio, REGLA #7)
    - Filtro de empleados respeta plan PRO (Issue #402)

    **Parámetros de Query:**
    - `start_date`: Filtrar ventas desde esta fecha (opcional)
    - `end_date`: Filtrar ventas hasta esta fecha (opcional)
    - `employee_names`: Filtrar por empleados específicos (opcional, PRO)
    - `max_partners`: Máximo de partners a retornar (1-20, default 8)

    **Respuesta:**
    - Lista ordenada de partners por contribución de cobertura
    - Métricas de cobertura (individual, incremental, acumulada)
    - Metadata del análisis (período, ventas totales, empleados)

    **Casos Especiales:**
    - Sin partners configurados: Retorna lista vacía
    - Sin ventas sustituibles: Retorna lista vacía
    - 100% cobertura alcanzada: Detiene selección antes de max_partners

    **Ejemplo de Respuesta:**
    ```json
    {
        "optimal_partners": [
            {
                "partner_id": "uuid",
                "partner_name": "CINFA",
                "coverage_percentage": 61.5,
                "incremental_coverage": 61.5,
                "cumulative_coverage": 61.5,
                "position": 1,
                "substitutable_sales": 1600.0,
                "potential_savings": 280.0,
                "products_covered": 15,
                "is_primary": true
            },
            {
                "partner_name": "KERN PHARMA",
                "coverage_percentage": 38.5,
                "incremental_coverage": 38.5,
                "cumulative_coverage": 100.0,
                "position": 2,
                ...
            }
        ],
        "analysis_metadata": {
            "total_substitutable_sales": 2600.0,
            "total_partners_analyzed": 8,
            "analysis_period": {
                "start": "2022-10-01",
                "end": "2024-10-31"
            },
            "employees_included": ["Juan", "Maria"]
        }
    }
    ```
    """
    logger.info(
        f"Optimal partners request: pharmacy={pharmacy_id}, "
        f"start={start_date}, end={end_date}, employees={employee_names}"
    )

    # Verify user has access to pharmacy (REGLA #7 - Security First)
    pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
    if not pharmacy:
        logger.warning(f"Pharmacy {pharmacy_id} not found")
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Farmacia {pharmacy_id} no encontrada"
        )

    # Verify user owns this pharmacy (REGLA #10 - 1:1 relationship)
    if current_user.pharmacy_id != pharmacy_id:
        logger.warning(
            f"User {current_user.id} attempted to access pharmacy {pharmacy_id} "
            f"but owns pharmacy {current_user.pharmacy_id}"
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="No tienes permisos para acceder a esta farmacia"
        )

    # Validate date range
    if start_date and end_date and start_date > end_date:
        logger.warning(f"Invalid date range: start={start_date} > end={end_date}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="La fecha de inicio no puede ser posterior a la fecha de fin"
        )

    # Calculate optimal partners
    try:
        result = await calculate_optimal_partners(
            db=db,
            pharmacy_id=pharmacy_id,
            start_date=start_date,
            end_date=end_date,
            employee_names=employee_names,
            max_partners=max_partners
        )

        logger.info(
            f"Optimal partners calculated: {len(result.optimal_partners)} partners, "
            f"total_sales={result.analysis_metadata.total_substitutable_sales}"
        )

        return result

    except Exception as e:
        logger.error(f"Error calculating optimal partners: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error al calcular partners óptimos: {str(e)}"
        )
