# backend/app/api/narrative.py
"""
API Router para NarrativeService (Issue #509)

Endpoints para Executive Summary con NLG:
- GET /narrative/{pharmacy_id}/summary: Obtener resumen ejecutivo
- POST /narrative/{pharmacy_id}/feedback: Registrar feedback (👍/👎)
- GET /narrative/{pharmacy_id}/feedback/history: Historial de feedback
"""

import hashlib
import json
from datetime import datetime, timezone
from typing import List, Optional
from uuid import UUID, uuid4

import structlog
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.narrative_feedback import NarrativeFeedback
from app.models.user import User
from app.schemas.narrative import (
    NarrativeFeedbackCreate,
    NarrativeFeedbackResponse,
    NarrativeKPIs,
    NarrativeResponse,
)
from app.services.insight_engine import insight_engine_service
from app.services.narrative_service import narrative_service
from app.services.ventalibre_service import ventalibre_service

logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/narrative", tags=["narrative"])


@router.get("/{pharmacy_id}/summary", response_model=NarrativeResponse)
async def get_executive_summary(
    pharmacy_id: UUID,
    date_from: str = Query(..., description="Fecha inicio (YYYY-MM-DD)"),
    date_to: str = Query(..., description="Fecha fin (YYYY-MM-DD)"),
    language: str = Query("es", description="Idioma (es, en, ca)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Genera resumen ejecutivo para el dashboard de Venta Libre.

    Combina KPIs + Insight Engine → Narrativa estructurada en 3 capas:
    1. EL QUÉ (Hechos)
    2. EL PORQUÉ (Diagnóstico)
    3. EL CÓMO (Acción)

    Cache: 24h (invalidado por nueva carga de datos).

    Args:
        pharmacy_id: UUID de la farmacia
        date_from: Fecha inicio del período
        date_to: Fecha fin del período
        language: Idioma de la narrativa

    Returns:
        NarrativeResponse con resumen ejecutivo

    Raises:
        403: Si el usuario no tiene acceso a la farmacia
        500: Error interno
    """
    # Verify user has access to pharmacy
    if current_user.pharmacy_id != pharmacy_id and current_user.role.name != "admin":
        raise HTTPException(
            status_code=403, detail="No tienes acceso a esta farmacia"
        )

    logger.info(
        "narrative.summary_request",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
        date_from=date_from,
        date_to=date_to,
    )

    try:
        # 1. Get KPIs from VentaLibreService (sync function - no await)
        kpis_data = ventalibre_service.get_kpis(
            db=db,
            pharmacy_id=pharmacy_id,
            date_from=date_from,
            date_to=date_to,
        )

        # Map VentaLibreService response fields to NarrativeKPIs
        kpis = NarrativeKPIs(
            total_sales=kpis_data.get("total_sales", 0.0),
            avg_margin=kpis_data.get("coverage_percent", 0.0),  # Use coverage as proxy
            unique_products=kpis_data.get("total_products", 0),  # Fixed field name
            trend_pct=kpis_data.get("yoy_growth", 0.0),  # Fixed field name
            date_from=date_from,
            date_to=date_to,
            sales_by_category=kpis_data.get("sales_by_category"),
        )

        # 2. Get insights from Insight Engine
        audit_result = await insight_engine_service.run_audit(
            db=db,
            pharmacy_id=pharmacy_id,
        )

        # 3. Generate narrative
        response = narrative_service.generate_summary(
            kpis=kpis,
            audit_result=audit_result,
            language=language,
        )

        logger.info(
            "narrative.summary_generated",
            pharmacy_id=str(pharmacy_id),
            insights_used=response.insights_used,
            grounding_valid=response.grounding_valid,
        )

        return response

    except Exception as e:
        logger.error(
            "narrative.summary_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500, detail="Error interno al generar resumen ejecutivo"
        )


@router.post("/{pharmacy_id}/feedback", response_model=NarrativeFeedbackResponse)
async def submit_feedback(
    pharmacy_id: UUID,
    feedback: NarrativeFeedbackCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Registra feedback del usuario sobre la narrativa (👍/👎).

    El feedback se almacena con un snapshot de los datos para debugging
    en caso de que el LLM haya alucinado.

    Args:
        pharmacy_id: UUID de la farmacia
        feedback: Datos del feedback

    Returns:
        NarrativeFeedbackResponse con confirmación

    Raises:
        403: Si el usuario no tiene acceso a la farmacia
    """
    # Verify user has access to pharmacy
    if current_user.pharmacy_id != pharmacy_id and current_user.role.name != "admin":
        raise HTTPException(
            status_code=403, detail="No tienes acceso a esta farmacia"
        )

    logger.info(
        "narrative.feedback_submit",
        pharmacy_id=str(pharmacy_id),
        is_helpful=feedback.is_helpful,
        narrative_hash=feedback.narrative_hash[:16] if feedback.narrative_hash else "empty",
    )

    try:
        # Create feedback record
        db_feedback = NarrativeFeedback(
            id=uuid4(),
            pharmacy_id=pharmacy_id,
            narrative_hash=feedback.narrative_hash,
            is_helpful=feedback.is_helpful,
            feedback_text=feedback.feedback_text,
            created_at=datetime.now(timezone.utc),
        )
        db.add(db_feedback)
        db.commit()
        db.refresh(db_feedback)

        return NarrativeFeedbackResponse(
            id=str(db_feedback.id),
            narrative_hash=db_feedback.narrative_hash,
            is_helpful=db_feedback.is_helpful,
            created_at=db_feedback.created_at.isoformat(),
        )

    except Exception as e:
        db.rollback()
        logger.error(
            "narrative.feedback_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500, detail=f"Error al guardar feedback: {str(e)}"
        )


@router.get(
    "/{pharmacy_id}/feedback/history",
    response_model=List[NarrativeFeedbackResponse],
)
async def get_feedback_history(
    pharmacy_id: UUID,
    limit: int = Query(20, ge=1, le=100, description="Límite de resultados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene historial de feedback de narrativas para una farmacia.

    Útil para debugging y mejora de prompts.

    Args:
        pharmacy_id: UUID de la farmacia
        limit: Número máximo de resultados

    Returns:
        Lista de NarrativeFeedbackResponse

    Raises:
        403: Si el usuario no tiene acceso a la farmacia
    """
    # Verify user has access (admin only for history)
    if current_user.role.name != "admin":
        raise HTTPException(
            status_code=403, detail="Solo administradores pueden ver el historial"
        )

    try:
        feedbacks = (
            db.query(NarrativeFeedback)
            .filter(NarrativeFeedback.pharmacy_id == pharmacy_id)
            .order_by(NarrativeFeedback.created_at.desc())
            .limit(limit)
            .all()
        )

        return [
            NarrativeFeedbackResponse(
                id=str(f.id),
                narrative_hash=f.narrative_hash,
                is_helpful=f.is_helpful,
                created_at=f.created_at.isoformat(),
            )
            for f in feedbacks
        ]

    except Exception as e:
        logger.error(
            "narrative.feedback_history_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500, detail=f"Error al obtener historial: {str(e)}"
        )


@router.get("/{pharmacy_id}/feedback/stats")
async def get_feedback_stats(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene estadísticas de feedback para una farmacia.

    Args:
        pharmacy_id: UUID de la farmacia

    Returns:
        Estadísticas de feedback (total, helpful, unhelpful, ratio)

    Raises:
        403: Si el usuario no es admin
    """
    # Admin only
    if current_user.role.name != "admin":
        raise HTTPException(
            status_code=403, detail="Solo administradores pueden ver estadísticas"
        )

    try:
        total = (
            db.query(NarrativeFeedback)
            .filter(NarrativeFeedback.pharmacy_id == pharmacy_id)
            .count()
        )

        helpful = (
            db.query(NarrativeFeedback)
            .filter(
                NarrativeFeedback.pharmacy_id == pharmacy_id,
                NarrativeFeedback.is_helpful.is_(True),
            )
            .count()
        )

        unhelpful = total - helpful
        ratio = (helpful / total * 100) if total > 0 else 0.0

        return {
            "total_feedbacks": total,
            "helpful_count": helpful,
            "unhelpful_count": unhelpful,
            "helpful_ratio": round(ratio, 1),
        }

    except Exception as e:
        logger.error(
            "narrative.feedback_stats_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500, detail=f"Error al obtener estadísticas: {str(e)}"
        )
