﻿# backend/app/api/laboratory_mapping.py
"""
API endpoint para mapeo de códigos de laboratorio ↔ nombres
Usado por el frontend para convertir códigos en nombres para mostrar al usuario

FASE 1.2: Implementa validación de inputs y seguridad según Issue #23
- Schemas Pydantic para validación de formato
- Rate limiting específico para laboratory endpoints
- Protección SQL injection y XSS
- Error handling estructurado
- Request size limits y timeouts

FASE 1.3: Cache layer con Redis para performance optimization
- Cache TTL: 24h para mappings, 12h para queries dinámicas
- Fallback graceful si Redis no disponible
- Cache statistics y monitoring
- Cache invalidation en catalog updates
"""

import logging
import time
import unicodedata
from typing import Any, Dict, List, Optional, Union

from fastapi import APIRouter, Depends, HTTPException, Query, Request
from fastapi.responses import JSONResponse
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

from ..api.deps import get_current_user, require_permission
from ..config.laboratory_config import LaboratoryConfig, LaboratorySecurityConfig
from ..core.subscription_limits import Permission
from ..core.rate_limiter import get_client_ip
from ..database import get_db
from ..models.user import User
from ..schemas.laboratory_validation import (
    LaboratoryValidationError,
    PaginatedResponse,
    calculate_pagination_metadata,
    sanitize_laboratory_name,
    validate_laboratory_code,
    validate_pagination_params,
)
from ..services.laboratory_cache_service import laboratory_cache_service

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/laboratory-mapping", tags=["laboratory-mapping"])


# Utilidades de seguridad
def check_security_headers(request: Request):
    """Verificar headers de seguridad y detectar requests sospechosos"""
    client_ip = get_client_ip(request)
    user_agent = request.headers.get("User-Agent", "")
    referer = request.headers.get("Referer", "")

    # Verificar IPs bloqueadas
    if client_ip in LaboratorySecurityConfig.get_blocked_ips():
        logger.warning(f"[SECURITY] IP bloqueada intentó acceso: {client_ip}")
        raise HTTPException(
            status_code=403, detail={"error_type": "access_forbidden", "message": "Acceso denegado desde esta IP"}
        )

    # Log de requests sospechosos
    if LaboratorySecurityConfig.is_suspicious_request(user_agent, referer):
        logger.warning(
            f"[SECURITY] Request sospechoso: IP={client_ip}, " f"UA={user_agent[:100]}, Referer={referer[:100]}"
        )


def create_error_response(
    status_code: int,
    error_type: str,
    message: str,
    details: Optional[Dict] = None,
    invalid_values: Optional[List[str]] = None,
) -> JSONResponse:
    """Crear respuesta de error estructurada"""
    error_data = LaboratoryValidationError(
        error_type=error_type, message=message, details=details or {}, invalid_values=invalid_values or []
    )

    response = JSONResponse(status_code=status_code, content=error_data.dict())

    # Agregar headers de seguridad
    for header, value in LaboratorySecurityConfig.REQUIRED_SECURITY_HEADERS.items():
        response.headers[header] = value

    return response


def validate_request_size(request: Request):
    """Validar tamaño del request"""
    content_length = request.headers.get("Content-Length")
    if content_length:
        size_bytes = int(content_length)
        max_size = LaboratoryConfig.MAX_REQUEST_SIZE_KB * 1024

        if size_bytes > max_size:
            raise HTTPException(
                status_code=413,
                detail={
                    "error_type": "payload_too_large",
                    "message": f"Request size {size_bytes} bytes excede límite {max_size} bytes",
                    "max_size_kb": LaboratoryConfig.MAX_REQUEST_SIZE_KB,
                },
            )


def execute_safe_query(db: Session, query: text, params: Dict) -> List:
    """Ejecutar query con protección SQL injection y timeout"""
    try:
        start_time = time.time()

        # Ejecutar con timeout
        result = db.execute(query, params).fetchall()

        execution_time = time.time() - start_time

        # Log de queries lentas
        if execution_time > 2.0:
            logger.warning(f"[PERFORMANCE] Query lenta detectada: {execution_time:.2f}s")

        return result

    except Exception as e:
        logger.error(f"[DATABASE] Error ejecutando query: {str(e)[:200]}")
        raise HTTPException(
            status_code=500, detail={"error_type": "database_error", "message": "Error interno del servidor"}
        )


@router.get(
    "/codes-to-names",
    summary="Mapeo códigos a nombres de laboratorios",
    description="Convierte códigos numéricos de laboratorios del nomenclátor español a nombres completos",
    responses={
        200: {
            "description": "Mapeo exitoso",
            "content": {
                "application/json": {
                    "examples": {
                        "generic_leaders": {
                            "summary": "Laboratorios genéricos principales",
                            "description": "Los 5 laboratorios genéricos más importantes de España",
                            "value": {
                                "111": "CINFA S.A.",
                                "426": "NORMON S.A.",
                                "863": "KERN PHARMA, S.L.",
                                "644": "SANDOZ FARMACEUTICA, S.A",
                                "1079": "TEVA PHARMA, S.L.U",
                            },
                        },
                        "with_pagination": {
                            "summary": "Respuesta con paginación",
                            "description": "Formato envelope con metadata de paginación",
                            "value": {
                                "items": {"111": "CINFA S.A.", "426": "NORMON S.A.", "863": "KERN PHARMA, S.L."},
                                "pagination": {
                                    "total": 1247,
                                    "page": 1,
                                    "per_page": 3,
                                    "total_pages": 416,
                                    "has_next": True,
                                    "has_previous": False,
                                },
                            },
                        },
                    }
                }
            },
        },
        400: {
            "description": "Parámetros inválidos",
            "content": {
                "application/json": {
                    "example": {
                        "error_type": "validation_error",
                        "message": "Demasiados códigos. Máximo permitido: 20",
                        "details": {"max_codes": 20, "received": 25},
                    }
                }
            },
        },
        422: {
            "description": "Códigos con formato inválido",
            "content": {
                "application/json": {
                    "example": {
                        "error_type": "format_error",
                        "message": "Códigos de laboratorio con formato inválido",
                        "details": {"expected_format": "1-4 dígitos numéricos", "pattern": "^\\d{1,4}$"},
                        "invalid_values": ["abc", "12345", "0000"],
                    }
                }
            },
        },
        429: {
            "description": "Rate limit excedido",
            "content": {
                "application/json": {
                    "example": {
                        "error_type": "rate_limit_exceeded",
                        "message": "Rate limit excedido: 60 requests por minuto",
                        "retry_after": 45,
                        "limit_type": "per_minute",
                    }
                }
            },
        },
    },
)
async def get_laboratory_codes_to_names(
    request: Request,
    current_user: User = Depends(get_current_user),
    codes: List[str] = Query(None, description="Códigos específicos a mapear (máx 20)", max_length=20),
    page: int = Query(None, ge=1, le=10000, description="Número de página (opcional para paginación)"),
    per_page: int = Query(None, ge=1, le=200, description="Elementos por página (opcional para paginación)"),
    envelope: bool = Query(False, description="Devolver respuesta en envelope con metadata de paginación"),
    db: Session = Depends(get_db),
) -> Union[Dict[str, str], PaginatedResponse[Dict[str, str]]]:
    """
    Obtiene mapeo de códigos de laboratorio a nombres.

    **FASE 2.2 - PAGINACIÓN IMPLEMENTADA**
    - Soporte de paginación opcional con page/per_page
    - Cursor-based pagination automática para páginas > 100
    - Backward compatibility completa (sin parámetros = comportamiento original)
    - Envelope opcional para metadata de paginación

    **FASE 1.2 - VALIDADO Y SEGURO**
    - Validación formato códigos: /^\\d{1,4}$/
    - Máximo 20 códigos por request
    - Rate limiting: 60 req/min, 500 req/hour
    - Protección SQL injection y XSS
    - Headers de seguridad obligatorios

    **Query parameters:**
    - codes: Lista de códigos específicos (opcional, máximo 20)
    - page: Número de página (opcional, 1-10000)
    - per_page: Elementos por página (opcional, 1-200)
    - envelope: Devolver con metadata de paginación (opcional, default: false)

    **Response tradicional (sin envelope):**
    ```json
    {
        "426": "NORMON S.A.",
        "111": "CINFA S.A.",
        "1079": "TEVA PHARMA, S.L.U"
    }
    ```

    **Response con envelope (envelope=true):**
    ```json
    {
        "items": {
            "426": "NORMON S.A.",
            "111": "CINFA S.A."
        },
        "pagination": {
            "total": 1247,
            "page": 2,
            "per_page": 50,
            "total_pages": 25,
            "has_next": true,
            "has_previous": true
        }
    }
    ```

    **Errores:**
    - 400: Códigos con formato inválido o parámetros de paginación inconsistentes
    - 413: Request demasiado grande
    - 422: Parámetros de paginación fuera de rango
    - 429: Rate limit excedido
    - 500: Error interno del servidor
    """
    # Verificaciones de seguridad
    check_security_headers(request)
    validate_request_size(request)

    client_ip = get_client_ip(request)

    try:
        # Validar códigos si están presentes
        validated_codes = []
        invalid_codes = []

        if codes:
            # Validar límite de cantidad
            if len(codes) > LaboratoryConfig.MAX_CODES_PER_REQUEST:
                return create_error_response(
                    status_code=400,
                    error_type="validation_error",
                    message=f"Demasiados códigos. Máximo permitido: {LaboratoryConfig.MAX_CODES_PER_REQUEST}",
                    details={"max_codes": LaboratoryConfig.MAX_CODES_PER_REQUEST, "received": len(codes)},
                )

            # Validar formato de cada código
            for code in codes:
                try:
                    validated_code = validate_laboratory_code(code)
                    validated_codes.append(validated_code)
                except ValueError:
                    invalid_codes.append(code)

            # Si hay códigos inválidos, devolver error
            if invalid_codes:
                return create_error_response(
                    status_code=422,
                    error_type="format_error",
                    message="Códigos de laboratorio con formato inválido",
                    details={
                        "expected_format": "1-4 dígitos numéricos",
                        "pattern": LaboratoryConfig.LABORATORY_CODE_PATTERN,
                    },
                    invalid_values=invalid_codes,
                )

        # FASE 2.2 - Validar parámetros de paginación si están presentes
        if (page is not None) != (per_page is not None):
            return create_error_response(
                status_code=400,
                error_type="validation_error",
                message="Si especifica paginación, debe incluir tanto 'page' como 'per_page'",
                details={"page": page, "per_page": per_page},
            )

        # Validar parámetros de paginación
        if page is not None and per_page is not None:
            pagination_errors = validate_pagination_params(page, per_page)
            if pagination_errors:
                return create_error_response(
                    status_code=422,
                    error_type="pagination_error",
                    message="Parámetros de paginación inválidos",
                    details=pagination_errors,
                )

        # FASE 2.2 - Usar cache service con soporte de paginación
        use_pagination = page is not None and per_page is not None
        return_total = envelope or use_pagination

        if return_total:
            mapping, cache_hit, total_count = laboratory_cache_service.get_codes_to_names_cached(
                db, validated_codes, page, per_page, return_total=True
            )
        else:
            mapping, cache_hit = laboratory_cache_service.get_codes_to_names_cached(
                db, validated_codes, page, per_page, return_total=False
            )

        # Aplicar sanitización adicional para seguridad (post-cache)
        sanitized_mapping = {}
        for code, name in mapping.items():
            if code and name and len(code) <= 4 and code.isdigit():
                try:
                    sanitized_name = sanitize_laboratory_name(name)
                    sanitized_mapping[code] = sanitized_name
                except ValueError as e:
                    logger.warning(f"[SECURITY] Nombre de laboratorio rechazado post-cache: {name} - {e}")
                    continue

        mapping = sanitized_mapping

        # FASE 2.2 - Devolver respuesta con o sin envelope basado en parámetros
        if envelope and return_total:
            # Respuesta con metadata de paginación
            pagination_metadata = calculate_pagination_metadata(total_count, page or 1, per_page or len(mapping))

            # Log exitoso con detalles de seguridad, cache y paginación
            logger.info(
                f"[LABORATORY_MAPPING] Mapeo paginado generado exitosamente: "
                f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
                f"IP={client_ip}, codes_requested={len(validated_codes) if validated_codes else 0}, "
                f"results={len(mapping)}, total={total_count}, page={page or 1}, "
                f"cache_hit={cache_hit}"
            )

            return PaginatedResponse[Dict[str, str]](items=mapping, pagination=pagination_metadata)
        else:
            # Respuesta tradicional (backward compatible)
            logger.info(
                f"[LABORATORY_MAPPING] Mapeo generado exitosamente: "
                f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
                f"IP={client_ip}, codes_requested={len(validated_codes) if validated_codes else 0}, "
                f"results={len(mapping)}, cache_hit={cache_hit}"
            )

            return mapping

    except HTTPException:
        # Re-raise HTTP exceptions (ya tienen formato correcto)
        raise
    except Exception as e:
        logger.error(f"[LABORATORY_MAPPING] Error inesperado: IP={client_ip}, " f"error={str(e)[:200]}")

        return create_error_response(status_code=500, error_type="internal_error", message="Error interno del servidor")


@router.get("/names-to-codes")
async def get_laboratory_names_to_codes(
    request: Request,
    current_user: User = Depends(get_current_user),
    names: List[str] = Query(None, description="Nombres específicos a mapear (máx 20)", max_length=20),
    page: int = Query(None, ge=1, le=10000, description="Número de página (opcional para paginación)"),
    per_page: int = Query(None, ge=1, le=200, description="Elementos por página (opcional para paginación)"),
    envelope: bool = Query(False, description="Devolver respuesta en envelope con metadata de paginación"),
    db: Session = Depends(get_db),
) -> Union[Dict[str, str], PaginatedResponse[Dict[str, str]]]:
    """
    Obtiene mapeo de nombres de laboratorio a códigos.

    **FASE 1.2 - VALIDADO Y SEGURO**
    - Validación nombres: máx 200 chars, sanitización XSS
    - Máximo 20 nombres por request
    - Rate limiting: 60 req/min, 500 req/hour
    - Protección SQL injection y XSS
    - Headers de seguridad obligatorios

    **Uso secundario:**
    - Para convertir nombres históricos a códigos
    - Para búsqueda por nombre de laboratorio

    **Query parameters:**
    - names: Lista de nombres específicos (opcional, máximo 20)

    **Response:**
    ```json
    {
        "NORMON S.A.": "426",
        "CINFA S.A.": "111",
        "TEVA PHARMA, S.L.U": "1079"
    }
    ```

    **Errores:**
    - 400: Nombres con formato inválido
    - 413: Request demasiado grande
    - 422: Nombres contienen caracteres no permitidos
    - 429: Rate limit excedido
    - 500: Error interno del servidor
    """
    # Verificaciones de seguridad
    check_security_headers(request)
    validate_request_size(request)

    client_ip = get_client_ip(request)

    try:
        # Validar nombres si están presentes
        validated_names = []
        invalid_names = []

        if names:
            # Validar límite de cantidad
            if len(names) > LaboratoryConfig.MAX_NAMES_PER_REQUEST:
                return create_error_response(
                    status_code=400,
                    error_type="validation_error",
                    message=f"Demasiados nombres. Máximo permitido: {LaboratoryConfig.MAX_NAMES_PER_REQUEST}",
                    details={"max_names": LaboratoryConfig.MAX_NAMES_PER_REQUEST, "received": len(names)},
                )

            # Validar formato de cada nombre
            for name in names:
                try:
                    # FIX: Normalizar encoding UTF-8 para manejar doble-encoding
                    # Issue: Algunos navegadores/proxies pre-codifican caracteres especiales
                    # causando doble encoding (ej: Ñ → %C3%83%C2%91 en lugar de %C3%91)
                    # Solución: Normalizar a UTF-8 canonical antes de validar
                    normalized_name = unicodedata.normalize('NFC', name)

                    validated_name = sanitize_laboratory_name(normalized_name)
                    validated_names.append(validated_name)
                except ValueError as e:
                    invalid_names.append(name)
                    # Logging estructurado para debugging de errores 422
                    logger.warning(
                        f"[VALIDATION] Nombre de laboratorio rechazado: '{name}' - {e}",
                        extra={
                            "endpoint": "names-to-codes",
                            "client_ip": client_ip,
                            "validation_error": str(e),
                            "name_length": len(name),
                            "name_preview": name[:50] if len(name) > 50 else name,
                            "normalized_preview": unicodedata.normalize('NFC', name)[:50],
                        },
                    )

            # Si hay nombres inválidos, devolver error
            if invalid_names:
                return create_error_response(
                    status_code=422,
                    error_type="format_error",
                    message="Nombres de laboratorio con formato inválido o caracteres no permitidos",
                    details={
                        "max_length": LaboratoryConfig.MAX_NAME_LENGTH,
                        "forbidden_chars": LaboratoryConfig.get_dangerous_chars(),
                    },
                    invalid_values=invalid_names,
                )

        # FASE 2.2 - Validar parámetros de paginación si están presentes
        if (page is not None) != (per_page is not None):
            return create_error_response(
                status_code=400,
                error_type="validation_error",
                message="Si especifica paginación, debe incluir tanto 'page' como 'per_page'",
                details={"page": page, "per_page": per_page},
            )

        # Validar parámetros de paginación
        if page is not None and per_page is not None:
            pagination_errors = validate_pagination_params(page, per_page)
            if pagination_errors:
                return create_error_response(
                    status_code=422,
                    error_type="pagination_error",
                    message="Parámetros de paginación inválidos",
                    details=pagination_errors,
                )

        # FASE 2.2 - Usar cache service con soporte de paginación
        use_pagination = page is not None and per_page is not None
        return_total = envelope or use_pagination

        if return_total:
            mapping, cache_hit, total_count = laboratory_cache_service.get_names_to_codes_cached(
                db, validated_names, page, per_page, return_total=True
            )
        else:
            mapping, cache_hit = laboratory_cache_service.get_names_to_codes_cached(
                db, validated_names, page, per_page, return_total=False
            )

        # Aplicar sanitización adicional para seguridad (post-cache)
        sanitized_mapping = {}
        for name, code in mapping.items():
            # Validar que el código es numérico y el nombre es seguro
            if code and name and len(code) <= 4 and code.isdigit() and len(name) <= LaboratoryConfig.MAX_NAME_LENGTH:

                # Doble sanitización del nombre
                try:
                    sanitized_name = sanitize_laboratory_name(name)
                    sanitized_mapping[sanitized_name] = code
                except ValueError as e:
                    logger.warning(f"[SECURITY] Nombre de laboratorio rechazado post-cache: {name} - {e}")
                    continue

        mapping = sanitized_mapping

        # FASE 2.2 - Devolver respuesta con o sin envelope basado en parámetros
        if envelope and return_total:
            # Respuesta con metadata de paginación
            pagination_metadata = calculate_pagination_metadata(total_count, page or 1, per_page or len(mapping))

            # Log exitoso con detalles de seguridad, cache y paginación
            logger.info(
                f"[LABORATORY_MAPPING] Mapeo inverso paginado generado exitosamente: "
                f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
                f"IP={client_ip}, names_requested={len(validated_names) if validated_names else 0}, "
                f"results={len(mapping)}, total={total_count}, page={page or 1}, "
                f"cache_hit={cache_hit}"
            )

            return PaginatedResponse[Dict[str, str]](items=mapping, pagination=pagination_metadata)
        else:
            # Respuesta tradicional (backward compatible)
            logger.info(
                f"[LABORATORY_MAPPING] Mapeo inverso generado exitosamente: "
                f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
                f"IP={client_ip}, names_requested={len(validated_names) if validated_names else 0}, "
                f"results={len(mapping)}, cache_hit={cache_hit}"
            )

            return mapping

    except HTTPException:
        # Re-raise HTTP exceptions (ya tienen formato correcto)
        raise
    except Exception as e:
        logger.error(
            f"[LABORATORY_MAPPING] Error inesperado en mapeo inverso: IP={client_ip}, " f"error={str(e)[:200]}"
        )

        return create_error_response(status_code=500, error_type="internal_error", message="Error interno del servidor")


@router.get("/generic-laboratories")
async def get_generic_laboratories(
    request: Request,
    current_user: User = Depends(get_current_user),
    page: int = Query(1, ge=1, le=10000, description="Número de página"),
    per_page: int = Query(50, ge=1, le=200, description="Elementos por página"),
    search: Optional[str] = Query(None, max_length=100, description="Término de búsqueda"),
    envelope: bool = Query(False, description="Devolver respuesta en envelope con metadata de paginación"),
    db: Session = Depends(get_db),
) -> Union[Dict[str, str], PaginatedResponse[Dict[str, str]]]:
    """
    Obtiene mapeo específico de laboratorios genéricos principales.

    **FASE 1.2 - VALIDADO Y SEGURO**
    - Paginación: 1-100 elementos por página
    - Búsqueda opcional con sanitización XSS
    - Rate limiting: 60 req/min, 500 req/hour
    - Protección SQL injection y XSS
    - Headers de seguridad obligatorios

    **Uso específico:**
    - Para componentes de selección de partners
    - Filtrado a laboratorios genéricos conocidos
    - Dashboard de análisis de genéricos

    **Query parameters:**
    - page: Número de página (1-1000)
    - per_page: Elementos por página (1-100)
    - search: Término de búsqueda opcional (máx 100 chars)

    **Response:**
    ```json
    {
        "111": "CINFA S.A.",
        "426": "NORMON S.A.",
        "863": "KERN PHARMA, S.L.",
        "644": "SANDOZ FARMACEUTICA, S.A",
        "1079": "TEVA PHARMA, S.L.U"
    }
    ```

    **Errores:**
    - 400: Parámetros de paginación inválidos
    - 413: Request demasiado grande
    - 422: Término de búsqueda contiene caracteres no permitidos
    - 429: Rate limit excedido
    - 500: Error interno del servidor
    """
    # Verificaciones de seguridad
    check_security_headers(request)
    validate_request_size(request)

    client_ip = get_client_ip(request)

    try:
        # FASE 2.2 - Validar parámetros de paginación con nuevos límites
        pagination_errors = validate_pagination_params(page, per_page)
        if pagination_errors:
            return create_error_response(
                status_code=422,
                error_type="pagination_error",
                message="Parámetros de paginación inválidos",
                details=pagination_errors,
            )

        # Validar término de búsqueda
        validated_search = None
        if search:
            try:
                validated_search = sanitize_laboratory_name(search)
            except ValueError as e:
                return create_error_response(
                    status_code=422,
                    error_type="format_error",
                    message="Término de búsqueda contiene caracteres no permitidos",
                    details={
                        "max_length": LaboratoryConfig.MAX_SEARCH_LENGTH,
                        "forbidden_chars": LaboratoryConfig.get_dangerous_chars(),
                    },
                    invalid_values=[search],
                )

        # FASE 2.2 - Usar cache service con soporte de conteo total
        if envelope:
            mapping, cache_hit, total_count = laboratory_cache_service.get_generic_laboratories_cached(
                db, page, per_page, validated_search, return_total=True
            )
        else:
            mapping, cache_hit = laboratory_cache_service.get_generic_laboratories_cached(
                db, page, per_page, validated_search, return_total=False
            )
            total_count = None

        # Aplicar sanitización adicional para seguridad (post-cache)
        sanitized_mapping = {}
        for code, name in mapping.items():
            if code and name and len(code) <= 4 and code.isdigit() and len(name) <= LaboratoryConfig.MAX_NAME_LENGTH:

                # Sanitizar nombre (protección XSS)
                try:
                    sanitized_name = sanitize_laboratory_name(name)
                    sanitized_mapping[code] = sanitized_name
                except ValueError as e:
                    logger.warning(f"[SECURITY] Laboratorio genérico rechazado post-cache: {name} - {e}")
                    continue

        mapping = sanitized_mapping

        # FASE 2.2 - Devolver respuesta con o sin envelope
        if envelope:
            # Respuesta con metadata de paginación
            pagination_metadata = calculate_pagination_metadata(total_count or 0, page, per_page)

            # Log exitoso con detalles de seguridad, paginación, cache y total
            logger.info(
                f"[LABORATORY_MAPPING] Laboratorios genéricos paginados obtenidos exitosamente: "
                f"IP={client_ip}, page={page}, per_page={per_page}, "
                f"search='{validated_search or 'none'}', results={len(mapping)}, "
                f"total={total_count or 0}, cache_hit={cache_hit}"
            )

            return PaginatedResponse[Dict[str, str]](items=mapping, pagination=pagination_metadata)
        else:
            # Respuesta tradicional (backward compatible)
            logger.info(
                f"[LABORATORY_MAPPING] Laboratorios genéricos obtenidos exitosamente: "
                f"IP={client_ip}, page={page}, per_page={per_page}, "
                f"search='{validated_search or 'none'}', results={len(mapping)}, cache_hit={cache_hit}"
            )

            return mapping

    except HTTPException:
        # Re-raise HTTP exceptions (ya tienen formato correcto)
        raise
    except Exception as e:
        logger.error(
            f"[LABORATORY_MAPPING] Error inesperado en laboratorios genéricos: IP={client_ip}, "
            f"page={page}, per_page={per_page}, error={str(e)[:200]}"
        )

        return create_error_response(status_code=500, error_type="internal_error", message="Error interno del servidor")


@router.get("/cache-stats")
async def get_laboratory_cache_statistics(
    request: Request,
    current_user: User = Depends(require_permission(Permission.VIEW_SYSTEM_STATS)),
) -> Dict[str, Any]:
    """
    Obtener estadísticas de performance del cache de laboratory mappings.

    **FASE 1.3 - Monitoreo de Cache Performance**
    - Hit/miss rates y performance metrics
    - Información de memoria y conexiones Redis
    - Estadísticas de warming y invalidación
    - Tiempo promedio de respuesta

    **Endpoint de monitoreo interno - NO requiere rate limiting**

    **Response:**
    ```json
    {
        "hits": 1250,
        "misses": 200,
        "total_operations": 1450,
        "hit_rate_percent": 86.21,
        "error_rate_percent": 0.0,
        "avg_response_time_ms": 12.5,
        "cache_size_estimate_mb": 2.3,
        "redis_connected_clients": 5,
        "redis_used_memory_human": "15.2M",
        "redis_hit_rate_percent": 89.4,
        "timestamp": "2025-01-21T10:30:00Z"
    }
    ```

    **Errores:**
    - 500: Error interno obteniendo estadísticas
    """
    client_ip = get_client_ip(request)

    try:
        # Obtener estadísticas del cache service
        stats = laboratory_cache_service.get_cache_statistics()

        logger.info(
            f"[LABORATORY_MAPPING] Estadísticas de cache solicitadas: "
            f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
            f"IP={client_ip}, hit_rate={stats.get('hit_rate_percent', 0):.2f}%"
        )

        return stats

    except Exception as e:
        logger.error(
            f"[LABORATORY_MAPPING] Error obteniendo estadísticas de cache: IP={client_ip}, " f"error={str(e)[:200]}"
        )

        return create_error_response(
            status_code=500, error_type="internal_error", message="Error obteniendo estadísticas de cache"
        )


@router.post("/invalidate-cache")
async def invalidate_laboratory_cache(
    request: Request,
    current_user: User = Depends(require_permission(Permission.MANAGE_DATABASE)),
) -> Dict[str, Any]:
    """
    Invalidar manualmente el cache de laboratory mappings.

    **FASE 1.3 - Cache Management**
    - Invalidación completa del cache de laboratory mappings
    - Útil para forzar refresh después de updates de catálogo
    - Endpoint interno de administración

    **Endpoint administrativo - Requiere permisos especiales en producción**

    **Response:**
    ```json
    {
        "invalidated_keys": 45,
        "message": "Cache invalidado exitosamente",
        "timestamp": "2025-01-21T10:30:00Z"
    }
    ```

    **Errores:**
    - 500: Error invalidando cache
    """
    client_ip = get_client_ip(request)

    try:
        # Invalidar cache de laboratory mappings
        invalidated_count = laboratory_cache_service.invalidate_all_laboratory_cache()

        result = {
            "invalidated_keys": invalidated_count,
            "message": "Cache de laboratory mappings invalidado exitosamente",
            "timestamp": utc_now().isoformat(),
        }

        logger.info(
            f"[LABORATORY_MAPPING] Cache invalidado manualmente: "
            f"user_id={current_user.id}, pharmacy_id={current_user.pharmacy_id}, "
            f"IP={client_ip}, keys_invalidated={invalidated_count}"
        )

        return result

    except Exception as e:
        logger.error(f"[LABORATORY_MAPPING] Error invalidando cache: IP={client_ip}, " f"error={str(e)[:200]}")

        return create_error_response(status_code=500, error_type="internal_error", message="Error invalidando cache")
