﻿# backend/app/api/invitations.py
"""
API endpoints para el sistema de invitaciones
"""

import logging
from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query, status
from sqlalchemy.orm import Session

from app.core.security import get_current_user
from app.database import get_db
from app.models.user import User
from app.schemas.invitation import (
    InvitationAccept,
    InvitationAcceptResponse,
    InvitationCreate,
    InvitationListResponse,
    InvitationResponse,
    InvitationValidate,
    InvitationValidationResponse,
)
from app.services.invitation_service import InvitationService
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

router = APIRouter()


@router.post("/create", response_model=InvitationResponse)
async def create_invitation(
    pharmacy_id: UUID,
    invitation_data: InvitationCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Crea una nueva invitación para un usuario

    Permisos requeridos:
    - Ser admin del sistema
    - Tener permiso de manage_users
    """
    try:
        # Crear invitación
        invitation = InvitationService.create_invitation(
            db=db,
            pharmacy_id=pharmacy_id,
            invited_by=current_user,
            email=invitation_data.email,
            role=invitation_data.role,
            full_name=invitation_data.full_name,
            dni_nie=invitation_data.dni_nie,
            message=invitation_data.message,
            expires_in_days=invitation_data.expires_in_days,
        )

        return InvitationResponse(**invitation)

    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    except Exception as e:
        logger.error(f"Error creando invitación: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error creando invitación",
        )


@router.get("/list/{pharmacy_id}", response_model=InvitationListResponse)
async def list_invitations(
    pharmacy_id: UUID,
    status: Optional[str] = Query(None, pattern="^(pending|accepted|expired|cancelled)$"),
    limit: int = Query(100, ge=1, le=500),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista las invitaciones de una farmacia

    Permisos requeridos:
    - Ser admin del sistema
    - Pertenecer a la farmacia
    """
    try:
        # Verificar permisos
        if not current_user.is_superuser:
            if str(current_user.pharmacy_id) != str(pharmacy_id):
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN,
                    detail="No tienes acceso a las invitaciones de esta farmacia",
                )

        # Obtener invitaciones
        invitations = InvitationService.list_invitations(
            db=db,
            pharmacy_id=pharmacy_id,
            status=status,
            limit=limit,
        )

        # Contar por estado
        pending_count = sum(1 for inv in invitations if inv.get("status") == "pending")
        accepted_count = sum(1 for inv in invitations if inv.get("status") == "accepted")

        return InvitationListResponse(
            invitations=[InvitationResponse(**inv) for inv in invitations],
            total=len(invitations),
            pending_count=pending_count,
            accepted_count=accepted_count,
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error listando invitaciones: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error obteniendo invitaciones",
        )


@router.post("/validate", response_model=InvitationValidationResponse)
async def validate_invitation(
    data: InvitationValidate,
    db: Session = Depends(get_db),
):
    """
    Valida un token de invitación

    Endpoint público (no requiere autenticación)
    """
    try:
        result = InvitationService.validate_token(db=db, token=data.token)
        return InvitationValidationResponse(**result)

    except Exception as e:
        logger.error(f"Error validando invitación: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error validando invitación",
        )


@router.post("/accept", response_model=InvitationAcceptResponse)
async def accept_invitation(
    data: InvitationAccept,
    db: Session = Depends(get_db),
):
    """
    Acepta una invitación y crea un nuevo usuario

    Endpoint público (no requiere autenticación)
    """
    try:
        result = InvitationService.accept_invitation(
            db=db,
            token=data.token,
            password=data.password,
            username=data.username,
        )

        return InvitationAcceptResponse(**result)

    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    except Exception as e:
        logger.error(f"Error aceptando invitación: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error procesando invitación",
        )


@router.delete("/cancel/{invitation_id}")
async def cancel_invitation(
    invitation_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Cancela una invitación pendiente

    Permisos requeridos:
    - Ser admin del sistema
    - Tener permiso de manage_users de la invitación
    """
    try:
        result = InvitationService.cancel_invitation(
            db=db,
            invitation_id=invitation_id,
            user=current_user,
        )

        return result

    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    except Exception as e:
        logger.error(f"Error cancelando invitación: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error cancelando invitación",
        )


@router.post("/resend/{invitation_id}")
async def resend_invitation(
    invitation_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Reenvía una invitación (cuando el servicio de email esté activo)

    Por ahora solo actualiza la fecha de expiración
    """
    try:
        from datetime import timedelta

        from app.models.invitation import Invitation

        # Buscar invitación
        invitation = db.query(Invitation).filter(Invitation.id == invitation_id).first()

        if not invitation:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Invitación no encontrada",
            )

        # Verificar permisos
        if not current_user.is_superuser:
            if not current_user.has_permission("manage_users"):
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN,
                    detail="No tienes permisos para reenviar invitaciones",
                )

            if str(current_user.pharmacy_id) != str(invitation.pharmacy_id):
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN,
                    detail="No puedes reenviar invitaciones de otras farmacias",
                )

        # Actualizar expiración
        invitation.expires_at = utc_now() + timedelta(days=7)
        invitation.updated_at = utc_now()
        db.commit()

        # TODO: Enviar email cuando el servicio esté activo

        return {
            "message": "Invitación actualizada. El email se enviará cuando el servicio esté activo",
            "expires_at": invitation.expires_at.isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error reenviando invitación: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error reenviando invitación",
        )
