# backend/app/api/intercambiable_groups.py
"""
API endpoints para grupos intercambiables de venta libre.

Issue #446: Taxonomia jerarquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Endpoints:
- GET  /intercambiable-groups/              - Lista todos los grupos
- GET  /intercambiable-groups/summary       - Resumen general
- GET  /intercambiable-groups/slug/{slug}   - Obtener grupo por slug
- GET  /intercambiable-groups/{group_id}    - Obtener grupo por ID
- GET  /intercambiable-groups/{group_id}/products - Productos del grupo
- POST /intercambiable-groups/{group_id}/assign   - Asignar productos (admin)
- POST /intercambiable-groups/assign-all          - Asignar todos (admin)
"""

import logging
from typing import Any, Dict, List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from pydantic import BaseModel, Field
from sqlalchemy.orm import Session

from ..api.deps import get_current_admin_user, get_current_user
from ..database import get_db
from ..models.user import User
from ..services.intercambiable_group_service import IntercambiableGroupService

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/intercambiable-groups", tags=["intercambiable-groups"])


# =============================================================================
# PYDANTIC SCHEMAS
# =============================================================================


class GroupBasicResponse(BaseModel):
    """Schema basico de grupo para listados"""

    id: str
    name: str
    slug: str
    necesidad: str
    subcategory: Optional[str] = None
    product_count: int = 0
    brand_count: int = 0
    total_sales_amount: float = 0.0
    validated: bool = False

    class Config:
        from_attributes = True


class GroupDetailResponse(GroupBasicResponse):
    """Schema detallado de grupo con productos"""

    description: Optional[str] = None
    total_sales_count: Optional[int] = 0
    validated_by: Optional[str] = None
    products: List[Dict[str, Any]] = []


class GroupListResponse(BaseModel):
    """Schema para lista de grupos"""

    groups: List[GroupBasicResponse]
    total: int


class GroupProductResponse(BaseModel):
    """Schema para producto de un grupo"""

    product_name: str
    brand: Optional[str] = None
    brand_line: Optional[str] = None
    total_ventas: float = 0.0
    num_ventas: int = 0


class GroupProductsListResponse(BaseModel):
    """Schema para lista de productos de un grupo"""

    group_id: str
    group_name: str
    products: List[GroupProductResponse]
    total: int


class GroupSummaryResponse(BaseModel):
    """Schema para resumen de grupos"""

    total_groups: int
    validated_groups: int
    total_products_in_groups: int
    total_sales_in_groups: float
    by_necesidad: Dict[str, Dict[str, int]]


class AssignProductsRequest(BaseModel):
    """Schema para asignacion de productos a un grupo"""

    pharmacy_id: Optional[UUID] = Field(
        None, description="Filtrar por farmacia especifica"
    )
    limit: int = Field(1000, ge=1, le=10000, description="Maximo de productos a asignar")


class AssignProductsResponse(BaseModel):
    """Schema para respuesta de asignacion"""

    group_id: str
    group_name: str
    necesidad: str
    assigned_count: int


class AssignAllGroupsRequest(BaseModel):
    """Schema para asignar todos los grupos"""

    pharmacy_id: Optional[UUID] = Field(
        None, description="Filtrar por farmacia especifica"
    )
    validated_only: bool = Field(True, description="Solo usar grupos validados")


class AssignAllGroupsResponse(BaseModel):
    """Schema para respuesta de asignacion masiva"""

    groups_processed: int
    total_assigned: int
    details: List[AssignProductsResponse]


# =============================================================================
# ENDPOINTS
# =============================================================================


@router.get("", response_model=GroupListResponse)
async def list_groups(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    validated_only: bool = Query(False, description="Solo grupos validados"),
    necesidad: Optional[str] = Query(None, description="Filtrar por necesidad"),
) -> Dict[str, Any]:
    """
    Lista todos los grupos intercambiables.

    Args:
        validated_only: Solo retornar grupos validados por farmaceutico
        necesidad: Filtrar por necesidad especifica (ej: 'proteccion_solar')

    Returns:
        Lista de grupos con estadisticas basicas
    """
    try:
        service = IntercambiableGroupService(db)
        groups = service.get_all(validated_only=validated_only, necesidad=necesidad)

        return {
            "groups": [
                {
                    "id": str(g.id),
                    "name": g.name,
                    "slug": g.slug,
                    "necesidad": g.necesidad,
                    "subcategory": g.subcategory,
                    "product_count": g.product_count,
                    "brand_count": g.brand_count,
                    "total_sales_amount": float(g.total_sales_amount or 0),
                    "validated": g.validated,
                }
                for g in groups
            ],
            "total": len(groups),
        }

    except Exception as e:
        logger.error(f"Error listing groups: {e}")
        raise HTTPException(status_code=500, detail="Error al listar grupos")


@router.get("/summary", response_model=GroupSummaryResponse)
async def get_summary(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Obtiene resumen de grupos intercambiables.

    Returns:
        Estadisticas agregadas de todos los grupos
    """
    try:
        service = IntercambiableGroupService(db)
        return service.get_summary()

    except Exception as e:
        logger.error(f"Error getting summary: {e}")
        raise HTTPException(status_code=500, detail="Error al obtener resumen")


@router.get("/slug/{slug}", response_model=GroupDetailResponse)
async def get_group_by_slug(
    slug: str,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Obtiene un grupo por su slug.

    Args:
        slug: Identificador URL-friendly del grupo

    Returns:
        Detalles del grupo con lista de productos
    """
    try:
        service = IntercambiableGroupService(db)
        group = service.get_by_slug(slug)

        if not group:
            raise HTTPException(status_code=404, detail="Grupo no encontrado")

        products = service.get_group_products(group.id, limit=50)

        return {
            "id": str(group.id),
            "name": group.name,
            "slug": group.slug,
            "description": group.description,
            "necesidad": group.necesidad,
            "subcategory": group.subcategory,
            "product_count": group.product_count,
            "brand_count": group.brand_count,
            "total_sales_amount": float(group.total_sales_amount or 0),
            "total_sales_count": group.total_sales_count,
            "validated": group.validated,
            "validated_by": group.validated_by,
            "products": products,
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting group by slug {slug}: {e}")
        raise HTTPException(status_code=500, detail="Error al obtener grupo")


@router.get("/{group_id}", response_model=GroupDetailResponse)
async def get_group(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Obtiene un grupo por ID.

    Args:
        group_id: UUID del grupo

    Returns:
        Detalles del grupo con lista de productos
    """
    try:
        service = IntercambiableGroupService(db)
        group = service.get_by_id(group_id)

        if not group:
            raise HTTPException(status_code=404, detail="Grupo no encontrado")

        products = service.get_group_products(group_id, limit=50)

        return {
            "id": str(group.id),
            "name": group.name,
            "slug": group.slug,
            "description": group.description,
            "necesidad": group.necesidad,
            "subcategory": group.subcategory,
            "product_count": group.product_count,
            "brand_count": group.brand_count,
            "total_sales_amount": float(group.total_sales_amount or 0),
            "total_sales_count": group.total_sales_count,
            "validated": group.validated,
            "validated_by": group.validated_by,
            "products": products,
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting group {group_id}: {e}")
        raise HTTPException(status_code=500, detail="Error al obtener grupo")


@router.get("/{group_id}/products", response_model=GroupProductsListResponse)
async def get_group_products(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    limit: int = Query(100, ge=1, le=1000, description="Maximo de productos"),
) -> Dict[str, Any]:
    """
    Obtiene los productos de un grupo intercambiable.

    Args:
        group_id: UUID del grupo
        limit: Numero maximo de productos a retornar

    Returns:
        Lista de productos con nombre, marca, linea y ventas
    """
    try:
        service = IntercambiableGroupService(db)
        group = service.get_by_id(group_id)

        if not group:
            raise HTTPException(status_code=404, detail="Grupo no encontrado")

        products = service.get_group_products(group_id, limit=limit)

        return {
            "group_id": str(group_id),
            "group_name": group.name,
            "products": products,
            "total": len(products),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting products for group {group_id}: {e}")
        raise HTTPException(status_code=500, detail="Error al obtener productos del grupo")


@router.post("/{group_id}/assign", response_model=AssignProductsResponse)
async def assign_products_to_group(
    group_id: UUID,
    request: AssignProductsRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),
) -> Dict[str, Any]:
    """
    Asigna productos a un grupo basandose en la necesidad.

    **Requiere permisos de administrador.**

    Args:
        group_id: UUID del grupo objetivo
        request: Parametros de asignacion

    Returns:
        Estadisticas de asignacion
    """
    try:
        service = IntercambiableGroupService(db)
        result = service.assign_products_by_necesidad(
            group_id=group_id,
            pharmacy_id=request.pharmacy_id,
            limit=request.limit,
        )

        logger.info(
            f"Admin {current_user.email} assigned {result['assigned_count']} products "
            f"to group {result['group_name']}"
        )

        return result

    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))
    except Exception as e:
        logger.error(f"Error assigning products to group {group_id}: {e}")
        raise HTTPException(status_code=500, detail="Error al asignar productos al grupo")


@router.post("/assign-all", response_model=AssignAllGroupsResponse)
async def assign_all_groups(
    request: AssignAllGroupsRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),
) -> Dict[str, Any]:
    """
    Asigna productos a TODOS los grupos intercambiables.

    **Requiere permisos de administrador.**

    Args:
        request: Parametros de asignacion

    Returns:
        Estadisticas de asignacion por grupo
    """
    try:
        service = IntercambiableGroupService(db)
        result = service.assign_all_groups(
            pharmacy_id=request.pharmacy_id,
            validated_only=request.validated_only,
        )

        logger.info(
            f"Admin {current_user.email} triggered assign_all_groups: "
            f"{result['groups_processed']} groups, {result['total_assigned']} products"
        )

        return result

    except Exception as e:
        logger.error(f"Error in assign_all_groups: {e}")
        raise HTTPException(status_code=500, detail="Error al asignar productos a todos los grupos")
