# backend/app/api/insights.py
"""
API Router para Insight Engine v2.0 (Issue #506)

Endpoints para gestión de insights automáticos:
- GET /insights/{pharmacy_id}: Obtener insights activos
- POST /insights/{pharmacy_id}/audit: Forzar re-ejecución del audit
- POST /insights/feedback: Registrar feedback (snooze/dismiss/resolve)
- GET /insights/feedback/{pharmacy_id}/history: Historial de feedback
- GET /insights/config: Configuración de reglas disponibles
"""

from typing import Optional
from uuid import UUID

import structlog
from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.schemas.insights import (
    FeedbackHistoryItem,
    FeedbackHistoryResponse,
    InsightFeedbackRequest,
    InsightFeedbackResponse,
    InsightItem,
    InsightsResponse,
    RuleInfo,
    RulesConfigResponse,
    TriggerAuditRequest,
    TriggerAuditResponse,
)
from app.services.insight_engine import insight_engine_service

logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/insights", tags=["insights"])


@router.get("/{pharmacy_id}", response_model=InsightsResponse)
async def get_insights(
    pharmacy_id: UUID,
    category: Optional[str] = Query(
        None,
        description="Filtrar por categoría: stock, margin, hhi, trend, surtido"
    ),
    severity: Optional[str] = Query(
        None,
        description="Filtrar por severidad: high, medium, low"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener insights activos para una farmacia.

    Ejecuta el audit si han pasado más de 5 minutos desde el último,
    o usa el cache si está disponible.

    Los insights se ordenan por economic_value DESC (mayor oportunidad primero).

    Args:
        pharmacy_id: UUID de la farmacia
        category: Filtrar por categoría (opcional)
        severity: Filtrar por severidad (opcional)

    Returns:
        InsightsResponse con lista de insights y metadata

    Raises:
        403: Si el usuario no tiene acceso a la farmacia
    """
    # Verify user has access to pharmacy
    if current_user.pharmacy_id != pharmacy_id and current_user.role.name != "admin":
        raise HTTPException(
            status_code=403,
            detail="No tienes acceso a esta farmacia"
        )

    logger.info(
        "insights.get_request",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
        category=category,
        severity=severity,
    )

    try:
        result = await insight_engine_service.get_active_insights(
            db=db,
            pharmacy_id=pharmacy_id,
            category=category,
            severity=severity,
        )

        # Convert to response model
        insights_items = [
            InsightItem(
                rule_code=i.rule_code,
                category=i.category,
                severity=i.severity,
                title=i.title,
                description=i.description,
                impact_score=i.impact_score,
                economic_impact=i.economic_impact,
                economic_value=i.economic_value,
                action_label=i.action_label,
                deeplink=i.deeplink,
                affected_items=i.affected_items,
                insight_hash=i.insight_hash,
            )
            for i in result.insights
        ]

        return InsightsResponse(
            insights=insights_items,
            total_opportunity=result.total_opportunity,
            by_category=result.by_category,
            by_severity=result.by_severity,
            suppressed_count=result.suppressed_count,
            audit_duration_ms=result.audit_duration_ms,
        )

    except Exception as e:
        logger.error(
            "insights.get_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500,
            detail=f"Error al obtener insights: {str(e)}"
        )


@router.post("/{pharmacy_id}/audit", response_model=TriggerAuditResponse)
async def trigger_audit(
    pharmacy_id: UUID,
    request: TriggerAuditRequest = TriggerAuditRequest(),
    background_tasks: BackgroundTasks = None,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Forzar re-ejecución del audit de insights.

    Útil para actualizar insights después de cambios significativos
    (nueva carga de datos, cambios de inventario, etc).

    Args:
        pharmacy_id: UUID de la farmacia
        request: Opciones de ejecución

    Returns:
        TriggerAuditResponse con estado y resultados
    """
    # Verify user has access to pharmacy
    if current_user.pharmacy_id != pharmacy_id and current_user.role.name != "admin":
        raise HTTPException(
            status_code=403,
            detail="No tienes acceso a esta farmacia"
        )

    logger.info(
        "insights.trigger_audit",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
        force=request.force,
    )

    try:
        result = await insight_engine_service.run_audit(
            db=db,
            pharmacy_id=pharmacy_id,
            custom_config=request.custom_config,
            force=request.force,
        )

        return TriggerAuditResponse(
            status="completed",
            message=f"Audit completado: {len(result.insights)} insights generados",
            insights_count=len(result.insights),
            total_opportunity=result.total_opportunity,
            duration_ms=result.audit_duration_ms,
        )

    except Exception as e:
        logger.error(
            "insights.trigger_audit_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        return TriggerAuditResponse(
            status="error",
            message=f"Error durante el audit: {str(e)}",
        )


@router.post("/feedback", response_model=InsightFeedbackResponse)
async def submit_feedback(
    request: InsightFeedbackRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Registrar feedback sobre un insight.

    Acciones disponibles:
    - snooze: Posponer el insight por N días (default 7)
    - dismiss: Descartar permanentemente (reconocido pero no accionado)
    - resolve: Marcar como resuelto (acción tomada)

    Args:
        request: Datos del feedback

    Returns:
        InsightFeedbackResponse con resultado
    """
    pharmacy_id = current_user.pharmacy_id

    if not pharmacy_id:
        raise HTTPException(
            status_code=400,
            detail="Usuario sin farmacia asignada"
        )

    logger.info(
        "insights.submit_feedback",
        pharmacy_id=str(pharmacy_id),
        user_id=str(current_user.id),
        insight_hash=request.insight_hash,
        action=request.action,
    )

    try:
        success = await insight_engine_service.submit_feedback(
            db=db,
            pharmacy_id=pharmacy_id,
            insight_hash=request.insight_hash,
            action=request.action,
            rule_code=request.rule_code,
            notes=request.notes,
            snooze_days=request.snooze_days,
        )

        if success:
            return InsightFeedbackResponse(
                success=True,
                message=f"Feedback '{request.action}' registrado correctamente",
            )
        else:
            return InsightFeedbackResponse(
                success=False,
                message="No se pudo registrar el feedback",
            )

    except Exception as e:
        logger.error(
            "insights.feedback_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500,
            detail=f"Error al registrar feedback: {str(e)}"
        )


@router.get("/feedback/{pharmacy_id}/history", response_model=FeedbackHistoryResponse)
async def get_feedback_history(
    pharmacy_id: UUID,
    limit: int = Query(50, ge=1, le=200, description="Máximo de registros"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener historial de feedback para una farmacia.

    Args:
        pharmacy_id: UUID de la farmacia
        limit: Máximo de registros a retornar

    Returns:
        FeedbackHistoryResponse con lista de feedbacks
    """
    # Verify user has access to pharmacy
    if current_user.pharmacy_id != pharmacy_id and current_user.role.name != "admin":
        raise HTTPException(
            status_code=403,
            detail="No tienes acceso a esta farmacia"
        )

    try:
        feedbacks = await insight_engine_service.get_feedback_history(
            db=db,
            pharmacy_id=pharmacy_id,
            limit=limit,
        )

        items = [
            FeedbackHistoryItem(
                id=f["id"],
                rule_code=f["rule_code"],
                insight_hash=f["insight_hash"],
                action=f["action"],
                snoozed_until=f["snoozed_until"],
                notes=f["notes"],
                created_at=f["created_at"],
                is_active=f["is_active"],
            )
            for f in feedbacks
        ]

        return FeedbackHistoryResponse(
            feedbacks=items,
            total_count=len(items),
        )

    except Exception as e:
        logger.error(
            "insights.history_error",
            pharmacy_id=str(pharmacy_id),
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500,
            detail=f"Error al obtener historial: {str(e)}"
        )


@router.get("/config", response_model=RulesConfigResponse)
async def get_rules_config(
    current_user: User = Depends(get_current_user),
):
    """
    Obtener configuración de reglas disponibles.

    Retorna información sobre todas las reglas de insights
    incluyendo su configuración por defecto.

    Returns:
        RulesConfigResponse con lista de reglas
    """
    try:
        rules = insight_engine_service.get_available_rules()

        items = [
            RuleInfo(
                rule_code=r["rule_code"],
                category=r["category"],
                severity=r["severity"],
                default_config=r["default_config"],
            )
            for r in rules
        ]

        return RulesConfigResponse(
            rules=items,
            total_rules=len(items),
        )

    except Exception as e:
        logger.error(
            "insights.config_error",
            error=str(e),
            exc_info=True,
        )
        raise HTTPException(
            status_code=500,
            detail=f"Error al obtener configuración: {str(e)}"
        )
